package org.bidib.wizard.mvc.stepcontrol.model;

import java.util.function.LongSupplier;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.model.stepcontrol.TurnTableType;
import org.bidib.wizard.mvc.stepcontrol.view.wizard.SummaryPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

public class ConfigurationWizardModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigurationWizardModel.class);

    public enum WizardStatus {
        started, finished, aborted;
    }

    public static final String PROPERTYNAME_TURNTABLE_TYPE = "turnTableType";

    public static final String PROPERTYNAME_MOTORSIZE_TYPE = "motorSizeType";

    public static final String PROPERTYNAME_STEPCOUNT = "stepCount";

    public static final String PROPERTYNAME_GEARING = "gearing";

    public static final String PROPERTYNAME_MICROSTEPPING = "microStepping";

    public static final String PROPERTYNAME_TOTALSTEPCOUNT = "totalStepCount";

    public static final String PROPERTYNAME_WIZARDSTATUS = "wizardStatus";

    public static final String PROPERTYNAME_HTMLCONTENT = "htmlContent";

    public static final String PROPERTYNAME_SPEED_SCALE = "speedScale";

    public static final String PROPERTYNAME_ACCELERATION_SCALE = "accelerationScale";

    public static final String PROPERTYNAME_SPEED = "speed";

    public static final String PROPERTYNAME_ACCEL = "accel";

    public static final String PROPERTYNAME_DECEL = "decel";

    private TurnTableType turnTableType;

    private MotorSizeType motorSizeType;

    private Integer stepCount;

    private Gearing gearing;

    private MicroStepsEnum microStepping = MicroStepsEnum.steps64;

    private MovementScaleEnum speedScale = MovementScaleEnum.scale1;

    private AccelarationScaleEnum accelerationScale = AccelarationScaleEnum.scale1;

    private WizardStatus wizardStatus = WizardStatus.started;

    private Integer speed;

    private Integer accel;

    private Integer decel;

    private final NodeInterface selectedNode;

    public ConfigurationWizardModel(final NodeInterface selectedNode) {
        this.selectedNode = selectedNode;
    }

    /**
     * @return the selected node
     */
    public NodeInterface getSelectedNode() {
        return selectedNode;
    }

    /**
     * @return the turnTableType
     */
    public TurnTableType getTurnTableType() {
        return turnTableType;
    }

    /**
     * @param turnTableType
     *            the turnTableType to set
     */
    public void setTurnTableType(TurnTableType turnTableType) {
        LOGGER.info("Set the turnTableType: {}", turnTableType);

        TurnTableType oldValue = this.turnTableType;
        this.turnTableType = turnTableType;

        firePropertyChange(PROPERTYNAME_TURNTABLE_TYPE, oldValue, turnTableType);
    }

    /**
     * @return the motorSizeType
     */
    public MotorSizeType getMotorSizeType() {
        return motorSizeType;
    }

    /**
     * @param motorSizeType
     *            the motorSizeType to set
     */
    public void setMotorSizeType(MotorSizeType motorSizeType) {
        LOGGER.info("Set the motorSizeType: {}", motorSizeType);
        MotorSizeType oldValue = this.motorSizeType;

        this.motorSizeType = motorSizeType;

        firePropertyChange(PROPERTYNAME_MOTORSIZE_TYPE, oldValue, motorSizeType);
    }

    /**
     * @return the stepCount
     */
    public Integer getStepCount() {
        return stepCount;
    }

    /**
     * @param stepCount
     *            the stepCount to set
     */
    public void setStepCount(Integer stepCount) {
        Integer oldValue = this.stepCount;
        Integer oldTotalStepCount = getTotalStepCount();

        this.stepCount = stepCount;

        firePropertyChange(PROPERTYNAME_STEPCOUNT, oldValue, stepCount);
        firePropertyChange(PROPERTYNAME_TOTALSTEPCOUNT, oldTotalStepCount, getTotalStepCount());
    }

    /**
     * @return the gearing
     */
    public Gearing getGearing() {
        if (gearing == null) {
            gearing = new Gearing("no");
        }
        return gearing;
    }

    /**
     * @param gearing
     *            the gearing to set
     */
    public void setGearing(Gearing gearing) {
        Gearing oldValue = this.gearing;
        Integer oldTotalStepCount = getTotalStepCount();

        this.gearing = gearing;

        firePropertyChange(PROPERTYNAME_GEARING, oldValue, gearing);
        firePropertyChange(PROPERTYNAME_TOTALSTEPCOUNT, oldTotalStepCount, getTotalStepCount());
    }

    /**
     * @return the microStepping
     */
    public MicroStepsEnum getMicroStepping() {
        return microStepping;
    }

    /**
     * @param microStepping
     *            the microStepping to set
     */
    public void setMicroStepping(MicroStepsEnum microStepping) {
        LOGGER.info("Set the new microstepping value: {}", microStepping);
        MicroStepsEnum oldValue = this.microStepping;
        Integer oldTotalStepCount = getTotalStepCount();

        this.microStepping = microStepping;

        firePropertyChange(PROPERTYNAME_MICROSTEPPING, oldValue, microStepping);
        firePropertyChange(PROPERTYNAME_TOTALSTEPCOUNT, oldTotalStepCount, getTotalStepCount());
    }

    /**
     * @return the total step count
     */
    public Integer getTotalStepCount() {
        try {
            if (gearing != null && Gearing.YES.equals(gearing.getKey())) {
                // the calculation of total steps with gearRatio primary and secondary
                int totalStepCount =
                    (stepCount * microStepping.getSteps() * gearing.getGearRatioPrimary())
                        / gearing.getGearRatioSecondary();

                LOGGER.info("Calculated total step count: {}", totalStepCount);

                return totalStepCount;
            }
            return stepCount * microStepping.getSteps();
        }
        catch (Exception ex) {
            LOGGER.warn("Calculate total step count failed.");
        }
        return null;
    }

    public void triggerUpdateTotalSteps() {
        firePropertyChange(PROPERTYNAME_TOTALSTEPCOUNT, null, getTotalStepCount());
    }

    /**
     * @return the speedScale
     */
    public MovementScaleEnum getSpeedScale() {
        return speedScale;
    }

    /**
     * @param speedScale
     *            the speedScale to set
     */
    public void setSpeedScale(MovementScaleEnum speedScale) {
        MovementScaleEnum oldValue = this.speedScale;
        this.speedScale = speedScale;

        firePropertyChange(PROPERTYNAME_SPEED_SCALE, oldValue, speedScale);
    }

    /**
     * @return the accelerationScale
     */
    public AccelarationScaleEnum getAccelerationScale() {
        return accelerationScale;
    }

    /**
     * @param accelerationScale
     *            the accelerationScale to set
     */
    public void setAccelerationScale(AccelarationScaleEnum accelerationScale) {
        AccelarationScaleEnum oldValue = this.accelerationScale;
        this.accelerationScale = accelerationScale;

        firePropertyChange(PROPERTYNAME_ACCELERATION_SCALE, oldValue, accelerationScale);
    }

    /**
     * @return the wizardStatus
     */
    public WizardStatus getWizardStatus() {
        return wizardStatus;
    }

    /**
     * @param wizardStatus
     *            the wizardStatus to set
     */
    public void setWizardStatus(WizardStatus wizardStatus) {
        WizardStatus oldValue = this.wizardStatus;
        this.wizardStatus = wizardStatus;

        firePropertyChange(PROPERTYNAME_WIZARDSTATUS, oldValue, wizardStatus);
    }

    /**
     * @return the speed
     */
    public Integer getSpeed() {
        return speed;
    }

    /**
     * @param speed
     *            the speed to set
     */
    public void setSpeed(Integer speed) {
        LOGGER.info("Set speed: {}", speed);

        Integer oldValue = this.speed;
        this.speed = speed;
        firePropertyChange(PROPERTYNAME_SPEED, oldValue, speed);
    }

    /**
     * @return the accel
     */
    public Integer getAccel() {
        return accel;
    }

    /**
     * @param accel
     *            the accel to set
     */
    public void setAccel(Integer accel) {
        Integer oldValue = this.accel;
        this.accel = accel;
        firePropertyChange(PROPERTYNAME_ACCEL, oldValue, accel);
    }

    /**
     * @return the decel
     */
    public Integer getDecel() {
        return decel;
    }

    /**
     * @param decel
     *            the decel to set
     */
    public void setDecel(Integer decel) {
        Integer oldValue = this.decel;
        this.decel = decel;
        firePropertyChange(PROPERTYNAME_DECEL, oldValue, decel);
    }

    /**
     * Check if the current configuration values for accel and decel are valid
     * 
     * @throws IllegalArgumentException
     *             thrown if invalid values detected
     */
    public void checkValidConfigurationValues(String valueIdentifier, final LongSupplier newValueSupplier) {

        validateConfigurationValue(valueIdentifier, newValueSupplier);
    }

    private void validateConfigurationValue(String valueIdentifier, final LongSupplier newValueSupplier) {

        if (getTotalStepCount() == null) {
            LOGGER.warn("The totalSteps are not available.");
            throw new IllegalArgumentException("The totalSteps are not available.");
        }

        long totalSteps = getTotalStepCount().longValue();

        long minDecel = totalSteps;
        minDecel = minDecel * newValueSupplier.getAsLong();
        if (minDecel < F_STEP_ACCEL_LIMIT) {
            LOGGER.warn("The configured '{}' value is too small: {}", valueIdentifier, newValueSupplier.getAsLong());
            throw new IllegalArgumentException(
                "The configured '" + valueIdentifier + "' value is too small: " + newValueSupplier.getAsLong());
        }
    }

    // = 1/((65536 / F_STEP_TIMER_676) ^2 / 720)
    private static final long F_STEP_ACCEL_LIMIT = 19150L;

    /**
     * Prepare the content of the SummaryPanel.
     */
    public String getHtmlContent() {
        StringBuilder sb = new StringBuilder("<html>");
        sb
            .append(Resources
                .getString(SummaryPanel.class, "turntableType",
                    Resources.getString(TurnTableType.class, turnTableType.name())))
            .append("<br/>");
        sb
            .append(Resources
                .getString(SummaryPanel.class, "motorSize",
                    Resources.getString(MotorSizeType.class, motorSizeType.name())))
            .append("<br/>");
        // sb
        // .append("&nbsp;&nbsp;")
        // .append(Resources
        // .getString(SummaryPanel.class, "current", motorSizeType.getCurrentMoving(),
        // motorSizeType.getCurrentStopped()))
        // .append("<br/>");
        // gearing
        if (gearing != null && !Gearing.NO.equals(gearing.getKey())) {
            sb
                .append(Resources
                    .getString(SummaryPanel.class, "gearing", gearing.getGearRatioPrimary(),
                        gearing.getGearRatioSecondary(), gearing.getBackLash()))
                .append("<br/>");
        }
        else {
            sb.append(Resources.getString(SummaryPanel.class, "gearingNone")).append("<br/>");
        }
        sb.append(Resources.getString(SummaryPanel.class, "stepCount", stepCount)).append("<br/>");
        sb
            .append(Resources
                .getString(SummaryPanel.class, "microStepping",
                    Resources.getString(MicroStepsEnum.class, microStepping.getKey())))
            .append("<br/>");
        sb.append(Resources.getString(SummaryPanel.class, "totalStepCount", getTotalStepCount())).append("<br/>");
        sb.append("</html>");
        return sb.toString();
    }

    public void triggerUpdateHtmlContent() {
        firePropertyChange(PROPERTYNAME_HTMLCONTENT, null, getHtmlContent());
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.SHORT_PREFIX_STYLE);
    }
}
