package org.bidib.wizard.mvc.loco.view;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.function.Consumer;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.WindowConstants;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.model.loco.LocoModel;
import org.bidib.wizard.mvc.common.view.RegisteredDialog;
import org.bidib.wizard.mvc.common.view.ViewCloseListener;
import org.bidib.wizard.mvc.loco.controller.LocoControlListener;
import org.bidib.wizard.mvc.loco.model.LocoConfigModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocoDialog extends JDialog implements RegisteredDialog {

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoDialog.class);

    private static final long serialVersionUID = 1L;

    private LocoView locoView;

    private final LocoModel locoModel;

    private final Consumer<LocoModel> locoModelConsumer;

    private final Consumer<Integer> locoAddressChangeListener;

    private final LocoConfigModel locoConfigModel;

    public LocoDialog(JFrame parent, final LocoModel locoModel, final LocoConfigModel locoConfigModel,
        final SettingsService settingsService, final LocoControlListener locoControlListener,
        final Consumer<LocoModel> locoModelConsumer, final Consumer<Integer> locoAddressChangeListener) {
        super(parent);
        this.locoModel = locoModel;
        this.locoConfigModel = locoConfigModel;
        this.locoModelConsumer = locoModelConsumer;
        this.locoAddressChangeListener = locoAddressChangeListener;

        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        LOGGER.info("Create the locoDialog for locoModel: {}", locoModel);

        if (this.locoModel != null) {
            this.locoModelConsumer.accept(this.locoModel);
        }

        updateDialogTitle(locoModel.getAddress());

        addContentPanel(locoModel, locoConfigModel, settingsService, locoControlListener);

        setLocationRelativeTo(parent);
    }

    public void updateDialogTitle(final Integer locoAddress) {

        if (this.locoConfigModel.isCarControlEnabled()) {
            if (locoAddress != null) {
                setTitle(Resources.getString(LocoView.class, "title.car") + " - " + locoAddress);
            }
            else {
                setTitle(Resources.getString(LocoView.class, "title.car"));
            }
        }
        else {
            if (locoAddress != null) {
                setTitle(Resources.getString(LocoView.class, "title.loco") + " - " + locoAddress);
            }
            else {
                setTitle(Resources.getString(LocoView.class, "title.loco"));
            }
        }
    }

    public static String prepareKey(Integer address) {
        if (address != null) {
            return LocoDialog.class.getSimpleName() + "-" + address;
        }
        return LocoDialog.class.getSimpleName() + "-";
    }

    @Override
    public String getKey() {
        LOGGER.info("Prepare the key: {}", locoModel);
        Integer address = locoModel != null ? locoModel.getAddress() : null;
        return prepareKey(address);
    }

    @Override
    public Window getWindow() {
        return this;
    }

    private void addContentPanel(
        final LocoModel locoModel, final LocoConfigModel locoConfigModel, final SettingsService settingsService,
        final LocoControlListener locoControlListener) {

        locoView =
            new LocoView(locoModel, locoConfigModel, settingsService, locoControlListener,
                this.locoAddressChangeListener);

        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                LOGGER.info("The window is closing: {}", e.getWindow());

                cleanup(settingsService);

                super.windowClosing(e);
            }

            @Override
            public void windowClosed(WindowEvent e) {
                LOGGER.info("The window is closed: {}", e.getWindow());

                super.windowClosed(e);
            }
        });

        getContentPane().setLayout(new BorderLayout());
        getContentPane().add(locoView.getComponent(), BorderLayout.CENTER);

        pack();

        setMinimumSize(new Dimension(locoView.getComponent().getPreferredSize().width + 20,
            locoView.getComponent().getPreferredSize().height + 20));

        setVisible(true);
    }

    public void addViewCloseListener(final ViewCloseListener listener) {
        locoView.addViewCloseListener(listener);
    }

    public void close(final SettingsService settingsService) {
        LOGGER.info("The LocoView should be closed.");

        cleanup(settingsService);

        LOGGER.info("Cleanup has passed, set visible false.");
        setVisible(false);
    }

    private void cleanup(final SettingsService settingsService) {
        LOGGER.info("The LocoDialog is disposed.");

        if (locoView != null) {
            locoView.cleanup(settingsService);
        }
    }
}
