package org.bidib.wizard.mvc.main.model;

import java.beans.IndexedPropertyChangeEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.FeedbackConfidenceData;
import org.bidib.jbidibc.messages.FeedbackDynStateData;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.FeedbackPortListener;
import org.bidib.wizard.api.model.listener.PortListListener;
import org.bidib.wizard.client.common.controller.FeedbackPortStatusChangeProvider;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

public class FeedbackPortModel extends Model implements FeedbackPortStatusChangeProvider {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPortModel.class);

    private NodeInterface selectedNode;

    private PropertyChangeListener pclFeedbackPorts;

    private PropertyChangeListener pclFeedbackPort;

    private final List<FeedbackPortListener> feedbackPortListeners = new LinkedList<>();

    private final List<PortListListener> portListListeners = new LinkedList<>();

    public FeedbackPortModel() {

        pclFeedbackPort = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.debug("Property has changed, evt: {}", evt);

                if (evt instanceof IndexedPropertyChangeEvent) {
                    final IndexedPropertyChangeEvent ipce = (IndexedPropertyChangeEvent) evt;

                    SwingUtilities.invokeLater(() -> {
                        final int portId = ipce.getIndex();
                        final NodeInterface node = (NodeInterface) ipce.getSource();
                        LOGGER.debug("The port status has changed: {}, emitting node: {}", portId, node);

                        if (node.equals(selectedNode)) {

                            if (FeedbackPort.GLOBAL_FEEDBACK_PORT_NUMBER == portId) {
                                // global detector
                                LOGGER.info("Received feedback port change for global detector.");
                            }
                            else {

                                List<FeedbackPort> feedbackPorts = selectedNode.getFeedbackPorts();
                                FeedbackPort port = feedbackPorts.get(portId);
                                if (port != null) {
                                    for (FeedbackPortListener feedbackPortListener : feedbackPortListeners) {
                                        switch (ipce.getPropertyName()) {
                                            case NodeInterface.PROPERTY_FEEDBACKPORT_STATUS:
                                                feedbackPortListener.statusChanged(selectedNode, port);
                                                break;
                                            case NodeInterface.PROPERTY_FEEDBACKPORT_ADDRESSES:
                                                List<FeedbackAddressData> addresses = port.getAddresses();
                                                feedbackPortListener.addressesChanged(port, addresses);
                                                break;
                                            case NodeInterface.PROPERTY_FEEDBACKPORT_CONFIDENCE:
                                                FeedbackConfidenceData confidence = port.getConfidence();
                                                feedbackPortListener.confidenceChanged(port, confidence);
                                                break;
                                            case NodeInterface.PROPERTY_FEEDBACKPORT_DYNSTATES:
                                                Collection<FeedbackDynStateData> dynStates = port.getDynStates();
                                                feedbackPortListener.dynStatesChanged(port, dynStates);
                                                break;
                                            default:
                                                break;
                                        }
                                    }
                                }
                                else {
                                    LOGGER.warn("No feedback port found for portId: {}", portId);
                                }
                            }
                        }
                    });

                }
            }
        };
        pclFeedbackPorts = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("Property has changed, evt: {}", evt);
                SwingUtilities.invokeLater(() -> {

                    for (PortListListener listener : portListListeners) {
                        LOGGER.info("Notify listener that the ports have changed: {}", listener);
                        listener.listChanged();
                    }
                });
            }
        };

    }

    @Override
    public NodeInterface getSelectedNode() {
        return selectedNode;
    }

    /**
     * @param selectedNode
     *            the selectedNode to set
     */
    public void setSelectedNode(final NodeInterface selectedNode) {

        if (this.selectedNode != null) {
            NodeInterface node = this.selectedNode;
            // unregister node listener
            LOGGER.trace("Remove pcl from previous selected node: {}", node);
            node.removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORTS, pclFeedbackPorts);
            node.removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_STATUS, pclFeedbackPort);
            node.removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_ADDRESSES, pclFeedbackPort);
            node.removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_CONFIDENCE, pclFeedbackPort);
            node.removePropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_DYNSTATES, pclFeedbackPort);
        }

        this.selectedNode = selectedNode;

        if (this.selectedNode != null) {
            NodeInterface node = this.selectedNode;
            // register node listener
            LOGGER.trace("Add pcl to currently selected node: {}", node);
            node.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORTS, pclFeedbackPorts);
            node.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_STATUS, pclFeedbackPort);
            node.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_ADDRESSES, pclFeedbackPort);
            node.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_CONFIDENCE, pclFeedbackPort);
            node.addPropertyChangeListener(NodeInterface.PROPERTY_FEEDBACKPORT_DYNSTATES, pclFeedbackPort);
        }

    }

    @Override
    public void addFeedbackPortListener(final FeedbackPortListener feedbackPortListener) {
        feedbackPortListeners.add(feedbackPortListener);
    }

    @Override
    public void removeFeedbackPortListener(final FeedbackPortListener feedbackPortListener) {
        feedbackPortListeners.remove(feedbackPortListener);
    }

    @Override
    public void addPortListListener(PortListListener portListListener) {
        portListListeners.add(portListListener);
    }

    @Override
    public void removePortListListener(PortListListener portListListener) {
        portListListeners.remove(portListListener);
    }
}
