package org.bidib.wizard.mvc.main.model;

import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.IoBehaviourInputEnum;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.status.InputPortStatus;
import org.bidib.wizard.mvc.main.model.listener.InputPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InputPortTableModel
    extends SimpleHierarchicalPortTableModel<InputPortStatus, InputPort, InputPortModelListener> {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(InputPortTableModel.class);

    public static final int COLUMN_IO_BEHAVIOUR = 1;

    // public static final int COLUMN_SWITCH_OFF_TIME = 2;

    public static final int COLUMN_PORT_IDENTIFIER = 2;

    public static final int COLUMN_STATUS = 3;

    public static final int COLUMN_PORT_INSTANCE = 4;

    public InputPortTableModel() {
        super();
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(getClass(), "label"), //
                Resources.getString(getClass(), "ioBehaviour"), //
                // Resources.getString(getClass(), "switchOffTime"), //
                Resources.getString(getClass(), "portIdentifier"), //
                Resources.getString(getClass(), "status"), null }; //
    }

    @Override
    public void addRow(InputPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.getLabel();
            rowData[COLUMN_IO_BEHAVIOUR] = port.getInputBehaviour();
            // rowData[COLUMN_SWITCH_OFF_TIME] = port.getSwitchOffTime();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();
            rowData[COLUMN_PORT_INSTANCE] = port;
            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        return false;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_IO_BEHAVIOUR:
                return IoBehaviourInputEnum.class;
            default:
                return Object.class;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
            case COLUMN_STATUS:
                // case COLUMN_SWITCH_OFF_TIME:
            case COLUMN_IO_BEHAVIOUR:
                column = COLUMN_PORT_INSTANCE;
                break;
            default:
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final Object o = getValueAt(row, COLUMN_PORT_INSTANCE);

        if (o instanceof InputPort) {
            final InputPort port = (InputPort) o;

            switch (column) {
                case COLUMN_LABEL:
                    port.setLabel((String) value);
                    super.setValueAt(port.toString(), row, column);
                    fireLabelChanged(port, port.getLabel());
                    break;
                case COLUMN_IO_BEHAVIOUR:
                    IoBehaviourInputEnum ioBehaviour = (IoBehaviourInputEnum) value;

                    if (ioBehaviour != port.getInputBehaviour()) {
                        port.setInputBehaviour(ioBehaviour);
                        super.setValueAt(value, row, column);
                        fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_INPUT_CTRL);
                    }
                    else {
                        LOGGER.debug("The IOBehaviour has not been changed.");
                    }
                    break;
                // case COLUMN_SWITCH_OFF_TIME:
                // int switchOffTime = (Integer) value;
                // if (port.getSwitchOffTime() != switchOffTime) {
                // port.setSwitchOffTime(switchOffTime);
                // super.setValueAt(value, row, column);
                // fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TICKS);
                // }
                // else {
                // LOGGER.debug("The switchOff time has not been changed.");
                // }
                // break;
                case COLUMN_STATUS:
                    LOGGER.debug("Status of input port is updated: {}, port: {}", value, port);
                    if (value instanceof InputPortStatus) {
                        super.setValueAt(value, row, column);
                    }
                    else {
                        LOGGER.warn("Set an invalid value: {}", value);
                        super.setValueAt(value, row, column);
                    }
                    break;
                default:
                    super.setValueAt(value, row, column);
                    break;
            }
        }
        else {
            super.setValueAt(value, row, column);
        }
    }

    @Override
    public void notifyPortStatusChanged(final InputPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(final InputPort port) {
        LOGGER.info("The port config was changed for port: {}", port.getDebugString());

        // update the port status
        notifyPortStatusChanged(port);
    }

    @Override
    public void notifyPortLabelChanged(final InputPort port) {
        LOGGER.info("The port label was changed for port: {}", port);

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, InputPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, InputPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final InputPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final InputPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            InputPort current = (InputPort) getValueAt(row, InputPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(final InputPort port) {

        LOGGER.info("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(final InputPort servoPort, final PortConfigKeys... portConfigKeys) {

        // provide the changed PortConfigKeys to the listener
        portListener.configChanged(servoPort, portConfigKeys);
    }

    @Override
    public void changePortType(LcOutputType portType, InputPort port) {
        portListener.changePortType(portType, port);
    }

    @Override
    protected InputPort getRow(int rowIndex) {
        InputPort port = (InputPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return port;
    }

}
