package org.bidib.wizard.mvc.main.view.exchange;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.ProtocolVersion;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeHelper;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeTableModel;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.InputPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.ServoPort;
import org.bidib.wizard.model.ports.SwitchPairPort;
import org.bidib.wizard.model.ports.SwitchPort;
import org.bidib.wizard.mvc.common.view.cvdefinition.CvDefinitionPanelProvider;
import org.bidib.wizard.mvc.common.view.cvdefinition.CvValueUtils;
import org.bidib.wizard.mvc.main.view.component.SaveNodeConfigurationDialog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.DefaultExpandableRow;

public class SaveConfigOnNodeHelper {

    private static final Logger LOGGER = LoggerFactory.getLogger(SaveConfigOnNodeHelper.class);

    public void saveOnNode(
        final String connectionId, final NodeService nodeService, final SwitchingNodeService switchingNodeService,
        final NodeInterface node, Map<String, Object> params) {

        LOGGER.info("Save the configuration to the node: {}", node);

        if (params == null) {
            // no params provided, save all configuration values
            params = new HashMap<String, Object>();
            params.put(SaveNodeConfigurationDialog.SAVE_MACROS, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_ACCESSORIES, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_BACKLIGHTPORTS, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_LIGHTPORTS, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_SERVOPORTS, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_SWITCHPORTS, Boolean.TRUE);
            params.put(SaveNodeConfigurationDialog.SAVE_FEATURES, Boolean.FALSE);
            params.put(SaveNodeConfigurationDialog.SAVE_CVS, Boolean.FALSE);
        }

        if (params.get(SaveNodeConfigurationDialog.SAVE_FEATURES) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_FEATURES)).booleanValue()) {

            LOGGER.info("Save the features on the node.");
            List<Feature> features = node.getNode().getFeatures();

            if (CollectionUtils.isNotEmpty(features)) {
                // use new list to prevent ConcurrentModificationException
                features = new LinkedList<>(features);

                for (Feature feature : features) {
                    try {
                        nodeService.setFeatures(connectionId, node, Arrays.asList(feature));
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Write feature to node failed: {}", feature, ex);
                        StringBuilder sb =
                            new StringBuilder("Save Feature ")
                                .append(feature.getFeatureName()).append(" on node failed, value: ")
                                .append(feature.getValue());
                        addError(params, sb.toString());
                    }
                }
            }
            else {
                LOGGER.warn("No features to transfer available.");
            }
        }

        if (params.get(SaveNodeConfigurationDialog.SAVE_CVS) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_CVS)).booleanValue()) {
            LOGGER.info("Save the CVs on the node.");

            ApplicationContext applicationContext = DefaultApplicationContext.getInstance();
            CvDefinitionTreeTableModel treeModel =
                org.bidib.wizard.utils.NodeUtils.getCvDefinitionTreeTableModel(node, applicationContext);
            if (treeModel != null) {

                boolean hasPendingCVChanges = CvDefinitionTreeHelper.hasPendingChanges(treeModel);
                LOGGER.info("Before transfer CV values to node, hasPendingCVChanges: {}", hasPendingCVChanges);

                // write the values to the node
                final CvDefinitionPanelProvider provider = new CvDefinitionPanelProvider() {

                    @Override
                    public void writeConfigVariables(List<ConfigurationVariable> cvList) {
                        LOGGER.info("Write cv variables to node.");

                        // write the cv values to the node
                        List<ConfigurationVariable> configVars =
                            nodeService.setConfigVariables(connectionId, node, cvList);

                        // iterate over the collection of stored variables in the model and update the values.
                        // After that notify the tree and delete the new values that are now stored in the node
                        node.updateConfigVariableValues(configVars, false);

                        LOGGER.info("Write cv variables to node finished.");
                    }

                    @Override
                    public void checkPendingChanges() {
                    }
                };
                DefaultExpandableRow root = (DefaultExpandableRow) treeModel.getRoot();
                CvValueUtils
                    .writeCvValues(node, root,
                        org.bidib.wizard.utils.NodeUtils.getCvNumberToNodeMap(node, applicationContext), provider);

            }
        }

        if (params.get(SaveNodeConfigurationDialog.SAVE_LIGHTPORTS) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_LIGHTPORTS)).booleanValue()) {

            for (LightPort lightPort : node.getLightPorts()) {
                try {
                    switchingNodeService.setPortConfig(connectionId, node.getSwitchingNode(), lightPort);
                }
                catch (Exception ex) {
                    LOGGER.warn("Save lightport configuration on node failed: {}", lightPort, ex);
                    addError(params, "Save lightport configuration on node failed: " + lightPort);
                }
            }
        }
        else {
            LOGGER.info("Don't save lightports.");
        }

        if (params.get(SaveNodeConfigurationDialog.SAVE_BACKLIGHTPORTS) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_BACKLIGHTPORTS)).booleanValue()) {

            for (BacklightPort backlightPort : node.getBacklightPorts()) {
                try {
                    switchingNodeService.setPortConfig(connectionId, node.getSwitchingNode(), backlightPort);
                }
                catch (Exception ex) {
                    LOGGER.warn("Save backlightport configuration on node failed: {}", backlightPort, ex);
                    addError(params, "Save backlightport configuration on node failed: " + backlightPort);
                }
            }
        }
        else {
            LOGGER.info("Don't save backlightports.");
        }

        if (params.get(SaveNodeConfigurationDialog.SAVE_SERVOPORTS) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_SERVOPORTS)).booleanValue()) {
            LOGGER.info("Set the servo port configuration on the node.");

            for (ServoPort servoPort : node.getServoPorts()) {
                try {
                    switchingNodeService.setPortConfig(connectionId, node.getSwitchingNode(), servoPort);
                }
                catch (Exception ex) {
                    LOGGER.warn("Save servoport configuration on node failed: {}", servoPort, ex);
                    addError(params, "Save servoport configuration on node failed: " + servoPort);
                }
            }
        }
        else {
            LOGGER.info("Don't save servoports.");
        }

        boolean hasSwitchPortConfig = true;

        // boolean hasSwitchPortConfig = false;
        // Feature switchPortConfigAvailable =
        // Feature.findFeature(node.getNode().getFeatures(), BidibLibrary.FEATURE_SWITCH_CONFIG_AVAILABLE);
        // if (switchPortConfigAvailable != null) {
        // hasSwitchPortConfig = (switchPortConfigAvailable.getValue() > 0);
        // }

        if (hasSwitchPortConfig) {
            if (params.get(SaveNodeConfigurationDialog.SAVE_SWITCHPORTS) != null
                && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_SWITCHPORTS)).booleanValue()) {
                LOGGER.info("Set the switch port configuration on the node.");

                if (node != null && node.getNode().isPortFlatModelAvailable()
                    && node.getNode().getProtocolVersion().isHigherThan(ProtocolVersion.VERSION_0_5)) {
                    LOGGER.info("The current node supports the flat port model.");

                    // we must iterate over the generic ports and find out the current port type
                    final List<GenericPort> genericPorts = new ArrayList<>();
                    genericPorts
                        .addAll(node
                            .getGenericPorts().stream()
                            .filter(gp -> gp.isSupportsSwitchPort() || gp.isSupportsSwitchPairPort())
                            .collect(Collectors.toList()));

                    for (GenericPort genericPort : genericPorts) {
                        // set the configured port type

                        LOGGER.info("Prepare the generic port: {}", genericPort);

                        Map<Byte, PortConfigValue<?>> values = genericPort.getPortConfigX();

                        if (!genericPort.isRemappingEnabled() && values.containsKey(BidibLibrary.BIDIB_PCFG_RECONFIG)) {
                            LOGGER
                                .info("Remove the reconfig param because the port has not remapping enabled: {}",
                                    genericPort);
                            values.remove(BidibLibrary.BIDIB_PCFG_RECONFIG);
                        }

                        switchingNodeService
                            .setPortConfig(connectionId, node.getSwitchingNode(), genericPort,
                                genericPort.getCurrentPortType(), values);
                    }
                }
                else {
                    LOGGER.info("The current node supports the typed port model.");

                    if (node.getNode().getProtocolVersion().isLowerThan(ProtocolVersion.VERSION_0_6)) {
                        // we must check the feature
                        hasSwitchPortConfig = false;
                        Feature switchPortConfigAvailable =
                            Feature
                                .findFeature(node.getNode().getFeatures(),
                                    BidibLibrary.FEATURE_SWITCH_CONFIG_AVAILABLE);
                        if (switchPortConfigAvailable != null) {
                            hasSwitchPortConfig = (switchPortConfigAvailable.getValue() > 0);
                        }
                    }

                    if (hasSwitchPortConfig) {

                        List<SwitchPort> switchPorts = new LinkedList<>();
                        switchPorts.addAll(node.getSwitchPorts());
                        for (SwitchPort switchPort : switchPorts) {
                            try {
                                if (switchPort.isEnabled()) {
                                    switchingNodeService
                                        .setPortConfig(connectionId, node.getSwitchingNode(), switchPort);
                                }
                                else {
                                    LOGGER.info("The current switchPort is not enabled: {}", switchPort);
                                }
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Save switchPort configuration on node failed: {}", switchPort, ex);
                                addError(params, "Save switchPort configuration on node failed: " + switchPort);
                            }
                        }

                        List<SwitchPairPort> switchPairPorts = new LinkedList<>();
                        switchPairPorts.addAll(node.getSwitchPairPorts());
                        for (SwitchPairPort switchPairPort : switchPairPorts) {
                            try {
                                if (switchPairPort.isEnabled()) {
                                    switchingNodeService
                                        .setPortConfig(connectionId, node.getSwitchingNode(), switchPairPort);
                                }
                                else {
                                    LOGGER.info("The current switchPairPort is not enabled: {}", switchPairPort);
                                }
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Save switchPairPort configuration on node failed: {}", switchPairPort, ex);
                                addError(params, "Save switchPairPort configuration on node failed: " + switchPairPort);
                            }
                        }

                        List<InputPort> inputPorts = new LinkedList<>();
                        inputPorts.addAll(node.getInputPorts());
                        for (InputPort inputPort : inputPorts) {
                            try {
                                if (inputPort.isEnabled()) {
                                    switchingNodeService
                                        .setPortConfig(connectionId, node.getSwitchingNode(), inputPort);
                                }
                                else {
                                    LOGGER.info("The current inputPort is not enabled: {}", inputPort);
                                }
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Save inputPort configuration on node failed: {}", inputPort, ex);
                                addError(params, "Save inputPort configuration on node failed: " + inputPort);
                            }
                        }
                    }
                    else {
                        LOGGER.info("Don't save switchports because the node does not support it.");
                    }
                }
            }
            else {
                LOGGER.info("Don't save switchports.");
            }
        }
        else {
            LOGGER.info("Node does not support switch port configuration.");
        }

        // save the node string username
        if (node.getNode().getStringSize() > 0) {
            LOGGER.info("Transfer the node strings.");

            String userName = node.getNode().getStoredString(StringData.INDEX_USERNAME);
            LOGGER.info("Transfer the node string username: {}", userName);

            try {
                nodeService.setNodeDetails(connectionId, node, userName);
            }
            catch (Exception ex) {
                LOGGER.warn("Save userName on node failed: {}", userName, ex);
                addError(params, "Save userName on node failed: " + userName);
            }
        }
        else {
            LOGGER.info("The node does not support store node string values.");
        }

        // save the macros on the node
        if (params.get(SaveNodeConfigurationDialog.SAVE_MACROS) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_MACROS)).booleanValue()) {
            LOGGER.info("Save macros on the node.");

            for (Macro macro : node.getMacros()) {
                LOGGER.info("Save macro: {}", macro);
                try {
                    // create a clone of the macro
                    final Macro macroClone = Macro.cloneMacro(macro);

                    switchingNodeService.saveMacro(connectionId, node.getSwitchingNode(), macroClone);
                }
                catch (Exception ex) {
                    LOGGER.warn("Save macro on node failed: {}", macro, ex);
                    addError(params, "Save macro on node failed: " + macro);
                }
            }
        }
        else {
            LOGGER.info("Don't save macros.");
        }

        // save the accessories on the node
        if (params.get(SaveNodeConfigurationDialog.SAVE_ACCESSORIES) != null
            && ((Boolean) params.get(SaveNodeConfigurationDialog.SAVE_ACCESSORIES)).booleanValue()) {
            LOGGER.info("Save accessories on the node.");

            // Communication communication = CommunicationFactory.getInstance();

            for (Accessory accessory : node.getAccessories()) {
                try {
                    // create a clone of the accessory
                    final Accessory accessoryClone = Accessory.cloneAccessoryData(accessory);

                    // communication.saveAccessory(node, accessory);
                    switchingNodeService.saveAccessory(connectionId, node.getSwitchingNode(), accessoryClone);
                }
                catch (Exception ex) {
                    LOGGER.warn("Save accessory on node failed: {}", accessory, ex);
                    addError(params, "Save accessory on node failed: " + accessory);
                }
            }
        }
        else {
            LOGGER.info("Don't save accessories.");
        }
    }

    private void addError(Map<String, Object> params, String message) {
        List<String> saveErrors = (List<String>) params.get(SaveNodeConfigurationDialog.SAVE_ERRORS);
        if (saveErrors == null) {
            saveErrors = new LinkedList<String>();
            params.put(SaveNodeConfigurationDialog.SAVE_ERRORS, saveErrors);
        }
        saveErrors.add(message);
    }

}
