package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.enums.FeatureEnum;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.CvDefinitionRequestListener;
import org.bidib.wizard.api.model.listener.PortListener;
import org.bidib.wizard.api.model.listener.PortListenerProvider;
import org.bidib.wizard.client.common.controller.FeedbackPortStatusChangeProvider;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeHelper;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeTableModel;
import org.bidib.wizard.client.common.view.cvdef.CvNode;
import org.bidib.wizard.client.common.view.cvdef.CvValueEditor;
import org.bidib.wizard.client.common.view.cvdef.GBM16TReverserCvNode;
import org.bidib.wizard.client.common.view.cvdef.GBM16TReverserEditor;
import org.bidib.wizard.client.common.view.cvdef.GBM16TReverserPresentationModel;
import org.bidib.wizard.client.common.view.cvdef.GBM16TReverserValueBean;
import org.bidib.wizard.client.common.view.cvdef.KeywordNodeNode;
import org.bidib.wizard.client.common.view.listener.TabStatusListener;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.mvc.common.view.cvdefinition.CvDefinitionPanelProvider;
import org.bidib.wizard.mvc.common.view.cvdefinition.CvValueUtils;
import org.bidib.wizard.mvc.main.controller.ReverserPanelController;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabSelectionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.Trigger;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.grid.DefaultExpandableRow;

public class ReverserPanel
    implements TabPanelProvider, TabVisibilityProvider, CvDefinitionPanelProvider, CvDefinitionRequestListenerAware,
    PortListenerProvider<FeedbackPort>, PendingChangesAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReverserPanel.class);

    private final ReverserPanelController reverserPanelController;

    private final MainModel mainModel;

    private final TabStatusListener tabStatusListener;

    private NodeInterface selectedNode;

    private JPanel contentPanel;

    private GBM16TReverserEditor gbm16TReverserEditor;

    private JButton saveButton;

    private Map<String, CvNode> cvNumberToNodeMap;

    private Map<String, ConfigurationVariable> configVariables;

    private CvDefinitionTreeTableModel treeModel;

    public static class ReverserFormPanel extends JPanel implements TabSelectionListener {

        private static final long serialVersionUID = 1L;

        private final ReverserPanel reverserPanel;

        public ReverserFormPanel(final ReverserPanel reverserPanel) {
            super(new BorderLayout());
            this.reverserPanel = reverserPanel;
        }

        @Override
        public void tabSelected(boolean selected) {
            LOGGER.info("Tab is selected: {}", selected);

            // toolbarCvDefinition.setVisible(selected);
            if (selected) {
                this.reverserPanel.updateComponentState();
            }
        }
    }

    private final ValueHolder valueModel = new ValueHolder(false);

    public ReverserPanel(final ReverserPanelController reverserPanelController,
        final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider, final MainModel model,
        final TabStatusListener tabStatusListener) {
        this.reverserPanelController = reverserPanelController;
        this.mainModel = model;
        this.tabStatusListener = tabStatusListener;

        // create the 'detail panel'
        final JPanel panel = new JPanel();

        final FormBuilder detailFormBuilder = FormBuilder.create().columns("p").rows("p").panel(panel);

        detailFormBuilder.border(Paddings.DLU4);

        // create the button panel to save the changed CV values

        this.gbm16TReverserEditor = createGBM16TReverserEditor(feedbackPortStatusChangeProvider);

        detailFormBuilder.add(this.gbm16TReverserEditor.getComponent()).xy(1, 1);

        JPanel builderPanel = detailFormBuilder.build();
        JScrollPane scrollPane = new JScrollPane(builderPanel);

        this.contentPanel = new ReverserFormPanel(this);
        this.contentPanel.setName(getName());
        this.contentPanel.add(scrollPane, BorderLayout.CENTER);

        // final ValueHolder valueModel = new ValueHolder(false);
        PropertyConnector.connect(this.gbm16TReverserEditor.getSaveButtonEnabledModel(), "value", valueModel, "value");

        PropertyConnector.connect(valueModel, "value", saveButton, "enabled");

        valueModel.addValueChangeListener(evt -> {
            setPendingChanges((boolean) valueModel.getValue());
        });
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    public void setPendingChanges(boolean pendingChanges) {
        LOGGER.info("set pending changes flag: {}", pendingChanges);

        // hook in here and set the pending changes flag on the tab
        // if (CvDefinitionTreeHelper.hasPendingChanges(treeModel)) {
        tabStatusListener.updatePendingChanges(this.contentPanel, pendingChanges);
        // }
    }

    // public boolean getPendingChanges() {
    // return false;
    // }

    private GBM16TReverserEditor createGBM16TReverserEditor(
        final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider) {
        final GBM16TReverserEditor gbm16TReverserEditor =
            new GBM16TReverserEditor(feedbackPortStatusChangeProvider, new Trigger(), createButtonPanel(),
                "/images/reverser/reverser_DUAL_small.png", "/images/reverser/reverser_DUAL_table_small.png");

        return gbm16TReverserEditor;
    }

    private JPanel createButtonPanel() {
        saveButton = new JButton(Resources.getString(getClass(), "editor.save"));
        saveButton.setEnabled(false);
        saveButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("save pressed.");

                saveValues();
            }
        });
        JButton resetButton = new JButton(Resources.getString(getClass(), "editor.reset"));
        resetButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("Reset was pressed.");
                Trigger trigger = getTrigger();
                if (trigger != null) {
                    trigger.triggerFlush();

                    // call 2x
                    trigger.triggerFlush();
                }
            }
        });
        // create the button panel
        JPanel buttonBar = new ButtonBarBuilder().addGlue().addButton(saveButton, resetButton).build();

        return buttonBar;
    }

    private void saveValues() {
        LOGGER.info("Save the values.");

        try {
            Trigger trigger = getTrigger();
            if (trigger == null) {
                LOGGER.info("No trigger available. Do not save!");
                return;
            }
            trigger.triggerCommit();

            CvValueEditor valueEditor = ReverserPanel.this.gbm16TReverserEditor;

            GBM16TReverserPresentationModel presentationModel = ReverserPanel.this.gbm16TReverserEditor.getCvAdapter();
            LOGGER.info("Current presentationModel: {}", presentationModel);

            final GBM16TReverserValueBean valueBean = presentationModel.getBean();
            LOGGER.info("Current valueBean: {}", valueBean);

            CvNode cvNode = valueEditor.getCvNode();
            LOGGER.debug("Current selected cvNode: {}", cvNode);
            if (cvNode != null) {
                // let the editor update the CV value(s) ...
                valueEditor.updateCvValues(cvNumberToNodeMap, treeModel);

                // collect the new values from the tree
                DefaultExpandableRow root = (DefaultExpandableRow) treeModel.getRoot();

                final NodeInterface selectedNode = mainModel.getSelectedNode();

                // write the values to the node
                CvValueUtils.writeCvValues(selectedNode, root, cvNumberToNodeMap, ReverserPanel.this);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Commit failed.", ex);
        }
    }

    @Override
    public boolean isTabVisible() {
        NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {

            int feedbackSize =
                Feature.getIntFeatureValue(node.getNode().getFeatures(), FeatureEnum.FEATURE_BM_SIZE.getNumber());

            // int accessoryCount =
            // Feature
            // .getIntFeatureValue(node.getNode().getFeatures(), FeatureEnum.FEATURE_ACCESSORY_COUNT.getNumber());

            boolean isTabVisible = ProductUtils.isGBM16TS(node.getUniqueId()) && feedbackSize > 0;

            LOGGER.info("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    public void nodeChanged() {
        LOGGER.info("The selected node has changed.");

        updateComponentState();
    }

    private void updateComponentState() {

        NodeInterface node = mainModel.getSelectedNode();

        // release not clear
        cvNumberToNodeMap = null;

        configVariables = null;
        treeModel = null;

        // TODO this will load the CV values

        if (isTabVisible()) {
            updateNodeInfo(node);
        }
        else {
            updateNodeInfo(null);
        }

        selectedNode = node;
        if (node != null) {

            CvDefinitionTreeTableModel cvDefinitionTreeTableModel =
                reverserPanelController.getCvDefinitionTreeTableModel(node);
            treeModel = cvDefinitionTreeTableModel;

            // fetch the current map from the node
            cvNumberToNodeMap = reverserPanelController.getCvNumberToNodeMap(node);
            configVariables = node.getConfigVariables();

        }
    }

    private void updateNodeInfo(final NodeInterface node) {
        if (node == null) {
            // no node available
            LOGGER.info("Clear the value of the gbm16TReverserEditor.");
            this.gbm16TReverserEditor.setValue(null, Collections.emptyMap());
            return;
        }
        LOGGER.info("refresh the feedback ports for node: {}", node);

        final Map<String, CvNode> cvNumberToNodeMap = reverserPanelController.getCvNumberToNodeMap(node);

        final CvDefinitionTreeTableModel cvDefinitionTreeTableModel =
            reverserPanelController.getCvDefinitionTreeTableModel(node);

        if (cvNumberToNodeMap == null || cvDefinitionTreeTableModel == null) {
            LOGGER.info("No cvNumberToNodeMap or cvDefinitionTreeTableModel available for node: {}", cvNumberToNodeMap);

            // must use setValue to initialise the editor correct
            this.gbm16TReverserEditor.setValue(null, Collections.emptyMap());

            return;
        }

        // search the Reverser KeywordNodeNode in the cvDefinitionTreeTableModel
        final List<KeywordNodeNode> keywordNodeNodes = new LinkedList<>();
        CvDefinitionTreeHelper
            .findNodeByKeyword((DefaultExpandableRow) cvDefinitionTreeTableModel.getRoot(),
                KeywordNodeNode.KEYWORD_REVERSER, keywordNodeNodes);

        LOGGER.info("Found reversers in cvDefinitionTreeTableModel: {}", keywordNodeNodes);

        if (keywordNodeNodes.isEmpty()) {
            // no keyword node available

            // must use setValue to initialise the editor correct
            this.gbm16TReverserEditor.setValue(null, Collections.emptyMap());
            return;
        }

        final KeywordNodeNode keywordNodeNode = keywordNodeNodes.get(0);
        GBM16TReverserCvNode gbm16tReverserCvNode = null;
        if (keywordNodeNode != null) {
            gbm16tReverserCvNode = (GBM16TReverserCvNode) keywordNodeNode.getMasterNode();

            LOGGER.info("Set the gbm16tReverserCvNode: {}", gbm16tReverserCvNode);

            List<CvNode> cvNodes = new LinkedList<CvNode>();
            CvDefinitionTreeHelper.addCvOfSubnode(keywordNodeNode, cvNodes, cvNumberToNodeMap);

            LOGGER.info("Nodes to update: {}", cvNodes);

            List<ConfigurationVariable> cvSet = new LinkedList<ConfigurationVariable>();
            for (CvNode cvNode : cvNodes) {
                cvSet.add(cvNode.getConfigVar());
            }

            try {
                readCvValues(cvSet);
            }
            catch (RuntimeException ex) {
                LOGGER.warn("Load CV values has failed!", ex);

                JOptionPane
                    .showMessageDialog(JOptionPane.getFrameForComponent(null),
                        Resources.getString(ReverserPanel.class, "loadcvfailed.message"),
                        Resources.getString(ReverserPanel.class, "loadcvfailed.title"), JOptionPane.ERROR_MESSAGE);
            }
        }

        // must use setValue to initialise the editor correct
        this.gbm16TReverserEditor.setValue(gbm16tReverserCvNode, cvNumberToNodeMap);
    }

    private void readCvValues(List<ConfigurationVariable> configVariables) {
        LOGGER.info("Read the CV values.");

        // Get the CV values from the node
        if (configVariables != null && configVariables.size() > 0) {
            LOGGER.info("Get the CV values for configuration variables, count: {}", configVariables.size());
            fireLoadConfigVariables(configVariables);
        }
        else {
            LOGGER.warn("No configuration variables available.");
        }
    }

    private List<CvDefinitionRequestListener> cvDefinitionRequestListeners =
        new LinkedList<CvDefinitionRequestListener>();

    @Override
    public void addCvDefinitionRequestListener(CvDefinitionRequestListener l) {
        cvDefinitionRequestListeners.add(l);
    }

    private void fireLoadConfigVariables(List<ConfigurationVariable> configVariables) {
        // TODO decouple the AWT-thread from this work?

        for (CvDefinitionRequestListener l : cvDefinitionRequestListeners) {
            l.loadCvValues(configVariables);
        }
    }

    @Override
    public void writeConfigVariables(List<ConfigurationVariable> cvList) {
        fireWriteConfigVariables(cvList);
    }

    private void fireWriteConfigVariables(List<ConfigurationVariable> cvList) {
        // TODO decouple the AWT-thread from this work?

        for (CvDefinitionRequestListener l : cvDefinitionRequestListeners) {
            l.writeCvValues(cvList);
        }
    }

    @Override
    public PortListener<FeedbackPort> getPortListener() {
        return this.gbm16TReverserEditor.getPortListener();
    }

    private Trigger getTrigger() {
        return gbm16TReverserEditor.getTrigger();
    }

    @Override
    public void checkPendingChanges() {
        // TODO Auto-generated method stub

    }

    @Override
    public boolean hasPendingChanges() {
        return (boolean) valueModel.getValue();
    }

    @Override
    public void savePendingChanges() {

        if (hasPendingChanges()) {
            LOGGER.info("Save the pending changes.");

            saveValues();
        }
        else {
            LOGGER.info("No pending changes to save available.");
        }

    }

}
