package org.bidib.wizard.mvc.ping.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

public class NodePingModel extends Model {
    private static final Logger LOGGER = LoggerFactory.getLogger(NodePingModel.class);

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_NODE_PING_STATUS = "nodePingStatus";

    public static final String PROPERTY_PING_INTERVAL = "pingInterval";

    public static final String PROPERTY_LAST_PING_TIMESTAMP = "lastPingTimestamp";

    public static final String PROPERTY_LAST_PONG_MARKER = "lastPongMarker";

    public static final String PROPERTY_NODE_LABEL = "nodeLabel";

    public static final String PROPERTY_ADDITIONAL_FILL_BYTES_COUNT = "additionalFillBytesCount";

    public static final String PROPERTY_ADDITIONAL_TOTAL_BYTES_COUNT = "additionalTotalBytesCount";

    public static final String PROPERTY_ADDITIONAL_PAYLOAD_START_VALUE = "additionalPayloadStartValue";

    public static final String PROPERTY_IDENTIFY_PROCESSING_WAIT_DURATION = "identifyProcessingWaitDuration";

    private final NodeInterface node;

    private String nodeLabel;

    private NodePingState nodePingState;

    private Integer pingInterval;

    private long lastPingTimestamp;

    private int lastPongMarker = -1;

    private int data;

    private int additionalFillBytesCount;

    private int additionalTotalBytesCount;

    private int additionalPayloadStartValue = 0xB0;

    private byte[] additionalPayload;

    private int identifyProcessingWaitDuration;

    private PropertyChangeListener propertyChangeListener;

    private final Object preparePayloadLock = new Object();

    public NodePingModel(final NodeInterface node) {
        this.node = node;

        LOGGER.info("Create new NodePingModel for node: {}", node);
    }

    public void registerNode() {
        propertyChangeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // update the label with the name ...
                LOGGER.debug("Property has changed, name: {}, value: {}", evt.getPropertyName(), evt.getNewValue());

                String nodeLabel = prepareNodeLabel();
                setNodeLabel(nodeLabel);
            }
        };
        this.node.addPropertyChangeListener(NodeInterface.PROPERTY_LABEL, propertyChangeListener);
        this.node
            .addPropertyChangeListener(
                NodeInterface.PROPERTY_NODE_PREFIX + org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME,
                propertyChangeListener);
    }

    public void freeNode() {
        this.node.removePropertyChangeListener(NodeInterface.PROPERTY_LABEL, propertyChangeListener);
        this.node
            .removePropertyChangeListener(
                NodeInterface.PROPERTY_NODE_PREFIX + org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME,
                propertyChangeListener);
    }

    /**
     * @return the node
     */
    public NodeInterface getNode() {
        return node;
    }

    /**
     * @return the nodePingState
     */
    public NodePingState getNodePingState() {
        return nodePingState;
    }

    /**
     * @param nodePingState
     *            the nodePingState to set
     */
    public void setNodePingState(NodePingState nodePingState) {
        NodePingState oldValue = this.nodePingState;

        this.nodePingState = nodePingState;

        firePropertyChange(PROPERTY_NODE_PING_STATUS, oldValue, this.nodePingState);
    }

    /**
     * @param nodeLabel
     *            the node label to set
     */
    public void setNodeLabel(String nodeLabel) {
        String oldValue = this.nodeLabel;
        this.nodeLabel = nodeLabel;

        firePropertyChange(PROPERTY_NODE_LABEL, oldValue, this.nodeLabel);
    }

    /**
     * @return the node label
     */
    public String getNodeLabel() {
        return nodeLabel;
    }

    /**
     * @return the pingInterval in milliseconds
     */
    public Integer getPingInterval() {
        return pingInterval;
    }

    /**
     * @param pingInterval
     *            the pingInterval in milliseconds to set
     */
    public void setPingInterval(Integer pingInterval) {
        Integer oldValue = this.pingInterval;

        this.pingInterval = pingInterval;
        firePropertyChange(PROPERTY_PING_INTERVAL, oldValue, this.pingInterval);
    }

    /**
     * @return the lastPing timestamp
     */
    public long getLastPingTimestamp() {
        return lastPingTimestamp;
    }

    /**
     * @param lastPingTimestamp
     *            the lastPing timestamp to set
     */
    public void setLastPingTimestamp(long lastPingTimestamp) {
        long oldValue = this.lastPingTimestamp;

        this.lastPingTimestamp = lastPingTimestamp;

        SwingUtils
            .executeInEDT(() -> firePropertyChange(PROPERTY_LAST_PING_TIMESTAMP, oldValue, this.lastPingTimestamp));
    }

    /**
     * @return the last pong marker
     */
    public int getLastPongMarker() {
        return lastPongMarker;
    }

    /**
     * @param lastPongMarker
     *            the last pong marker to set
     */
    public void setLastPongMarker(int lastPongMarker) {
        int oldValue = this.lastPongMarker;

        this.lastPongMarker = lastPongMarker;

        SwingUtils.executeInEDT(() -> firePropertyChange(PROPERTY_LAST_PONG_MARKER, oldValue, this.lastPongMarker));
    }

    /**
     * @return the additionalFillBytesCount
     */
    public int getAdditionalFillBytesCount() {
        return additionalFillBytesCount;
    }

    /**
     * @param additionalFillBytesCount
     *            the additionalFillBytesCount to set
     */
    public void setAdditionalFillBytesCount(int additionalFillBytesCount) {
        int oldValue = this.additionalFillBytesCount;
        this.additionalFillBytesCount = additionalFillBytesCount;

        synchronized (preparePayloadLock) {

            prepareAdditionalPayload(this.additionalTotalBytesCount, this.additionalFillBytesCount);
        }

        firePropertyChange(PROPERTY_ADDITIONAL_FILL_BYTES_COUNT, oldValue, this.additionalFillBytesCount);
    }

    private void prepareAdditionalPayload(int additionalTotalBytesCount, int additionalFillBytesCount) {

        if (additionalTotalBytesCount > 0) {

            if (additionalFillBytesCount == 0) {
                additionalFillBytesCount = 16;
            }
        }
        if (additionalFillBytesCount > 0) {

            this.additionalPayload = new byte[additionalFillBytesCount];
            for (int index = 0; index < additionalFillBytesCount; index++) {
                this.additionalPayload[index] = (byte) ((this.additionalPayloadStartValue + index) & 0xFF);
            }
        }
        else {
            this.additionalPayload = null;
        }
    }

    /**
     * @return the additionalTotalBytesCount
     */
    public int getAdditionalTotalBytesCount() {
        return additionalTotalBytesCount;
    }

    /**
     * @param additionalTotalBytesCount
     *            the additionalTotalBytesCount to set
     */
    public void setAdditionalTotalBytesCount(int additionalTotalBytesCount) {
        int oldValue = this.additionalTotalBytesCount;
        this.additionalTotalBytesCount = additionalTotalBytesCount;

        synchronized (preparePayloadLock) {
            prepareAdditionalPayload(this.additionalTotalBytesCount, this.additionalFillBytesCount);
        }

        firePropertyChange(PROPERTY_ADDITIONAL_TOTAL_BYTES_COUNT, oldValue, this.additionalTotalBytesCount);
    }

    /**
     * @param additionalPayloadStartValue
     *            the start value of the additional payload bytes
     */
    public void setAdditionalPayloadStartValue(int additionalPayloadStartValue) {
        int oldValue = this.additionalPayloadStartValue;
        this.additionalPayloadStartValue = additionalPayloadStartValue;

        if (oldValue != additionalPayloadStartValue) {
            synchronized (preparePayloadLock) {
                prepareAdditionalPayload(this.additionalTotalBytesCount, this.additionalFillBytesCount);
            }
        }

        firePropertyChange(PROPERTY_ADDITIONAL_PAYLOAD_START_VALUE, oldValue, this.additionalPayloadStartValue);
    }

    /**
     * @return the start value of the additional payload bytes
     */
    public int getAdditionalPayloadStartValue() {
        return additionalPayloadStartValue;
    }

    public void setData(int data) {
        this.data = data;
    }

    public int getData() {
        return this.data;
    }

    public byte[] getNextIncData() {
        int data = ++this.data;

        LOGGER.info("Next inc data: {} .. {}", this.data, data);

        if (data > 255) {
            data = 0;
            this.data = data;

            LOGGER.info("Adjusted data: {}", this.data);
        }

        synchronized (preparePayloadLock) {
            if (additionalPayload == null) {
                return new byte[] { (byte) (data & 0xFF) };
            }

            return ByteUtils.prepend((byte) (data & 0xFF), additionalPayload);
        }
    }

    /**
     * @return the identifyProcessingWaitDuration
     */
    public int getIdentifyProcessingWaitDuration() {
        return identifyProcessingWaitDuration;
    }

    /**
     * @param identifyProcessingWaitDuration
     *            the identifyProcessingWaitDuration to set
     */
    public void setIdentifyProcessingWaitDuration(int identifyProcessingWaitDuration) {
        int oldValue = this.identifyProcessingWaitDuration;
        this.identifyProcessingWaitDuration = identifyProcessingWaitDuration;

        firePropertyChange(PROPERTY_IDENTIFY_PROCESSING_WAIT_DURATION, oldValue, this.identifyProcessingWaitDuration);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof NodePingModel) {
            NodePingModel other = (NodePingModel) obj;
            if (node.equals(other.getNode())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return node.hashCode();
    }

    public String prepareNodeLabel() {
        String nodeLabel = node.getLabel();
        if (StringUtils.isBlank(nodeLabel)) {
            // try to get the product name
            String productString = node.getNode().getStoredString(StringData.INDEX_PRODUCTNAME);
            if (StringUtils.isNotBlank(productString)) {
                nodeLabel = productString;
            }
        }
        return nodeLabel;
    }

    @Override
    public String toString() {
        return node.toString();
    }
}
