package org.bidib.wizard.mvc.pt.controller;

import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JFrame;

import org.bidib.jbidibc.messages.enums.CommandStationProgState;
import org.bidib.jbidibc.messages.enums.CommandStationPt;
import org.bidib.jbidibc.messages.enums.CommandStationState;
import org.bidib.jbidibc.messages.enums.PtOperation;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.connection.AbstractMessageEvent;
import org.bidib.wizard.api.model.connection.event.CommandStationProgStateMessageEvent;
import org.bidib.wizard.api.model.connection.event.CommandStationStateMessageEvent;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.core.model.connection.MessageAdapter;
import org.bidib.wizard.core.model.connection.MessageEventConsumer;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.pt.controller.listener.PtProgrammerControllerListener;
import org.bidib.wizard.mvc.pt.model.PtProgrammerModel;
import org.bidib.wizard.mvc.pt.view.PtProgrammerView;
import org.bidib.wizard.mvc.pt.view.listener.PtProgrammerViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.vlsolutions.swing.docking.DockingDesktop;

public class PtProgrammerController {
    private static final Logger LOGGER = LoggerFactory.getLogger(PtProgrammerController.class);

    private final Collection<PtProgrammerControllerListener> listeners =
        new LinkedList<PtProgrammerControllerListener>();

    private final JFrame parent;

    private final NodeInterface node;

    private final int x;

    private final int y;

    private final PtProgrammerModel model = new PtProgrammerModel();

    private PtProgrammerView ptProgrammerView;

    @Autowired
    private ConnectionService connectionService;

    private MessageAdapter messageAdapter;

    private static AtomicBoolean singleton = new AtomicBoolean();

    public PtProgrammerController(final NodeInterface node, JFrame parent, int x, int y) {
        this.parent = parent;
        this.node = node;
        this.x = x;
        this.y = y;

        // this.compDispMessages = new CompositeDisposable();
    }

    public static boolean isOpened() {
        return singleton.get();
    }

    private void setOpened(boolean opened) {
        singleton.set(opened);
    }

    public void addPtProgrammerControllerListener(PtProgrammerControllerListener l) {
        listeners.add(l);
    }

    private void fireClose() {
        for (PtProgrammerControllerListener l : listeners) {
            l.close();
        }

        // reset the opened flag
        setOpened(false);
    }

    private void fireSendRequest(PtOperation operation, int cvNumber, int cvValue) {

        LOGGER.info("Send CV request, operation: {}, cvNumber: {}, value: {}", operation, cvNumber, cvValue);

        CommandStationPt opCode = CommandStationPt.valueOf(ByteUtils.getLowByte(operation.getType()));

        // clear the stored cv value in the programmer model
        model.clearCvValue();

        LOGGER.info("Prepared opCode: {}", opCode);

        for (PtProgrammerControllerListener l : listeners) {
            l.sendRequest(node, opCode, cvNumber, cvValue);
        }
    }

    /**
     * @param activateProgMode
     *            activate the programming mode
     */
    private void fireSendCommandStationStateRequest(boolean activateProgMode) {

        // TODO if we switch to PROG mode we must make sure the booster is on! Check!
        CommandStationStatus commandStationState = null;
        if (activateProgMode) {
            commandStationState = CommandStationStatus.PROG;
        }
        else {
            CommandStationState requestedCommandStationState = node.getNode().getRequestedCommandStationState();
            if (requestedCommandStationState == null) {
                commandStationState = CommandStationStatus.GO_IGN_WD;
            }
            else {
                commandStationState = CommandStationStatus.valueOf(requestedCommandStationState);
            }
        }
        LOGGER
            .info("Set the new command station state: {}, activateProgMode: {}", commandStationState, activateProgMode);

        for (PtProgrammerControllerListener l : listeners) {
            l.sendCommandStationStateRequest(node, commandStationState);
        }
    }

    private CommandStationStatus fireGetCommandStationStateRequest() {

        LOGGER.info("Get the commandStationState");

        for (PtProgrammerControllerListener l : listeners) {
            CommandStationStatus commandStationState = l.getCurrentCommandStationState(node);
            return commandStationState;
        }
        return null;
    }

    public void start(DockingDesktop desktop) {

        this.messageAdapter = new MessageAdapter(connectionService) {

            @Override
            protected void prepareMessageMap(
                Map<Class<? extends AbstractMessageEvent>, MessageEventConsumer<AbstractMessageEvent, NodeInterface>> messageActionMap) {
                LOGGER.info("Prepare the message map.");

                messageActionMap.put(CommandStationStateMessageEvent.class, (evt, node) -> {
                    CommandStationStateMessageEvent event = (CommandStationStateMessageEvent) evt;

                    CommandStationState commandStationState = event.getCommandStationState();
                    byte[] address = event.getAddress();

                    LOGGER
                        .info("The command station state has changed, address: {}, state: {}", address,
                            commandStationState);

                    if (Arrays.equals(node.getNode().getAddr(), address)) {
                        LOGGER.info("The state of the selected command station node has changed.");

                        model.setCommandStationState(commandStationState);
                    }
                    else {
                        LOGGER.warn("Another command station has changed the state.");
                    }
                });

                messageActionMap.put(CommandStationProgStateMessageEvent.class, (evt, node) -> {
                    CommandStationProgStateMessageEvent event = (CommandStationProgStateMessageEvent) evt;

                    CommandStationProgState commandStationProgState = event.getCommandStationProgState();
                    byte[] address = event.getAddress();
                    int remainingTime = event.getRemainingTime();
                    int cvNumber = event.getCvNumber();
                    int cvData = event.getCvData();

                    LOGGER
                        .info(
                            "CV was received, node addr: {}, progState: {}, remainingTime: {}, cvNumber: {}, cvData: {}",
                            address, commandStationProgState, remainingTime, cvNumber, cvData);

                    writeCommandStationProgState(commandStationProgState, remainingTime, cvNumber, cvData);
                });
            }

            @Override
            protected void onDisconnect() {

                if (ptProgrammerView != null) {

                    final PtProgrammerView view = ptProgrammerView;
                    SwingUtils.executeInEDT(() -> view.close());

                    ptProgrammerView = null;
                }

                super.onDisconnect();
            }
        };
        messageAdapter.setNode(node);
        messageAdapter.start();

        // create the view
        ptProgrammerView = new PtProgrammerView(model);
        ptProgrammerView.addPtProgrammerViewListener(new PtProgrammerViewListener() {
            @Override
            public void close() {

                // compDispMessages.dispose();
                messageAdapter.dispose();

                fireClose();
            }

            @Override
            public void sendRequest(PtOperation operation, int cvNumber, int cvValue) {
                fireSendRequest(operation, cvNumber, cvValue);
            }

            @Override
            public void sendCommandStationStateRequest(boolean activateProgMode) {
                fireSendCommandStationStateRequest(activateProgMode);
            }

            @Override
            public CommandStationStatus getCurrentCommandStationState() {
                return fireGetCommandStationStateRequest();
            }
        });

        LOGGER.info("Initialize the view.");
        setOpened(true);

        ptProgrammerView.showDialog(parent, x, y);

    }

    private void writeCommandStationProgState(
        CommandStationProgState commandStationProgState, int remainingTime, int cvNumber, int cvValue) {
        model.updateCommandStationProgResult(commandStationProgState, remainingTime, cvNumber, cvValue);
    }
}
