package org.bidib.wizard.mvc.accessory.view.panel;

import java.text.Format;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.enums.AccessoryAddressingEnum;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.value.BindingConverter;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.base.Preconditions;

/**
 * Converts Values to Strings and vice-versa using a given Format.
 */
public final class RCN213Converter implements BindingConverter<Number, String> {
    private static final Logger LOGGER = LoggerFactory.getLogger(RCN213Converter.class);

    /**
     * Holds the {@code Format} used to format and parse.
     */
    private final Format format;

    private final ValueModel addressingModel;

    // Instance Creation **************************************************

    /**
     * Constructs a {@code StringConverter} on the given subject using the specified {@code Format}.
     * 
     * @param format
     *            the {@code Format} used to format and parse
     * @throws NullPointerException
     *             if the subject or the format is null.
     */
    public RCN213Converter(ValueModel addressingModel, Format format) {
        this.format = Preconditions.checkNotNull(format, "The format must not be null.");
        this.addressingModel = addressingModel;
    }

    // Implementing Abstract Behavior *************************************

    /**
     * Formats the source value and returns a String representation.
     * 
     * @param sourceValue
     *            the source value
     * @return the formatted sourceValue
     */
    @Override
    public String targetValue(Number sourceValue) {
        if (sourceValue != null) {
            if (sourceValue instanceof Byte) {
                Byte byteValue = (Byte) sourceValue;
                sourceValue = ByteUtils.getInt(byteValue);
            }

            LOGGER
                .info("Converting sourceValue: {}, addressingModel.value: {}", sourceValue, addressingModel.getValue());
            if (AccessoryAddressingEnum.RCN_213 == addressingModel.getValue()) {
                // Aus Gruenden der Kompatibilitaet zu existierenden Zentralen ist die erste angesprochene
                // Adresse 4 = 1000-0001 1111-D00R. Diese Adresse wird in Anwenderdialogen als
                // Adresse 1 dargestellt.

                int value = sourceValue.intValue() + 3;
                LOGGER.info("RCN-213 is active, added 3 to get the direct address: {}", value);
                return format.format(value);
            }

            return format.format(sourceValue);
        }
        return null;
    }

    /**
     * Parses the given String encoding and sets it as the subject's new value. Silently catches {@code ParseException}.
     * 
     * @param targetValue
     *            the value to be converted and set as new subject value
     */
    @Override
    public Number sourceValue(String targetValue) {
        try {
            if (StringUtils.isNotBlank(targetValue)) {
                // return format.parseObject((String) targetValue);
                return Integer.valueOf(targetValue);
            }
        }
        catch (NumberFormatException e) {
            LOGGER.warn("Cannot parse the target value {}", targetValue);
        }
        return null;
    }
}
