package org.bidib.wizard.mvc.locolist.model;

import java.util.Collections;
import java.util.List;
import java.util.UUID;

import javax.swing.SwingUtilities;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.bidib.jbidibc.messages.DriveState;
import org.bidib.jbidibc.messages.enums.DirectionEnum;
import org.bidib.jbidibc.messages.utils.ConversionUtils;
import org.bidib.wizard.api.notification.LocoListChangeAction;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.locolist.controller.listener.LocoTableControllerListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class LocoTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(LocoTableModel.class);

    public static final String PROPERTY_CS_NODE_SELECTED = "csNodeSelected";

    private ArrayListModel<LocoListModel> locoArrayList;

    private boolean csNodeSelected;

    public LocoTableModel(final LocoTableControllerListener locoTableController) {
        this.locoArrayList = new ArrayListModel<>();
    }

    public void setLocoListModel(final List<LocoListModel> locoModelList) {
        this.locoArrayList.clear();
        this.locoArrayList.addAll(locoModelList);

        // locoArrayList.fireContentsChanged(0, this.locoArrayList.isEmpty() ? 0 : this.locoArrayList.getSize() - 1);
    }

    public void processLocoAction(final LocoListChangeAction ca) {
        LOGGER.debug("Process the locoList action: {}", ca);

        switch (ca.getLocoAction()) {
            case CLEAR:
                this.locoArrayList.clear();
                break;
            case ADD:
                this.locoArrayList.add(ca.getLocoListModel());
                break;
            case DELETE:
                this.locoArrayList.remove(ca.getLocoListModel());
                break;
            case UPDATE:
                // TODO keep the entry and just update the properties
                this.locoArrayList.remove(ca.getLocoListModel());
                this.locoArrayList.add(ca.getLocoListModel());
                break;
        }
    }

    public void addLoco(final DriveState driveState) {
        synchronized (locoArrayList) {
            int locoAddress = driveState.getAddress();
            final LocoListModel loco = new LocoListModel(locoAddress, UUID.randomUUID().toString());
            if (!locoArrayList.contains(loco)) {
                LOGGER.info("Add loco to loco list: {}, driveState: {}", loco, driveState);
                toLocoModel(driveState, loco);
                locoArrayList.add(loco);
            }
            else {
                LOGGER.warn("Loco is already in loco list: {}", loco);
                LocoListModel existing = IterableUtils.find(locoArrayList, new Predicate<LocoListModel>() {

                    @Override
                    public boolean evaluate(LocoListModel model) {
                        return model.equals(loco);
                    }

                });
                toLocoModel(driveState, existing);

                fireLocoChanged(existing);
            }
        }
    }

    private void fireLocoChanged(final LocoListModel locoModel) {
        LOGGER.debug("The loco address has been changed.");
        int index = locoArrayList.indexOf(locoModel);
        locoArrayList.fireContentsChanged(index);
    }

    private void toLocoModel(final DriveState driveState, final LocoListModel loco) {
        // update members of loco
        loco.setSpeed(driveState.getSpeed() & 0x7F);

        loco
            .setDirection((driveState.getDirection() == DirectionEnum.FORWARD ? DirectionStatus.FORWARD
                : DirectionStatus.BACKWARD));

        loco.setSpeedSteps(SpeedSteps.fromBidibFormat(driveState.getAddressFormat()));

        loco.setFunctions(ConversionUtils.convertFunctions(driveState.getFunctions()));

        loco.setOutputActive(driveState.getOutputActive() > 0);
    }

    public void removeLoco(int locoAddress) {
        synchronized (locoArrayList) {
            LOGGER.info("Remove loco from loco list: {}", locoAddress);

            final LocoListModel loco = findLocoByAddress(locoArrayList, locoAddress);

            // int index = locoArrayList.indexOf(new LocoListModel(locoAddress));
            // if (index > -1) {
            if (loco != null) {
                // LocoListModel loco = locoArrayList.remove(index);
                boolean removed = locoArrayList.remove(loco);
                LOGGER.info("Removed loco from locoArrayList: {}", removed);

                // TODO remove loco from service
            }
        }
    }

    private static LocoListModel findLocoByAddress(final List<LocoListModel> locoList, int locoAddress) {
        return locoList.stream().filter(llm -> llm.getAddress() == locoAddress).findFirst().orElse(null);
    }

    public void removeAllLocos() {
        synchronized (locoArrayList) {
            LOGGER.info("Remove all locos from loco list.");

            if (SwingUtilities.isEventDispatchThread()) {

                internalRemoveAllLocos();
            }
            else {
                SwingUtilities.invokeLater(() -> internalRemoveAllLocos());
            }
        }
    }

    private void internalRemoveAllLocos() {
        locoArrayList.clear();
    }

    public void setDriveState(byte[] address, DriveState driveState) {
        LOGGER.info("Drive state was delivered: {}", driveState);

        if (driveState.getAddress() > 0) {
            addLoco(driveState);
        }

    }

    public ArrayListModel<LocoListModel> getLocoListModel() {
        return locoArrayList;
    }

    public List<LocoListModel> getLocos() {
        if (locoArrayList == null) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(locoArrayList);
    }

    public boolean isCsNodeSelected() {
        return csNodeSelected;
    }

    public void setCsNodeSelected(boolean csNodeSelected) {
        boolean oldValue = this.csNodeSelected;
        this.csNodeSelected = csNodeSelected;

        if (SwingUtilities.isEventDispatchThread()) {
            firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected);
        }
        else {
            SwingUtilities.invokeLater(() -> firePropertyChange(PROPERTY_CS_NODE_SELECTED, oldValue, csNodeSelected));
        }
    }
}
