package org.bidib.wizard.mvc.main.controller;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Supplier;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.message.netbidib.NetBidibLinkData.PairingStatus;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.jbidibc.netbidib.NetBidibContextKeys;
import org.bidib.jbidibc.netbidib.client.pairingstates.PairingStateEnum;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.dialog.PairingDialog;
import org.bidib.wizard.dialog.PairingDialog.PairingDialogType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultPairingController implements PairingController {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultPairingController.class);

    private final Supplier<JFrame> parent;

    private final StatusBar statusBar;

    private PairingDialog pairingDialog;

    private final ConnectionService connectionService;

    private final Supplier<String> connectionId;

    private final ScheduledExecutorService netBidibPublisherWorker;

    public DefaultPairingController(final Supplier<JFrame> parent, final StatusBar statusBar,
        final ConnectionService connectionService, final Supplier<String> connectionId) {
        LOGGER.info("Created PairingController, provided connectionId: {}", connectionId);
        this.parent = parent;
        this.statusBar = statusBar;
        this.connectionService = connectionService;
        this.connectionId = connectionId;

        netBidibPublisherWorker =
            Executors
                .newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().setNameFormat("netBidibPublisherWorkers-thread-%d").build());
    }

    /**
     * Publish connection action.
     * 
     * @param connectionId
     *            the connectionId.
     * @param messageKey
     *            the message key
     */
    @Override
    public void publishAction(final String connectionId, String messageKey, final Context context) {
        LOGGER
            .info("Notify the action, connectionId: {}, messageKey: {}, context: {}", connectionId, messageKey,
                context);

        if (connectionId
            .equals(this.connectionId.get()) /* || ConnectionRegistry.CONNECTION_ID_GATEWAY.equals(connectionId) */) {

            if (messageKey.startsWith(NetBidibContextKeys.KEY_ACTION_PAIRING_PREFIX)) {
                LOGGER.info("This is a pairing action: {}", messageKey);

                SwingUtilities.invokeLater(() -> {

                    Long uniqueId = context.get(NetBidibContextKeys.KEY_DESCRIPTOR_UID, Long.class, null);
                    String connectionType = context.get(NetBidibContextKeys.KEY_CONNECTION_TYPE, String.class, null);

                    LOGGER
                        .info("Current uniqueId: {}, connectionType: {}", ByteUtils.formatHexUniqueId(uniqueId),
                            connectionType);

                    switch (messageKey) {
                        case NetBidibContextKeys.KEY_ACTION_PAIRING_STATE:

                            if (context
                                .get("PAIRING_STATE", PairingStateEnum.class,
                                    PairingStateEnum.Unpaired) == PairingStateEnum.Unpaired) {
                                LOGGER
                                    .info(
                                        "Unpaired status was signalled. Send the pairing request to the remote partner.");

                                // check if this is only required if we are client

                                if (!"server".equals(connectionType)) {
                                    // signal that the pairing request is sent to the remote partner
                                    signalPairingRequest(connectionId);
                                }
                                else {
                                    LOGGER.info("Don't send pairing request to remote partner if we are server.");
                                }

                                // show the pairing required dialog
                                if (pairingDialog == null) {
                                    LOGGER.info("Show the pairing dialog.");

                                    String statusText = Resources.getString(DefaultPairingController.class, messageKey);
                                    statusBar.setStatusText(statusText, StatusBar.DISPLAY_NORMAL);

                                    pairingDialog =
                                        new PairingDialog(this.parent.get(), PairingDialogType.PairingRequired, context,
                                            true, (accepted) -> {
                                                // if (pairingDialog != null && pairingDialog
                                                // .getPairingDialogType() == PairingDialogType.PairingRequested) {

                                                if (Boolean.FALSE == accepted) {

                                                    LOGGER
                                                        .info(
                                                            "Disconnect the connection because the user cancelled the pairing dialog in state KEY_ACTION_PAIRING_STATE. Current connectionId: {}",
                                                            connectionId);
                                                    connectionService.disconnect(connectionId);
                                                }
                                                // }
                                                // else {
                                                // LOGGER.info("Skip disconnect after close pairing dialog.");
                                                // }

                                            });

                                    pairingDialog.setAlwaysOnTop(true);
                                    pairingDialog.setVisible(true);

                                    statusBar.setStatusText(null, StatusBar.DISPLAY_NORMAL);
                                }
                                else {
                                    LOGGER.warn("Pairing dialog is already assigned.");
                                }
                            }
                            else {
                                LOGGER
                                    .info(
                                        "Paired status was signalled. Hide the pairing dialog if shown, pairingDialog: {}",
                                        pairingDialog);

                                // show the pairing passed dialog
                                if (pairingDialog != null) {
                                    LOGGER.info("The pairing has passed. Hide the pairing dialog.");
                                    pairingDialog.signalPaired();
                                    // pairingDialog.setVisible(false);
                                    pairingDialog = null;
                                }
                            }

                            break;
                        case NetBidibContextKeys.KEY_ACTION_PAIRING_REQUIRED:
                            // show the pairing required dialog
                            if (pairingDialog == null) {
                                LOGGER.info("Show the pairing dialog.");

                                String statusText = Resources.getString(DefaultPairingController.class, messageKey);
                                this.statusBar.setStatusText(statusText, StatusBar.DISPLAY_NORMAL);

                                pairingDialog =
                                    new PairingDialog(this.parent.get(), PairingDialogType.PairingRequired, context,
                                        true, (accepted) -> {

                                            // if (pairingDialog != null && pairingDialog
                                            // .getPairingDialogType() == PairingDialogType.PairingRequested) {
                                            if (Boolean.FALSE == accepted) {

                                                LOGGER
                                                    .info(
                                                        "Disconnect the connection because the user cancelled the pairing dialog in state KEY_ACTION_PAIRING_REQUIRED. Current connectionId: {}",
                                                        connectionId);
                                                connectionService.disconnect(connectionId);
                                            }
                                            // }
                                            // else {
                                            // LOGGER.info("Skip disconnect after close pairing dialog.");
                                            // }
                                        });

                                pairingDialog.setAlwaysOnTop(true);
                                pairingDialog.setVisible(true);

                                this.statusBar.setStatusText(null, StatusBar.DISPLAY_NORMAL);
                            }
                            else {
                                LOGGER.warn("Pairing dialog is already assigned.");
                            }
                            break;
                        case NetBidibContextKeys.KEY_ACTION_PAIRING_REQUESTED:
                            // show the pairing requested dialog

                            if (NetBidibContextKeys.VALUE_CONNECTION_TYPE_SERVER.equals(connectionType)) {

                                // we act as server for this connection

                                if (pairingDialog != null) {
                                    LOGGER.info("The pairing is requested. Hide the existing pairing dialog.");
                                    pairingDialog.setVisible(false);
                                    pairingDialog = null;
                                }

                                if (pairingDialog == null) {
                                    LOGGER.info("Show the pairing requested dialog.");

                                    String statusText = Resources.getString(DefaultPairingController.class, messageKey);
                                    this.statusBar.setStatusText(statusText, StatusBar.DISPLAY_NORMAL);

                                    pairingDialog =
                                        new PairingDialog(this.parent.get(), PairingDialogType.PairingRequested,
                                            context, true, (accepted) -> {
                                                if (Boolean.FALSE == accepted) {
                                                    if (pairingDialog != null && pairingDialog
                                                        .getPairingDialogType() == PairingDialogType.PairingRequested) {

                                                        LOGGER
                                                            .info(
                                                                "Disconnect the connection because the user cancelled the pairing dialog in state KEY_ACTION_PAIRING_REQUESTED. Current connectionId: {}",
                                                                connectionId);

                                                        // we must signal the pairing status unpaired
                                                        signalPairingStatus(connectionId, uniqueId,
                                                            PairingStatus.UNPAIRED);

                                                        // TODO this disconnects the gateway connection instead of the
                                                        // connection to the remote partner
                                                        // connectionService.disconnect(connectionId);
                                                    }
                                                    else {
                                                        LOGGER.info("Skip disconnect after close pairing dialog.");
                                                    }

                                                }
                                                else {
                                                    LOGGER.info("We accepted the pairing.");
                                                    signalPairingStatus(connectionId, uniqueId, PairingStatus.PAIRED);

                                                }
                                            });

                                    pairingDialog.setAlwaysOnTop(true);
                                    pairingDialog.setVisible(true);

                                    this.statusBar.setStatusText(null, StatusBar.DISPLAY_NORMAL);
                                }
                                else {
                                    LOGGER.warn("Pairing dialog is already assigned.");

                                    LOGGER.info("We accept the pairing immediately.");
                                    signalPairingStatus(connectionId, uniqueId, PairingStatus.PAIRED);
                                }
                            }
                            else {
                                LOGGER.info("We accept the pairing immediately.");
                                signalPairingStatus(connectionId, uniqueId, PairingStatus.PAIRED);
                            }
                            break;
                        case NetBidibContextKeys.KEY_ACTION_PAIRING_PASSED:
                            // show the pairing passed dialog
                            if (pairingDialog != null) {
                                LOGGER.info("The pairing has passed. Hide the pairing dialog.");
                                pairingDialog.setVisible(false);
                                pairingDialog = null;
                            }
                            break;
                        case NetBidibContextKeys.KEY_ACTION_PAIRING_FAILED:
                            // show the pairing failed dialog
                            if (pairingDialog != null) {
                                LOGGER.info("The pairing has failed. Hide the pairing dialog.");
                                pairingDialog.setVisible(false);
                                pairingDialog = null;
                            }
                            break;
                        default:
                            break;
                    }

                });
            }
        }
    }

    private void signalPairingStatus(String connectionId, Long uniqueId, PairingStatus pairingStatus) {
        LOGGER
            .info("Signal the pairing status to the connection: {}, uniqueId: {}", pairingStatus,
                ByteUtils.getUniqueIdAsString(uniqueId));

        netBidibPublisherWorker
            .submit(() -> connectionService.signalPairingStatus(connectionId, uniqueId, pairingStatus));

    }

    private void signalPairingRequest(String connectionId) {
        LOGGER.info("Signal the pairing request to the connection.");

        netBidibPublisherWorker.submit(() -> connectionService.signalPairingRequest(connectionId));

    }

    @Override
    public void closeDialogs() {
        LOGGER.info("Close the pairing dialogs.");

        if (pairingDialog != null) {
            LOGGER.info("The pairing has passed. Hide the pairing dialog.");
            pairingDialog.setVisible(false);
            pairingDialog = null;
        }
    }
}
