package org.bidib.wizard.mvc.main.controller;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.CommandStationPom;
import org.bidib.jbidibc.messages.enums.PomAcknowledge;
import org.bidib.jbidibc.messages.enums.PomAddressTypeEnum;
import org.bidib.jbidibc.messages.enums.PomOperation;
import org.bidib.jbidibc.messages.enums.PomProgState;
import org.bidib.jbidibc.messages.exception.NoAnswerException;
import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.helpers.DefaultContext;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.connection.AbstractMessageEvent;
import org.bidib.wizard.api.model.connection.event.OccupancyCvMessageEvent;
import org.bidib.wizard.api.model.connection.event.OccupancyDynStateMessageEvent;
import org.bidib.wizard.api.model.connection.event.OccupancySpeedMessageEvent;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.core.LocoService;
import org.bidib.wizard.api.service.node.CommandStationService;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.connection.MessageAdapter;
import org.bidib.wizard.core.model.connection.MessageEventConsumer;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.model.loco.LocoModel;
import org.bidib.wizard.model.loco.listener.LocoModelListener;
import org.bidib.wizard.model.locolist.LocoListModel;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.RfBasisMode;
import org.bidib.wizard.model.status.SpeedLevel;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.loco.controller.LocoControlListener;
import org.bidib.wizard.mvc.loco.model.LocoConfigModel;
import org.bidib.wizard.mvc.loco.model.SpeedometerModel;
import org.bidib.wizard.mvc.loco.model.SpeedometerModel.SpeedMeasurementStage;
import org.bidib.wizard.mvc.loco.model.command.PomRequestProcessor;
import org.bidib.wizard.mvc.loco.model.command.SpeedometerPomCommand;
import org.bidib.wizard.mvc.loco.view.PomProgrammerRequestListener;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.LocoPanelView;
import org.bidib.wizard.mvc.pom.model.PomProgrammerModel;
import org.bidib.wizard.mvc.pom.model.ProgCommandAwareBeanModel;
import org.bidib.wizard.mvc.pom.model.command.PomOperationCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import io.reactivex.rxjava3.core.SingleObserver;
import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;

public class LocoPanelController {
    private static final Logger LOGGER = LoggerFactory.getLogger(LocoPanelController.class);

    private final MainModel mainModel;

    private final LocoConfigModel locoConfigModel = new LocoConfigModel();

    private final SpeedometerModel speedometerModel = new SpeedometerModel();

    private LocoModelListener locoModelListener;

    private LocoPanelView locoPanelView;

    @Autowired
    private ConnectionService connectionService;

    @Autowired
    private CommandStationService commandStationService;

    @Autowired
    private NodeService nodeService;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private ConsoleService consoleService;

    @Autowired
    private LocoService locoService;

    private MessageAdapter messageAdapter;

    private final ScheduledExecutorService pomRequestProcessorWorker;

    private final LocoModel locoModel;

    public LocoPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        this.locoModel = new LocoModel(null);

        pomRequestProcessorWorker =
            Executors
                .newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().setNameFormat("pomRequestProcessorWorkers-thread-%d").build());
    }

    public LocoPanelView createPanel() {

        final NodeInterface node = mainModel.getSelectedNode();
        LOGGER.info("Create the LocoView panel, selected node: {}", node);

        if (node != null) {
            if (ProductUtils.isRFBasisNode(node.getUniqueId()) || ProductUtils.isSpeedometer(node.getUniqueId())) {
                LOGGER.info("The default speed steps for the decoders on the RF Basis Node or Speedometer is 128.");
                locoConfigModel.setSpeedSteps(SpeedSteps.DCC128);
                locoConfigModel.setCarControlEnabled(true);
            }
            else {
                locoConfigModel.setSpeedSteps(SpeedSteps.DCC128);
                locoConfigModel.setCarControlEnabled(false);
            }
        }
        else {
            LOGGER.info("No node selected while create the LocoView panel. Do not initialize the speed steps.");
        }

        this.messageAdapter = new MessageAdapter(connectionService) {

            @Override
            protected void prepareMessageMap(
                Map<Class<? extends AbstractMessageEvent>, MessageEventConsumer<AbstractMessageEvent, NodeInterface>> messageActionMap) {
                LOGGER.info("Prepare the message map.");

                messageActionMap.put(OccupancyCvMessageEvent.class, (evt, node) -> {
                    OccupancyCvMessageEvent event = (OccupancyCvMessageEvent) evt;

                    byte[] address = event.getAddress();
                    PomAddressData decoderAddress = event.getAddressData();
                    int cvNumber = event.getCvNumber();
                    int cvData = event.getCvValue();

                    LOGGER
                        .debug("CV was received, node addr: {}, decoder address: {}, cvNumber: {}, cvData: {}", address,
                            decoderAddress, cvNumber, cvData);

                    updatePomProgState(PomProgState.POM_PROG_OKAY, decoderAddress, cvNumber, cvData);
                });

                messageActionMap.put(OccupancyDynStateMessageEvent.class, (evt, node) -> {
                    OccupancyDynStateMessageEvent event = (OccupancyDynStateMessageEvent) evt;

                    final AddressData decoderAddress = event.getDecoderAddress();
                    int dynNumber = event.getDynNumber();
                    int dynValue = event.getDynValue();

                    LOGGER
                        .debug("dynState, decoderAddress: {}, dynNumber: {}, dynValue: {}", decoderAddress, dynNumber,
                            dynValue);

                    SwingUtilities.invokeLater(() -> {
                        if (locoModel.getAddress() != null && decoderAddress.getAddress() == locoModel.getAddress()
                            && dynNumber == 3) {
                            LOGGER.info("The dynState is forwarded to loco model.");
                            locoModel.setDynStateEnergy(dynValue);

                            if (speedometerModel.isActive()) {
                                LOGGER.info("Update the DYN_STATE in speedometer model.");
                                speedometerModel.setDynStateEnergy(dynValue);
                            }
                            else {
                                LOGGER.info("Speedometer model is not in state active.");
                            }
                        }
                        else {
                            LOGGER.debug("The dynState ist not forwarded to model");
                        }
                    });
                });

                messageActionMap.put(OccupancySpeedMessageEvent.class, (evt, node) -> {
                    OccupancySpeedMessageEvent event = (OccupancySpeedMessageEvent) evt;

                    final AddressData addressData = event.getAddressData();
                    int speed = event.getSpeed();

                    // TODO must check the address and the address of the selected node and make sure the speed is
                    // delivered from the correct decoder because we can have multiple domains

                    LOGGER.debug("Update the reported speed: {}", speed);

                    SwingUtilities.invokeLater(() -> {
                        if (locoModel.getAddress() != null && addressData.getAddress() == locoModel.getAddress()) {

                            if (speedometerModel.isActive()) {
                                LOGGER.info("Update the reported speed in speedometer model: {}", speed);

                                speedometerModel.setReportedSpeed(speed);

                                // TODO remove after testing
                                Integer scale = speedometerModel.getCv37Scale();
                                if (scale == null) {
                                    scale = 1;
                                }
                                double factor = ((double) (scale * 3600)) / 1000000;
                                LOGGER.debug("Current scale: {}, factor: {}", scale, factor);

                                ConsoleController.ensureConsoleVisible();

                                if (scale > 1) {
                                    consoleService
                                        .addConsoleLine(ConsoleColor.blue, "Current reported speed: " + speed
                                            + " mm/s -> " + (int) (speed * factor) + " km/h");
                                }
                                else {
                                    consoleService
                                        .addConsoleLine(ConsoleColor.blue,
                                            "Current reported speed: " + speed + " mm/s");
                                }
                            }
                            else {
                                LOGGER.debug("Speedometer model is not in state active.");
                                locoModel.setReportedSpeed(speed);
                            }
                        }
                        else {
                            LOGGER.debug("Reported speed: {}, address does not match.", speed);
                        }
                    });
                });
            }

            @Override
            protected void onDisconnect() {

                // if (ptProgrammerView != null) {
                // ptProgrammerView.close();
                //
                // ptProgrammerView = null;
                // }

                super.onDisconnect();
            }
        };

        LOGGER.info("Set the node in the messageAdapter: {}", node);
        messageAdapter.setNode(node);
        // delayed start
        SwingUtilities.invokeLater(() -> messageAdapter.start());

        final PomProgrammerRequestListener pomProgrammerRequestListener = new PomProgrammerRequestListener() {

            @Override
            public void sendRequest(PomAddressData decoderAddress, PomOperation operation, int cvNumber, int cvValue) {
                LOGGER.info("Send the POM request.");

                CommandStationPom opCode = CommandStationPom.valueOf(ByteUtils.getLowByte(operation.getType()));

                final NodeInterface selectedNode = mainModel.getSelectedNode();

                if (selectedNode != null && selectedNode.getCommandStationNode() != null) {
                    final CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();

                    PomAcknowledge pomAck =
                        commandStationService
                            .sendCvPomRequest(ConnectionRegistry.CONNECTION_ID_MAIN, commandStationNode, decoderAddress,
                                opCode, cvNumber, cvValue);
                    LOGGER.debug("Received pomAck: {}", pomAck);
                }
                else {
                    LOGGER.warn("No command station node selected.");
                }
            }
        };

        final LocoControlListener locoControlListener = new LocoControlListener() {

            @Override
            public void setSpeed(SpeedLevel speedLevel) {
                // LOGGER.info("Set speed: {}", speed);

                LocoPanelController.this.setSpeed(speedLevel);
            }

            @Override
            public void setFunction(int index, boolean value) {
                // TODO Auto-generated method stub

            }

            @Override
            public void setBinaryState(int state, boolean value) {
                // TODO Auto-generated method stub

            }

            @Override
            public void setSpeedSteps(SpeedSteps speedSteps) {
                // LOGGER.info("Set the speedSteps: {}", speedSteps);

                LocoPanelController.this.setSpeedSteps(speedSteps);
            };

            @Override
            public void clearLoco() {
            };
        };

        locoPanelView =
            new LocoPanelView(locoModel, speedometerModel, pomProgrammerRequestListener, settingsService,
                locoControlListener, newLocoAddress -> changeLocoAddress(newLocoAddress));

        locoModelListener = new LocoModelListener() {

            @Override
            public void functionChanged(int index, boolean value) {

                NodeInterface selectedNode = mainModel.getSelectedNode();

                if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

                    if (locoModel == null) {
                        LOGGER.warn("No locoModel and therefore address available.");
                        return;
                    }

                    // Prepare the function group index
                    int functionGroupIndex = 0;

                    if (index < 5) {
                        functionGroupIndex = 0;
                    }
                    else if (index < 9) {
                        functionGroupIndex = 1;
                    }
                    else if (index < 13) {
                        functionGroupIndex = 2;
                    }
                    else if (index < 21) {
                        functionGroupIndex = 3;
                    }
                    else if (index < 29) {
                        functionGroupIndex = 4;
                    }
                    else {
                        LOGGER.warn("Function > 28 uses binary state operation.");

                        binaryStateChanged(index, value);
                        return;
                    }

                    BitSet activeFunctions = new BitSet(functionGroupIndex + 1);
                    LOGGER.debug("functions have changed, functionGroupIndex: {}", functionGroupIndex);
                    activeFunctions.set(functionGroupIndex, true);

                    final CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();

                    // create the context
                    final Context context = new DefaultContext();
                    registerActiveRfBasis(context);

                    // create speedLevel without speed
                    final SpeedLevel speedLevel =
                        new SpeedLevel(null, locoModel.getSpeedSteps(), locoModel.getDirection());

                    locoService
                        .setSpeed(ConnectionRegistry.CONNECTION_ID_MAIN, commandStationNode, locoModel.getAddress(),
                            speedLevel, activeFunctions, locoModel.getFunctions(), context);
                }
                else {
                    LOGGER.warn("No command station node selected.");
                }
            }

            @Override
            public void binaryStateChanged(int stateNumber, boolean value) {
                LOGGER
                    .info("Send the binary state, address: {}, stateNumber: {}, value: {}", locoModel.getAddress(),
                        stateNumber, value);

                if (locoModel.getAddress() == null) {
                    LOGGER.warn("No address available.");
                    return;
                }

                final NodeInterface selectedNode = mainModel.getSelectedNode();
                if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

                    final CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();

                    // TODO we must send the binary state for the change of active RF base to all bases but the others
                    // only to the selected base

                    final Context context = new DefaultContext();

                    if (stateNumber < 512 && stateNumber > 517) {
                        registerActiveRfBasis(context);
                    }

                    // count the CS_BIN_STATE messages that are sent
                    locoModel.incCounterCsBinState();

                    commandStationService
                        .setBinaryState(ConnectionRegistry.CONNECTION_ID_MAIN, commandStationNode,
                            locoModel.getAddress(), stateNumber, value, context);
                }
                else {
                    LOGGER.warn("No node selected.");
                }
            }
        };
        locoModel.addLocoModelListener(locoModelListener);

        speedometerModel
            .addPropertyChangeListener(SpeedometerModel.PROPERTYNAME_SPEEDMEASUREMENTSTAGE,
                new PropertyChangeListener() {

                    @Override
                    public void propertyChange(PropertyChangeEvent evt) {
                        SpeedMeasurementStage stage = speedometerModel.getSpeedMeasurementStage();
                        LOGGER
                            .info("The speedmeasurement stage has changed: {}, active: {}", stage,
                                speedometerModel.isActive());

                        final NodeInterface selectedNode = mainModel.getSelectedNode();
                        if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

                            final CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();
                            switch (stage) {
                                case WAIT_FOR_CAR_SPEED_AND_BATTERY_RESPONSE:
                                    LOGGER
                                        .info("Activate lights and send speed 0 to decoder with address: {}",
                                            locoModel.getAddress());

                                    // reset the speed and dynstate in the speedo model
                                    speedometerModel.setSpeed(null);
                                    speedometerModel.setDynStateEnergy(null);

                                    // TODO set the POM repeat to 1: "CP 1" --> FEATURE_GEN_POM_REPEAT

                                    // set the lights on
                                    changeLightsAndSpeed(commandStationNode, true);
                                    try {
                                        Thread.sleep(500);
                                    }
                                    catch (InterruptedException ex) {
                                        LOGGER.warn("Wait between to lights failed.", ex);
                                    }
                                    // set the lights off
                                    changeLightsAndSpeed(commandStationNode, false);

                                    triggerSpeedRequest(10);
                                    break;
                                case READ_CURRENT_CV_VALUES:
                                    LOGGER.info("Current stage is READ_CURRENT_CV_VALUES.");
                                    fireReadCvValues();
                                    break;
                                case WRITE_CV_VALUES:
                                    LOGGER.info("Current stage is WRITE_CV_VALUES.");
                                    fireWriteCvValues();
                                    break;
                                case SET_MOTOR_PID_SOFT:
                                    LOGGER.info("Current stage is SET_MOTOR_PID_SOFT.");

                                    triggerSetMotorPID(selectedNode);
                                    break;
                                case START_MEASUREMENT:
                                    LOGGER.info("Current stage is START_MEASUREMENT.");
                                    triggerMeasurement(selectedNode, speedometerModel.getCv37Scale(), true, true);

                                    // triggerSpeedRequest(126);
                                    break;
                                case SET_SPEED_MAX:
                                    LOGGER.info("Current stage is SET_SPEED_MAX.");
                                    // max speed of car decoder is 126
                                    triggerSpeedRequest(126);
                                    break;

                                case ABORTED:
                                    LOGGER.info("Current stage is ABORTED.");

                                    triggerSpeedRequest(0);
                                    waitBetweenCommands(10);
                                    triggerMeasurement(selectedNode, null, false, false);
                                    break;
                                case FINISHED:
                                    LOGGER.info("Current stage is FINISHED.");

                                    triggerSpeedRequest(0);

                                    waitBetweenCommands(10);
                                    triggerMeasurement(selectedNode, null, false, false);
                                    break;

                                case STOP_FOR_USER_INTERACTION:
                                    LOGGER.info("Current stage is STOP_FOR_USER_INTERACTION.");

                                    triggerSpeedRequest(0);

                                    waitBetweenCommands(10);
                                    triggerMeasurement(selectedNode, null, false, false);
                                    break;

                                case MEASURE_ON:
                                    triggerMeasurement(selectedNode, 1, true, false);
                                    break;
                                case MEASURE_OFF:
                                    triggerMeasurement(selectedNode, 1, false, false);
                                    break;
                                default:
                                    LOGGER.warn("unhandled stage: {}", stage);
                                    break;
                            }
                        }
                        else {
                            LOGGER.warn("No node selected.");
                        }
                    }
                });

        mainModel.addNodeListListener(new DefaultNodeListListener() {

            @Override
            public void listChanged() {

                // // check if no more nodes
                // final NodeInterface selectedNode = mainModel.getSelectedNode();
                // if (selectedNode == null) {
                // LOGGER.info("No nodes in model. Clear speedometer values.");
                // speedometerModel.clearValues();
                // locoPanelView.clearModelValues();
                // }

            }

            @Override
            public void nodeChanged(final NodeInterface node) {
                final NodeInterface selectedNode = mainModel.getSelectedNode();
                LOGGER.info("The selected node has changed, selectedNode: {}", selectedNode);

                // clear the speedometer model
                speedometerModel.clearValues();
                locoPanelView.clearModelValues();

                if (selectedNode != null) {
                    if (ProductUtils.isRFBasisNode(selectedNode.getUniqueId())
                        || ProductUtils.isSpeedometer(selectedNode.getUniqueId())) {
                        LOGGER
                            .info(
                                "The default speed steps for the decoders on the RF Basis Node or Speedometer is 128.");
                        locoModel.setSpeedSteps(SpeedSteps.DCC128);
                        locoConfigModel.setCarControlEnabled(true);
                    }
                    else {
                        locoModel.setSpeedSteps(SpeedSteps.DCC128);
                        locoConfigModel.setCarControlEnabled(false);
                    }

                    LOGGER.info("Set the selected node in the messageAdapter: {}", selectedNode);
                    messageAdapter.setNode(selectedNode);

                    if (selectedNode.getCommandStationNode() != null) {
                        // query the current state of the command station
                        try {

                            final CompositeDisposable disp = new CompositeDisposable();

                            Disposable dispCsState =
                                selectedNode.getCommandStationNode().subscribeSubjectCommandStationState(csState -> {

                                    LOGGER.debug("Current command station state: {}", csState);

                                    // this will start the command station if the node is a speedometer

                                    if (ProductUtils.isSpeedometer(selectedNode.getUniqueId())
                                        && CommandStationStatus.isOffState(csState)) {
                                        LOGGER.info("Switch the command station of speedometer on.");

                                        commandStationService
                                            .setCommandStationState(ConnectionRegistry.CONNECTION_ID_MAIN,
                                                selectedNode.getCommandStationNode(), CommandStationStatus.GO);
                                    }
                                    disp.dispose();
                                }, error -> {

                                    LOGGER.warn("Get the current command station state failed.");
                                    disp.dispose();
                                }, () -> {
                                    LOGGER
                                        .warn(
                                            "Get the current command station state subscription has completed, node: {}",
                                            node);
                                });
                            disp.add(dispCsState);

                            commandStationService
                                .queryCommandStationState(ConnectionRegistry.CONNECTION_ID_MAIN,
                                    selectedNode.getCommandStationNode());
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Get command station state failed.", ex);
                        }
                    }
                    else {
                        LOGGER.debug("The current node is not a command station node.");
                    }
                }
                else {
                    LOGGER.debug("No node selected.");
                }
            }
        });

        return locoPanelView;
    }

    private void changeLocoAddress(Integer locoAddress) {
        LOGGER.info("Change the locoAddress: {}", locoAddress);

        if (locoAddress != null) {
            final NodeInterface selectedNode = mainModel.getSelectedNode();
            if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

                LocoListModel locoListModel =
                    locoService
                        .getOrCreateLoco(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getCommandStationNode(),
                            locoAddress);
                locoModel.setLocoListModel(locoListModel);
            }
            else {
                LOGGER.warn("Selected node is not a command station: {}", selectedNode);
                // reset the loco
                locoModel.setLocoListModel(null);
            }
        }
        else {
            // reset the loco
            locoModel.setLocoListModel(null);
        }

        // must reset the reported cell number
        locoModel.setReportedCellNumber(null);
    }

    private void setSpeed(SpeedLevel speedLevel) {
        LOGGER.info("Set loco speedLevel: {}", speedLevel);

        if (speedLevel != null) {
            try {

                final NodeInterface selectedNode = mainModel.getSelectedNode();
                if (selectedNode != null && selectedNode.getCommandStationNode() != null
                    && locoModel.getAddress() != null) {
                    LOGGER.debug("The speed in LocoModel has changed. Set the new speed value: {}", speedLevel);

                    final CommandStationNodeInterface commandStationNode = selectedNode.getCommandStationNode();

                    // create the context
                    final Context context = new DefaultContext();
                    registerActiveRfBasis(context);

                    locoService
                        .setSpeed(ConnectionRegistry.CONNECTION_ID_MAIN, commandStationNode, locoModel.getAddress(),
                            speedLevel, null, null, context);
                }
                else {
                    LOGGER.debug("No node selected or address is not available.");
                }
            }
            catch (NoAnswerException ex) {
                LOGGER.warn("Set speed failed.", ex);
            }
            catch (Exception ex) {
                LOGGER.warn("Set speed failed.", ex);
            }
        }
        else {
            LOGGER.info("No speed value delivered.");
        }
    }

    private void setSpeedSteps(SpeedSteps speedSteps) {
        LOGGER.info("Set the speedSteps: {}, locoModel: {}", speedSteps, locoModel);
        locoModel.setSpeedSteps(speedSteps);
    }

    private void triggerSetMotorPID(final NodeInterface selectedNode) {
        LOGGER.info("Set the motor PID values.");

    }

    private void triggerMeasurement(
        final NodeInterface selectedNode, Integer scale, boolean activate, boolean sendMaxSpeed) {
        LOGGER.info("Trigger the measurement, scale: {}, activate: {}", scale, activate);

        List<ConfigurationVariable> cvList = new ArrayList<>();

        cvList.add(new ConfigurationVariable("Scale", scale != null ? scale.toString() : "-1"));
        cvList.add(new ConfigurationVariable("SpeedMeasurement", activate ? "MS 1" : "MS 0"));

        LOGGER.info("Prepared cvList to send: {}", cvList);

        List<ConfigurationVariable> configVars =
            nodeService.setConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, cvList);

        LOGGER.info("Returned configVars: {}", configVars);

        if (activate && sendMaxSpeed) {
            LOGGER.info("Activate sets the new stage to SET_SPEED_MAX.");
            SwingUtilities.invokeLater(() -> {
                speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.SET_SPEED_MAX);
            });
        }
    }

    private void triggerSpeedRequest(int speed) {
        LOGGER.info("Set the speed to: {}", speed);

        try {
            locoModel.setSpeed(speed);

            locoPanelView.addLogMessage("Set the speed value to {}.", speed);
        }
        catch (NoAnswerException ex) {
            LOGGER.warn("Set speed failed.", ex);
        }
        catch (Exception ex) {
            LOGGER.warn("Set speed failed.", ex);
        }

    }

    private void updatePomProgState(
        PomProgState pomProgState, PomAddressData decoderAddress, int cvNumber, int cvValue) {
        if (speedometerModel != null) {
            PomProgrammerModel pomProgrammerModel = speedometerModel.getPomProgrammerModel();
            if (pomProgrammerModel != null) {
                pomProgrammerModel.updatePomProgResult(pomProgState, decoderAddress, cvNumber, cvValue);
            }
            else {
                LOGGER.warn("No pomProgrammerModel available in speedoMeterModel.");
            }
        }
    }

    private void changeLightsAndSpeed(final CommandStationNodeInterface selectedNode, boolean lightsOn) {
        LOGGER.info("Change the lights value and set speed to 0, lightsOn: {}", lightsOn);

        if (locoModel.getAddress() == null) {
            LOGGER.warn("No address available.");
            return;
        }

        int functionGroupIndex = 0;
        BitSet activeFunctions = new BitSet(functionGroupIndex + 1);
        activeFunctions.set(functionGroupIndex, true);

        BitSet functions = new BitSet(29);
        functions.set(0, lightsOn ? 1 : 0);

        Integer speed = 0;

        // create the context
        final Context context = new DefaultContext();
        registerActiveRfBasis(context);

        final SpeedLevel speedLevel = new SpeedLevel(speed, locoModel.getSpeedSteps(), DirectionStatus.FORWARD);

        this.locoService
            .setSpeed(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, locoModel.getAddress(), speedLevel,
                activeFunctions, functions, context);
    }

    private void waitBetweenCommands(int delay) {
        try {
            Thread.sleep(delay);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait between commands was interrupted.", ex);
        }
    }

    private void fireReadCvValues() {
        LOGGER.info("Read the values from the decoder.");

        final List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands = new ArrayList<>();

        Integer dccAddress = locoModel.getAddress();

        LOGGER.info("Prepared DCC address: {}", dccAddress);

        if (dccAddress == null) {
            LOGGER.warn("No dccAddress available.");
            return;
        }

        PomAddressData addressData = new PomAddressData(dccAddress, PomAddressTypeEnum.LOCOMOTIVE);

        LOGGER.info("Prepared addressData for CV read: {}", addressData);

        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 2, ByteUtils.getLowByte(0)));
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 5, ByteUtils.getLowByte(0)));
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 37, ByteUtils.getHighByte(0)));
        pomProgCommands
            .add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 111, ByteUtils.getHighByte(0)));

        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 35, ByteUtils.getHighByte(0)));
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 36, ByteUtils.getHighByte(0)));

        // V3: load control
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 59, ByteUtils.getHighByte(0)));
        // load the decoder hardware version before the firmware is read
        pomProgCommands
            .add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 111, ByteUtils.getHighByte(0)));

        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 61, ByteUtils.getHighByte(0)));
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 62, ByteUtils.getHighByte(0)));
        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 63, ByteUtils.getHighByte(0)));

        pomProgCommands.add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 7, ByteUtils.getHighByte(0)));
        pomProgCommands
            .add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 109, ByteUtils.getHighByte(0)));
        pomProgCommands
            .add(new SpeedometerPomCommand(addressData, PomOperation.RD_BYTE, 110, ByteUtils.getHighByte(0)));

        SingleObserver<String> completeAction = speedometerModel.getCompleteAction();
        speedometerModel.setCompleteAction(null);

        final PomRequestProcessor<?> pomRequestProcessor = speedometerModel.getPomRequestProcessor();

        pomRequestProcessorWorker
            .schedule(() -> pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction), 0,
                TimeUnit.MILLISECONDS);
        // pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction);
    }

    private void fireWriteCvValues() {
        LOGGER.info("Write the values to the decoder.");

        final List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands =
            speedometerModel.getPomProgCommands();
        speedometerModel.setPomProgCommands(null);

        SingleObserver<String> completeAction = speedometerModel.getCompleteAction();
        speedometerModel.setCompleteAction(null);

        LOGGER.info("Current pomProgCommands: {}", pomProgCommands);

        final PomRequestProcessor<?> pomRequestProcessor = speedometerModel.getPomRequestProcessor();

        pomRequestProcessorWorker
            .schedule(() -> pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction), 0,
                TimeUnit.MILLISECONDS);

    }

    private void registerActiveRfBasis(final Context context) {
        if (locoConfigModel.isCarControlEnabled() && locoModel.getActiveBase() != null) {

            RfBasisMode activeRfBase = locoModel.getActiveBase();
            if (activeRfBase.getBaseNumber() != null) {
                int activeBaseNumber = activeRfBase.getBaseNumber();

                final NodeInterface rfBasisNode = mainModel.getNodeProvider().getNodes().stream().filter(bidibNode -> {
                    return (ProductUtils.isRFBasisNode(bidibNode.getUniqueId())
                        && bidibNode.getBaseNumber() == activeBaseNumber);
                }).findFirst().orElse(null);

                if (rfBasisNode != null) {
                    LOGGER.info("Found the active rfBasisNode: {}", rfBasisNode);

                    context.register("activeRfBasis", rfBasisNode.getNode());
                }
                else {
                    LOGGER.warn("No active rfBasisNode found for activeBaseNumber: {}", activeBaseNumber);
                }
            }
            else {
                LOGGER.info("No base number for active rf base available.");
            }
        }
    }

}
