package org.bidib.wizard.mvc.main.controller;

import java.awt.Desktop;
import java.io.File;
import java.io.FileOutputStream;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.bidib2.BiDiB;
import org.bidib.jbidibc.messages.AddressData;
import org.bidib.jbidibc.simulation.SimulatorRegistry;
import org.bidib.jbidibc.simulation.nodes.Simulation;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.RfBasisNodeInterface;
import org.bidib.wizard.api.model.event.CheckForWizardUpdatesEvent;
import org.bidib.wizard.api.utils.NodeUtils;
import org.bidib.wizard.client.common.event.BidibConnectionEvent;
import org.bidib.wizard.client.common.event.MainControllerBoosterOnEvent;
import org.bidib.wizard.client.common.event.MainControllerEvent;
import org.bidib.wizard.client.common.event.MenuEvent;
import org.bidib.wizard.client.common.mvc.firmware.controller.FirmwareArchiveValidationController;
import org.bidib.wizard.client.common.view.DefaultBusyFrame;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.config.BackupControllerFactory;
import org.bidib.wizard.config.ComparisonControllerFactory;
import org.bidib.wizard.config.DccAdvControllerFactory;
import org.bidib.wizard.config.DebugConsoleControllerFactory;
import org.bidib.wizard.config.DebugInterfaceControllerFactory;
import org.bidib.wizard.config.FeedbackPositionControllerFactory;
import org.bidib.wizard.config.LocoControllerFactory;
import org.bidib.wizard.config.LocoTableControllerFactory;
import org.bidib.wizard.config.NetDebugControllerFactory;
import org.bidib.wizard.config.SimulationControllerFactory;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.dcca.client.controller.DccAdvController;
import org.bidib.wizard.dialog.AboutDialog;
import org.bidib.wizard.discovery.service.ZeroConfService;
import org.bidib.wizard.firmwarerepo.client.controller.FirmwareRepoController;
import org.bidib.wizard.localhost.client.controller.LocalHostClientController;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.backup.controller.BackupController;
import org.bidib.wizard.mvc.booster.controller.BoosterTableController;
import org.bidib.wizard.mvc.common.DialogRegistry;
import org.bidib.wizard.mvc.comparison.controller.ComparisonController;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.debug.controller.DebugInterfaceController;
import org.bidib.wizard.mvc.loco.controller.LocoController;
import org.bidib.wizard.mvc.locolist.controller.LocoTableController;
import org.bidib.wizard.mvc.logger.controller.LogPanelController;
import org.bidib.wizard.mvc.main.controller.exception.CloseAbortedException;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.MainView;
import org.bidib.wizard.mvc.main.view.component.BoosterOnConfirmDialog;
import org.bidib.wizard.mvc.main.view.menu.listener.MainMenuListener;
import org.bidib.wizard.mvc.netdebug.controller.NetDebugController;
import org.bidib.wizard.mvc.nodedebug.controller.DebugConsoleController;
import org.bidib.wizard.mvc.ping.controller.PingTableController;
import org.bidib.wizard.mvc.position.controller.FeedbackPositionController;
import org.bidib.wizard.mvc.preferences.controller.PreferencesController;
import org.bidib.wizard.mvc.worklist.controller.WorkListController;
import org.bidib.wizard.nodes.client.controller.NodesClientController;
import org.bidib.wizard.nodescript.client.controller.NodeScriptController;
import org.bidib.wizard.script.client.controller.ScriptClientController;
import org.bidib.wizard.simulation.client.controller.SimulationController;
import org.bidib.wizard.tracer.client.controller.TracerClientController;
import org.bidib.wizard.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;

import com.vlsolutions.swing.docking.DockingDesktop;

public class DefaultMainMenuListener implements MainMenuListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultMainMenuListener.class);

    private final MainView viewSupplier;

    private final DockingDesktop dockingDesktop;

    private final MainModel mainModel;

    private final SettingsService settingsService;

    private final ApplicationContext applicationContext;

    private final LocoTableControllerFactory locoTableControllerFactory;

    public DefaultMainMenuListener(final MainView viewSupplier, final DockingDesktop dockingDesktop,
        final MainModel mainModel, final SettingsService settingsService, final ApplicationContext applicationContext,
        final LocoTableControllerFactory locoTableControllerFactory) {
        this.viewSupplier = viewSupplier;
        this.dockingDesktop = dockingDesktop;
        this.mainModel = mainModel;
        this.settingsService = settingsService;
        this.applicationContext = applicationContext;
        this.locoTableControllerFactory = locoTableControllerFactory;
    }

    @Override
    public void about() {
        new AboutDialog(viewSupplier.getFrame());
    }

    @Override
    public void exit() {
        this.applicationContext.publishEvent(new MainControllerEvent(MainControllerEvent.Action.stop));
    }

    @Override
    public void preferences() {

        PreferencesController preferencesController = this.applicationContext.getBean(PreferencesController.class);
        try {
            preferencesController.start();
        }
        catch (Exception ex) {
            LOGGER.warn("Start preferences controller failed.", ex);
        }
    }

    @Override
    public void connect() {
        LOGGER.info("Let the mainController open the connection.");

        this.applicationContext
            .publishEvent(
                new BidibConnectionEvent(ConnectionRegistry.CONNECTION_ID_MAIN, BidibConnectionEvent.Action.connect));
    }

    @Override
    public void disconnect() {
        LOGGER.info("Let the mainController close the connection.");

        try {
            this.applicationContext
                .publishEvent(new BidibConnectionEvent(ConnectionRegistry.CONNECTION_ID_MAIN,
                    BidibConnectionEvent.Action.disconnect));
        }
        catch (CloseAbortedException ex) {
            LOGGER.warn("Close connection was aborted.");
        }
    }

    @Override
    public void listenNetBidib() {
        LOGGER.info("Let the mainController listen for incoming netBidib connections.");
        this.applicationContext.publishEvent(new MainControllerEvent(MainControllerEvent.Action.listenNetBidib));
    }

    @Override
    public void allBoosterOff() {
        LOGGER.info("Switch all boosters off!");
        this.applicationContext.publishEvent(new MainControllerEvent(MainControllerEvent.Action.allBoosterOff));
    }

    @Override
    public void allBoosterOn() {
        LOGGER.info("Switch all boosters on!");
        int result = JOptionPane.CANCEL_OPTION;
        CommandStationStatus requestedCommandStationState = CommandStationStatus.GO;

        if (!settingsService.getWizardSettings().isAllBoosterOnDoNotConfirmSwitch()) {
            // show a confirm dialog
            BoosterOnConfirmDialog allBoosterOnConfirmDialog =
                new BoosterOnConfirmDialog((JFrame) JOptionPane.getFrameForComponent(viewSupplier.getFrame()),
                    settingsService);

            result = allBoosterOnConfirmDialog.getResult();
            requestedCommandStationState = allBoosterOnConfirmDialog.getCommandStationStatus();
        }
        else {
            result = settingsService.getWizardSettings().getAllBoosterOnSavedAction();
            requestedCommandStationState =
                settingsService.getWizardSettings().getAllBoosterOnRequestedCommandStationState();

            LOGGER.info("Fetched action for all booster on from preferences: {}", result);
        }

        if (JOptionPane.CANCEL_OPTION == result) {
            LOGGER.info("User cancelled allBoosterOnConfirmDialog.");
            return;
        }

        this.applicationContext
            .publishEvent(new MainControllerBoosterOnEvent(
                result == BoosterOnConfirmDialog.RESULT_CONTINUE_BOOSTER_AND_COMMANDSTATION,
                requestedCommandStationState));
    }

    @Override
    public void checkForUpdates() {
        LOGGER.info("Check for updates.");

        this.applicationContext.publishEvent(new CheckForWizardUpdatesEvent());
    }

    private static final String WORKING_DIR_LOGFILES_COLLECT_KEY = "logfilesCollect";

    @Override
    public void collectLogFiles() {
        LOGGER.info("Collect the logfiles.");

        final String logFilePath = settingsService.getMiscSettings().getLogFilePath();
        if (StringUtils.isNotBlank(logFilePath)) {
            final File logfile = new File(new File(logFilePath), "BiDiBWizard2.log");

            if (logfile.exists()) {
                LOGGER.info("Found logfile to copy: {}", logfile);

                final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
                String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_LOGFILES_COLLECT_KEY);

                final FileFilter[] ff = null;
                FileDialog dialog =
                    new FileDialog(viewSupplier.getFrame(), FileDialog.SAVE, storedWorkingDirectory, (String) null,
                        ff) {
                        @Override
                        public void approve(final String selectedFile) {
                            File file = new File(selectedFile);
                            if (file != null && file.isDirectory()) {

                                String zipFilePath = file.getPath();
                                LOGGER.info("Copy the logfiles to directory: {}", zipFilePath);

                                try {
                                    File targetWizardLog = new File(file, "BiDiBWizard2.log");
                                    FileUtils.silentlyDeleteFile(targetWizardLog, 200L);
                                    FileUtils.copyFile(logfile, targetWizardLog);

                                    File logfileRXTX = new File(new File(logFilePath), "BiDiBWizard2-RXTX.log");
                                    File targetWizardRXTXLog = null;
                                    if (logfileRXTX.exists()) {
                                        targetWizardRXTXLog = new File(file, "BiDiBWizard2-RXTX.log");
                                        FileUtils.silentlyDeleteFile(targetWizardRXTXLog, 200L);
                                        FileUtils.copyFile(logfileRXTX, targetWizardRXTXLog);
                                    }

                                    File logfileRXTX_NET = new File(new File(logFilePath), "BiDiBWizard2-RXTX-NET.log");
                                    File targetWizardRXTX_NETLog = null;
                                    if (logfileRXTX_NET.exists()) {
                                        targetWizardRXTX_NETLog = new File(file, "BiDiBWizard2-RXTX-NET.log");
                                        FileUtils.silentlyDeleteFile(targetWizardRXTX_NETLog, 200L);
                                        FileUtils.copyFile(logfileRXTX_NET, targetWizardRXTX_NETLog);
                                    }

                                    File logfileRAW = new File(new File(logFilePath), "BiDiBWizard2-RAW.log");
                                    File targetWizardRAWLog = null;
                                    if (logfileRAW.exists()) {
                                        targetWizardRAWLog = new File(file, "BiDiBWizard2-RAW.log");
                                        FileUtils.silentlyDeleteFile(targetWizardRAWLog, 200L);
                                        FileUtils.copyFile(logfileRAW, targetWizardRAWLog);
                                    }

                                    File logfileSIM = new File(new File(logFilePath), "BiDiBWizard2-SIM.log");
                                    File targetWizardSIMLog = null;
                                    if (logfileSIM.exists()) {
                                        targetWizardSIMLog = new File(file, "BiDiBWizard2-SIM.log");
                                        FileUtils.silentlyDeleteFile(targetWizardSIMLog, 200L);
                                        FileUtils.copyFile(logfileSIM, targetWizardSIMLog);
                                    }

                                    File logfileMDNS = new File(new File(logFilePath), "BiDiBWizard2-MDNS.log");
                                    File targetWizardMDNSLog = null;
                                    if (logfileMDNS.exists()) {
                                        targetWizardMDNSLog = new File(file, "BiDiBWizard2-MDNS.log");
                                        FileUtils.silentlyDeleteFile(targetWizardMDNSLog, 200L);
                                        FileUtils.copyFile(logfileMDNS, targetWizardMDNSLog);
                                    }

                                    File logfileZ21 = new File(new File(logFilePath), "BiDiBWizard2-Z21.log");
                                    File targetWizardZ21Log = null;
                                    if (logfileZ21.exists()) {
                                        targetWizardZ21Log = new File(file, "BiDiBWizard2-Z21.log");
                                        FileUtils.silentlyDeleteFile(targetWizardZ21Log, 200L);
                                        FileUtils.copyFile(logfileZ21, targetWizardZ21Log);
                                    }

                                    // add files to zip
                                    SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd-HHmmSS");
                                    File zipFile =
                                        new File(file, "BiDiBWizard2-logs-" + sdf.format(new Date()) + ".zip");

                                    LOGGER.info("Zip the logfiles to file: {}", zipFile);

                                    FileOutputStream fileOutputStream = new FileOutputStream(zipFile);
                                    ZipOutputStream zipOutputStream = new ZipOutputStream(fileOutputStream);

                                    ZipEntry zipEntry = new ZipEntry(targetWizardLog.getName());
                                    zipOutputStream.putNextEntry(zipEntry);

                                    org.apache.commons.io.FileUtils.copyFile(targetWizardLog, zipOutputStream);

                                    // close ZipEntry to store the stream to the file
                                    zipOutputStream.closeEntry();
                                    FileUtils.silentlyDeleteFile(targetWizardLog, null);

                                    // copy the RXTX log
                                    if (targetWizardRXTXLog != null) {
                                        zipEntry = new ZipEntry(targetWizardRXTXLog.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils.copyFile(targetWizardRXTXLog, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardRXTXLog, null);
                                    }

                                    // copy the RXTX-NET log
                                    if (targetWizardRXTX_NETLog != null) {
                                        zipEntry = new ZipEntry(targetWizardRXTX_NETLog.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils
                                            .copyFile(targetWizardRXTX_NETLog, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardRXTX_NETLog, null);
                                    }

                                    // copy the RAW log
                                    if (targetWizardRAWLog != null) {
                                        zipEntry = new ZipEntry(targetWizardRAWLog.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils.copyFile(targetWizardRAWLog, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardRAWLog, null);
                                    }

                                    // copy the SIM log
                                    if (targetWizardSIMLog != null) {
                                        zipEntry = new ZipEntry(targetWizardSIMLog.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils.copyFile(targetWizardSIMLog, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardSIMLog, null);
                                    }

                                    // copy the MDNS log
                                    if (targetWizardMDNSLog != null) {
                                        zipEntry = new ZipEntry(targetWizardMDNSLog.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils.copyFile(targetWizardMDNSLog, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardMDNSLog, null);
                                    }

                                    // copy the Z21 log
                                    if (targetWizardZ21Log != null) {
                                        zipEntry = new ZipEntry(targetWizardZ21Log.getName());
                                        zipOutputStream.putNextEntry(zipEntry);

                                        org.apache.commons.io.FileUtils.copyFile(targetWizardZ21Log, zipOutputStream);

                                        // close ZipEntry to store the stream to the file
                                        zipOutputStream.closeEntry();
                                        FileUtils.silentlyDeleteFile(targetWizardZ21Log, null);
                                    }

                                    zipOutputStream.close();

                                    fileOutputStream.close();

                                    final String workingDir =
                                        Paths.get(selectedFile).getParent() != null
                                            ? Paths.get(selectedFile).getParent().toString()
                                            : Paths.get(selectedFile).toString();
                                    LOGGER.info("Save current workingDir: {}", workingDir);

                                    wizardSettings.setWorkingDirectory(WORKING_DIR_LOGFILES_COLLECT_KEY, workingDir);

                                    LOGGER.info("Open folder: {}", file);
                                    Desktop.getDesktop().open(file);
                                }
                                catch (Exception ex) {
                                    LOGGER.warn("Copy logfile to backup location failed.", ex);

                                    JOptionPane
                                        .showMessageDialog(viewSupplier.getFrame(),
                                            Resources
                                                .getString(DefaultMainMenuListener.class, "copy-logfiles-failed",
                                                    new Object[] { ex.getLocalizedMessage() }),
                                            Resources.getString(DefaultMainMenuListener.class, "title-error"),
                                            JOptionPane.ERROR_MESSAGE);
                                }
                            }
                        }
                    };
                dialog.setApproveButtonText(Resources.getString(DefaultMainMenuListener.class, "save-under"));
                dialog.showDialog();
            }
            else {
                JOptionPane
                    .showMessageDialog(viewSupplier.getFrame(),
                        Resources
                            .getString(DefaultMainMenuListener.class, "copy-logfiles-not-found",
                                new Object[] { logFilePath }),
                        Resources.getString(DefaultMainMenuListener.class, "title-error"), JOptionPane.ERROR_MESSAGE);
            }

        }
        else {
            LOGGER.warn("No logFilePath available.");
        }
    }

    @Override
    public void showRxTxLoggerView() {
        LOGGER.info("Open the RXTX log panel.");

        LogPanelController logPanelController = this.applicationContext.getBean(LogPanelController.class);
        logPanelController.start();
    }

    @Override
    public void showTracerClient() {
        LOGGER.info("Open the tracer client view.");

        final TracerClientController tracerClientController =
            this.applicationContext.getBean(TracerClientController.class);

        tracerClientController.start();
    }

    @Override
    public void boosterTable() {
        LOGGER.info("Open the booster table.");

        BoosterTableController boosterTableController = this.applicationContext.getBean(BoosterTableController.class);
        boosterTableController.start();
    }

    @Override
    public void feedbackPositionTable() {
        LOGGER.info("Open the feedbackPosition table.");

        FeedbackPositionControllerFactory feedbackPositionControllerFactory =
            this.applicationContext.getBean(FeedbackPositionControllerFactory.class);

        FeedbackPositionController feedbackPositionController =
            feedbackPositionControllerFactory.createController(dockingDesktop, this.mainModel);
        feedbackPositionController.start();
    }

    @Override
    public void pingTable() {
        LOGGER.info("Open the ping table.");

        PingTableController pingTableController = this.applicationContext.getBean(PingTableController.class);
        pingTableController.start();
    }

    @Override
    public void console() {
        LOGGER.info("Open the console.");

        ConsoleController.ensureConsoleVisible();
    }

    @Override
    public void locoTable() {
        LOGGER.info("Open the loco table.");

        final NodeInterface selectedNode = this.mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

            CommandStationNodeInterface node = selectedNode.getCommandStationNode();
            LOGGER.info("Open the loco table for node: {}", node);

            DefaultBusyFrame.setWaitCursor(viewSupplier.getFrame());

            try {

                final MainControllerInterface mainController =
                    this.applicationContext.getBean(MainControllerInterface.class);
                LocoTableController locoTableController =
                    locoTableControllerFactory
                        .createLocoTableController(selectedNode, viewSupplier.getDesktop(), viewSupplier.getFrame(),
                            this.applicationContext);

                locoTableController.start(mainController);
                locoTableController.queryLocoList();

            }
            catch (Exception ex) {
                LOGGER.warn("Open loco table failed.", ex);
                // show dialog for user
                JOptionPane
                    .showMessageDialog(viewSupplier.getFrame(), "Open loco table failed.", "Open loco table",
                        JOptionPane.ERROR_MESSAGE);
            }
            finally {
                DefaultBusyFrame.setDefaultCursor(viewSupplier.getFrame());
            }
        }
    }

    @Override
    public void nodeScriptView() {
        LOGGER.info("Open the nodeScript editor.");

        NodeScriptController nodeScriptController = this.applicationContext.getBean(NodeScriptController.class);
        nodeScriptController.showView(dockingDesktop, this.mainModel);
    }

    @Override
    public void debugInterface() {
        LOGGER.info("Open the debug interface panel.");

        DebugInterfaceControllerFactory debugInterfaceControllerFactory =
            this.applicationContext.getBean(DebugInterfaceControllerFactory.class);

        DebugInterfaceController debugInterfaceController =
            debugInterfaceControllerFactory.createController(this.dockingDesktop, this.settingsService);
        debugInterfaceController.start();
    }

    @Override
    public void netDebug() {
        LOGGER.info("Open the netDebug panel.");

        NetDebugControllerFactory netDebugControllerFactory =
            this.applicationContext.getBean(NetDebugControllerFactory.class);

        NetDebugController netDebugController = netDebugControllerFactory.createController(dockingDesktop);
        netDebugController.start();
    }

    @Override
    public void debugConsole() {
        LOGGER.info("Open the debug console panel.");

        DebugConsoleControllerFactory debugConsoleControllerFactory =
            this.applicationContext.getBean(DebugConsoleControllerFactory.class);

        final DebugConsoleController debugConsoleController = debugConsoleControllerFactory.createController();
        debugConsoleController.start();
    }

    private static final String WORKING_DIR_SIMULATION_KEY = "simulation";

    private static final String SUFFIX_JSON = "json";

    @Override
    public void saveNodeTreeForSimulation() {
        LOGGER.info("Save the node tree for simulation.");

        // create the simulation controller
        try {
            final SimulationControllerFactory simulationControllerFactory =
                this.applicationContext.getBean(SimulationControllerFactory.class);

            final SimulationController simulationController =
                simulationControllerFactory.createController(dockingDesktop);

            final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
            String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_SIMULATION_KEY);

            final Simulation simulation = simulationController.exportNodeTreeToSimulation();

            final FileFilter ff = new FileNameExtensionFilter("Simulation", "xml", "json");
            FileDialog dialog =
                new FileDialog(viewSupplier.getFrame(), FileDialog.SAVE, storedWorkingDirectory, (String) null, ff) {
                    @Override
                    public void approve(final String fileName) {
                        LOGGER.info("Save the simulation to file: {}", fileName);

                        String selectedFile = fileName;
                        if (!FilenameUtils.isExtension(fileName, SUFFIX_XML)
                            && !FilenameUtils.isExtension(fileName, SUFFIX_JSON)) {
                            LOGGER.info("Add default extension 'xml'.");
                            selectedFile = fileName + "." + SUFFIX_XML;
                        }

                        try {
                            File file = new File(selectedFile);

                            if (FilenameUtils.isExtension(fileName, SUFFIX_JSON)) {
                                SimulatorRegistry.saveSimulationConfigurationJson(simulation, file);
                            }
                            else {
                                SimulatorRegistry.saveSimulationConfiguration(simulation, file);
                            }

                            final String workingDir = Paths.get(selectedFile).getParent().toString();
                            LOGGER.info("Save current workingDir: {}", workingDir);

                            wizardSettings.setWorkingDirectory(WORKING_DIR_SIMULATION_KEY, workingDir);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Save simulation to file failed: {}", selectedFile, ex);

                            JOptionPane
                                .showMessageDialog(viewSupplier.getFrame(),
                                    Resources
                                        .getString(DefaultMainMenuListener.class, "save-nodes-for-simulation.failed",
                                            new Object[] { ex.getLocalizedMessage() }),
                                    Resources
                                        .getString(DefaultMainMenuListener.class, "save-nodes-for-simulation.title"),
                                    JOptionPane.ERROR_MESSAGE);
                        }
                    }
                };
            dialog.showDialog();

        }
        catch (Exception ex) {
            LOGGER.warn("Save node tree for simulation failed.", ex);

            JOptionPane
                .showMessageDialog(viewSupplier.getFrame(),
                    Resources.getString(DefaultMainMenuListener.class, "save-nodes-for-simulation.failed"),
                    Resources.getString(DefaultMainMenuListener.class, "save-nodes-for-simulation.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void backupAllNodes() {
        LOGGER.info("Backup all nodes.");

        // create the backup controller
        try {
            final BackupControllerFactory backupControllerFactory =
                this.applicationContext.getBean(BackupControllerFactory.class);

            final BackupController backupController =
                backupControllerFactory.createController(dockingDesktop, this.mainModel);
            backupController.start();
        }
        catch (Exception ex) {
            LOGGER.warn("Backup node tree failed.", ex);

            JOptionPane
                .showMessageDialog(viewSupplier.getFrame(),
                    Resources.getString(DefaultMainMenuListener.class, "backup-all-nodes.failed"),
                    Resources.getString(DefaultMainMenuListener.class, "backup-all-nodes.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void dccAdv() {
        LOGGER.info("Open the DCC-A panel.");

        final DccAdvControllerFactory dccAdvControllerFactory =
            this.applicationContext.getBean(DccAdvControllerFactory.class);

        DccAdvController dccAdvController =
            dccAdvControllerFactory.createController(this.mainModel.getSelectedNode(), dockingDesktop);
        dccAdvController.start();
    }

    @Override
    public void showNodeScriptWizard() {

        LOGGER.info("Open the nodeScript wizard.");

        final NodeScriptController nodeScriptController = this.applicationContext.getBean(NodeScriptController.class);
        nodeScriptController.showWizard(dockingDesktop, this.mainModel);
    }

    @Override
    public void showFirmwareRepo() {
        LOGGER.info("Open the firmware repo view.");

        final FirmwareRepoController firmwareRepoController =
            this.applicationContext.getBean(FirmwareRepoController.class);

        firmwareRepoController.start(false);
    }

    @Override
    public void showNodesClient() {
        LOGGER.info("Open the nodes client view.");

        final NodesClientController nodesClientController =
            this.applicationContext.getBean(NodesClientController.class);

        nodesClientController.start();
    }

    @Override
    public void showScriptClient() {
        LOGGER.info("Open the script client view.");

        final ScriptClientController scriptClientController =
            this.applicationContext.getBean(ScriptClientController.class);

        scriptClientController.start();
    }

    @Override
    public void showLocalHostClient() {
        LOGGER.info("Open the localHost client view.");

        final LocalHostClientController localHostClientController =
            this.applicationContext.getBean(LocalHostClientController.class);

        localHostClientController.start();
    }

    @Override
    public void discoverySingleListServices() {

        try {
            final ZeroConfService bonjourService = this.applicationContext.getBean(ZeroConfService.class);
            bonjourService.listServices(ZeroConfService.NETBIDIB_SERVICETYPE + ZeroConfService.LOCAL_SUFFIX);
        }
        catch (Exception ex) {
            LOGGER.warn("List services in bonjour service failed.", ex);
            JOptionPane
                .showMessageDialog(viewSupplier.getFrame(),
                    "List services in bonjour service failed:\n" + ex.getMessage(), "Bonjour Developer",
                    JOptionPane.ERROR_MESSAGE);
        }

    }

    @Override
    public void showLocoDialog() {
        LOGGER.info("Open the loco controller dialog.");

        // find the first command station node
        CommandStationNodeInterface node =
            NodeUtils.findFirstCommandStationNode(this.mainModel.getNodeProvider().getNodes());
        if (node != null) {

            final LocoControllerFactory locoControllerFactory =
                this.applicationContext.getBean(LocoControllerFactory.class);
            final DialogRegistry dialogRegistry = this.applicationContext.getBean(DialogRegistry.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node, this.viewSupplier.getFrame(), mainModel.getNodeProvider(),
                        dialogRegistry);

            AddressData initialAddress = null;
            locoController.start(initialAddress, null, null);
        }
    }

    @Override
    public void showCarDialog() {
        LOGGER.info("Open the car controller dialog.");

        // find the master RF basis node
        RfBasisNodeInterface node = NodeUtils.findFirstRfBasisMasterNode(this.mainModel.getNodeProvider().getNodes());
        if (node != null) {

            final LocoControllerFactory locoControllerFactory =
                this.applicationContext.getBean(LocoControllerFactory.class);
            final DialogRegistry dialogRegistry = this.applicationContext.getBean(DialogRegistry.class);
            final LocoController locoController =
                locoControllerFactory
                    .createLocoController(node.getNode().getCommandStationNode(), this.viewSupplier.getFrame(),
                        mainModel.getNodeProvider(), dialogRegistry);

            AddressData initialAddress = null;
            locoController.start(initialAddress, null, null);
        }
        else {
            JOptionPane
                .showMessageDialog(viewSupplier.getFrame(),
                    Resources.getString(DefaultMainMenuListener.class, "open-car-dialog.failed-no-basis"),
                    Resources.getString(DefaultMainMenuListener.class, "open-car-dialog.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @EventListener
    public void handleMenuEvent(final MenuEvent event) {
        LOGGER.info("Handle the menu event: {}", event);

        SwingUtilities.invokeLater(() -> {

            switch (event.getAction()) {
                case nodeScriptWizard:
                    showNodeScriptWizard();
                    break;
                case boosterTable:
                    boosterTable();
                    break;
                case rxtxView:
                    showRxTxLoggerView();
                    break;
                default:
                    LOGGER.warn("Unhandled MenuEvent detected: {}", event);
                    break;
            }
        });
    }

    @Override
    public void showWorkItemList() {
        LOGGER.info("Open the workList view.");

        final WorkListController workListController = this.applicationContext.getBean(WorkListController.class);

        workListController.start();
    }

    private static final String WORKING_DIR_COMPARISON_KEY = "comparison";

    private static final String SUFFIX_XML = "xml";

    @Override
    public void saveNodeTreeForComparison() {
        LOGGER.info("Save the node tree for comparison.");

        // create the comparison controller
        try {
            final ComparisonControllerFactory comparisonControllerFactory =
                this.applicationContext.getBean(ComparisonControllerFactory.class);

            final ComparisonController comparisonController =
                comparisonControllerFactory.createController(dockingDesktop);

            final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
            String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_COMPARISON_KEY);

            final BiDiB bidib = comparisonController.exportNodeTreeToComparison();

            final FileFilter ff = new FileNameExtensionFilter("Comparison", "xml");
            FileDialog dialog =
                new FileDialog(viewSupplier.getFrame(), FileDialog.SAVE, storedWorkingDirectory, (String) null,
                    SUFFIX_XML, ff) {
                    @Override
                    public void approve(final String fileName) {
                        LOGGER.info("Save the comparison data to file: {}", fileName);

                        String selectedFile = fileName;
                        if (!FilenameUtils.isExtension(fileName, SUFFIX_XML)) {
                            LOGGER.info("Add default extension 'xml'.");
                            selectedFile = fileName + "." + SUFFIX_XML;
                        }
                        try {
                            File file = new File(selectedFile);
                            BidibFactory.saveBiDiB(bidib, file, false);

                            final String workingDir = Paths.get(selectedFile).getParent().toString();
                            LOGGER.info("Save current workingDir: {}", workingDir);

                            wizardSettings.setWorkingDirectory(WORKING_DIR_COMPARISON_KEY, workingDir);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Save comparison data to file failed: {}", selectedFile, ex);

                            JOptionPane
                                .showMessageDialog(viewSupplier.getFrame(),
                                    Resources
                                        .getString(DefaultMainMenuListener.class, "save-nodes-for-comparison.failed",
                                            new Object[] { ex.getLocalizedMessage() }),
                                    Resources
                                        .getString(DefaultMainMenuListener.class, "save-nodes-for-comparison.title"),
                                    JOptionPane.ERROR_MESSAGE);
                        }
                    }
                };
            dialog.showDialog();
        }
        catch (Exception ex) {
            LOGGER.warn("Save node tree for comparison failed.", ex);

            JOptionPane
                .showMessageDialog(viewSupplier.getFrame(),
                    Resources.getString(DefaultMainMenuListener.class, "save-nodes-for-comparison.failed"),
                    Resources.getString(DefaultMainMenuListener.class, "save-nodes-for-comparison.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    @Override
    public void showFirmwareArchiveValidationView() {
        LOGGER.info("Open the FirmwareArchiveValidation view.");

        FirmwareArchiveValidationController firmwareArchiveValidationController =
            this.applicationContext.getBean(FirmwareArchiveValidationController.class);
        firmwareArchiveValidationController.start();
    }
}
