package org.bidib.wizard.mvc.main.controller;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.enums.IdentifyState;
import org.bidib.jbidibc.messages.exception.NoAnswerException;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.controller.CvDefinitionPanelControllerInterface;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.script.node.NodeScripting;
import org.bidib.wizard.common.script.node.types.CvType;
import org.bidib.wizard.common.script.node.types.FeatureType;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.mvc.common.exception.NodeSelectionChangeException;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.nodescript.client.view.listener.NodeTreeScriptingListener;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.bidib.wizard.nodescript.script.node.NodeScriptingSupportProvider;
import org.bidib.wizard.utils.NodeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultNodeScripting implements NodeScripting {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultNodeScripting.class);

    private final MainModel mainModel;

    private final MainControllerInterface mainController;

    private final CvDefinitionPanelControllerInterface cvDefinitionPanelController;

    private final SwitchingNodeService switchingNodeService;

    private final NodeService nodeService;

    private final NodeScriptingSupportProvider nodeScriptingSupportProvider;

    public DefaultNodeScripting(final MainModel mainModel, final MainControllerInterface mainController,
        final CvDefinitionPanelControllerInterface cvDefinitionPanelController,
        final SwitchingNodeService switchingNodeService, final NodeService nodeService,
        final NodeScriptingSupportProvider nodeScriptingSupportProvider) {
        this.mainModel = mainModel;
        this.mainController = mainController;
        this.cvDefinitionPanelController = cvDefinitionPanelController;
        this.switchingNodeService = switchingNodeService;
        this.nodeService = nodeService;
        this.nodeScriptingSupportProvider = nodeScriptingSupportProvider;
    }

    @Override
    public void setLabel(Long uuid, TargetType portType) {

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set CV can only be performed on the selected node!");
            return;
        }

        if (SwingUtilities.isEventDispatchThread()) {
            LOGGER.info("Change the label on the selected node.");
            setLabelInternal(uuid, portType);
        }
        else {
            try {
                SwingUtilities.invokeAndWait(() -> {
                    LOGGER.info("Change the label on the selected node, uuid: {}, portType: {}", uuid, portType);
                    setLabelInternal(uuid, portType);
                });
            }
            catch (InvocationTargetException | InterruptedException ex) {
                LOGGER.warn("Change label failed.", ex);
            }
        }
    }

    private void setLabelInternal(Long uuid, TargetType targetType) {
        LOGGER.info("Set label, uuid: {}, targetType: {}", uuid, targetType);

        ChangeLabelSupport changeLabelSupport =
            nodeScriptingSupportProvider.lookup(targetType.getScriptingTargetType());
        if (changeLabelSupport != null) {
            changeLabelSupport.changeLabel(targetType);
        }
        else {
            LOGGER.warn("No changeLabelSupport available for targetType: {}", targetType);
        }
    }

    @Override
    public void setCv(Long uuid, CvType... cvTypes) {
        LOGGER.info("Set the CV, uuid: {}, cvTypes: {}", uuid, cvTypes);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set CV can only be performed on the selected node!");
            return;
        }

        List<ConfigurationVariable> cvList = new LinkedList<>();

        // prepare the list of CV values
        for (CvType cvType : cvTypes) {

            cvList.add(new ConfigurationVariable(cvType.getCvNumber(), cvType.getCvValue()));
        }
        selectedNode.setConfigVariables(cvList);

        // transfer to node
        List<ConfigurationVariable> configVars =
            this.cvDefinitionPanelController.setConfigVariables(selectedNode, cvList);

        // iterate over the collection of stored variables in the model and update the values.
        // After that notify the tree and delete the new values that are now stored in the node
        selectedNode.updateConfigVariableValues(configVars, false);
    }

    @Override
    public List<ConfigurationVariable> getCv(Long uniqueId, CvType... cvTypes) {
        LOGGER.info("Get the CV directly from the node, uniqueId: {}, cvTypes: {}", uniqueId, cvTypes);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uniqueId.longValue()) {
            LOGGER.warn("Get CV can only be performed on the selected node!");
            return Collections.emptyList();
        }

        List<ConfigurationVariable> cvList = new LinkedList<>();

        // prepare the list of CV values
        for (CvType cvType : cvTypes) {

            cvList.add(new ConfigurationVariable(cvType.getCvNumber(), cvType.getCvValue()));
        }

        // transfer to node
        List<ConfigurationVariable> configVars =
            this.cvDefinitionPanelController.setConfigVariables(selectedNode, cvList);

        // iterate over the collection of stored variables in the model and update the values.
        // After that notify the tree and delete the new values that are now stored in the node
        selectedNode.updateConfigVariableValues(configVars, true);

        return configVars;
    }

    @Override
    public void setFeature(Long uuid, FeatureType... featureTypes) {
        LOGGER.info("Set the feature, uuid: {}, featureTypes: {}", uuid, featureTypes);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set features can only be performed on the selected node!");
            return;
        }

        final List<Feature> features = new LinkedList<>();
        for (FeatureType featureType : featureTypes) {
            features.add(new Feature(featureType.getFeatureNumber(), featureType.getFeatureValue()));
        }

        nodeService.setFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode, features);
    }

    @Override
    public List<Feature> featuresGetAll(Long uniqueId, boolean discardCache) {
        LOGGER.info("Get all features directly from the node, uniqueId: {}, discardCache: {}", uniqueId, discardCache);

        final NodeInterface node = this.nodeService.getNode(ConnectionRegistry.CONNECTION_ID_MAIN, uniqueId);
        if (node != null && node.getUniqueId() != uniqueId.longValue()) {
            LOGGER.warn("Get all features can only be performed on an available node!");
            return Collections.emptyList();
        }

        return this.nodeService.queryAllFeatures(ConnectionRegistry.CONNECTION_ID_MAIN, node, discardCache);
    }

    @Override
    public boolean isNodeHasRestartPending(Long uuid) {
        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() == uuid.longValue()) {
            return selectedNode.isNodeHasRestartPendingError();
        }
        return false;
    }

    @Override
    public void resetNode(Long uuid) {
        LOGGER.info("Reset node with uuid: {}", uuid);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Reset node can only be performed on the selected node!");
            return;
        }

        // let the main controller reset the node
        mainController.resetNode(selectedNode);
    }

    @Override
    public void reselectNode(Long uuid) {
        LOGGER.info("Reselect node with uuid: {}", uuid);

        final NodeInterface node = NodeUtils.findNodeByUuid(mainModel.getNodeProvider().getNodes(), uuid);

        if (node != null) {
            // select the node
            LOGGER.info("Set the selected node in the mainModel: {}", node);

            NodeTreeScriptingListener nodeListPanel =
                DefaultApplicationContext.getInstance().get("nodeListPanel", NodeTreeScriptingListener.class);

            if (nodeListPanel != null) {

                if (SwingUtilities.isEventDispatchThread()) {
                    try {
                        LOGGER.info("Change the selected node.");
                        nodeListPanel.setSelectedNode(node);
                    }
                    catch (IllegalArgumentException | NodeSelectionChangeException ex) {
                        LOGGER.warn("Select node failed.", ex);
                        throw ex;
                    }
                }
                else {
                    try {
                        SwingUtilities.invokeAndWait(() -> {
                            try {
                                LOGGER.info("Change the selected node.");
                                nodeListPanel.setSelectedNode(node);
                            }
                            catch (IllegalArgumentException | NodeSelectionChangeException ex) {
                                LOGGER.warn("Select node failed.", ex);
                                throw ex;
                            }
                        });
                    }
                    catch (InvocationTargetException | InterruptedException ex) {
                        LOGGER.warn("Select node failed.", ex);
                    }
                }
            }
        }
    }

    @Override
    public void setAccessory(Long uuid, final Accessory accessory) {
        LOGGER.info("Set the accessory, uuid: {}, accessory: {}", uuid, accessory);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set accessory can only be performed on the selected node!");
            return;
        }

        LOGGER.info("Transfer accessory to node: {}", accessory);
        // let the main controller replace the accessory
        mainController.transferAccessoryToNode(selectedNode, accessory);
    }

    @Override
    public void setMacro(Long uuid, final Macro macro) {
        LOGGER.info("Set the macro, uuid: {}, macro: {}", uuid, macro);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set macro can only be performed on the selected node!");
            return;
        }

        LOGGER.info("Transfer the macro to node and save permanently, node: {}, macro: {}", selectedNode, macro);
        try {
            switchingNodeService.saveMacro(mainModel.getConnectionId(), selectedNode.getSwitchingNode(), macro);
        }
        catch (NoAnswerException ex) {
            LOGGER.warn("Transfer macro and save macro failed.", ex);

            selectedNode.setNodeHasError(true);
            selectedNode.setReasonData("Transfer macro and save macro failed.");
        }
        catch (/* InvalidConfigurationException */ RuntimeException ex) {
            LOGGER.warn("Transfer macro and save macro failed.", ex);

            selectedNode.setNodeHasError(true);
            selectedNode.setReasonData("Transfer macro and save macro failed.");
        }
    }

    @Override
    public void setPortConfig(Long uuid, TargetType portType, final Map<Byte, PortConfigValue<?>> portConfig) {
        LOGGER.info("Set the port config, uuid: {}, portConfig: {}", uuid, portConfig);

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set port config can only be performed on the selected node!");
            return;
        }

        if (selectedNode != null) {

            // TODO support all port types
            switch (portType.getScriptingTargetType()) {
                case ANALOGPORT:
                    selectedNode.setAnalogPortConfig(portType.getPortNum(), portConfig);
                    break;
                case BACKLIGHTPORT:
                    selectedNode.setBacklightPortConfig(portType.getPortNum(), portConfig);
                    break;
                case LIGHTPORT:
                    selectedNode.setLightPortConfig(portType.getPortNum(), portConfig);
                    break;
                case SERVOPORT:
                    selectedNode.setServoPortConfig(portType.getPortNum(), portConfig);
                    break;
                case SWITCHPORT:
                    selectedNode.setSwitchPortConfig(portType.getPortNum(), portConfig);
                    break;
                case SWITCHPAIRPORT:
                    selectedNode.setSwitchPairPortConfig(portType.getPortNum(), portConfig);
                    break;
                case SOUNDPORT:
                    selectedNode.setSoundPortConfig(portType.getPortNum(), portConfig);
                    break;
                // TODO add support for other port types
                default:
                    LOGGER.error("Unsupported port type detected: {}", portType);
                    break;
            }
        }

        // write config to node
        mainController.replacePortConfig(portType, portConfig);
    }

    @Override
    public void assertPortType(Long uuid, TargetType portType) {

        LOGGER.info("Assert that the port type matches, uuid: {}, portType: {}", uuid, portType);

        final NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uuid.longValue()) {
            LOGGER.warn("Set port config can only be performed on the selected node!");
            return;
        }

        // map the port to the new type
        mainController.mapPortType(portType);

    }

    @Override
    public void setIdentifyState(Long uniqueId, IdentifyState identifyState) {
        LOGGER.info("Set the identify state, uniqueId: {}, identifyState: {}", uniqueId, identifyState);

        final NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null && selectedNode.getUniqueId() != uniqueId.longValue()) {
            LOGGER.warn("Set identify state can only be performed on the selected node!");
            return;
        }

        selectedNode.setIdentifyState(identifyState);
    }

    @Override
    public IdentifyState queryIdentifyState(Long uniqueId) {
        final NodeInterface node = this.mainModel.getSelectedNode();
        return node.getIdentifyState();
    }

    @Override
    public String setString(Long uniqueId, int namespace, int index, String value) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public String getString(Long uniqueId, int namespace, int index) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public String sendNodeDebugString(
        Long uniqueId, int namespace, int index, String value, int expectedResponseCount) {
        // TODO Auto-generated method stub
        return null;
    }

}
