package org.bidib.wizard.mvc.main.model;

import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.IoBehaviourSwitchEnum;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.LoadTypeEnum;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.SwitchPort;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.model.listener.SwitchPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SwitchPortTableModel
    extends SimpleHierarchicalPortTableModel<SwitchPortStatus, SwitchPort, SwitchPortModelListener> {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPortTableModel.class);

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_IO_BEHAVIOUR = 1;

    public static final int COLUMN_SWITCH_OFF_TIME = 2;

    public static final int COLUMN_LOAD_TYPE = 3;

    public static final int COLUMN_PORT_IDENTIFIER = 4;

    public static final int COLUMN_STATUS = 5;

    public static final int COLUMN_TEST = 6;

    public static final int COLUMN_PORT_INSTANCE = 7;

    public SwitchPortTableModel(final MainModel model) {
        super();
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Integer getTestColumnIndex() {
        return COLUMN_TEST;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(getClass(), "label"), Resources.getString(getClass(), "ioBehaviour"),
                Resources.getString(getClass(), "switchOffTime"), Resources.getString(getClass(), "loadType"),
                Resources.getString(getClass(), "portIdentifier"), Resources.getString(getClass(), "status"),
                Resources.getString(getClass(), "test"), null };
    }

    @Override
    public void addRow(SwitchPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.getLabel();
            rowData[COLUMN_IO_BEHAVIOUR] = port.getOutputBehaviour();
            rowData[COLUMN_SWITCH_OFF_TIME] = port.getSwitchOffTime();
            rowData[COLUMN_LOAD_TYPE] = port.getLoadType();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();

            SwitchPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;
            rowData[COLUMN_PORT_INSTANCE] = port;
            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        SwitchPort switchPort = (SwitchPort) getValueAt(row, SwitchPortTableModel.COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_TEST:

                if (Port.getConfiguredPortType(switchPort) == LcOutputType.SWITCHPORT) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_PORT_INSTANCE:
            case COLUMN_PORT_IDENTIFIER:
                return Object.class;
            case COLUMN_TEST:
                return Object.class;
            case COLUMN_IO_BEHAVIOUR:
                return IoBehaviourSwitchEnum.class;
            case COLUMN_LOAD_TYPE:
                return LoadTypeEnum.class;

            default:
                return Object.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final SwitchPort port = (SwitchPort) getValueAt(row, SwitchPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_IO_BEHAVIOUR:
                IoBehaviourSwitchEnum ioBehaviour = (IoBehaviourSwitchEnum) value;
                if (ioBehaviour != port.getOutputBehaviour()) {
                    port.setOutputBehaviour(ioBehaviour);
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_SWITCH_CTRL);
                }
                else {
                    LOGGER.debug("The IOBehaviour has not been changed.");
                }
                break;
            case COLUMN_SWITCH_OFF_TIME:
                int switchOffTime = (Integer) value;
                if (port.getSwitchOffTime() != switchOffTime) {
                    port.setSwitchOffTime(switchOffTime);
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TICKS);
                }
                else {
                    LOGGER.debug("The switchOff time has not been changed.");
                }
                break;
            case COLUMN_LOAD_TYPE:
                LoadTypeEnum loadType = (LoadTypeEnum) value;
                if (loadType != port.getLoadType()) {
                    port.setLoadType(loadType);
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_LOAD_TYPE);
                }
                else {
                    LOGGER.debug("The load type has not been changed.");
                }
                break;
            case COLUMN_STATUS:
                port.setStatus((SwitchPortStatus) value);
                super.setValueAt(value, row, column);
                break;
            case COLUMN_TEST:
                LOGGER.info("Status of switch port is updated: {}, port: {}", value, port);

                SwitchPortStatus portStatus = (SwitchPortStatus) value;

                LOGGER.info("Set the test port status: {}", portStatus);

                super.setValueAt(portStatus, row, column);

                // set the value on the temporary port
                final SwitchPort switchPort = new SwitchPort();
                switchPort.setId(port.getId());
                switchPort.setStatus(portStatus);
                fireTestButtonPressed(switchPort);
                break;
            default:
                super.setValueAt(value, row, column);
                break;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        final SwitchPort port = (SwitchPort) super.getValueAt(row, SwitchPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_IO_BEHAVIOUR:
                break;
            case COLUMN_LOAD_TYPE:
                return port.getLoadType();
            case COLUMN_SWITCH_OFF_TIME:
                return port.getSwitchOffTime();
            case COLUMN_STATUS:
                return port.getStatus();
            case COLUMN_PORT_INSTANCE:
                return port;
            case COLUMN_TEST:
            default:
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortStatusChanged(final SwitchPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.info("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);

                // get the opposite status and set it
                SwitchPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());

                LOGGER.info("Update port status: {}, oppositeStatus: {}", port.getStatus(), oppositeStatus);

                super.setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(final SwitchPort port) {
        LOGGER.debug("The port config was changed for port: {}", port);

        // update the port status
        notifyPortStatusChanged(port);
    }

    @Override
    public void notifyPortLabelChanged(final SwitchPort port) {
        LOGGER.info("The port label was changed for port: {}", port);

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, SwitchPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, SwitchPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final SwitchPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final SwitchPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            SwitchPort current = (SwitchPort) getValueAt(row, SwitchPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(SwitchPort port) {
        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(final SwitchPort port, PortConfigKeys... portConfigKeys) {

        LOGGER.info("The values of the port have changed: {}", port);
        portListener.configChanged(port, portConfigKeys);
    }

    @Override
    public void changePortType(LcOutputType portType, Port<? extends BidibStatus> port) {

        portListener.changePortType(portType, port);
    }

    @Override
    protected SwitchPort getRow(int rowIndex) {
        SwitchPort port = (SwitchPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return port;
    }

    public void fireTestButtonPressed(SwitchPort port) {
        LOGGER.info("Port status has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }

}
