package org.bidib.wizard.mvc.loco.view.speedo;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.PomAddressData;
import org.bidib.jbidibc.messages.enums.PomAddressTypeEnum;
import org.bidib.jbidibc.messages.enums.PomOperation;
import org.bidib.jbidibc.messages.enums.PomProgState;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.converter.StringConverter;
import org.bidib.wizard.client.common.rxjava2.PropertyChangeEventSource;
import org.bidib.wizard.client.common.rxjava2.PropertyChangeEventSource.Pair;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.IntegerRangeFilter;
import org.bidib.wizard.client.common.text.LabeledTextField;
import org.bidib.wizard.client.common.text.TrailingLabeledTextField;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.model.status.DirectionStatus;
import org.bidib.wizard.model.status.SpeedSteps;
import org.bidib.wizard.mvc.common.view.ViewCloseListener;
import org.bidib.wizard.mvc.common.view.converter.CalculatingConverter;
import org.bidib.wizard.mvc.common.view.graph.LedBarGraph;
import org.bidib.wizard.mvc.common.view.graph.LedBarGraph.Orientation;
import org.bidib.wizard.mvc.loco.model.LocoModel;
import org.bidib.wizard.mvc.loco.model.SpeedometerModel;
import org.bidib.wizard.mvc.loco.model.SpeedometerModel.SpeedMeasurementStage;
import org.bidib.wizard.mvc.loco.model.SpeedometerProgBeanModel;
import org.bidib.wizard.mvc.loco.model.command.PomRequestProcessor;
import org.bidib.wizard.mvc.loco.model.command.SpeedometerPomCommand;
import org.bidib.wizard.mvc.loco.model.listener.LocoModelListener;
import org.bidib.wizard.mvc.loco.view.EmptyAddressHandler;
import org.bidib.wizard.mvc.loco.view.PomProgrammerRequestListener;
import org.bidib.wizard.mvc.loco.view.command.PomProgResultListener;
import org.bidib.wizard.mvc.loco.view.listener.LogPaneProvider;
import org.bidib.wizard.mvc.loco.view.listener.MeasurementViewListener;
import org.bidib.wizard.mvc.loco.view.listener.ProgressStatusCallback;
import org.bidib.wizard.mvc.pom.model.PomProgrammerModel;
import org.bidib.wizard.mvc.pom.model.ProgCommandAwareBeanModel;
import org.bidib.wizard.mvc.pom.model.command.PomOperationCommand;
import org.bidib.wizard.mvc.pom.view.panel.AbstractPomPanel;
import org.bidib.wizard.mvc.pom.view.panel.PomProgStateConverter;
import org.bidib.wizard.mvc.pom.view.panel.PomResultProxyModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

import com.jgoodies.binding.PresentationModel;
import com.jgoodies.binding.beans.PropertyAdapter;
import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.Trigger;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.FormsSetup;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.Severity;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.ValidationResultModel;
import com.jgoodies.validation.util.DefaultValidationResultModel;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.InfiniteProgressPanel;
import com.jidesoft.swing.OverlayTextArea;

import eu.hansolo.steelseries.extras.Led;
import eu.hansolo.steelseries.tools.LedColor;
import io.reactivex.rxjava3.core.SingleObserver;
import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.functions.Action;
import io.reactivex.rxjava3.functions.Consumer;

public class SpeedometerPanel<M extends ProgCommandAwareBeanModel> implements ViewCloseListener, LogPaneProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(SpeedometerPanel.class);

    private static final Logger LOGGER_MEASUREMENT = LoggerFactory.getLogger("MEASUREMENT");

    public static final int DEFAULT_TIMEOUT = 1000;

    private final static String NEWLINE = "\n";

    private final JPanel contentPanel;

    private ValueModel speedoFunctionsEnabled;

    private final JButton readCvButton;

    private final JButton writeCvButton;

    private JButton stopButton;

    private final JTextArea loggerArea;

    private final SpeedometerModel speedometerModel;

    private final LocoModel locoModel;

    private ValueModel addressValueModel;

    private JTextField address;

    private JComboBox<SpeedSteps> speedStepsCombo;

    private JTextField firmware;

    private final SpeedometerProgProxyBeanModel speedometerProgProxyBeanModel;

    private final SpeedometerProgBeanModel speedometerProgBeanModel;

    private final PresentationModel<SpeedometerProgBeanModel> speedometerPresentationModel;

    protected ValueModel currentOperationModel;

    protected JLabel currentOperationLabel;

    protected final PomProgrammerModel cvProgrammerModel;

    protected final PomResultProxyModel pomResultProxyModel;

    private ImageIcon progOperationErrorIcon;

    private ImageIcon progOperationSuccessfulIcon;

    private ImageIcon progOperationWaitIcon;

    private ImageIcon progOperationUnknownIcon;

    private ImageIcon progOperationEmptyIcon;

    private final InfiniteProgressPanel progressPanel;

    private final PomRequestProcessor<SpeedometerProgBeanModel> pomRequestProcessor;

    private boolean manualMeasurement = true;

    private final Led measurementLed;

    private TrailingLabeledTextField vSpeedStep1;

    private TrailingLabeledTextField vMiddle;

    private Trigger trigger;

    private JPanel locoAddressPanel;

    private ValidationResultModel validationModel;

    private LabeledTextField scale;

    private JToggleButton measureButton;

    private PropertyValidationI18NSupport validationSupport;

    private LedBarGraph ledBarGraph;

    private final Holder<MeasurementProgressDialog> measurementProgressDialogHolder = new Holder<>();

    private final SpeedometerProgBeanModel backupSpeedometerProgBeanModel;

    public SpeedometerPanel(final SpeedometerModel speedometerModel, final LocoModel locoModel,
        final PomProgrammerRequestListener pomProgrammerRequestListener, final SettingsService settingsService) {
        this.speedometerModel = speedometerModel;
        this.locoModel = locoModel;

        speedoFunctionsEnabled = new ValueHolder(false);

        initializeIcons();

        validationModel = new DefaultValidationResultModel();

        cvProgrammerModel = new PomProgrammerModel();
        this.speedometerModel.setPomProgrammerModel(cvProgrammerModel);

        // create a proxy model that receives updates from the programmer model but does not update values if not active
        pomResultProxyModel = new PomResultProxyModel();
        pomResultProxyModel.setAllowUpdate(true);

        speedometerProgBeanModel = new SpeedometerProgBeanModel();
        speedometerProgProxyBeanModel = new SpeedometerProgProxyBeanModel(speedometerProgBeanModel);

        backupSpeedometerProgBeanModel = new SpeedometerProgBeanModel();

        trigger = new Trigger();
        speedometerPresentationModel = new PresentationModel<>(speedometerProgBeanModel, trigger);

        validationSupport = new PropertyValidationI18NSupport(speedometerProgBeanModel, "validation");

        final PomProgResultListener pomProgResultListener = new PomProgResultListener() {

            @Override
            public void addLogText(String logLine, Object... args) {
                SpeedometerPanel.this.addLogText(logLine, args);
            }

            @Override
            public void enableInputElements() {
                SpeedometerPanel.this.enableInputElements();
            }

            @Override
            public void disableInputElements() {
                SpeedometerPanel.this.disableInputElements();
            }
        };

        this.pomRequestProcessor =
            new PomRequestProcessor<>(speedometerProgProxyBeanModel, locoModel, cvProgrammerModel,
                pomProgrammerRequestListener, pomProgResultListener, settingsService);

        this.speedometerModel.setPomRequestProcessor(pomRequestProcessor);

        this.pomRequestProcessor.addPomProgrammerModelListeners();

        FormBuilder formBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(SpeedometerPanel.class, "title");
                }

                @Override
                public Dimension getPreferredSize() {
                    return new Dimension(super.getPreferredSize().width + 20, super.getPreferredSize().height);
                }
            };
            // formBuilder = new DefaultFormBuilder(new FormLayout("pref, 3dlu, pref:grow"), panel);
            formBuilder =
                FormBuilder.create().columns("pref, 3dlu, pref:grow").rows("pref, 5dlu, fill:100dlu:grow").panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout()) {

                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(SpeedometerPanel.class, "title");
                }
            };
            // formBuilder = new DefaultFormBuilder(new FormLayout("pref, 3dlu, pref:grow"), panel);
            formBuilder =
                FormBuilder.create().columns("pref, 3dlu, pref:grow").rows("pref, 5dlu, fill:100dlu:grow").panel(panel);
        }
        formBuilder.border(Paddings.TABBED_DIALOG);

        measurementLed = new Led();
        measurementLed.setLedColor(LedColor.ORANGE_LED);
        measurementLed.setSize(24, 24);

        readCvButton = new JButton(Resources.getString(getClass(), "readCv"));
        readCvButton.setEnabled(false);

        readCvButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireReadCv();
            }
        });

        writeCvButton = new JButton(Resources.getString(getClass(), "writeCv"));
        writeCvButton.setEnabled(false);

        writeCvButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireWriteCv();
            }
        });

        stopButton = new JButton(Resources.getString(getClass(), "stop"));

        stopButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireStop();
            }
        });

        // add address panel
        locoAddressPanel = createLocoAddressPanel();

        PropertyValidationSupport support = new PropertyValidationI18NSupport(locoModel, "validation");

        validationModel.setResult(support.getResult());

        // Padding for overlay icon
        final Icon dirtyIcon = ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/bullet_wrench.png", 12, 12);
        IconFeedbackPanel iconPanel = new IconFeedbackPanel(this.validationModel, locoAddressPanel, dirtyIcon);

        formBuilder.add(iconPanel).xyw(1, 1, 3);

        // formBuilder.appendRow("5dlu");
        // formBuilder.appendRow("fill:100dlu:grow");
        // formBuilder.nextLine(2);

        FormBuilder formBuilderProgResult = null;
        boolean debugDialogProgResult = false;
        if (debugDialogProgResult) {
            // formBuilderProgResult =
            // new DefaultFormBuilder(new FormLayout("pref, 3dlu, pref:grow"), new FormDebugPanel());
            formBuilderProgResult =
                FormBuilder
                    .create().columns("pref, 3dlu, pref:grow").rows("pref, 5dlu, fill:100dlu:grow")
                    .panel(new FormDebugPanel());
        }
        else {
            // formBuilderProgResult = new DefaultFormBuilder(new FormLayout("pref, 3dlu, pref:grow"));
            formBuilderProgResult =
                FormBuilder.create().columns("pref, 3dlu, pref:grow").rows("pref, 5dlu, fill:100dlu:grow");
        }

        // prepare the operation verdict
        currentOperationModel =
            new PropertyAdapter<PomResultProxyModel>(pomResultProxyModel, PomResultProxyModel.PROPERTYNAME_POMPROGSTATE,
                true);
        ValueModel valueConverterModel = new ConverterValueModel(currentOperationModel, new PomProgStateConverter());
        currentOperationLabel = WizardComponentFactory.createLabel(valueConverterModel);
        currentOperationLabel.setIcon(progOperationEmptyIcon);
        formBuilderProgResult.add(Resources.getString(AbstractPomPanel.class, "prog-result")).xy(1, 1);
        formBuilderProgResult.add(currentOperationLabel).xy(3, 1);

        // prepare the logger area
        loggerArea = new OverlayTextArea();
        loggerArea.setColumns(40);
        loggerArea.setRows(20);

        loggerArea.setFont(UIManager.getDefaults().getFont("Label.font"));

        JScrollPane scrollPane =
            new JScrollPane(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        scrollPane.getViewport().add(loggerArea);
        loggerArea.setEditable(false);
        scrollPane.setOpaque(FormsSetup.getOpaqueDefault());
        scrollPane.getViewport().setOpaque(FormsSetup.getOpaqueDefault());

        JPopupMenu popupMenu = new BasicPopupMenu();
        JMenuItem clearConsole = new JMenuItem(Resources.getString(getClass(), "clear_console"));
        clearConsole.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireClearConsole();
            }
        });
        popupMenu.add(clearConsole);

        loggerArea.setComponentPopupMenu(popupMenu);

        // formBuilderProgResult.appendRow("5dlu");
        // formBuilderProgResult.appendRow("fill:100dlu:grow");
        // formBuilderProgResult.nextLine(2);

        final DefaultOverlayable overlayTextArea = new DefaultOverlayable(scrollPane);
        progressPanel = new InfiniteProgressPanel() {
            private static final long serialVersionUID = 1L;

            @Override
            public Dimension getPreferredSize() {
                return new Dimension(20, 20);
            }
        };
        overlayTextArea.addOverlayComponent(progressPanel);
        progressPanel.stop();
        overlayTextArea.setOverlayVisible(false);

        formBuilderProgResult.add(overlayTextArea).xyw(1, 3, 3);

        JPanel progResultPanel = formBuilderProgResult.build();
        formBuilder.add(progResultPanel).xyw(1, 3, 3);

        // build the panel
        contentPanel = formBuilder.build();

        // add property change listener on speed measurement stage
        speedometerModel.addPropertyChangeListener(SpeedometerModel.PROPERTYNAME_SPEEDMEASUREMENTSTAGE, evt -> {
            LOGGER.info("The speedmeasurement stage has changed: {}", speedometerModel.getSpeedMeasurementStage());

            switch (speedometerModel.getSpeedMeasurementStage()) {
                case WAIT_FOR_CAR_SPEED_AND_BATTERY_RESPONSE:
                    addLogText("Wait for speed and battery status from car with address: {}", locoModel.getAddress());
                    break;
                case READ_CURRENT_CV_VALUES:
                    addLogText("Read the current Vmin and Scale values from car with address: {}",
                        locoModel.getAddress());
                    break;
                case ABORTED:
                    addLogText("Current measurement was aborted.");
                    break;
                case FINISHED:
                    addLogText("Current measurement has finished.");
                    break;
                default:
                    addLogText("Received update of speedmeasurement stage: {}",
                        speedometerModel.getSpeedMeasurementStage());
                    break;
            }
        });

        speedometerModel.addPropertyChangeListener(SpeedometerModel.PROPERTYNAME_DYNSTATEENERGY, evt -> {
            LOGGER.info("The DYN_STATE has changed: {}", speedometerModel.getDynStateEnergy());

        });
        speedometerModel.addPropertyChangeListener(SpeedometerModel.PROPERTYNAME_SPEED, evt -> {
            LOGGER.info("The SPEED has changed: {}", speedometerModel.getSpeed());

        });

        addPomProgrammerModelListeners();

        doBindButtons();

    }

    private void addPomProgrammerModelListeners() {

        cvProgrammerModel
            .addPropertyChangeListener(PomProgrammerModel.PROPERTYNAME_POMPROGSTATE, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    final PomProgState pomProgState = (PomProgState) evt.getNewValue();
                    LOGGER.info("The pomProgState has changed: {}", pomProgState);
                    if (SwingUtilities.isEventDispatchThread()) {
                        signalPomProgStateChanged(pomProgState);
                    }
                    else {
                        SwingUtilities.invokeLater(new Runnable() {
                            @Override
                            public void run() {
                                signalPomProgStateChanged(pomProgState);
                            }
                        });
                    }
                }
            });

    }

    /**
     * Initialize the icons
     */
    private void initializeIcons() {
        // Load the icons
        progOperationErrorIcon = ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/accessory-error.png");
        progOperationSuccessfulIcon =
            ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/accessory-successful.png");
        progOperationWaitIcon = ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/accessory-wait.png");
        progOperationUnknownIcon = ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/information.png");
        progOperationEmptyIcon = ImageUtils.createImageIcon(SpeedometerPanel.class, "/icons/empty.png");
    }

    protected void doBindButtons() {
        // add bindings for enable/disable the write button

        final EmptyAddressHandler handler = new EmptyAddressHandler(value -> {
            LOGGER.info("The address value is not empty: {}", value);

            speedoFunctionsEnabled.setValue(value.booleanValue());

            // speedRangeMeasurementEnabled.setValue(value.booleanValue());
        });
        PropertyConnector
            .connect(locoModel, LocoModel.PROPERTYNAME_ADDRESS, handler, EmptyAddressHandler.PROPERTYNAME_ADDRESS)
            .updateProperty2();

        PropertyConnector.connect(speedoFunctionsEnabled, "value", readCvButton, "enabled");
        PropertyConnector.connect(speedoFunctionsEnabled, "value", writeCvButton, "enabled");
    }

    public JPanel getComponent() {
        return contentPanel;
    }

    private void fireClearConsole() {
        LOGGER.info("clear the console.");

        loggerArea.setText(null);
    }

    private JPanel createLocoAddressPanel() {
        FormBuilder formBuilder =
            FormBuilder
                .create()
                .columns(
                    "pref, 3dlu, 50dlu, 10dlu, pref, 3dlu, 50dlu, 10dlu, pref, 3dlu, 50dlu, 10dlu, pref, 3dlu, 50dlu, 10dlu, pref, pref:grow")
                .rows(
                    "pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref, 5dlu, pref");

        formBuilder.add(readCvButton).xyw(1, 1, 3);
        formBuilder.add(writeCvButton).xyw(5, 1, 3);

        formBuilder.add(stopButton).xyw(11, 1, 3);
        // formBuilder.nextLine();

        formBuilder.addSeparator(Resources.getString(getClass(), "vehicle")).xyw(1, 3, 18);

        addressValueModel = new PropertyAdapter<LocoModel>(locoModel, LocoModel.PROPERTYNAME_ADDRESS, true);

        final ValueModel addressConverterModel =
            new ConverterValueModel(addressValueModel, new StringConverter(new DecimalFormat("#")));

        address = WizardComponentFactory.createTextField(addressConverterModel, false);
        // address.setEditable(false);

        address.setDocument(new InputValidationDocument(InputValidationDocument.NUMERIC));

        // set the default address
        if (locoModel.getAddress() != null) {
            address.setText(Integer.toString(locoModel.getAddress()));
        }
        else {
            address.setText(null);
        }

        address.getDocument().addDocumentListener(new DocumentListener() {
            @Override
            public void changedUpdate(DocumentEvent e) {
                valueChanged();
            }

            @Override
            public void removeUpdate(DocumentEvent e) {
                valueChanged();
            }

            @Override
            public void insertUpdate(DocumentEvent e) {
                valueChanged();
            }

            private void valueChanged() {
                Integer currentAddress = null;

                if (!address.getText().isEmpty()) {
                    try {
                        currentAddress = Integer.valueOf(address.getText());
                    }
                    catch (NumberFormatException e) {
                        LOGGER.warn("Parse decoder address failed.");
                    }
                }
                if (locoModel != null) {
                    LOGGER.info("Set the new address: {}", currentAddress);
                    locoModel.setAddress(currentAddress);
                }
            }
        });

        formBuilder.add(Resources.getString(getClass(), "address")).xy(1, 5);
        formBuilder.add(address).xy(3, 5);

        // speedsteps
        boolean m4SupportEnabled = false /* Preferences.getInstance().isM4SupportEnabled() */;

        if (m4SupportEnabled) {
            speedStepsCombo = new JComboBox<SpeedSteps>(SpeedSteps.values());
        }
        else {
            // filter the M4 speed steps

            List<SpeedSteps> speedSteps = new ArrayList<>();
            speedSteps.addAll(Arrays.asList(SpeedSteps.values()));
            CollectionUtils.filter(speedSteps, new Predicate<SpeedSteps>() {

                @Override
                public boolean evaluate(SpeedSteps speedStep) {
                    return (speedStep.ordinal() <= SpeedSteps.DCC128.ordinal());
                }
            });

            speedStepsCombo = new JComboBox<SpeedSteps>(speedSteps.toArray(new SpeedSteps[0]));
        }

        speedStepsCombo.setSelectedItem(locoModel.getSpeedSteps());

        speedStepsCombo.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("Speed steps combo has changed, update loco model.");
                locoModel.setSpeedSteps((SpeedSteps) ((JComboBox<SpeedSteps>) e.getSource()).getSelectedItem());
            }
        });

        formBuilder.add(Resources.getString(getClass(), "speedSteps")).xy(5, 5);
        formBuilder.add(speedStepsCombo).xy(7, 5);

        // firmware

        BufferedValueModel firmwareValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_FIRMWARE);
        firmware = WizardComponentFactory.createTextField(firmwareValueModel, false);
        firmware.setEditable(false);

        formBuilder.add(Resources.getString(getClass(), "firmware")).xy(9, 5);
        formBuilder.add(firmware).xy(11, 5);

        // add the battery led bargraph
        ledBarGraph = new LedBarGraph(10, Orientation.horizontal);
        ledBarGraph.setOpaque(FormsSetup.getOpaqueDefault());
        formBuilder.add(ledBarGraph).xyw(13, 5, 3);

        // add loco model listener
        LocoModelListener locoModelListener = new LocoModelListener() {

            @Override
            public void directionChanged(DirectionStatus direction) {

            }

            @Override
            public void functionChanged(int index, boolean value) {

            }

            @Override
            public void speedStepsChanged(SpeedSteps speedSteps) {

            }

            @Override
            public void dynStateEnergyChanged(int dynStateEnergy) {
                LOGGER.info("The dynStateEnergy has changed, dynStateEnergy: {}", dynStateEnergy);

                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {

                        if (ledBarGraph != null) {
                            ledBarGraph.setValue(dynStateEnergy);
                        }
                    }
                });
            }

            @Override
            public void binaryStateChanged(int state, boolean value) {

            }

            @Override
            public void triggerClearLoco() {
            };

        };
        locoModel.addLocoModelListener(locoModelListener);

        // formBuilder.nextLine();

        ValidationComponentUtils.setMandatory(firmware, true);
        ValidationComponentUtils.setMessageKey(firmware, "validation.firmware_key");

        prepareFirmwareValidationListener(firmware, firmwareValueModel, speedometerProgBeanModel, "firmware_key");

        // Vmin, Vmax

        BufferedValueModel vMinValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV2_VMIN);

        final ValueModel vMinConverterModel =
            new ConverterValueModel(vMinValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv2Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv2Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        final TrailingLabeledTextField vMin =
            WizardComponentFactory.createTrailingLabeledTextField(vMinConverterModel, false, "PWM%", cv2Document);
        vMin.getTextField().setEditable(manualMeasurement);
        vMin.setHintText("CV2");
        // vMin.setLabelText("km/h");
        vMin.setToolTipText(Resources.getString(getClass(), "vMinPref.tooltip"));

        final JLabel vMinPrefLabel = new JLabel(Resources.getString(getClass(), "vMinPref"));
        vMinPrefLabel.setToolTipText(Resources.getString(getClass(), "vMinPref.tooltip"));
        formBuilder.add(vMinPrefLabel).xy(1, 7);
        formBuilder.add(vMin).xy(3, 7);

        ValidationComponentUtils.setMandatory(vMin, true);
        ValidationComponentUtils.setMessageKey(vMin, "validation.cv2_key");

        prepareDirtyListener(vMin, vMinValueModel, speedometerProgBeanModel,
            SpeedometerProgBeanModel.PROPERTYNAME_CV2_VMIN, "cv2_key");

        BufferedValueModel vMaxValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV5_VMAX);

        final ValueModel vMaxConverterModel =
            new ConverterValueModel(vMaxValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv5Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv5Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        TrailingLabeledTextField vMax =
            WizardComponentFactory.createTrailingLabeledTextField(vMaxConverterModel, false, "PWM%", cv5Document);
        vMax.getTextField().setEditable(manualMeasurement);
        vMax.setHintText("CV5");
        // vMax.setLabelText("%");
        vMax.setToolTipText(Resources.getString(getClass(), "vMax.tooltip"));
        final JLabel vMaxLabel = new JLabel(Resources.getString(getClass(), "vMax"));
        vMaxLabel.setToolTipText(Resources.getString(getClass(), "vMax.tooltip"));
        formBuilder.add(vMaxLabel).xy(5, 7);
        formBuilder.add(vMax).xy(7, 7);

        // vMaxConverterModel.addValueChangeListener(evt -> validate());
        ValidationComponentUtils.setMandatory(vMax, true);
        ValidationComponentUtils.setMessageKey(vMax, "validation.cv5_key");

        prepareDirtyListener(vMax, vMaxValueModel, speedometerProgBeanModel,
            SpeedometerProgBeanModel.PROPERTYNAME_CV5_VMAX, "cv5_key");

        // add the button for automatic speed adjustment
        // formBuilder.nextColumn(2);
        JButton autoAdjustButton = new JButton(Resources.getString(getClass(), "autoAdjust"));
        autoAdjustButton.setToolTipText(Resources.getString(getClass(), "autoAdjust.tooltip"));
        autoAdjustButton.setEnabled(false);

        autoAdjustButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireStartAutoSpeedAdjustment();
            }
        });

        formBuilder.add(autoAdjustButton).xyw(11, 7, 3);

        // formBuilder.nextLine();

        // speedometer
        formBuilder.addSeparator("Speedometer").xyw(1, 9, 18);

        // measurement
        measureButton = new JToggleButton(Resources.getString(getClass(), "measurement"));
        measureButton.setEnabled(false);
        measureButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireToggleMeasure(measureButton.isSelected());
            }
        });
        measureButton.setEnabled(false);

        ValueModel speedValueModel =
            new PropertyAdapter<SpeedometerModel>(speedometerModel, SpeedometerModel.PROPERTYNAME_REPORTEDSPEED, true);

        final ValueModel speedConverterModel =
            new ConverterValueModel(speedValueModel, new StringConverter(new DecimalFormat("#")));

        TrailingLabeledTextField speedValue =
            WizardComponentFactory.createTrailingLabeledTextField(speedConverterModel, false, "mm/s", null);
        speedValue.getTextField().setEditable(false);
        speedValue.setHintText("Speed");

        formBuilder.add(Resources.getString(getClass(), "speed")).xy(1, 11);
        formBuilder.add(speedValue).xy(3, 11);

        BufferedValueModel scaleValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV37_SCALE);

        final ValueModel scaleConverterModel =
            new ConverterValueModel(scaleValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv37Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv37Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        scale = WizardComponentFactory.createLabeledTextField(scaleConverterModel, false, "1:", cv37Document);
        scale.getTextField().setEditable(manualMeasurement);
        scale.setHintText("CV37");

        formBuilder.add(Resources.getString(getClass(), "scale")).xy(5, 11);
        formBuilder.add(scale).xy(7, 11);

        ValidationComponentUtils.setMandatory(scale, true);
        ValidationComponentUtils.setMessageKey(scale, "validation.scale_key");

        prepareDirtyListener(scale, scaleValueModel, speedometerProgBeanModel,
            SpeedometerProgBeanModel.PROPERTYNAME_CV37_SCALE, "scale_key");

        speedometerProgBeanModel.addPropertyChangeListener(SpeedometerProgBeanModel.PROPERTYNAME_CV37_SCALE, evt -> {
            // measurement button

            boolean scaleAvailable = speedometerProgBeanModel.getCv37Scale() != null;
            LOGGER.info("The scale property has changed: {}", scaleAvailable);

            measureButton
                .setEnabled((scaleAvailable | measureButton.isSelected())
                    && Boolean.TRUE.equals(speedoFunctionsEnabled.getValue()));

        });

        ValueModel speedKmHValueModel =
            new PropertyAdapter<SpeedometerModel>(speedometerModel, SpeedometerModel.PROPERTYNAME_REPORTEDSPEED, true);

        final ValueModel speedKmHConverterModel =
            new ConverterValueModel(speedKmHValueModel, new CalculatingConverter(new DecimalFormat("#"), val -> {
                Integer scaleVal = speedometerProgBeanModel.getCv37Scale();
                int speedVal = 0;
                if (scaleVal != null && scaleVal > 1) {
                    speedVal = (int) (((double) val * scaleVal) / 277.778);
                    // speedVal = (int) ((double) val * 0.3132);

                    LOGGER.info("Converted speed, {} mm/s --> {} km/h", val, speedVal);
                }
                else {
                    speedVal = val;
                }

                return speedVal;
            }));

        TrailingLabeledTextField speedKmHValue =
            WizardComponentFactory.createTrailingLabeledTextField(speedKmHConverterModel, false, "km/h", null);
        speedKmHValue.getTextField().setEditable(false);
        speedKmHValue.setHintText("Speed");

        formBuilder.add(Resources.getString(getClass(), "speed")).xy(9, 11);
        formBuilder.add(speedKmHValue).xy(11, 11);

        formBuilder.add(measureButton).xy(13, 11);
        formBuilder.add(measurementLed).xy(15, 11);

        PropertyConnector.connect(speedoFunctionsEnabled, "value", measureButton, "enabled");

        // formBuilder.nextLine();

        // velocity
        formBuilder.addSeparator(Resources.getString(getClass(), "velocity")).xyw(1, 13, 18);

        // speed values in mm/s

        BufferedValueModel vSpeedStep1ValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV35);

        final ValueModel vSpeedStep1ConverterModel =
            new ConverterValueModel(vSpeedStep1ValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv35Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv35Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        vSpeedStep1 =
            WizardComponentFactory
                .createTrailingLabeledTextField(vSpeedStep1ConverterModel, false, "mm/s", cv35Document);
        vSpeedStep1.getTextField().setEditable(manualMeasurement);
        vSpeedStep1.setHintText("CV35");

        JButton speedStep2Button = new JButton(Resources.getString(getClass(), "speedStep2"));
        speedStep2Button.setEnabled(false);
        speedStep2Button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSpeedStep(2);
            }
        });

        JButton speedToCv35Button = new JButton(Resources.getString(getClass(), "speedToCv35"));
        speedToCv35Button.setToolTipText(Resources.getString(getClass(), "speedToCv35.tooltip"));
        speedToCv35Button.setEnabled(false);

        speedToCv35Button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSpeedToCv35();
            }
        });

        formBuilder.add(speedStep2Button).xyw(1, 15, 3);
        formBuilder.add(speedToCv35Button).xyw(5, 15, 3);
        formBuilder.add(Resources.getString(getClass(), "vMin")).xy(9, 15);
        formBuilder.add(vSpeedStep1).xy(11, 15);

        PropertyConnector.connect(speedoFunctionsEnabled, "value", speedStep2Button, "enabled");
        PropertyConnector.connect(speedoFunctionsEnabled, "value", speedToCv35Button, "enabled");

        // vSpeedStep1ValueModel.addValueChangeListener(evt -> validate());
        ValidationComponentUtils.setMandatory(vSpeedStep1, true);
        ValidationComponentUtils.setMessageKey(vSpeedStep1, "validation.cv35_key");

        prepareDirtyListener(vSpeedStep1, vSpeedStep1ValueModel, speedometerProgBeanModel,
            SpeedometerProgBeanModel.PROPERTYNAME_CV35, "cv35_key");

        // formBuilder.nextLine();

        BufferedValueModel vMiddleValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV36);

        final ValueModel vMiddleConverterModel =
            new ConverterValueModel(vMiddleValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv36Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv36Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        vMiddle =
            WizardComponentFactory.createTrailingLabeledTextField(vMiddleConverterModel, false, "mm/s", cv36Document);
        vMiddle.getTextField().setEditable(manualMeasurement);
        vMiddle.setHintText("CV36");

        JButton speedStep64Button = new JButton(Resources.getString(getClass(), "speedStep64"));
        speedStep64Button.setEnabled(false);
        speedStep64Button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSpeedStep(64);
            }
        });

        JButton speedToCv36Button = new JButton(Resources.getString(getClass(), "speedToCv36"));
        speedToCv36Button.setEnabled(false);
        speedToCv36Button.setToolTipText(Resources.getString(getClass(), "speedToCv36.tooltip"));

        speedToCv36Button.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSpeedToCv36();
            }
        });

        formBuilder.add(speedStep64Button).xyw(1, 17, 3);
        formBuilder.add(speedToCv36Button).xyw(5, 17, 3);
        formBuilder.add(Resources.getString(getClass(), "vMid")).xy(9, 17);
        formBuilder.add(vMiddle).xy(11, 17);

        // vMiddleConverterModel.addValueChangeListener(evt -> validate());
        // ValidationComponentUtils.setMandatory(vMiddle, true);
        ValidationComponentUtils.setMessageKey(vMiddle, "validation.cv36_key");

        prepareDirtyListener(vMiddle, vMiddleValueModel, speedometerProgBeanModel,
            SpeedometerProgBeanModel.PROPERTYNAME_CV36, "cv36_key");

        PropertyConnector.connect(speedoFunctionsEnabled, "value", speedStep64Button, "enabled");
        PropertyConnector.connect(speedoFunctionsEnabled, "value", speedToCv36Button, "enabled");

        // enable the autoAdjustButton
        PropertyConnector.connect(speedoFunctionsEnabled, "value", autoAdjustButton, "enabled");

        // formBuilder.nextLine();

        // PID values
        formBuilder.addSeparator(Resources.getString(getClass(), "pid")).xyw(1, 19, 18);

        // P value
        BufferedValueModel pValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV61);

        final ValueModel pConverterModel =
            new ConverterValueModel(pValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv61Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv61Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        TrailingLabeledTextField pValue =
            WizardComponentFactory.createTrailingLabeledTextField(pConverterModel, false, "   ", cv61Document);
        pValue.getTextField().setEditable(manualMeasurement);
        pValue.setHintText("CV61");

        formBuilder.add(Resources.getString(getClass(), "pValue")).xy(1, 21);
        formBuilder.add(pValue).xy(3, 21);

        // pValueModel.addValueChangeListener(evt -> validate());
        ValidationComponentUtils.setMandatory(pValue, true);
        ValidationComponentUtils.setMessageKey(pValue, "validation.cv61_key");

        prepareDirtyListener(pValue, pValueModel, speedometerProgBeanModel, SpeedometerProgBeanModel.PROPERTYNAME_CV61,
            "cv61_key");

        // I value

        BufferedValueModel iValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV62);

        final ValueModel iConverterModel =
            new ConverterValueModel(iValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv62Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv62Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        TrailingLabeledTextField iValue =
            WizardComponentFactory.createTrailingLabeledTextField(iConverterModel, false, "   ", cv62Document);
        iValue.getTextField().setEditable(manualMeasurement);
        iValue.setHintText("CV62");

        formBuilder.add(Resources.getString(getClass(), "iValue")).xy(5, 21);
        formBuilder.add(iValue).xy(7, 21);

        // iValueModel.addValueChangeListener(evt -> validate());
        ValidationComponentUtils.setMandatory(iValue, true);
        ValidationComponentUtils.setMessageKey(iValue, "validation.cv62_key");

        prepareDirtyListener(iValue, iValueModel, speedometerProgBeanModel, SpeedometerProgBeanModel.PROPERTYNAME_CV62,
            "cv62_key");

        // D value
        BufferedValueModel dValueModel =
            speedometerPresentationModel.getBufferedModel(SpeedometerProgBeanModel.PROPERTYNAME_CV63);

        final ValueModel dConverterModel =
            new ConverterValueModel(dValueModel, new StringConverter(new DecimalFormat("#")));

        InputValidationDocument cv63Document = new InputValidationDocument(3, InputValidationDocument.NUMERIC);
        cv63Document.setDocumentFilter(new IntegerRangeFilter(0, 255));

        TrailingLabeledTextField dValue =
            WizardComponentFactory.createTrailingLabeledTextField(dConverterModel, false, "   ", cv63Document);
        dValue.getTextField().setEditable(manualMeasurement);
        dValue.setHintText("CV63");

        formBuilder.add(Resources.getString(getClass(), "dValue")).xy(9, 21);
        formBuilder.add(dValue).xy(11, 21);

        // dValueModel.addValueChangeListener(evt -> validate());
        ValidationComponentUtils.setMandatory(dValue, true);
        ValidationComponentUtils.setMessageKey(dValue, "validation.cv63_key");

        prepareDirtyListener(dValue, dValueModel, speedometerProgBeanModel, SpeedometerProgBeanModel.PROPERTYNAME_CV63,
            "cv63_key");

        // formBuilder.nextLine();

        return formBuilder.build();
    }

    // // open the progress dialog
    // final MeasurementProgressDialog progressDialog = new MeasurementProgressDialog(contentPanel, modal, listeners);
    // measurementProgressDialogHolder.value = progressDialog;

    public void validate() {
        LOGGER.trace("Validate the values.");
        ValidationResult validationResult = validationSupport.getResult();

        validationModel.setResult(validationResult);
    }

    private void prepareDirtyListener(
        final JComponent textField, BufferedValueModel valueModel, SpeedometerProgBeanModel speedometerProgBeanModel,
        String cvNumber, String validationKey) {

        final Method getterMethod;
        try {
            getterMethod = SpeedometerProgBeanModel.class.getDeclaredMethod("get" + StringUtils.capitalize(cvNumber));

            valueModel.addValueChangeListener(evt -> {
                LOGGER.trace("Value of {} in model has changed, new value: {}", cvNumber, evt.getNewValue());

                try {
                    if (Objects.equals(valueModel.getValue(), getterMethod.invoke(speedometerProgBeanModel))) {
                        // remove dirty marker
                        validationSupport.remove(validationKey);
                    }
                    else {
                        // add dirty marker
                        validationSupport.add(Severity.INFO, validationKey, "value_dirty");
                    }
                    validate();
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the overlay visible/invisible failed.", ex);
                }
            });
        }
        catch (NoSuchMethodException | SecurityException ex) {
            LOGGER.warn("Get the getter method failed: {}", cvNumber, ex);
            throw new RuntimeException("Get the getter method failed: " + cvNumber);
        }

        valueModel.addPropertyChangeListener(BufferedValueModel.PROPERTY_BUFFERING, evt -> {

            if (Boolean.FALSE == evt.getNewValue()) {
                LOGGER.debug("{} is not buffering, hide overlay.", cvNumber);
                // remove dirty marker
                validationSupport.remove(validationKey);

                validate();
            }
        });
    }

    private void prepareFirmwareValidationListener(
        final JComponent textField, BufferedValueModel valueModel, SpeedometerProgBeanModel speedometerProgBeanModel,
        String validationKey) {

        // final Method getterMethod;
        try {

            valueModel.addValueChangeListener(evt -> {
                LOGGER.debug("Value of firmware in model has changed, new value: {}", evt.getNewValue());
                LOGGER
                    .debug("CV111: {}, CV59: {}", speedometerProgBeanModel.getCv111(),
                        speedometerProgBeanModel.getCv59());
                if (speedometerProgBeanModel.getCv111() != null && (speedometerProgBeanModel.getCv111() / 10 == 3)) {
                    if (speedometerProgBeanModel.getCv59() != null
                        && ((speedometerProgBeanModel.getCv59() & 0x80) == 0x00)) {
                        // no load control enabled
                        LOGGER.warn("No load control enabled.");
                        // remove dirty marker
                        validationSupport.remove(validationKey);
                        // add dirty marker
                        validationSupport.add(Severity.WARNING, validationKey, "load_control_not_active");
                    }
                }
                else {
                    // remove dirty marker
                    validationSupport.remove(validationKey);
                }

                validate();
            });
        }
        catch (Exception ex) {
            LOGGER.warn("Check if load control is enabled failed: {}", ex);
        }

        valueModel.addPropertyChangeListener(BufferedValueModel.PROPERTY_BUFFERING, evt -> {

            if (Boolean.FALSE == evt.getNewValue()) {
                LOGGER.info("Firmware is not buffering, hide overlay.");
                // remove dirty marker
                validationSupport.remove(validationKey);

                validate();
            }
        });
    }

    protected void fireSpeedToCv35() {
        Integer reportedSpeed = speedometerModel.getReportedSpeed();

        LOGGER.info("Set the speed step 2 speed: {}", reportedSpeed);

        vSpeedStep1.setText(reportedSpeed != null ? reportedSpeed.toString() : null);
    }

    protected void fireSpeedToCv36() {
        Integer reportedSpeed = speedometerModel.getReportedSpeed();

        LOGGER.info("Set the speed step 64 speed: {}", reportedSpeed);

        vMiddle.setText(reportedSpeed != null ? reportedSpeed.toString() : null);
    }

    @Override
    public void addLogText(String logMessage, Object... args) {
        // LOGGER.info("Add text to loggerArea, logLine: {}, args: {}", logLine, args);

        LOGGER.info(logMessage, args);

        if (args != null) {
            logMessage = MessageFormatter.arrayFormat(logMessage, args).getMessage();
        }
        SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss.SSS - ");
        final StringBuffer message = new StringBuffer(sdf.format(new Date()));
        message.append(logMessage).append(NEWLINE);

        if (SwingUtilities.isEventDispatchThread()) {

            loggerArea.append(message.toString());
            loggerArea.setCaretPosition(loggerArea.getDocument().getLength());
        }
        else {
            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                    loggerArea.append(message.toString());
                    loggerArea.setCaretPosition(loggerArea.getDocument().getLength());
                }
            });
        }
    }

    public void clearModelValues() {

        LOGGER.info("Clear the model values.");
        clearAllCvValues();

        fireClearConsole();
    }

    private void clearAllCvValues() {

        // clear the vMin and scale value
        speedometerProgProxyBeanModel.setCv2Vmin(null);
        speedometerProgProxyBeanModel.setCv5Vmax(null);
        speedometerProgProxyBeanModel.setCv37Scale(null);

        speedometerProgProxyBeanModel.setCv35(null);
        speedometerProgProxyBeanModel.setCv36(null);

        // V3: load control
        speedometerProgProxyBeanModel.setCv59(null);
        speedometerProgProxyBeanModel.setCv111(null);

        speedometerProgProxyBeanModel.setCv61(null);
        speedometerProgProxyBeanModel.setCv62(null);
        speedometerProgProxyBeanModel.setCv63(null);

        speedometerProgProxyBeanModel.setCv7(null);
        speedometerProgProxyBeanModel.setCv109(null);
        speedometerProgProxyBeanModel.setCv110(null);
    }

    private void backupCurrentCvValues() {
        LOGGER.info("Backup the current CV values from the speedometer bean model.");

        // TODO
        try {
            BeanUtils.copyProperties(backupSpeedometerProgBeanModel, speedometerProgBeanModel);
            LOGGER.info("Created a backup of the current cv values: {}", backupSpeedometerProgBeanModel);
        }
        catch (IllegalAccessException | InvocationTargetException ex) {
            LOGGER.warn("Backup current cv values failed.", ex);
        }
    }

    private void fireReadCvValues(final CallbackAction finishCallback) {
        LOGGER.info("Read the values from the decoder");

        if (!SwingUtilities.isEventDispatchThread()) {

            SwingUtilities.invokeLater(() -> internalFireReadCvValues(finishCallback));
        }
        else {
            internalFireReadCvValues(finishCallback);
        }

    }

    /**
     * Clear all CV values from proxy model and let the controller read the CV values.
     * 
     * @param finishCallback
     */
    private void internalFireReadCvValues(final CallbackAction finishCallback) {
        LOGGER.info("internalFireReadCvValues");

        speedoFunctionsEnabled.setValue(false);

        clearAllCvValues();

        SingleObserver<String> completeAction = new SingleObserver<String>() {

            @Override
            public void onSuccess(String t) {
                LOGGER.info("Read CV values was successful, t: {}", t);

                if (!isMeasurementAborted()) {

                    backupCurrentCvValues();

                    if (finishCallback != null) {
                        LOGGER
                            .info(
                                "Inside success handler of read CV values. Invoke callback to get the full speed range of the car.");

                        finishCallback.invoke();
                    }
                    else {

                        LOGGER.info("Set the speed measurement stage to finished.");
                        speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.FINISHED);

                        speedoFunctionsEnabled.setValue(true);
                    }
                }
                else {
                    LOGGER.info("After read cv values. The speed measurement is aborted already.");
                }
            }

            @Override
            public void onSubscribe(Disposable d) {
                LOGGER.info("Subscribed the complete action for read CV values, d: {}", d);

            }

            @Override
            public void onError(Throwable e) {
                LOGGER.warn("Read the CV values failed. Set the speed measurement stage to aborted.", e);

                speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.ABORTED);
                speedoFunctionsEnabled.setValue(true);

                fireMeasurementStop();
            }
        };

        speedometerModel.setCompleteAction(completeAction);

        LOGGER.info("Let the pomRequestProcessor load the related CV values.");

        // this will trigger the controller to perform the operation
        speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.READ_CURRENT_CV_VALUES);
    }

    private boolean isMeasurementAborted() {
        return SpeedMeasurementStage.ABORTED == speedometerModel.getSpeedMeasurementStage();
    }

    private void getFullSpeedRange() {
        LOGGER.info("Get the full speed range.");

        if (isMeasurementAborted()) {
            LOGGER.info("The measurement is aborted. Skip get full speed range.");
            return;
        }

        LOGGER.info("Set the speed measurement stage to STOP_FOR_USER_INTERACTION.");

        // this will set the speed to 0 and deactivate the measurement
        speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.STOP_FOR_USER_INTERACTION);

        speedoFunctionsEnabled.setValue(true);

        if (speedScaleBeanModelSaved == null) {
            speedScaleBeanModelSaved = new SpeedScaleBeanModel();
        }

        // set the scale value from the CV
        speedScaleBeanModelSaved.setCv37Scale(speedometerProgBeanModel.getCv37Scale());

        // ask the user if the scale is correct
        int result = JOptionPane.CANCEL_OPTION;
        final ApproveScaleDialog dialog =
            new ApproveScaleDialog(JOptionPane.getFrameForComponent(contentPanel), speedScaleBeanModelSaved);

        result = dialog.getResult();
        LOGGER.info("Result: {}", result);

        switch (result) {
            case JOptionPane.OK_OPTION:

                LOGGER.info("The approved scale value: {}", speedScaleBeanModelSaved.getCv37Scale());

                final CvParamsBeanModel cvParamsBeanModel = new CvParamsBeanModel();

                // check if the scale was changed
                if (speedScaleBeanModelSaved.getCv37Scale() != null
                    && speedScaleBeanModelSaved.getCv37Scale() != speedometerProgBeanModel.getCv37Scale()) {
                    LOGGER
                        .info("The scale value was changed. Write new scale to decoder: {}",
                            speedScaleBeanModelSaved.getCv37Scale());

                    cvParamsBeanModel.setCv37(speedScaleBeanModelSaved.getCv37Scale());

                }

                speedometerModel.setCv37Scale(speedScaleBeanModelSaved.getCv37Scale());

                // set the CV2 to a value of 1
                cvParamsBeanModel.setCv2(Integer.valueOf(1));
                // set the CV5 to a value of 100
                cvParamsBeanModel.setCv5(Integer.valueOf(100));

                // prepare write changed CV values to decoder

                LOGGER.info("Write default CV 2 and CV 5 values.");
                writeCvValues(cvParamsBeanModel, speedometerProgBeanModel, () -> {
                    LOGGER.info("The finish callback is called after change the scale CV");

                    updateProgressText("Detect full speed range.");

                    // start the full speed range detection
                    scheduleFullSpeedRangeDetection();

                }, () -> {
                    LOGGER.info("The error callback is called after change the scale CV");
                    speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.ABORTED);

                    fireMeasurementStop();
                });

                break;
            default:
                LOGGER.warn("User aborted the speed measurement operation in approve scale dialog.");

                fireMeasurementStop();
                break;
        }

    }

    private void updateProgressText(String progressText) {

        LOGGER_MEASUREMENT.info(progressText);

        if (SpeedometerPanel.this.measurementProgressCallback != null) {
            try {
                SpeedometerPanel.this.measurementProgressCallback.textChanged("<html>" + progressText + "</html>");
            }
            catch (Exception ex) {
                LOGGER.warn("Signal new progress text to callback failed.", ex);
            }
        }
    }

    private void scheduleFullSpeedRangeDetection() {
        LOGGER.info("Schedule the worker for the full speed range detection.");

        int delay = 0;
        measurementWorker.schedule(new Runnable() {
            @Override
            public void run() {

                LOGGER.info("Make sure the speed measurement is active.");
                updateProgressText("Activate measurement of speedometer.");
                try {
                    SwingUtilities.invokeAndWait(() -> fireToggleMeasure(true));
                }
                catch (Exception ex) {
                    LOGGER.warn("toggle measurement to on failed.", ex);
                }

                // wait 2s until the speed measurement has found a stable value
                try {
                    Thread.sleep(2000);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
                }

                final SpeedScaleBeanModel speedScaleBeanModel = speedScaleBeanModelSaved;
                Integer scaleVal = speedometerProgBeanModel.getCv37Scale();

                addLogText("Set the speed step 2.");
                fireSpeedStep(2);

                updateProgressText("Detect the minimum speed at speed step 2.");

                // wait for stable value
                speedAdjustmentLoop(speedConverterToKmH, scaleVal, 20,
                    val -> speedScaleBeanModel.setMeasuredMinSpeed(val), "Detect the minimum speed at speed step 2.");

                addLogText("The measured min speed: {}", speedScaleBeanModel.getMeasuredMinSpeed());

                addLogText("Set the speed step 126.");
                fireSpeedStep(126);

                updateProgressText("Wait 5s before start detect the maximum speed at speed step 126.");

                // wait 5s until the speed measurement has found a stable value
                try {
                    Thread.sleep(5000);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
                }

                if (isMeasurementAborted()) {
                    fireMeasurementStop();
                    return;
                }

                updateProgressText("Detect the maximum speed at speed step 126.");

                // wait for stable value
                speedAdjustmentLoop(speedConverterToKmH, scaleVal, 20,
                    val -> speedScaleBeanModel.setMeasuredMaxSpeed(val), "Detect the maximum speed at speed step 126.");

                addLogText("The measured max speed: {}", speedScaleBeanModel.getMeasuredMaxSpeed());

                // stop the car
                fireSpeedStep(0);

                LOGGER.info("Validate the min and max speed and scale value.");
                SwingUtilities.invokeLater(() -> validateVMinAndScaleValue());
            }
        }, delay, TimeUnit.MILLISECONDS);

    }

    private boolean speedAdjustmentLoop(
        final BiFunction<Integer, Integer, Integer> speedConverter, Integer scaleVal, int maxAdjustmentDuration,
        java.util.function.Consumer<Integer> speedValueConsumer, String logTextPrefix) {
        boolean speedValueReached = false;

        final LocalDateTime maxAdjustmentTime = LocalDateTime.now().plusSeconds(maxAdjustmentDuration);

        int lastSpeedValue = 0;
        do {
            // wait 2s until the speed measurement has found a stable value
            try {
                addLogText("Wait 2s before compare speed value.");
                Thread.sleep(2000);
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
            }

            // get the current speed value
            Integer currentSpeedValue = speedometerModel.getReportedSpeed();

            if (currentSpeedValue == null) {
                currentSpeedValue = 0;
            }

            Integer convertedSpeed = speedConverter.apply(currentSpeedValue, scaleVal);

            LOGGER.info("Current speed value: {}, convertedSpeed: {}", currentSpeedValue, convertedSpeed);

            updateProgressText(logTextPrefix + "<br>Current speed: " + convertedSpeed + " km/h");

            if (currentSpeedValue > lastSpeedValue) {
                lastSpeedValue = currentSpeedValue;

                addLogText("Changed speed value detected: {} mm/s ({} km/h). Wait for final speed.", lastSpeedValue,
                    convertedSpeed);
            }
            else {
                addLogText("Final speed detected: {} mm/s ({} km/h)", lastSpeedValue, convertedSpeed);
                speedValueReached = true;

                speedValueConsumer.accept(Integer.valueOf(lastSpeedValue));
            }
        }
        while (!speedValueReached && LocalDateTime.now().isBefore(maxAdjustmentTime));

        return speedValueReached;
    }

    private SpeedScaleBeanModel speedScaleBeanModelSaved;

    private void validateVMinAndScaleValue() {
        LOGGER.info("Validate the speed range and scale values.");

        // ask the user for the scale and Vmin and Vmax values
        final SpeedScaleBeanModel speedScaleBeanModelIn = new SpeedScaleBeanModel();

        speedScaleBeanModelIn.setCv37Scale(speedometerProgBeanModel.getCv37Scale());

        // set some default values
        speedScaleBeanModelIn.setTargetMinSpeed(6);
        speedScaleBeanModelIn.setTargetMaxSpeed(80);

        if (this.speedScaleBeanModelSaved != null) {
            speedScaleBeanModelIn.setMeasuredMinSpeed(speedScaleBeanModelSaved.getMeasuredMinSpeed());
            speedScaleBeanModelIn.setMeasuredMaxSpeed(speedScaleBeanModelSaved.getMeasuredMaxSpeed());

            speedScaleBeanModelIn.setTargetMinSpeed(this.speedScaleBeanModelSaved.getTargetMinSpeed());
            speedScaleBeanModelIn.setTargetMaxSpeed(this.speedScaleBeanModelSaved.getTargetMaxSpeed());

            // keep the reference
            // this.speedScaleBeanModelSaved = speedScaleBeanModelIn;
        }

        int result = JOptionPane.CANCEL_OPTION;
        ApproveSpeedRangeDialog dialog = null;
        if (!isMeasurementAborted()) {
            updateProgressText("Validate the speed values.");

            dialog = new ApproveSpeedRangeDialog(JOptionPane.getFrameForComponent(contentPanel), speedScaleBeanModelIn);

            result = dialog.getResult();
            LOGGER.info("Result: {}", result);

            if (isMeasurementAborted()) {
                result = JOptionPane.CANCEL_OPTION;
            }
        }

        switch (result) {
            case JOptionPane.OK_OPTION:

                // check if the values have changed
                List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands = new ArrayList<>();
                PomAddressData addressData = new PomAddressData(locoModel.getAddress(), PomAddressTypeEnum.LOCOMOTIVE);

                final SpeedScaleBeanModel speedScaleBeanModel = dialog.getSpeedScaleBeanModel();

                if (!Objects.equals(speedScaleBeanModel.getCv37Scale(), speedometerProgBeanModel.getCv37Scale())) {
                    int scale = speedScaleBeanModel.getCv37Scale();
                    LOGGER.info("scale value has changed: {}", scale);
                    pomProgCommands
                        .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 37,
                            ByteUtils.getLowByte(scale)));
                }

                SingleObserver<String> completeAction = new SingleObserver<String>() {

                    @Override
                    public void onSuccess(String t) {
                        LOGGER.info("Update changed values, onSuccess, t: {}", t);

                        // fireSetPIDValues();

                        // start adjusting the CV2 value to reach the target speed of 6-8 km/h
                        fireAdjustCv2AndCv5(speedScaleBeanModel);
                    }

                    @Override
                    public void onSubscribe(Disposable d) {
                        LOGGER.info("Update changed values, onSubscribe, d: {}", d);

                    }

                    @Override
                    public void onError(Throwable e) {
                        LOGGER.warn("Update changed values, onError, e: {}", e);

                        fireMeasurementStop();
                    }
                };

                if (CollectionUtils.isNotEmpty(pomProgCommands)) {
                    LOGGER.info("Update the changed values.");

                    SwingUtilities
                        .invokeLater(() -> pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction));
                }
                else {
                    LOGGER.info("No update of changed values required.");
                    completeAction.onSuccess(null);
                }

                break;
            default:
                LOGGER.info("Dialog was cancelled, stop the measurement.");

                updateProgressText("Adjusting speed values was cancelled.");

                fireMeasurementStop();
                break;
        }
    }

    private void fireStop() {

        if (speedometerModel.isActive()) {
            LOGGER.info("Stop the measurement.");

            fireMeasurementStop();
        }
        else {
            LOGGER.info("Stop the car.");
            locoModel.setSpeed(0);
        }

    }

    protected void fireMeasurementStop() {

        LOGGER.info("Stop the measurement.");

        fireToggleMeasure(false);

        // speedometerModel.setActive(false);

        final ProgressStatusCallback callback = this.measurementProgressCallback;

        SwingUtilities.invokeLater(() -> {
            LOGGER.info("Stop the car.");
            fireStop();

            speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.ABORTED);

            final CompositeDisposable compositeDisposable = speedometerModel.getCompositeDisposable();
            if (compositeDisposable != null) {
                LOGGER.info("Dispose the subscription on speed and battery message.");
                speedometerModel.setCompositeDisposable(null);
                compositeDisposable.dispose();
            }

            // stop the progress panel
            progressPanel.stop();

            if (callback != null) {
                LOGGER.info("Call actionFinished on callback to close the dialog.");
                callback.actionFinished();
            }

            speedoFunctionsEnabled.setValue(true);
        });
    }

    private AtomicBoolean continueMeasurementHolder;

    /**
     * Start the speed auto-adjustment process
     */
    private void fireStartAutoSpeedAdjustment() {

        // open the progress dialog
        final MeasurementProgressDialog progressDialog =
            new MeasurementProgressDialog(contentPanel, true, new MeasurementViewListener() {

                @Override
                public void startMeasurement(AtomicBoolean continueMeasurementHolder, ProgressStatusCallback callback) {
                    LOGGER.info("Start measurement is requested.");
                    try {

                        SpeedometerPanel.this.measurementProgressCallback = callback;
                        SpeedometerPanel.this.continueMeasurementHolder = continueMeasurementHolder;

                        updateProgressText("Start the speed adjustment process.");

                        SwingUtilities.invokeLater(() -> scheduleStartAutoSpeedAdjustment());
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Schedule auto speed adjustement failed.", ex);
                    }

                }

                @Override
                public void stopMeasurement(AtomicBoolean continueMeasurementHolder, ProgressStatusCallback callback) {
                    LOGGER.info("Stop measurement is requested. Fire measurement stop and release the callback.");

                    if (isMeasurementAborted()) {
                        return;
                    }

                    try {

                        SwingUtilities.invokeLater(() -> {
                            if (!isMeasurementAborted()) {

                                fireMeasurementStop();
                            }
                            else {
                                LOGGER.info("Skip send measurement stop. The speed measurement is aborted already.");
                            }
                        });

                        // SpeedometerPanel.this.measurementProgressCallback = null;
                        // SpeedometerPanel.this.continueMeasurementHolder = null;
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Stop auto speed adjustement failed.", ex);
                    }

                }
            });
        measurementProgressDialogHolder.value = progressDialog;

        progressDialog.startAction();
    }

    private ProgressStatusCallback measurementProgressCallback;

    /**
     * Start the speed auto-adjustment process
     */
    private void scheduleStartAutoSpeedAdjustment() {

        int delay = 0;
        // use executor to process the responses
        measurementWorker.schedule(new Runnable() {
            @Override
            public void run() {

                try {
                    SwingUtilities.invokeAndWait(() -> {

                        updateProgressText("Wait for speed and battery status message.");

                        progressPanel.start();

                    });
                }
                catch (InvocationTargetException | InterruptedException ex) {
                    LOGGER.warn("Start progress panel failed.", ex);
                }

                final CompositeDisposable compositeDisposable = new CompositeDisposable();

                Action complete = () -> {
                    LOGGER.info("Wait for speed and battery status message has completed.");
                };

                Consumer<Pair<PropertyChangeEvent>> action = pair -> {
                    LOGGER.info("Combined subscription has fired: {}", pair);

                    if (pair.getFirst() != null && pair.getFirst().getNewValue() != null && pair.getSecond() != null
                        && pair.getSecond().getNewValue() != null) {
                        LOGGER.info("The pair values are valid. Start read the CV values.");

                        complete.run();
                        compositeDisposable.dispose();

                        if (continueMeasurementHolder != null && !continueMeasurementHolder.get()) {
                            LOGGER.warn("The measurement was stopped!");

                            // stop the infinite progress panel on the log panel
                            if (!SwingUtilities.isEventDispatchThread()) {
                                try {
                                    SwingUtilities.invokeAndWait(() -> progressPanel.stop());
                                }
                                catch (InvocationTargetException | InterruptedException ex) {
                                    LOGGER.warn("Stop progress panel failed.", ex);
                                }
                            }
                            else {
                                progressPanel.stop();
                            }

                        }
                        else {
                            LOGGER
                                .info(
                                    "The speed and battery status message was received. Schedule a new task to read the CV values.");

                            measurementWorker.schedule(() -> {

                                addLogText("Start worker reads CV values.");
                                updateProgressText("Read all CV values.");

                                // TODO should keep the original values to restore if required

                                // let the pom request processor start to read all CV values
                                fireReadCvValues(() -> getFullSpeedRange());

                            }, delay, TimeUnit.MILLISECONDS);
                        }
                    }
                    else if (pair.getFirst() != null && pair.getFirst().getNewValue() != null) {
                        addLogText("Received speed message: {}", pair.getFirst().getNewValue());
                    }
                    else if (pair.getSecond() != null && pair.getSecond().getNewValue() != null) {
                        addLogText("Received battery message: {}", pair.getSecond().getNewValue());
                    }

                };
                Consumer<Throwable> error = th -> {
                    LOGGER.warn("Combined subscription has thrown an error: {}", th);

                    SwingUtilities.invokeLater(() -> {

                        progressPanel.stop();

                        if (th instanceof TimeoutException) {
                            addLogText(
                                "Timeout error while waiting for speed and battery response from car with address: {}",
                                locoModel.getAddress());

                            updateProgressText("Wait for speed and battery status message expired with timout.");
                        }
                        else {
                            addLogText("Error during wait for speed and battery response from car with address: : {}",
                                locoModel.getAddress());

                            updateProgressText("Wait for speed and battery status message failed with an error.");
                        }
                    });

                    speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.ABORTED);
                    fireMeasurementStop();
                };

                int timeoutWaitForSpeedAndBattery = 20;

                // after speed and dynState energy value we will continue the measurement
                Disposable combinedSubscription =
                    PropertyChangeEventSource
                        .fromPropertyChangeEventsOf(speedometerModel, SpeedometerModel.PROPERTYNAME_REPORTEDSPEED,
                            SpeedometerModel.PROPERTYNAME_DYNSTATEENERGY)
                        .timeout(timeoutWaitForSpeedAndBattery, TimeUnit.SECONDS).subscribe(action, error, complete);

                compositeDisposable.add(combinedSubscription);
                speedometerModel.setCompositeDisposable(compositeDisposable);

                SwingUtilities.invokeLater(() -> {

                    addLogText("Wait for speed and battery response from car with address: {}", locoModel.getAddress());

                    measurementWorker.schedule(() -> {
                        LOGGER.info("Start the measurement with trigger lights on and off.");

                        // this will trigger the lights on and off
                        speedometerModel.setActive(true);

                        // this will trigger lights on and off, and set the speed to 10
                        speedometerModel
                            .setSpeedMeasurementStage(SpeedMeasurementStage.WAIT_FOR_CAR_SPEED_AND_BATTERY_RESPONSE);
                    }, delay, TimeUnit.MILLISECONDS);
                });

                LOGGER.info("Start worker has finished.");
            }
        }, delay, TimeUnit.MILLISECONDS);
    }

    private void fireToggleMeasure(boolean activate) {
        LOGGER.info("Fire the measure, activate: {}", activate);

        if (activate) {
            speedometerModel.setActive(true);

            // clear the speed in the loco model
            locoModel.setReportedSpeed(null);

            speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.MEASURE_ON);
        }
        else {
            if (!isMeasurementAborted()) {
                LOGGER.info("Set the measurement off.");
                speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.MEASURE_OFF);
            }
            else {
                LOGGER.info("Do not change the speed measurement stage because it is aborted.");
            }
            speedometerModel.setActive(false);

            measurementLed.setLedOn(false);

            if (speedometerProgBeanModel.getCv37Scale() == null) {
                SwingUtilities.invokeLater(() -> {
                    measureButton.setEnabled(false);
                });
            }
        }

        SwingUtilities.invokeLater(() -> {
            measurementLed.setLedBlinking(activate);

            measureButton.setSelected(activate);
        });
    }

    private void fireAdjustCv2AndCv5(final SpeedScaleBeanModel speedScaleBeanModel) {
        addLogText("Start adjusting CV2 and CV5.");

        updateProgressText("Adjusting values for CV2 and CV5.");

        int delay = 0;
        measurementWorker.schedule(new Runnable() {
            @Override
            public void run() {

                LOGGER.info("Make sure the speed measurement is active.");
                try {
                    SwingUtilities.invokeAndWait(() -> fireToggleMeasure(true));
                }
                catch (Exception ex) {
                    LOGGER.warn("toggle measurement to on failed.", ex);
                }

                // wait 2s until the speed measurement has found a stable value
                try {
                    Thread.sleep(2000);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
                }

                addLogText("Set the speed step 2.");

                updateProgressText("Set speed step 2.");

                fireSpeedStep(2);

                Integer currentCv2Value = speedometerProgBeanModel.getCv2Vmin();
                Integer currentCv5Value = speedometerProgBeanModel.getCv5Vmax();

                Integer targetVminValue = speedScaleBeanModel.getTargetMinSpeed();

                // max speed because we measure at speed step 127
                Integer targetVmaxValue = speedScaleBeanModel.getTargetMaxSpeed();

                Integer scaleValue = speedometerProgBeanModel.getCv37Scale();
                LOGGER
                    .info("Fetched current CV2: {}, CV5: {}, targetVminValue: {}, targetVmaxValue: {}, scaleValue: {}",
                        currentCv2Value, currentCv5Value, targetVminValue, targetVmaxValue, scaleValue);

                addLogText("Set the speed step 127.");

                updateProgressText("Set speed step 127.");

                fireSpeedStep(127);

                // wait 5s until the speed measurement has found a stable value
                try {
                    addLogText("Wait 5s before enter the value adjustment loop for maximum speed.");
                    Thread.sleep(5000);
                }
                catch (InterruptedException ex) {
                    LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
                }

                addLogText("Start adjusting CV5 value to match targetVmaxValue: {}", targetVmaxValue);

                updateProgressText("Detect the CV5 value for Vmax=" + targetVmaxValue + "km/h.");

                // ask the user for the target speed range
                speedValueAdjustmentLoop(speedConverterToKmH, scaleValue, currentCv5Value, 100, 5, targetVmaxValue - 1,
                    targetVmaxValue + 2, "Detect the CV5 value for Vmax=" + targetVmaxValue + "km/h.");

                if (!speedometerModel.isActive()) {
                    LOGGER.warn("The measurement was aborted during CV5 measurement!");
                    return;
                }

                LOGGER.info("Adjusted the CV5 value.");

                // detect the min speed value

                if (currentCv2Value == null) {
                    currentCv2Value = 3;
                    LOGGER.info("Adjusted invalid CV2: {}", currentCv2Value);
                }

                updateProgressText("Detect the CV2 value for Vmin=" + targetVminValue + "km/h.");

                speedValueAdjustmentLoop(speedConverterToKmH, scaleValue, currentCv2Value, 1, 2, targetVminValue,
                    targetVminValue + 1, "Detect the CV2 value for Vmin=" + targetVminValue + "km/h.");

                if (!speedometerModel.isActive()) {
                    LOGGER.warn("The measurement was aborted during CV2 measurement!");
                    return;
                }

                LOGGER.info("Adjusted the CV2 value.");

                // set speed step 2 and get the measured speed in mm/s and store in CV35
                updateProgressText("Set speed step 2 and get value for CV35.");
                speedReadAndWriteCV(val -> val, 2, 35, 5);

                if (!speedometerModel.isActive()) {
                    LOGGER.warn("The measurement was aborted during CV35 measurement!");
                    return;
                }

                // set speed step 64 and get the measured speed in mm/s and store in CV36
                updateProgressText("Set speed step 64 and get value for CV36.");
                speedReadAndWriteCV(val -> val, 64, 36, 10);

                final StringBuilder sb = new StringBuilder("<html>Adjusted the CV2 value: ");
                sb.append(speedometerProgBeanModel.getCv2Vmin());
                sb.append("<br/>Adjusted the CV5 value: ");
                sb.append(speedometerProgBeanModel.getCv5Vmax());
                sb.append("<br/>Adjusted the CV35 value: ");
                sb.append(speedometerProgBeanModel.getCv35());
                sb.append("<br/>Adjusted the CV36 value: ");
                sb.append(speedometerProgBeanModel.getCv36());
                sb.append("</html>");

                updateProgressText("Speed adjustment has finished.");

                if (isMeasurementAborted()) {
                    LOGGER.warn("Speed measurement was aborted.");
                }
                else {
                    SwingUtilities.invokeLater(() -> {

                        // TODO allow to restore the old CV values

                        final FinishOrRestoreDialog dialog =
                            new FinishOrRestoreDialog(JOptionPane.getFrameForComponent(contentPanel),
                                speedometerProgBeanModel, backupSpeedometerProgBeanModel);
                        int result = dialog.getResult();
                        LOGGER.info("Result: {}", result);

                        // TODO evaluate restore option
                        if (result == JOptionPane.CANCEL_OPTION) {
                            LOGGER.info("Restore the old values.");
                            fireRestoreCv();
                        }

                        // JOptionPane.showMessageDialog(contentPanel, sb.toString());

                        if (SpeedometerPanel.this.measurementProgressCallback != null) {

                            try {
                                SpeedometerPanel.this.measurementProgressCallback.actionFinished();
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Signal action finished to callback failed.", ex);
                            }
                        }
                    });

                    LOGGER.info("Make sure the speed measurement is inactive.");
                    try {
                        SwingUtilities.invokeAndWait(() -> {
                            fireToggleMeasure(false);
                            fireSpeedStep(0);
                        });
                    }
                    catch (Exception ex) {
                        LOGGER.warn("toggle measurement to off failed.", ex);
                    }
                }
            }
        }, delay, TimeUnit.MILLISECONDS);
    }

    private boolean speedReadAndWriteCV(
        final Function<Integer, Integer> speedConverter, int speedStep, int cvNumber, int waitForSpeedDelay) {

        // set speed step and get the measured speed in mm/s and store in CV
        fireSpeedStep(speedStep);

        // wait 3s until the speed measurement has found a stable value
        try {
            addLogText("Wait {}s before get the current measured speed.", waitForSpeedDelay);
            Thread.sleep(waitForSpeedDelay * 1000);
        }
        catch (InterruptedException ex) {
            LOGGER.warn("Wait for a stable speed measurement value at speed step {} failed.", speedStep, ex);
        }

        // get the current speed value
        Integer currentSpeedValue = speedometerModel.getReportedSpeed();

        if (currentSpeedValue == null) {
            currentSpeedValue = 0;
        }

        Integer convertedSpeed = speedConverter.apply(currentSpeedValue);

        LOGGER
            .info("Current speed step {}, speed value: {}, convertedSpeed: {}", speedStep, currentSpeedValue,
                convertedSpeed);

        // write current speed value to the CV
        final Object continueLock = new Object();

        SingleObserver<String> completeAction = new SingleObserver<String>() {

            @Override
            public void onSuccess(String t) {
                LOGGER.info("Update changed values, onSuccess, t: {}", t);

                synchronized (continueLock) {
                    continueLock.notifyAll();
                }
            }

            @Override
            public void onSubscribe(Disposable d) {
                LOGGER.info("Update changed values, onSubscribe, d: {}", d);

            }

            @Override
            public void onError(Throwable e) {
                LOGGER.warn("Update changed values, onError was called.", e);

                fireMeasurementStop();
            }
        };

        if (convertedSpeed > 255) {
            convertedSpeed = 255;
            LOGGER.info("Set the convertedSpeed to the upper limit value: {}", convertedSpeed);
        }

        LOGGER.info("Update the changed values, CV: {}, convertedSpeed: {}", cvNumber, convertedSpeed);

        updateProgressText("Set CV" + cvNumber + " to value: " + convertedSpeed);

        List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands = new ArrayList<>();
        PomAddressData addressData = new PomAddressData(locoModel.getAddress(), PomAddressTypeEnum.LOCOMOTIVE);

        // set retry count on this command because it failed sometimes
        final SpeedometerPomCommand command =
            new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, cvNumber,
                ByteUtils.getLowByte(convertedSpeed));
        command.setRetryCount(1);
        pomProgCommands.add(command);

        pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction);

        synchronized (continueLock) {
            try {
                continueLock.wait(5000);
            }
            catch (InterruptedException e1) {
                LOGGER.warn("Wait for signal continue lock was interrupted.", e1);
                fireMeasurementStop();
            }
        }

        return true;
    }

    private boolean speedValueAdjustmentLoop(
        final BiFunction<Integer, Integer, Integer> speedConverter, Integer scaleValue, Integer currentCvValue,
        Integer currentCvRangeLimitValue, int cvNumber, int lowerRange, int upperRange, String progressTextPrefix) {
        boolean speedValueReached = false;

        do {

            // check if aborted
            if (isMeasurementAborted()) {
                break;
            }

            // wait 3s until the speed measurement has found a stable value
            try {
                addLogText("Wait 5s before compare speed value.");
                Thread.sleep(5000);
            }
            catch (InterruptedException ex) {
                LOGGER.warn("Wait for a stable speed measurement value failed.", ex);
            }

            // get the current speed value
            Integer currentSpeedValue = speedometerModel.getReportedSpeed();

            if (currentSpeedValue == null) {
                currentSpeedValue = 0;
            }

            Integer convertedSpeed = speedConverter.apply(currentSpeedValue, scaleValue);

            LOGGER
                .info("Current speed value: {}, convertedSpeed: {}, currentCvValue: {}", currentSpeedValue,
                    convertedSpeed, currentCvValue);

            updateProgressText(progressTextPrefix + "<br>Current speed: " + convertedSpeed + " km/h");

            // check if the values have changed
            List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands = new ArrayList<>();
            PomAddressData addressData = new PomAddressData(locoModel.getAddress(), PomAddressTypeEnum.LOCOMOTIVE);

            if (convertedSpeed < lowerRange) {
                currentCvValue++;

                addLogText("Increase the CV{} value: {}", cvNumber, currentCvValue);

                pomProgCommands
                    .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, cvNumber,
                        ByteUtils.getLowByte(currentCvValue)));
            }
            else if (convertedSpeed > upperRange) {

                if (convertedSpeed - upperRange > 5) {
                    currentCvValue = currentCvValue - 2;
                }
                else {
                    currentCvValue--;
                }

                addLogText("Decrease the CV{} value: {}", cvNumber, currentCvValue);

                pomProgCommands
                    .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, cvNumber,
                        ByteUtils.getLowByte(currentCvValue)));
            }
            else {
                addLogText("Adjusted the CV{} value to target speed: {}", cvNumber, convertedSpeed);
            }

            final Object continueLock = new Object();

            SingleObserver<String> completeAction = new SingleObserver<String>() {

                @Override
                public void onSuccess(String t) {
                    LOGGER.info("Update changed values, onSuccess, t: {}", t);

                    synchronized (continueLock) {
                        continueLock.notifyAll();
                    }
                }

                @Override
                public void onSubscribe(Disposable d) {
                    LOGGER.info("Update changed values, onSubscribe, d: {}", d);

                }

                @Override
                public void onError(Throwable e) {
                    LOGGER.warn("Update changed values, onError, e: {}", e);

                    fireMeasurementStop();
                }
            };

            if (CollectionUtils.isNotEmpty(pomProgCommands)) {
                LOGGER.info("Update the changed values.");

                pomRequestProcessor.submitProgCommands(pomProgCommands, completeAction);

                synchronized (continueLock) {
                    try {
                        continueLock.wait(5000);
                    }
                    catch (InterruptedException e1) {
                        LOGGER.warn("Wait for signal continue lock was interrupted.", e1);
                        fireMeasurementStop();
                    }
                }
            }
            else {
                LOGGER.info("No update of changed values required.");
                speedValueReached = true;
            }
        }
        while (!speedValueReached);

        return speedValueReached;
    }

    private void fireSetPIDValues() {
        LOGGER.info("Set the PID values.");

        // addLogText("Set the PID values.");
        //
        // speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.SET_MOTOR_PID_SOFT);

        int result = JOptionPane.CANCEL_OPTION;
        ApprovePidParamsDialog dialog =
            new ApprovePidParamsDialog(JOptionPane.getFrameForComponent(contentPanel), speedometerProgBeanModel);

        result = dialog.getResult();
        LOGGER.info("Result: {}", result);

        switch (result) {
            case JOptionPane.OK_OPTION:
                final PidParamsBeanModel pidParamsBeanModel = dialog.getPidParamsBeanModel();

                LOGGER.info("Write the PID values, pidParamsBeanModel: {}", pidParamsBeanModel);
                writeCvValues(pidParamsBeanModel, speedometerProgProxyBeanModel, () -> {
                    LOGGER.info("Write CV values passed. Start the measurement.");
                    fireMeasurementStart();
                }, () -> {
                    LOGGER.warn("Write the PID values failed.");
                    fireMeasurementStop();
                });

                // fireStop();
                break;
            default:
                fireMeasurementStop();
                break;
        }
    }

    protected void fireRestoreCv() {

        LOGGER.info("Write backup CV values.");

        LOGGER.info("Write CV values from backup model: {}", backupSpeedometerProgBeanModel);

        final CvParamsBeanModel cvParamsBeanModel = new CvParamsBeanModel();

        cvParamsBeanModel.setCv2(backupSpeedometerProgBeanModel.getCv2Vmin());
        cvParamsBeanModel.setCv5(backupSpeedometerProgBeanModel.getCv5Vmax());
        // cvParamsBeanModel.setCv37(backupSpeedometerProgBeanModel.getCv37Scale());
        // cvParamsBeanModel.setCv61(backupSpeedometerProgBeanModel.getCv61());
        // cvParamsBeanModel.setCv62(backupSpeedometerProgBeanModel.getCv62());
        // cvParamsBeanModel.setCv63(backupSpeedometerProgBeanModel.getCv63());

        cvParamsBeanModel.setCv35(backupSpeedometerProgBeanModel.getCv35());
        cvParamsBeanModel.setCv36(backupSpeedometerProgBeanModel.getCv36());

        writeCvValues(cvParamsBeanModel, speedometerProgProxyBeanModel, null, null);
    }

    protected void fireWriteCv() {

        LOGGER.info("Write CV values.");

        trigger.triggerCommit();

        LOGGER.info("Write CV values after trigger commit: {}", speedometerProgBeanModel);

        final CvParamsBeanModel cvParamsBeanModel = new CvParamsBeanModel();

        cvParamsBeanModel.setCv2(speedometerProgBeanModel.getCv2Vmin());
        cvParamsBeanModel.setCv5(speedometerProgBeanModel.getCv5Vmax());
        cvParamsBeanModel.setCv37(speedometerProgBeanModel.getCv37Scale());
        cvParamsBeanModel.setCv61(speedometerProgBeanModel.getCv61());
        cvParamsBeanModel.setCv62(speedometerProgBeanModel.getCv62());
        cvParamsBeanModel.setCv63(speedometerProgBeanModel.getCv63());

        cvParamsBeanModel.setCv35(speedometerProgBeanModel.getCv35());
        cvParamsBeanModel.setCv36(speedometerProgBeanModel.getCv36());

        writeCvValues(cvParamsBeanModel, speedometerProgProxyBeanModel, null, null);
    }

    private void writeCvValues(
        final PidParamsBeanModel pidParamsBeanModel, final SpeedometerProgBeanModel speedoProgBeanModel,
        final CallbackAction finishCallback, final CallbackAction errorCallback) {

        // check if the values have changed
        List<PomOperationCommand<? extends ProgCommandAwareBeanModel>> pomProgCommands = new ArrayList<>();
        PomAddressData addressData = new PomAddressData(locoModel.getAddress(), PomAddressTypeEnum.LOCOMOTIVE);

        if (pidParamsBeanModel instanceof CvParamsBeanModel) {
            final CvParamsBeanModel cvParamsBeanModel = (CvParamsBeanModel) pidParamsBeanModel;
            if (!Objects.equals(cvParamsBeanModel.getCv2(), speedoProgBeanModel.getCv2Vmin())) {
                Integer cv2 = cvParamsBeanModel.getCv2();
                LOGGER.info("CV2 value has changed: {}", cv2);
                if (cv2 != null) {
                    pomProgCommands
                        .add(
                            new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 2, ByteUtils.getLowByte(cv2)));
                }
                else {
                    addLogText("No value for CV2 available.");
                }
            }

            if (!Objects.equals(cvParamsBeanModel.getCv5(), speedoProgBeanModel.getCv5Vmax())) {
                Integer cv5 = cvParamsBeanModel.getCv5();
                LOGGER.info("CV5 value has changed: {}", cv5);
                if (cv5 != null) {
                    pomProgCommands
                        .add(
                            new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 5, ByteUtils.getLowByte(cv5)));
                }
                else {
                    addLogText("No value for CV5 available.");
                }
            }

            if (!Objects.equals(cvParamsBeanModel.getCv35(), speedoProgBeanModel.getCv35())) {
                Integer cv35 = cvParamsBeanModel.getCv35();
                LOGGER.info("CV35 value has changed: {}", cv35);
                if (cv35 != null) {
                    pomProgCommands
                        .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 35,
                            ByteUtils.getLowByte(cv35)));
                }
                else {
                    addLogText("No value for CV35 available.");
                }
            }

            if (!Objects.equals(cvParamsBeanModel.getCv36(), speedoProgBeanModel.getCv36())) {
                Integer cv36 = cvParamsBeanModel.getCv36();
                LOGGER.info("CV36 value has changed: {}", cv36);
                if (cv36 != null) {
                    pomProgCommands
                        .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 36,
                            ByteUtils.getLowByte(cv36)));
                }
                else {
                    addLogText("No value for CV36 available.");
                }
            }

            if (!Objects.equals(cvParamsBeanModel.getCv37(), speedoProgBeanModel.getCv37Scale())) {
                Integer cv37 = cvParamsBeanModel.getCv37();
                LOGGER.info("CV37 value has changed: {}", cv37);
                if (cv37 != null) {
                    pomProgCommands
                        .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 37,
                            ByteUtils.getLowByte(cv37)));
                }
                else {
                    addLogText("No value for CV37 available.");
                }
            }
        }

        if (!Objects.equals(pidParamsBeanModel.getCv61(), speedoProgBeanModel.getCv61())) {
            Integer cv61 = pidParamsBeanModel.getCv61();
            LOGGER.info("CV61 value has changed: {}", cv61);
            if (cv61 != null) {
                pomProgCommands
                    .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 61, ByteUtils.getLowByte(cv61)));
            }
            else {
                addLogText("No value for CV61 available.");
            }
        }
        if (!Objects.equals(pidParamsBeanModel.getCv62(), speedoProgBeanModel.getCv62())) {
            Integer cv62 = pidParamsBeanModel.getCv62();
            LOGGER.info("CV62 value has changed: {}", cv62);
            if (cv62 != null) {
                pomProgCommands
                    .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 62, ByteUtils.getLowByte(cv62)));
            }
            else {
                addLogText("No value for CV62 available.");
            }
        }
        if (!Objects.equals(pidParamsBeanModel.getCv63(), speedoProgBeanModel.getCv63())) {
            Integer cv63 = pidParamsBeanModel.getCv63();
            LOGGER.info("CV63 value has changed: {}", cv63);
            if (cv63 != null) {
                pomProgCommands
                    .add(new SpeedometerPomCommand(addressData, PomOperation.WR_BYTE, 63, ByteUtils.getLowByte(cv63)));
            }
            else {
                addLogText("No value for CV63 available.");
            }
        }

        final SingleObserver<String> completeAction = new SingleObserver<String>() {

            @Override
            public void onSuccess(String t) {
                LOGGER.info("Update changed values after write CV values, onSuccess, t: {}", t);

                // update the current values
                if (pidParamsBeanModel.getCv61() != null) {
                    speedoProgBeanModel.setCv61(pidParamsBeanModel.getCv61());
                }
                if (pidParamsBeanModel.getCv62() != null) {
                    speedoProgBeanModel.setCv62(pidParamsBeanModel.getCv62());
                }

                if (pidParamsBeanModel.getCv63() != null) {
                    speedoProgBeanModel.setCv63(pidParamsBeanModel.getCv63());
                }

                if (pidParamsBeanModel instanceof CvParamsBeanModel) {
                    final CvParamsBeanModel cvParamsBeanModel = (CvParamsBeanModel) pidParamsBeanModel;

                    if (cvParamsBeanModel.getCv35() != null) {
                        speedoProgBeanModel.setCv35(cvParamsBeanModel.getCv35());
                    }
                    if (cvParamsBeanModel.getCv36() != null) {
                        speedoProgBeanModel.setCv36(cvParamsBeanModel.getCv36());
                    }
                    if (cvParamsBeanModel.getCv37() != null) {
                        speedoProgBeanModel.setCv37Scale(cvParamsBeanModel.getCv37());
                    }

                    if (cvParamsBeanModel.getCv2() != null) {
                        speedoProgBeanModel.setCv2Vmin(cvParamsBeanModel.getCv2());
                    }
                    if (cvParamsBeanModel.getCv5() != null) {
                        speedoProgBeanModel.setCv5Vmax(cvParamsBeanModel.getCv5());
                    }
                }

                // call callback if available
                if (finishCallback != null) {
                    LOGGER.info("Call the finish callback after the CV values were updated.");
                    finishCallback.invoke();
                }
                else {
                    LOGGER.info("No finish callback available.");
                }
            }

            @Override
            public void onSubscribe(Disposable d) {
                LOGGER.info("Update changed values, onSubscribe, d: {}", d);

            }

            @Override
            public void onError(Throwable e) {
                LOGGER.warn("Update changed values, onError, e: {}", e);
                if (errorCallback != null) {
                    errorCallback.invoke();
                }
                else {
                    LOGGER.info("No finish callback available.");
                }
            }
        };

        if (CollectionUtils.isNotEmpty(pomProgCommands)) {
            LOGGER.info("Update the changed PID values.");

            speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.IDLE);

            // set the commands in the speedometer model
            speedometerModel.setPomProgCommands(pomProgCommands);
            speedometerModel.setCompleteAction(completeAction);

            // this will trigger the controller to perform the operation
            speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.WRITE_CV_VALUES);

        }
        else {
            LOGGER.info("No update of changed values required.");
            completeAction.onSuccess(null);
        }
    }

    private void fireMeasurementStart() {

        speedometerModel.setCv37Scale(speedometerProgBeanModel.getCv37Scale());

        addLogText("Start the measurement.");

        speedometerModel.setSpeedMeasurementStage(SpeedMeasurementStage.START_MEASUREMENT);
    }

    private void fireSpeedStep(int speedStep) {
        LOGGER.info("Set the speedStep: {}", speedStep);
        if (SwingUtilities.isEventDispatchThread()) {
            locoModel.setSpeed(speedStep);
        }
        else {
            SwingUtilities.invokeLater(() -> locoModel.setSpeed(speedStep));
        }
    }

    protected final ScheduledExecutorService measurementWorker =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("measurementWorkers-thread-%d").build());

    private void signalPomProgStateChanged(PomProgState pomProgState) {

        LOGGER.info("The POM prog state has changed: {}", pomProgState);
        switch (pomProgState) {
            // finished state
            case POM_PROG_OKAY:
                currentOperationLabel.setIcon(progOperationSuccessfulIcon);
                addLogText("Prog operation passed: " + getCurrentOperation());
                // enable the input elements
                enableInputElements();
                break;
            case POM_PROG_STOPPED:
                // case PROG_NO_LOCO:
            case POM_PROG_NO_ANSWER:
                // case PROG_SHORT:
                // case PROG_VERIFY_FAILED:
                currentOperationLabel.setIcon(progOperationErrorIcon);
                addLogText("Prog operation failed: {}", pomProgState.name());
                // enable the input elements
                enableInputElements();
                break;
            // pending
            case POM_PROG_START:
                currentOperationLabel.setIcon(progOperationUnknownIcon);
                addLogText("Prog operation started: " + getCurrentOperation());
                // disableInputElements();
                break;
            // pending
            case POM_PROG_RUNNING:
                currentOperationLabel.setIcon(progOperationWaitIcon);
                addLogText("Prog operation is running ...");
                break;
        }
    }

    protected void disableInputElements() {
        LOGGER.trace("Disable input elements.");
    }

    protected void enableInputElements() {
        LOGGER.trace("Enable input elements.");
    }

    private Object getCurrentOperation() {
        return speedometerProgProxyBeanModel.getCurrentOperation();
    }

    protected void fireReadCv() {

        LOGGER.info("Read CV values without show the dialog.");

        fireReadCvValues(null);
    }

    @Override
    public void close() {
        LOGGER.info("Close the panel.");

        LOGGER.info("Shutdown the measurementWorker.");
        measurementWorker.shutdownNow();
    }

    /**
     * Holds a value of type <code>T</code>.
     */
    public final class Holder<T> implements Serializable {

        private static final long serialVersionUID = 1L;

        /**
         * The value contained in the holder.
         */
        public T value;

        /**
         * Creates a new holder with a <code>null</code> value.
         */
        public Holder() {
        }

        /**
         * Create a new holder with the specified value.
         *
         * @param value
         *            The value to be stored in the holder.
         */
        public Holder(T value) {
            this.value = value;
        }
    }

    private final BiFunction<Integer, Integer, Integer> speedConverterToKmH = (val, scaleVal) -> {

        int speedVal = 0;
        if (scaleVal != null && scaleVal > 1) {
            // val is the speed in mm/s --> convert to km/h with the scale factor
            speedVal = (int) (((double) val * scaleVal) / 277.778);

            LOGGER.info("Converted speed: {} to speedVal: {}, scaleVal: {}", val, speedVal, scaleVal);
        }
        else {
            speedVal = val;
        }

        return speedVal;
    };

}
