package org.bidib.wizard.mvc.main.model;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.api.json.types.NodeInfo.NodeAction;
import org.bidib.jbidibc.core.schema.bidibbase.DefaultLabelsActionType;
import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.exchange.vendorcv.VendorCvData;
import org.bidib.jbidibc.messages.Node;
import org.bidib.jbidibc.messages.enums.PortModelEnum;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.AccessorySaveState;
import org.bidib.wizard.api.model.Flag;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroSaveState;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.NodeListProvider;
import org.bidib.wizard.api.model.NodeProvider;
import org.bidib.wizard.api.model.listener.CvDefinitionListener;
import org.bidib.wizard.api.model.listener.NodeErrorListener;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.model.listener.NodeSelectionListener;
import org.bidib.wizard.api.notification.NodeUpdate;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.labels.AccessoryLabelUtils;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.node.Node.ErrorStatePropertyChange;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.main.model.listener.AccessorySelectionListener;
import org.bidib.wizard.mvc.main.model.listener.MacroSelectionListener;
import org.bushe.swing.event.EventBus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.util.Assert;

import io.reactivex.rxjava3.core.Observable;
import io.reactivex.rxjava3.core.Observer;
import io.reactivex.rxjava3.disposables.Disposable;

public class MainModel implements NodeSelectionProvider, NodeListProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(MainModel.class);

    private Collection<MacroSelectionListener> macroSelectionListeners = new LinkedList<>();

    private List<AccessorySelectionListener> accessorySelectionListeners = new LinkedList<>();

    private Collection<NodeListListener> nodeListListeners = new LinkedList<>();

    private Collection<NodeSelectionListener> nodeSelectionListeners = new LinkedList<>();

    private Collection<CvDefinitionListener> cvDefinitionListeners = new LinkedList<>();

    // the node provider
    private NodeProvider nodeProvider;

    private Accessory selectedAccessory;

    private Macro selectedMacro;

    private NodeInterface selectedNode;

    private final String connectionId;

    private Object nodeLock = new Object();

    private final StatusModel statusModel;

    private NodeErrorListener nodeErrorListener;

    private AtomicBoolean initialLoadFinished = new AtomicBoolean(false);

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private ConsoleService consoleService;

    private final ApplicationEventPublisher applicationEventPublisher;

    public MainModel(final StatusModel statusModel, final String connectionId,
                     final ApplicationEventPublisher applicationEventPublisher) {

        this.statusModel = statusModel;
        this.connectionId = connectionId;
        this.applicationEventPublisher = applicationEventPublisher;
    }

    /**
     * @return the connection id
     */
    public String getConnectionId() {
        return this.connectionId;
    }

    public void setNodeProvider(final NodeProvider nodeProvider) {
        LOGGER.info("Set the node provider: {}", nodeProvider);
        this.nodeProvider = nodeProvider;

        this.nodeProvider.subscribeNodeListChanges(new Observer<NodeUpdate>() {

            @Override
            public void onSubscribe(Disposable disposable) {
                LOGGER.info("Subscribe to nodeProvider passed, disposable.isDisposed: {}", disposable.isDisposed());
            }

            @Override
            public void onNext(final NodeUpdate nodeUpdate) {
                LOGGER.info("Update received from node list: {}", nodeUpdate);

                if (SwingUtilities.isEventDispatchThread()) {
                    switch (nodeUpdate.getNodeAction()) {
                        case REMOVE:
                            handleNodeRemove(nodeUpdate);
                            break;
                        default:
                            handleNodeUpdate(nodeUpdate);
                            break;
                    }
                }
                else {
                    SwingUtilities.invokeLater(() -> {
                        switch (nodeUpdate.getNodeAction()) {
                            case REMOVE:
                                handleNodeRemove(nodeUpdate);
                                break;
                            default:
                                handleNodeUpdate(nodeUpdate);
                                break;
                        }
                    });
                }
            }

            @Override
            public void onError(Throwable e) {
                LOGGER.warn("Subscription on nodeProvider signalled an error: {}", e);
            }

            @Override
            public void onComplete() {
                LOGGER.info("Subscription on nodeProvider signalled complete");
            }
        });

        this.nodeProvider.subscribeNodePropertyChanges(npu -> {
            LOGGER.debug("The property has changed: {}", npu);

            switch (npu.getProperty()) {
                case NodeInterface.PROPERTY_ADDRESSMESSAGESENABLED:
                    LOGGER.info("Address messsages enabled was changed.");
                    break;
                case NodeInterface.PROPERTY_ERROR_STATE:
                case NodeInterface.PROPERTY_REASON_DATA:
                    LOGGER.info("The error state was changed.");
                    if (nodeErrorListener != null) {

                        final ErrorStatePropertyChange propertyChange = npu.getValue(ErrorStatePropertyChange.class);
                        nodeErrorListener
                            .nodeErrorChanged(npu.getNode(), propertyChange.getSysError(),
                                propertyChange.getReasonData());
                    }
                    break;
                case NodeInterface.PROPERTY_NODE_PREFIX + Node.PROPERTY_STALL:
                    LOGGER.info("The stall state was changed.");
                    if (nodeErrorListener != null) {
                        nodeErrorListener.nodeStallChanged(npu.getNode(), npu.getValue(Boolean.class));
                    }
                    break;

                case NodeInterface.PROPERTY_NODE_PREFIX + Node.PROPERTY_SOFTWARE_VERSION:
                    LOGGER.info("The firmware version of the node has changed: {}", npu.getNode().getNode().getSoftwareVersion());

                    final NodeUpdate nodeUpdate = new NodeUpdate(connectionId, NodeAction.NOTIFY, npu.getNode());
                    this.applicationEventPublisher.publishEvent(nodeUpdate);

                    break;

                default:
                    break;
            }

        }, error -> {
            LOGGER.warn("Subscription on nodeProvider for node property changes signalled an error: {}", error);
        });
    }

    private void handleNodeRemove(final NodeUpdate nodeUpdate) {
        NodeInterface currentNode = nodeUpdate.getNode();
        fireNodeListRemoved(currentNode);

        if (currentNode != null && currentNode.equals(getSelectedNode())) {
            LOGGER.info("The active node in model was removed: {}", getSelectedNode());
            setSelectedNode(null, true);
        }
        fireNodeListChanged();
    }

    private void handleNodeUpdate(final NodeUpdate nodeUpdate) {

        LOGGER.info("Node was updated: {}", nodeUpdate);

        // load the labels

        final NodeInterface node = nodeUpdate.getNode();
        Long uniqueId = Long.valueOf(node.getUniqueId());

        if (nodeUpdate.getNodeAction() == NodeAction.ADD) {
            checkDefaultLabelsApplied(node, uniqueId);
        }

        if (nodeUpdate.getNodeAction() == NodeAction.ADD) {
            fireNodeListChanged();

            fireNodeListAdded(node);
        }
    }

    private void checkDefaultLabelsApplied(final NodeInterface node, final Long uniqueId) {

        final DefaultLabelsActionType defaultLabelsApplied = wizardLabelWrapper.getDefaultLabelsApplied(uniqueId);
        LOGGER
                .info("The default labels have been applied for this node: {}, uniqueId: {}", defaultLabelsApplied,
                        ByteUtils.formatHexUniqueId(uniqueId));

        try {
            // and reload the labels
            reloadLabels(node);
        }
        catch (Exception ex) {
            LOGGER.warn("Load labels from BiDiB failed.", ex);

            // display warning in console
            SwingUtilities.invokeLater(() -> {
                ConsoleController.ensureConsoleVisible();

                // show an error message in the console
                consoleService.addConsoleLine(ConsoleColor.red,
                        String
                                .format("Load lables failed for node with uniqueId: %s",
                                        ByteUtils.getUniqueIdAsString(selectedNode.getUniqueId())));
            });
        }

        // apply labels to node before apply default labels
        final NodeLabels nodeLabels = wizardLabelWrapper.loadLabels(uniqueId);

        if (SwingUtilities.isEventDispatchThread()) {
            node.setLabel(BidibLabelUtils.getNodeLabel(nodeLabels));
        }
        else {
            SwingUtilities.invokeLater(() -> node.setLabel(BidibLabelUtils.getNodeLabel(nodeLabels)));
        }

        // check if we must apply the default labels
        if (defaultLabelsApplied != null && !(DefaultLabelsActionType.APPLIED.equals(defaultLabelsApplied)
                || DefaultLabelsActionType.IGNORED.equals(defaultLabelsApplied))) {
            int relevantPidBits = node.getNode().getRelevantPidBits();

            // check if default labels are available
            final Observable<Boolean> defaultLabelsAvailableObservable = Observable.create(emitter -> {
                try {
                    boolean available = wizardLabelWrapper.isDefaultLabelsAvailable(uniqueId, relevantPidBits);
                    LOGGER
                            .info("Default labels available: {}, uniqueId: {}", available,
                                    ByteUtils.formatHexUniqueId(uniqueId));
                    emitter.onNext(Boolean.valueOf(available));

                    emitter.onComplete();
                }
                catch (Exception ex) {
                    LOGGER.warn("Check if defaultLabels are available failed.", ex);
                    emitter.onError(new RuntimeException("Check if defaultLabels are available failed."));
                }
            });

            defaultLabelsAvailableObservable.subscribe(available -> {
                LOGGER.info("The defaultLabels for this node are available: {}", available);

                if (available) {
                    SwingUtilities.invokeLater(() -> {

                        NodeInterface changedNode =
                                org.bidib.wizard.utils.NodeUtils
                                        .findNodeByUuid(nodeProvider.getNodes(), uniqueId.longValue());
                        BidibNodeNameUtils.NodeLabelData labelData =
                                BidibNodeNameUtils.prepareLabel(changedNode, nodeLabels, true, false);

                        StringBuffer sb =
                                new StringBuffer(Resources.getString(MainModel.class, "apply-default-labels.text"));
                        sb.append(" ").append(labelData.getNodeLabel());

                        int result =
                                JOptionPane
                                        .showConfirmDialog(JOptionPane.getFrameForComponent(null), sb.toString(),
                                                Resources.getString(MainModel.class, "apply-default-labels.title"),
                                                JOptionPane.YES_NO_CANCEL_OPTION);

                        String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
                        final PortModelEnum portModel = PortModelEnum.getPortModel(node.getNode());

                        if (result == JOptionPane.YES_OPTION) {
                            LOGGER.info("Apply the default labels on node: {}", node);

                            // apply the default labels
                            wizardLabelWrapper
                                    .setDefaultLabelsApplied(lang, uniqueId, relevantPidBits, portModel,
                                            DefaultLabelsActionType.APPLIED);

                            try {
                                // and reload the labels
                                reloadLabels(node);
                            }
                            catch (Exception ex) {
                                LOGGER.warn("Load labels from BiDiB failed.", ex);

                                // display warning in console
                                ConsoleController.ensureConsoleVisible();

                                // show an error message in the console
                                consoleService.addConsoleLine(ConsoleColor.red,
                                        String
                                                .format("Load lables failed for node with uniqueId: %s",
                                                        ByteUtils.getUniqueIdAsString(selectedNode.getUniqueId())));
                            }

                            changedNode.setLabel(BidibLabelUtils.getNodeLabel(nodeLabels));

                            // change labels of accessories
                            if (CollectionUtils.isNotEmpty(changedNode.getAccessories())) {

                                for (Accessory accessory : changedNode.getAccessories()) {

                                    final AccessoryLabel labelType =
                                            AccessoryLabelUtils.getAccessoryLabel(nodeLabels, accessory.getId());
                                    String accessoryLabel = labelType.getLabel();
                                    LOGGER
                                            .info("Set the label of accessory id: {}, label: {}", accessory.getId(),
                                                    accessoryLabel);
                                    accessory.setLabel(accessoryLabel);
                                }
                            }

                            // force a reload of the changed labels
                            final LabelsChangedEvent labelsChangedEvent = new LabelsChangedEvent(uniqueId);
                            LOGGER.info("Publish the labelsChangedEvent: {}", labelsChangedEvent);
                            EventBus.publish(labelsChangedEvent);

                            MainModel.this.applicationEventPublisher.publishEvent(labelsChangedEvent);
                        }
                        else if (result == JOptionPane.NO_OPTION) {
                            LOGGER.info("User declined to apply the default labels on node: {}", node);

                            wizardLabelWrapper
                                    .setDefaultLabelsApplied(lang, uniqueId, relevantPidBits, portModel,
                                            DefaultLabelsActionType.IGNORED);
                        }
                    });
                }
            }, error -> {
                LOGGER.warn("Check for default labels failed.", error);
            });

        }
    }

    public void setNodeErrorListener(NodeErrorListener nodeErrorListener) {
        this.nodeErrorListener = nodeErrorListener;
    }

    /**
     * Clear the cached nodes.
     */
    public void clearNodes() {
        LOGGER.info("Clear the nodes and reset the selected node.");

        setSelectedNode(null, true);

        fireNodeListChanged();
    }

    @Override
    public Collection<NodeInterface> getNodes() {
        if (getNodeProvider() != null) {
            return getNodeProvider().getNodes();
        }
        return Collections.emptyList();
    }

    /**
     * @return the nodeProvider
     */
    @Override
    public NodeProvider getNodeProvider() {
        return nodeProvider;
    }

    @Override
    public void addNodeListListener(final NodeListListener listener) {
        nodeListListeners.add(listener);
    }

    @Override
    public void removeNodeListListener(final NodeListListener listener) {
        boolean removed = nodeListListeners.remove(listener);
        Assert.isTrue(removed, "Remove NodeListListener failed: " + listener);
    }

    @Override
    public void addNodeSelectionListener(NodeSelectionListener l) {
        nodeSelectionListeners.add(l);
    }

    @Override
    public void removeNodeSelectionListener(NodeSelectionListener l) {
        nodeSelectionListeners.remove(l);
    }

    public void addMacroSelectionListener(MacroSelectionListener l) {
        macroSelectionListeners.add(l);
    }

    public void removeMacroSelectionListener(MacroSelectionListener l) {
        macroSelectionListeners.remove(l);
    }

    public void addAccessorySelectionListener(AccessorySelectionListener l) {
        accessorySelectionListeners.add(l);
    }

    public void removeAccessorySelectionListener(AccessorySelectionListener l) {
        accessorySelectionListeners.remove(l);
    }

    public void addCvDefinitionListener(CvDefinitionListener l) {
        cvDefinitionListeners.add(l);
    }

    public void removeCvDefinitionListener(CvDefinitionListener l) {
        cvDefinitionListeners.remove(l);
    }
    private void fireMacroSelectionChanged() {
        if (SwingUtilities.isEventDispatchThread()) {
            for (MacroSelectionListener l : macroSelectionListeners) {
                l.macroChanged();
            }
        }
        else {
            SwingUtilities.invokeLater(new Runnable() {

                @Override
                public void run() {
                    for (MacroSelectionListener l : macroSelectionListeners) {
                        l.macroChanged();
                    }
                }
            });
        }
    }

    private void fireAccessorySelectionChanged() {
        if (SwingUtilities.isEventDispatchThread()) {
            for (AccessorySelectionListener l : accessorySelectionListeners) {
                l.accessoryChanged();
            }
        }
        else {
            SwingUtilities.invokeLater(() -> {
                for (AccessorySelectionListener l : accessorySelectionListeners) {
                    l.accessoryChanged();
                }
            });
        }
    }

    private void fireSelectedNodeWillChange(final NodeInterface node) {
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));

        for (NodeListListener l : listeners) {
            l.nodeWillChange(node);
        }
    }

    private void fireSelectedNodeChanged(final NodeInterface node) {
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));

        for (NodeListListener l : listeners) {
            l.nodeChanged(node);
        }

        final Collection<NodeSelectionListener> nodeSelectionListeners =
            Collections.unmodifiableCollection(this.nodeSelectionListeners);

        for (NodeSelectionListener l : nodeSelectionListeners) {
            l.selectedNodeChanged(selectedNode);
        }
    }

    private void fireNodeListChanged() {
        LOGGER.debug("Notify the listeners that the node list has changed.");
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));
        for (NodeListListener l : listeners) {
            l.listChanged();
        }
    }

    private void fireNodeListAdded(final NodeInterface node) {
        LOGGER.debug("Notify the listeners that the node list has added a node.");
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));

        for (NodeListListener l : listeners) {
            l.listNodeAdded(node);
        }
    }

    private void fireNodeListRemoved(final NodeInterface node) {
        LOGGER.debug("Notify the listeners that the node list has removed a node.");
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));

        for (NodeListListener l : listeners) {
            l.listNodeRemoved(node);
        }
    }

    private void fireNodeStateChanged(final NodeInterface node) {
        final Collection<NodeListListener> listeners =
            new LinkedList<>(Collections.unmodifiableCollection(nodeListListeners));

        for (NodeListListener l : listeners) {
            l.nodeStateChanged(node);
        }
    }

    private void fireCvDefinitionChanged() {
        notifyCvDefinitionListeners(cvDefinitionListeners);
    }

    private void notifyCvDefinitionListeners(final Collection<CvDefinitionListener> listeners) {
        if (SwingUtilities.isEventDispatchThread()) {
            for (CvDefinitionListener l : listeners) {
                l.cvDefinitionChanged();
            }
        }
        else {
            SwingUtilities.invokeLater(() -> {
                for (CvDefinitionListener l : listeners) {
                    l.cvDefinitionChanged();
                }
            });
        }
    }

    public List<Flag> getFlags() {
        NodeInterface node = getSelectedNode();
        if (node != null) {
            return node.getFlags();
        }
        LOGGER.warn("No node selected to get the flags from.");
        return Collections.emptyList();
    }

    @Override
    public NodeInterface getSelectedNode() {
        synchronized (nodeLock) {
            return selectedNode;
        }
    }

    public StatusModel getStatusModel() {
        return statusModel;
    }

    /**
     * Returns the currently selected macro.
     * 
     * @return the currently selected macro
     */
    public Macro getSelectedMacro() {
        return selectedMacro;
    }

    /**
     * Set the selected macro.
     * 
     * @param macro
     *            the selected macro
     */
    public void setSelectedMacro(Macro macro) {

        if (selectedMacro != null && MacroSaveState.PENDING_CHANGES.equals(selectedMacro.getMacroSaveState())) {
            // log warning
            LOGGER.warn("The current macro has pending changes.");
        }

        this.selectedMacro = macro;

        // notify the macro panel of the selection
        fireMacroSelectionChanged();
    }

    public List<Macro> getMacros() {

        NodeInterface node = getSelectedNode();
        if (node != null) {
            List<Macro> macros = node.getMacros();
            return macros;
        }
        LOGGER.warn("No node selected to get the macros from.");
        return Collections.emptyList();
    }

    /**
     * Replace the macro with the provided macro.
     * 
     * @param macro
     *            the new macro
     */
    public void replaceMacro(final Macro macro) {
        LOGGER.info("Replace the macro: {}", macro);

        NodeInterface node = getSelectedNode();
        if (node != null) {
            Macro replacedMacro = node.replaceMacro(macro, false);
            setSelectedMacro(replacedMacro);
            return;
        }

        // fire an exception
        throw new RuntimeException("Replace macro failed because no node is selected.");
    }

    /**
     * @return the selected accessory
     */
    public Accessory getSelectedAccessory() {
        return selectedAccessory;
    }

    /**
     * Set the selected accessory.
     * 
     * @param accessory
     *            the selected accessory
     */
    public void setSelectedAccessory(Accessory accessory) {
        if (selectedAccessory != null
            && (AccessorySaveState.PERMANENTLY_STORED_ON_NODE != selectedAccessory.getAccessorySaveState())) {
            // log warning
            LOGGER.warn("The current accessory has pending changes.");
        }

        this.selectedAccessory = accessory;

        // notify the accessory panel of the selection
        fireAccessorySelectionChanged();
    }

    public List<Accessory> getAccessories() {
        NodeInterface node = getSelectedNode();
        if (node != null) {
            List<Accessory> accessories = node.getAccessories();
            return accessories;
        }
        LOGGER.warn("No node selected to get the accessories from.");
        return Collections.emptyList();
    }

    public void replaceAccessory(Accessory accessory) {

        NodeInterface node = getSelectedNode();
        if (node != null) {
            Accessory replacedAccessory = node.replaceAccessory(accessory, false);
            setSelectedAccessory(replacedAccessory);
            return;
        }

        // fire an exception
        throw new RuntimeException("Replace accessory failed because no node is selected.");
    }

    /**
     * Set the selected node instance
     * 
     * @param node
     *            the selected node instance
     * @param forceChange
     *            force the change of the node
     */
    public void setSelectedNode(NodeInterface node, boolean forceChange) {
        LOGGER.info("Set the selected node in the main model: {}", node);

        synchronized (nodeLock) {

            if ((node != null && node.equals(selectedNode)) || (node == null && selectedNode == null)) {
                LOGGER.info("The selected node has not changed.");
                return;
            }

            if (!forceChange) {
                // allow the pending changes to prevent the change of the node
                fireSelectedNodeWillChange(node);
            }

            selectedNode = node;

            // clear the cached node values
            clearCachedNodeValues();
        }

        SwingUtilities.invokeLater(() -> fireSelectedNodeChanged(node));
    }

    private void clearCachedNodeValues() {
        LOGGER.info("Clear the cached node values.");

        setSelectedMacro(null);
        setSelectedAccessory(null);
    }

    private void reloadLabels(final NodeInterface node) {
        // load the labels of the node
        LOGGER.info("Load labels for node: {}", node);

        if (node == null) {
            LOGGER.info("No node selected to load the labels.");
            return;
        }

        // load all labels
        wizardLabelWrapper.loadLabels(node.getUniqueId());

        LOGGER.info("Finished load labels for node: {}", node);
    }

    public void setNodeHasError(final NodeInterface node, boolean nodeHasError) {
        setNodeHasError(node, nodeHasError, null);
    }

    public void setNodeHasError(final NodeInterface node, boolean nodeHasError, String reason) {
        LOGGER.info("setErrorState, node: {}, nodeHasError: {}", node, nodeHasError);
        node.setNodeHasError(nodeHasError);
        if (reason != null) {
            node.setReasonData(reason);
        }
        fireNodeStateChanged(node);
    }

    public void setCvDefinition(VendorCvData vendorCV) {
        LOGGER.info("Set the CV definition: {}", vendorCV);

        fireCvDefinitionChanged();
    }

    public void signalInitialLoadFinished() {
        LOGGER.info("The initial load has finished.");
        initialLoadFinished.set(true);
    }

    public void signalResetInitialLoadFinished() {
        LOGGER.info("The initial load is reset.");
        initialLoadFinished.set(false);
    }

    public boolean isInitialLoadFinished() {
        return initialLoadFinished.get();
    }
}
