package org.bidib.wizard.mvc.main.model;

import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.LoadTypeEnum;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.ports.SwitchPairPort;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.model.listener.SwitchPairPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SwitchPairPortTableModel
    extends SimpleHierarchicalPortTableModel<SwitchPortStatus, SwitchPairPort, SwitchPairPortModelListener> {
    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPairPortTableModel.class);

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_SWITCH_OFF_TIME = 1;

    public static final int COLUMN_LOAD_TYPE = 2;

    public static final int COLUMN_PORT_IDENTIFIER = 3;

    public static final int COLUMN_STATUS = 4;

    public static final int COLUMN_TEST = 5;

    public static final int COLUMN_PORT_INSTANCE = 6;

    public SwitchPairPortTableModel(final MainModel model) {
        super();
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Integer getTestColumnIndex() {
        return COLUMN_TEST;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(getClass(), "label"), Resources.getString(getClass(), "switchOffTime"),
                Resources.getString(getClass(), "loadType"), Resources.getString(getClass(), "portIdentifier"),
                Resources.getString(getClass(), "status"), Resources.getString(getClass(), "test"), null };
    }

    @Override
    public void addRow(SwitchPairPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.getLabel();
            rowData[COLUMN_SWITCH_OFF_TIME] = port.getSwitchOffTime();
            rowData[COLUMN_LOAD_TYPE] = port.getLoadType();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();

            SwitchPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;
            rowData[COLUMN_PORT_INSTANCE] = port;
            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        SwitchPairPort switchPairPort = (SwitchPairPort) getValueAt(row, SwitchPairPortTableModel.COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_TEST:
                if (Port.getConfiguredPortType(switchPairPort) == LcOutputType.SWITCHPAIRPORT) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_PORT_INSTANCE:
            case COLUMN_PORT_IDENTIFIER:
                return Object.class;
            case COLUMN_LOAD_TYPE:
                return LoadTypeEnum.class;
            case COLUMN_STATUS:
                return Object.class;
            case COLUMN_TEST:
                return Object.class;

            default:
                return Object.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final SwitchPairPort port = (SwitchPairPort) getValueAt(row, SwitchPairPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_SWITCH_OFF_TIME:
                int switchOffTime = (Integer) value;
                if (port.getSwitchOffTime() != switchOffTime) {
                    port.setSwitchOffTime(switchOffTime);
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TICKS);
                }
                else {
                    LOGGER.debug("The switchOff time has not been changed.");
                }
                break;
            case COLUMN_LOAD_TYPE:
                LoadTypeEnum loadType = (LoadTypeEnum) value;
                if (loadType != port.getLoadType()) {
                    port.setLoadType(loadType);
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_LOAD_TYPE);
                }
                else {
                    LOGGER.debug("The load type has not been changed.");
                }
                break;
            case COLUMN_STATUS:
                port.setStatus((SwitchPortStatus) value);
                super.setValueAt(value, row, column);
                break;
            case COLUMN_TEST:
                LOGGER.info("Status of switch port is updated: {}, port: {}", value, port);

                SwitchPortStatus portStatus = (SwitchPortStatus) value;

                LOGGER.info("Set the test port status: {}", portStatus);

                super.setValueAt(portStatus, row, column);

                // set the value on the temporary port
                final SwitchPairPort switchPairPort = new SwitchPairPort();
                switchPairPort.setId(port.getId());
                switchPairPort.setStatus(portStatus);
                fireTestButtonPressed(switchPairPort);

                break;
            default:
                super.setValueAt(value, row, column);
                break;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        final SwitchPairPort port =
            (SwitchPairPort) super.getValueAt(row, SwitchPairPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_LOAD_TYPE:
                return port.getLoadType();
            case COLUMN_SWITCH_OFF_TIME:
                return port.getSwitchOffTime();
            case COLUMN_STATUS:
                return port.getStatus();
            case COLUMN_TEST:
            default:
                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortStatusChanged(final SwitchPairPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.info("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);

                // get the opposite status and set it
                SwitchPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());

                LOGGER.info("Update port status: {}, oppositeStatus: {}", port.getStatus(), oppositeStatus);

                super.setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(SwitchPairPort port) {
        LOGGER.debug("The port config was changed for port: {}", port);

        // update the port status
        notifyPortStatusChanged(port);
    }

    @Override
    public void notifyPortLabelChanged(final SwitchPairPort port) {
        LOGGER.info("The port label was changed for port: {}", port);

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, SwitchPairPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, SwitchPairPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final SwitchPairPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final SwitchPairPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            SwitchPairPort current = (SwitchPairPort) getValueAt(row, SwitchPairPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(SwitchPairPort port) {

        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(final SwitchPairPort port, PortConfigKeys... portConfigKeys) {

        LOGGER.info("The values of the port have changed: {}", port);
        portListener.configChanged(port, portConfigKeys);
    }

    /**
     * Change the port type.
     * 
     * @param portType
     *            the new port type
     * @param port
     *            the port
     */
    @Override
    public void changePortType(LcOutputType portType, SwitchPairPort port) {

        portListener.changePortType(portType, port);
    }

    @Override
    protected SwitchPairPort getRow(int rowIndex) {
        SwitchPairPort port = (SwitchPairPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return port;
    }

    public void fireTestButtonPressed(SwitchPairPort port) {
        LOGGER.info("Port status has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }
}
