package org.bidib.wizard.utils;

import java.awt.Component;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;

import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.platform.win32.User32;
import com.sun.jna.platform.win32.WinDef.DWORD;
import com.sun.jna.platform.win32.WinDef.HWND;

public class WindowUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(WindowUtils.class);

    /**
     * Brings the provided window to the front.
     * 
     * @param window
     *            the window instance to bring to front
     */
    public static void bringWindowToFront(Window window) {
        LOGGER.info("Try to bring the window to front, window: {}", window);
        try {
            if (window != null) {

                try {
                    if (SystemUtils.IS_OS_WINDOWS) {
                        sendToTheTop(Native.getWindowPointer(window));

                        HWND hWnd = new HWND();
                        hWnd.setPointer(Native.getComponentPointer(window));

                        boolean passedSetForeground = User32.INSTANCE.SetForegroundWindow(hWnd);
                        LOGGER.info("passedSetForeground: {}", passedSetForeground);

                        HWND activeWindow = User32Ext.INSTANCE.SetActiveWindow(hWnd);
                        LOGGER.info("activeWindow: {}", activeWindow);
                    }
                    else {
                        LOGGER.info("Bring window to front only works on Windows OS.");
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Cannot send the window {} to the front.", window, ex);
                }
            }
        }
        catch (Throwable ex) {
            LOGGER.warn("bring window to front failed.");
        }
    }

    /**
     * Move the window to the top of the z-order.
     *
     * @param pointer
     *            The pointer to the window to bring into background.
     * @return true if successful.
     */
    public static boolean sendToTheTop(Pointer pointer) {
        if (SystemUtils.IS_OS_WINDOWS) {
            return sendToTheTopWin32(pointer);
        }
        throw new UnsupportedOperationException(
            String.format("The function sendToTheTop(Pointer) is not supported for the %s OS", SystemUtils.OS_NAME));
    }

    /**
     * Move the window to the top of the z-order.
     *
     * @param pointer
     *            The pointer to the window to bring into background.
     * @return true if the window could placed set on top.
     */
    public static boolean sendToTheTopWin32(Pointer pointer) {
        if (pointer != null) {
            try {
                HWND thWindow = new HWND(pointer);
                HWND fgWindow = User32.INSTANCE.GetForegroundWindow();

                if (fgWindow == null) {
                    return false;
                }

                int thId = User32.INSTANCE.GetWindowThreadProcessId(thWindow, null);
                int fgId = User32.INSTANCE.GetWindowThreadProcessId(fgWindow, null);

                if (thId == fgId) {

                    User32.INSTANCE
                        .SetWindowPos(thWindow, new HWND(new Pointer(0)), 0, 0, 0, 0,
                            WindowPosFlags.NOMOVE.flag() | WindowPosFlags.NOSIZE.flag());
                    User32.INSTANCE.SetForegroundWindow(thWindow);
                    User32.INSTANCE.SetFocus(thWindow);
                }
                else {

                    User32.INSTANCE.AttachThreadInput(new DWORD(fgId), new DWORD(thId), true);
                    User32.INSTANCE
                        .SetWindowPos(thWindow, new HWND(new Pointer(0)), 0, 0, 0, 0,
                            WindowPosFlags.NOMOVE.flag() | WindowPosFlags.NOSIZE.flag());
                    User32.INSTANCE.SetForegroundWindow(thWindow);
                    User32.INSTANCE.SetFocus(thWindow);
                    User32.INSTANCE.AttachThreadInput(new DWORD(fgId), new DWORD(thId), false);
                }
                return true;
            }
            catch (RuntimeException e) {
                LOGGER.warn("Cannot send the window to foreground.", e);
                return false;
            }
        }
        else {
            return false;
        }
    }

    public enum WindowPosFlags {
        ASYNCWINDOWPOS(0x4000), DEFERERASE(0x2000), DRAWFRAME(0x0020), FRAMECHANGED(0x0020), HIDEWINDOW(
            0x0080), NOACTIVATE(0x0010), NOCOPYBITS(0x0100), NOMOVE(0x0002), NOOWNERZORDER(0x0200), NOREDRAW(
                0x0008), NOREPOSITION(
                    0x0200), NOSENDCHANGING(0x400), NOSIZE(0x0001), NOZORDER(0x0004), SHOWWINDOW(0x0040);

        int flag;

        WindowPosFlags(int flag) {
            this.flag = flag;
        }

        public int flag() {
            return flag;
        }
    }

    public static boolean isValidScreenLocation(final Rectangle location) {

        GraphicsEnvironment environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
        for (GraphicsDevice device : environment.getScreenDevices()) {
            Rectangle bounds = device.getDefaultConfiguration().getBounds();

            if (location.x < bounds.x || location.x > (bounds.x + bounds.width)) {
                continue;
            }
            if (location.y < bounds.y || location.y > (bounds.y + bounds.height)) {
                continue;
            }

            return true;
        }
        return false;
    }

    public static Rectangle getMaximumScreenBounds() {
        int minx = 0, miny = 0, maxx = 0, maxy = 0;
        GraphicsEnvironment environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
        for (GraphicsDevice device : environment.getScreenDevices()) {
            Rectangle bounds = device.getDefaultConfiguration().getBounds();
            minx = Math.min(minx, bounds.x);
            miny = Math.min(miny, bounds.y);
            maxx = Math.max(maxx, bounds.x + bounds.width);
            maxy = Math.max(maxy, bounds.y + bounds.height);
        }
        return new Rectangle(minx, miny, maxx - minx, maxy - miny);
    }

    /**
     * getScreenInsets, This returns the insets of the screen, which are defined by any task bars that have been set up
     * by the user. This function accounts for multi-monitor setups. If a window is supplied, then the the monitor that
     * contains the window will be used. If a window is not supplied, then the primary monitor will be used.
     */
    static public Insets getScreenInsets(Window windowOrNull) {
        Insets insets;
        if (windowOrNull == null) {
            insets =
                Toolkit
                    .getDefaultToolkit().getScreenInsets(GraphicsEnvironment
                        .getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration());
        }
        else {
            insets = windowOrNull.getToolkit().getScreenInsets(windowOrNull.getGraphicsConfiguration());
        }
        return insets;
    }

    /**
     * getScreenWorkingArea, This returns the working area of the screen. (The working area excludes any task bars.)
     * This function accounts for multi-monitor setups. If a window is supplied, then the the monitor that contains the
     * window will be used. If a window is not supplied, then the primary monitor will be used.
     */
    static public Rectangle getScreenWorkingArea(Window windowOrNull) {
        Insets insets;
        Rectangle bounds;
        if (windowOrNull == null) {
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            insets = Toolkit.getDefaultToolkit().getScreenInsets(ge.getDefaultScreenDevice().getDefaultConfiguration());
            bounds = ge.getDefaultScreenDevice().getDefaultConfiguration().getBounds();
        }
        else {
            GraphicsConfiguration gc = windowOrNull.getGraphicsConfiguration();
            insets = windowOrNull.getToolkit().getScreenInsets(gc);
            bounds = gc.getBounds();
        }
        bounds.x += insets.left;
        bounds.y += insets.top;
        bounds.width -= (insets.left + insets.right);
        bounds.height -= (insets.top + insets.bottom);
        return bounds;
    }

    /**
     * getScreenTotalArea, This returns the total area of the screen. (The total area includes any task bars.) This
     * function accounts for multi-monitor setups. If a window is supplied, then the the monitor that contains the
     * window will be used. If a window is not supplied, then the primary monitor will be used.
     */
    static public Rectangle getScreenTotalArea(Window windowOrNull) {
        Rectangle bounds;
        if (windowOrNull == null) {
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            bounds = ge.getDefaultScreenDevice().getDefaultConfiguration().getBounds();
        }
        else {
            GraphicsConfiguration gc = windowOrNull.getGraphicsConfiguration();
            bounds = gc.getBounds();
        }
        return bounds;
    }

    /**
     * Center the view on the current screen of the parent component.
     * 
     * @param parent
     *            the parent
     * @param view
     *            the view
     */
    public static void centerOnCurrentScreen(final Component parent, final Component view) {
        Rectangle screen = parent.getGraphicsConfiguration().getBounds();
        view
            .setLocation(screen.x + (screen.width - view.getWidth()) / 2,
                screen.y + (screen.height - view.getHeight()) / 2);

    }
}
