package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import org.bidib.jbidibc.messages.CurrentValue;
import org.bidib.jbidibc.messages.CurrentValue.CurrentType;
import org.bidib.jbidibc.messages.enums.CommandStationState;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.BoosterNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.model.status.BoosterStatus;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.listener.StatusListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.time.Millisecond;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.FormsSetup;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.DefaultComponentFactory;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.JideSplitButton;
import com.jidesoft.swing.JideToggleButton;

import eu.hansolo.steelseries.gauges.Radial1Vertical;
import eu.hansolo.steelseries.tools.Section;

/**
 * The {@code BoosterPanel} displays data of the command station and booster (current, temperature and voltage).
 */
public class BoosterPanel implements TabVisibilityProvider, TabPanelProvider, TabComponentCreator {

    private static final Logger LOGGER = LoggerFactory.getLogger(BoosterPanel.class);

    private static final String ENCODED_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private Collection<StatusListener> statusListeners = new LinkedList<StatusListener>();

    private final JPanel contentPanel;

    private final MainModel mainModel;

    private Radial1Vertical ampereMeter;

    private Radial1Vertical thermoMeter;

    private Radial1Vertical voltMeter;

    private final JideToggleButton boosterStatusButton;

    private final JTextField boosterStatusText;

    private final JideSplitButton commandStationStatusButton;

    private final JTextField commandStationStatusText;

    private NodeInterface selectedNode;

    private final ImageIcon iconBoosterOn;

    private final ImageIcon iconBoosterOff;

    private TimeSeries series;

    private JFreeChart chart;

    private DisabledPanel disabledChartPanel;

    private int maximumItemCount = 500;

    private JLabel boosterStatusLabel;

    public BoosterPanel(final MainModel model) {
        this.mainModel = model;

        iconBoosterOff = ImageUtils.createImageIcon(BoosterPanel.class, "/icons/booster/boosterOff_00.png");
        iconBoosterOn = ImageUtils.createImageIcon(BoosterPanel.class, "/icons/booster/boosterOn_80.png");

        boosterStatusButton = createBoosterStatusButton();
        boosterStatusText = addStatusText();
        boosterStatusText.setToolTipText(Resources.getString(BoosterPanel.class, "booster-status-tooltip"));

        commandStationStatusButton = createCommandStationStatusButton();

        setStatusButtonState(commandStationStatusButton, CommandStationStatus.OFF);
        commandStationStatusText = new JTextField(20);
        commandStationStatusText.setEditable(false);
        commandStationStatusText.setToolTipText(Resources.getString(BoosterPanel.class, "cs-status-tooltip"));

        // prepare the form
        boolean debugDialog = false;
        final JPanel panel;
        if (debugDialog) {
            panel = new FormDebugPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    return Resources.getString(BoosterPanel.class, "name");
                }
            };
        }
        else {
            panel = new JPanel(new BorderLayout()) {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    return Resources.getString(BoosterPanel.class, "name");
                }
            };
        }
        final FormBuilder dialogBuilder = FormBuilder.create().columns(ENCODED_COLUMN_SPECS).rows("p").panel(panel);
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        contentPanel = new JPanel(new BorderLayout()) {
            private static final long serialVersionUID = 1L;

            @Override
            public String getName() {
                return panel.getName();
            }
        };

        SwingUtilities.invokeLater(() -> {

            LOGGER.info("Create the meter for current, temp and voltage.");

            ampereMeter = addAmpereMeter();
            thermoMeter = addThermoMeter();
            voltMeter = addVoltMeter();

            dialogBuilder.add(Resources.getString(BoosterPanel.class, "cs")).xy(1, 1);
            dialogBuilder.add(commandStationStatusText).xy(3, 1);
            dialogBuilder.add(commandStationStatusButton).xy(5, 1);

            boosterStatusLabel =
                DefaultComponentFactory.getInstance().createLabel(Resources.getString(BoosterPanel.class, "booster"));

            dialogBuilder.add(boosterStatusLabel).xy(7, 1);
            dialogBuilder.add(boosterStatusText).xy(9, 1);
            dialogBuilder.add(boosterStatusButton).xy(11, 1);

            dialogBuilder.appendRows("5dlu");
            dialogBuilder.appendRows("fill:150dlu:none");

            LOGGER.info("Prepare the meter panel for current, temp and voltage.");
            JPanel meterPanel = new JPanel();
            meterPanel.setLayout(new BoxLayout(meterPanel, BoxLayout.LINE_AXIS));

            meterPanel.add(ampereMeter);
            meterPanel.add(Box.createRigidArea(new Dimension(5, 0)));
            meterPanel.add(thermoMeter);
            meterPanel.add(Box.createRigidArea(new Dimension(5, 0)));
            meterPanel.add(voltMeter);

            dialogBuilder.add(meterPanel).xyw(1, 3, 11);

            dialogBuilder.appendRows("5dlu");
            dialogBuilder.appendRows("fill:100dlu:grow");

            LOGGER.info("Create the TimeSeries for the booster current.");

            this.series = new TimeSeries("Current");
            final TimeSeriesCollection dataset = new TimeSeriesCollection(this.series);

            series.setMaximumItemCount(maximumItemCount);

            chart = createChart(dataset);

            // Created Chartpanel for chart area
            final ChartPanel chartPanel = new ChartPanel(chart);
            // gray out depending on confidence
            disabledChartPanel = new DisabledPanel(chartPanel);
            disabledChartPanel.setEnabled(false);

            dialogBuilder.add(disabledChartPanel).xyw(1, 5, 13);

            // create the content panel
            final JPanel innerContentPanel = dialogBuilder.build();

            JScrollPane scrollPaneBoosterPanel = new JScrollPane();
            scrollPaneBoosterPanel.setViewportView(innerContentPanel);
            scrollPaneBoosterPanel.setOpaque(FormsSetup.getOpaqueDefault());
            scrollPaneBoosterPanel.getViewport().setOpaque(FormsSetup.getOpaqueDefault());

            contentPanel.add(scrollPaneBoosterPanel, BorderLayout.CENTER);
            contentPanel.setBorder(Paddings.TABBED_DIALOG);
            contentPanel.setOpaque(FormsSetup.getOpaqueDefault());

            LOGGER.info("Update the component state.");
            // update the state of the components
            updateComponentState();
        });
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    /**
     * Creates a sample chart.
     *
     * @param dataset
     *            the dataset.
     *
     * @return A sample chart.
     */
    private JFreeChart createChart(final XYDataset dataset) {

        final Color backgroundColor = UIManager.getColor("Chart.background");

        final JFreeChart chart =
            ChartFactory
                .createTimeSeriesChart(Resources.getString(BoosterPanel.class, "boosterCurrent"),
                    Resources.getString(BoosterPanel.class, "time"),
                    Resources.getString(BoosterPanel.class, "current") + " [mA]", dataset, false, true, false);

        chart.setBackgroundPaint(backgroundColor);
        chart.getTitle().setPaint(UIManager.getColor("Chart.title.foreground"));

        final XYPlot plot = chart.getXYPlot();

        plot.setBackgroundPaint(backgroundColor);
        plot.setDomainGridlinesVisible(true);
        plot.setDomainGridlinePaint(Color.lightGray);
        plot.setRangeGridlinesVisible(true);
        plot.setRangeGridlinePaint(Color.lightGray);

        ValueAxis xaxis = plot.getDomainAxis();
        xaxis.setAutoRange(true);

        xaxis.setTickLabelsVisible(false);

        ValueAxis yaxis = plot.getRangeAxis();
        yaxis.setRange(0, 5000);

        final Font titleFont = UIManager.getDefaults().getFont("Label.font");
        chart.getTitle().setFont(titleFont.deriveFont(Font.BOLD));

        xaxis.setLabelFont(titleFont);
        yaxis.setLabelFont(titleFont);

        xaxis.setLabelPaint(UIManager.getColor("Chart.title.foreground"));
        yaxis.setLabelPaint(UIManager.getColor("Chart.title.foreground"));

        xaxis.setTickLabelPaint(UIManager.getColor("Chart.title.foreground"));
        xaxis.setTickMarkPaint(UIManager.getColor("Chart.title.foreground"));

        yaxis.setTickLabelPaint(UIManager.getColor("Chart.title.foreground"));
        yaxis.setTickMarkPaint(UIManager.getColor("Chart.title.foreground"));

        XYLineAndShapeRenderer r1 = new XYLineAndShapeRenderer();
        r1.setSeriesPaint(0, UIManager.getColor("Chart.current.line"));
        r1.setSeriesShapesVisible(0, false);

        plot.setRenderer(0, r1);

        return chart;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof BoosterPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    public void commandStationStatusChanged(CommandStationStatus status) {
        LOGGER.debug("Status of command station has changed: {}", status);

        setStatusButtonState(commandStationStatusButton, status);
        setStatusText(commandStationStatusText, status);
    }

    public void boosterStateChanged(BoosterStatus status) {
        LOGGER.info("Status of command station has changed: {}", status);

        setStatusButtonState(boosterStatusButton, status);
        setStatusText(boosterStatusText, status);
    }

    public void boosterCurrentChanged(final CurrentValue current, final long timestamp) {
        LOGGER.debug("Booster current has changed: {}", current);

        if (current != null && current.getCurrentType() == CurrentType.normal && current.getCurrent() >= 0) {
            ampereMeter.setEnabled(true);
            ampereMeter.setValueAnimated(current.getCurrent());
        }
        else {
            LOGGER.info("The current value is no longer valid.");
            ampereMeter.setValue(0);
            ampereMeter.setEnabled(false);
        }

        if (current != null) {
            try {
                this.series.add(new Millisecond(new Date(timestamp)), current.getCurrent());
            }
            catch (Exception ex) {
                LOGGER.warn("Add current value to timeseries failed.", ex);
            }
        }
    }

    private ValueMarker maxBoosterCurrentMarker;

    private int DEFAULT_MAXCURRENT_LIMIT = 4000;

    public void boosterMaximumCurrentChanged(Integer maximumCurrent) {
        LOGGER.debug("Booster maximum current has changed: {}", maximumCurrent);

        if (maximumCurrent != null && maximumCurrent > 0) {
            int upperLimit = maximumCurrent + 1000;
            ampereMeter.setMaxValue(upperLimit);

            Section[] sections = new Section[3];

            sections[0] = new Section(0, /* upperLimit / 2 */maximumCurrent - 500, Color.GREEN);
            sections[1] =
                new Section(/* upperLimit / 2 */ maximumCurrent - 500, /* 3 * upperLimit / 4 */ maximumCurrent,
                    Color.ORANGE);
            sections[2] = new Section(/* 3 * upperLimit / 4 */ maximumCurrent, upperLimit, Color.RED);
            ampereMeter.setSections(sections);

            BoosterNodeInterface boosterNode = selectedNode.getBoosterNode();
            if (boosterNode != null && boosterNode.getBoosterCurrent() != null) {
                ampereMeter.setValue(boosterNode.getBoosterCurrent().getCurrent());
            }
            else {
                ampereMeter.setValue(0);
            }

            if (maximumCurrent > DEFAULT_MAXCURRENT_LIMIT) {
                chart.getXYPlot().getRangeAxis().setRange(0, upperLimit);
            }
            else {
                chart.getXYPlot().getRangeAxis().setRange(0, DEFAULT_MAXCURRENT_LIMIT + 1000);
            }

            if (maxBoosterCurrentMarker != null) {
                chart.getXYPlot().removeRangeMarker(maxBoosterCurrentMarker);
            }
            maxBoosterCurrentMarker = new ValueMarker(maximumCurrent, Color.RED, new BasicStroke(0.5f));
            chart.getXYPlot().addRangeMarker(maxBoosterCurrentMarker);
        }
        else {
            // Max value cannot be equal to min value
            ampereMeter.setMaxValue(0.1);
        }
    }

    public void temperatureChanged(Integer temperature) {
        if (temperature != null) {
            thermoMeter.setValueAnimated(temperature);
        }
        else {
            thermoMeter.setValueAnimated(0);
        }
    }

    public void voltageChanged(Integer voltage) {
        if (voltage != null && voltage > 0) {
            voltMeter.setValueAnimated(voltage / 10d);
        }
        else {
            voltMeter.setValueAnimated(0);
        }
    }

    private JideToggleButton createBoosterStatusButton() {
        final JideToggleButton statusButton = new JideToggleButton();
        statusButton.setButtonStyle(JideSplitButton.TOOLBOX_STYLE);

        statusButton.setIcon(iconBoosterOff);

        // set the preferred size to prevent change of width if the button text is changed
        Dimension dim = statusButton.getPreferredSize();
        statusButton.setPreferredSize(dim);

        statusButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                JideToggleButton button = (JideToggleButton) e.getSource();

                if (button.isSelected()) {
                    fireSwitchedOn();
                }
                else {
                    fireSwitchedOff();
                }
            }
        });
        setStatusButtonState(statusButton, BoosterStatus.OFF);

        return statusButton;
    }

    private JideSplitButton createCommandStationStatusButton() {
        final JideSplitButton statusButton = new JideSplitButton(Resources.getString(getClass(), "cs-go"));
        statusButton.setButtonStyle(JideSplitButton.TOOLBOX_STYLE);

        statusButton.setButtonEnabled(true);
        statusButton.setEnabled(true);

        // set the preferred size to prevent change of width if the button text is changed
        Dimension dim = statusButton.getPreferredSize();
        statusButton.setPreferredSize(new Dimension(150, dim.height));

        statusButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {

                String statusButtonText = statusButton.getText();
                LOGGER
                    .info("The command station status button was clicked, current statusButtonText: {}",
                        statusButtonText);

                if (statusButtonText.equals(Resources.getString(BoosterPanel.class, "cs-go"))) {
                    fireSwitchedCommandStationOn(false);
                }
                else if (statusButtonText.equals(Resources.getString(BoosterPanel.class, "cs-goIgnWd"))) {
                    fireSwitchedCommandStationOn(true);
                }
                else if (statusButtonText.equals(Resources.getString(BoosterPanel.class, "cs-stop"))) {
                    fireSwitchedCommandStationStop();
                }
                else if (statusButtonText.equals(Resources.getString(BoosterPanel.class, "cs-softStop"))) {
                    fireSwitchedCommandStationSoftStop();
                }
                else if (statusButtonText.equals(Resources.getString(BoosterPanel.class, "cs-off"))) {
                    fireSwitchedCommandStationOff();
                }
            }
        });

        // prepare the popup menu for the split button
        JMenuItem csGoItem = new JMenuItem(Resources.getString(getClass(), "cs-go"));

        csGoItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                // prepareUpdate(true);
                fireSwitchedCommandStationOn(false);
            }
        });
        statusButton.add(csGoItem);

        JMenuItem csGoIgnWdItem = new JMenuItem(Resources.getString(getClass(), "cs-goIgnWd"));

        csGoIgnWdItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSwitchedCommandStationOn(true);
            }
        });
        statusButton.add(csGoIgnWdItem);

        JMenuItem csSoftStopItem = new JMenuItem(Resources.getString(getClass(), "cs-softStop"));

        csSoftStopItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSwitchedCommandStationSoftStop();
            }
        });
        statusButton.add(csSoftStopItem);

        JMenuItem csOffItem = new JMenuItem(Resources.getString(getClass(), "cs-off"));

        csOffItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSwitchedCommandStationOff();
            }
        });
        statusButton.add(csOffItem);

        statusButton.setEnabled(false);
        setStatusButtonState(statusButton, CommandStationStatus.OFF);

        return statusButton;
    }

    private JTextField addStatusText() {
        JTextField statusText = new JTextField();

        statusText.setEditable(false);
        statusText.setColumns(20);

        BoosterStatus boosterStatus = null;
        if (selectedNode != null && selectedNode.getBoosterNode() != null) {
            boosterStatus = selectedNode.getBoosterNode().getBoosterStatus();
        }

        setStatusText(statusText, boosterStatus);

        JPopupMenu popup = new JPopupMenu("Reload status");
        JMenuItem mi = new JMenuItem(Resources.getString(getClass(), "booster-query-status"));
        mi.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                LOGGER.info("Query the booster state");
                fireQueryBoosterState();
            }
        });
        popup.add(mi);
        statusText.add(popup);
        statusText.setComponentPopupMenu(popup);

        return statusText;
    }

    public void addStatusListener(StatusListener l) {
        statusListeners.add(l);
    }

    private Radial1Vertical addAmpereMeter() {
        Radial1Vertical ampereMeter = new Radial1Vertical();
        ampereMeter.setPreferredSize(new Dimension(160, 160));

        ampereMeter.setFrameVisible(true);
        ampereMeter.setLcdDecimals(0);
        ampereMeter.setLcdUnitStringVisible(false);
        ampereMeter.setLedVisible(false);
        ampereMeter.setSectionsVisible(true);
        ampereMeter.setTitle(Resources.getString(getClass(), "current"));
        ampereMeter.setTrackVisible(false);
        ampereMeter.setUnitString("mA");

        return ampereMeter;
    }

    private Radial1Vertical addThermoMeter() {
        Radial1Vertical thermoMeter = new Radial1Vertical();
        thermoMeter.setPreferredSize(new Dimension(160, 160));

        thermoMeter.setFrameVisible(true);
        thermoMeter.setLcdDecimals(0);
        thermoMeter.setLcdUnitStringVisible(false);
        thermoMeter.setLedVisible(false);
        thermoMeter.setMaxValue(100);
        thermoMeter.setSectionsVisible(true);
        thermoMeter.setTitle(Resources.getString(getClass(), "temperature"));
        thermoMeter.setTrackVisible(false);
        thermoMeter.setUnitString("°C");

        Section[] sections = new Section[3];

        sections[0] = new Section(0, 70, Color.GREEN);
        sections[1] = new Section(70, 90, Color.ORANGE);
        sections[2] = new Section(90, thermoMeter.getMaxValue(), Color.RED);
        thermoMeter.setSections(sections);

        return thermoMeter;
    }

    private Radial1Vertical addVoltMeter() {
        Radial1Vertical voltMeter = new Radial1Vertical();
        voltMeter.setPreferredSize(new Dimension(160, 160));

        voltMeter.setFrameVisible(true);
        voltMeter.setLcdDecimals(0);
        voltMeter.setLcdUnitStringVisible(false);
        voltMeter.setLedVisible(false);
        voltMeter.setMaxValue(30);
        voltMeter.setSectionsVisible(true);
        voltMeter.setTitle(Resources.getString(getClass(), "voltage"));
        voltMeter.setTrackVisible(false);
        voltMeter.setUnitString("V");

        Section[] sections = new Section[5];

        sections[0] = new Section(0, 10, Color.RED);
        sections[1] = new Section(10, 11, Color.ORANGE);
        sections[2] = new Section(11, 22, Color.GREEN);
        sections[3] = new Section(22, 23, Color.ORANGE);
        sections[4] = new Section(23, voltMeter.getMaxValue(), Color.RED);
        voltMeter.setSections(sections);

        return voltMeter;
    }

    private void fireSwitchedOn() {
        for (StatusListener l : statusListeners) {
            l.switchedOn();
        }
    }

    private void fireSwitchedOff() {
        for (StatusListener l : statusListeners) {
            l.switchedOff();
        }
    }

    private void fireQueryBoosterState() {
        for (StatusListener l : statusListeners) {
            l.queryBoosterState();
        }
    }

    private void fireSwitchedCommandStationOn(boolean ignoreWatchDog) {
        LOGGER.info("Set the command station ON.");
        for (StatusListener l : statusListeners) {
            l.switchedCommandStationOn(ignoreWatchDog);
        }
    }

    private void fireSwitchedCommandStationStop() {
        LOGGER.info("Set the command station STOP.");
        for (StatusListener l : statusListeners) {
            l.switchedCommandStationStop();
        }
    }

    private void fireSwitchedCommandStationSoftStop() {
        LOGGER.info("Set the command station SOFT STOP.");
        for (StatusListener l : statusListeners) {
            l.switchedCommandStationSoftStop();
        }
    }

    private void fireSwitchedCommandStationOff() {
        LOGGER.info("Set the command station OFF.");
        for (StatusListener l : statusListeners) {
            l.switchedCommandStationOff();
        }
    }

    private void setStatusButtonState(JideToggleButton statusButton, BoosterStatus status) {
        final NodeInterface selectedNode = mainModel.getSelectedNode();
        if (status != null) {

            boolean wasEnabled = statusButton.isEnabled();
            if (selectedNode != null) {
                wasEnabled = wasEnabled && NodeUtils.hasBoosterFunctions(selectedNode.getUniqueId());
            }

            if (BoosterStatus.isOffState(status)) {
                // the booster is off

                statusButton.setSelected(false);
                statusButton.setIcon(iconBoosterOff);
                // statusButton.setText(Resources.getString(getClass(), "on"));
            }
            else {
                // the booster is on
                statusButton.setSelected(true);
                statusButton.setIcon(iconBoosterOn);
                // statusButton.setText(Resources.getString(getClass(), "off"));
            }

            // restore the enabled state because setSelected() enables the button
            LOGGER.info("Set boosterStatus button enabled: {}", wasEnabled);
            statusButton.setEnabled(wasEnabled);
        }
        else {
            if (selectedNode != null && NodeUtils.hasBoosterFunctions(selectedNode.getUniqueId())) {
                statusButton.setEnabled(true);
            }
            else {
                statusButton.setSelected(false);
                statusButton.setEnabled(false);
            }
        }
    }

    private void setStatusButtonState(JideSplitButton statusButton, CommandStationStatus status) {
        final NodeInterface selectedNode = mainModel.getSelectedNode();

        if (status != null) {

            boolean wasEnabled = statusButton.isEnabled();
            if (selectedNode != null) {
                wasEnabled = wasEnabled && NodeUtils.hasCommandStationFunctions(selectedNode.getUniqueId());
            }

            if (wasEnabled) {
                if (CommandStationState.isOffState(status.getType())) {
                    // the command station is off
                    // statusButton.setSelected(false);
                    statusButton.setText(Resources.getString(getClass(), "cs-go"));
                }
                else {
                    // the command station is on
                    // statusButton.setSelected(true);
                    statusButton.setText(Resources.getString(getClass(), "cs-stop"));
                }
            }

            // restore the enabled state because setSelected() enables the button
            LOGGER.debug("Set the command station status button enabled: {}", wasEnabled);
            statusButton.setEnabled(wasEnabled);
        }
        else {
            if (selectedNode != null && NodeUtils.hasCommandStationFunctions(selectedNode.getUniqueId())) {
                LOGGER.debug("Set the status button enabled because the node has command station functions.");
                statusButton.setEnabled(true);
            }
            else {
                LOGGER.debug("Set the status button disabled because the node has no command station functions.");
                // statusButton.setSelected(false);
                statusButton.setEnabled(false);
            }
        }
    }

    private void setStatusText(JTextField statusText, BoosterStatus status) {

        statusText.setText(status != null ? Resources.getString(BoosterStatus.class, status.getKey()) : "");
    }

    private void setStatusText(JTextField statusText, CommandStationStatus status) {
        LOGGER.debug("Set the new status of the command station: {}", status);

        statusText.setText(status != null ? Resources.getString(CommandStationStatus.class, status.getKey()) : "");
    }

    public void nodeChanged() {
        // LOGGER.debug("The selected node has changed.");

        updateComponentState();
    }

    private void updateComponentState() {

        final NodeInterface node = mainModel.getSelectedNode();

        if (selectedNode != null && selectedNode.equals(node)) {
            LOGGER.debug("Node is selected already: {}", node);
            return;
        }

        boolean isBooster = (node != null ? NodeUtils.hasBoosterFunctions(node.getUniqueId()) : false);
        boolean isCommandStation = (node != null ? NodeUtils.hasCommandStationFunctions(node.getUniqueId()) : false);

        // clear the booster current values
        series.clear();

        if (isBooster || isCommandStation) {
            selectedNode = node;
        }
        else {
            selectedNode = null;
        }

        LOGGER
            .debug("The selected node has booster functions: {}, has command station functions: {}, node: {}",
                isBooster, isCommandStation, node);

        if (SwingUtilities.isEventDispatchThread()) {
            updateComponentsVisibilityAndEnabledState(isBooster, isCommandStation);
        }
        else {
            SwingUtilities.invokeLater(() -> updateComponentsVisibilityAndEnabledState(isBooster, isCommandStation));
        }
    }

    private void updateComponentsVisibilityAndEnabledState(boolean isBooster, boolean isCommandStation) {
        boosterStatusText.setVisible(isBooster);
        boosterStatusLabel.setVisible(isBooster);

        boosterStatusButton.setVisible(isBooster);
        boosterStatusButton.setEnabled(isBooster);
        ampereMeter.setEnabled(isBooster);
        ampereMeter.setVisible(isBooster);
        voltMeter.setEnabled(isBooster);
        voltMeter.setVisible(isBooster);
        thermoMeter.setEnabled(isBooster);
        thermoMeter.setVisible(isBooster);

        disabledChartPanel.setEnabled(isBooster);
        disabledChartPanel.setVisible(isBooster);

        commandStationStatusButton.setEnabled(isCommandStation);
        commandStationStatusButton.setVisible(isCommandStation);
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible =
                NodeUtils.hasCommandStationFunctions(node.getUniqueId())
                    || NodeUtils.hasBoosterFunctions(node.getUniqueId());
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }
}
