package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Calendar;
import java.util.Date;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroRepeatDay;
import org.bidib.wizard.api.model.MacroRepeatTime;
import org.bidib.wizard.api.model.MacroSaveState;
import org.bidib.wizard.api.model.StartCondition;
import org.bidib.wizard.api.model.TimeStartCondition;
import org.bidib.wizard.api.model.function.Function;
import org.bidib.wizard.api.model.listener.MacroListener;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.MacroSelectionListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MacroParameterPanel extends JPanel implements MacroListener {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(MacroParameterPanel.class);

    private final JSpinner timeSpinner = new JSpinner(new SpinnerDateModel());

    private final DisabledPanel disabledRepeatPanel;

    private boolean silentUpdate;

    private Macro macro;

    private final SliderPanel cyclesPanel;

    private final SliderPanel slowdownPanel;

    private final JCheckBox timeButton;

    private final JComboBox<MacroRepeatTime> repeatTime;

    private final JComboBox<MacroRepeatDay> repeatDay;

    private static class MacroRepeatTimeRenderer extends DefaultListCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getListCellRendererComponent(
            JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {

            MacroRepeatTime macroRepeatTime = (MacroRepeatTime) value;
            setText(Resources.getString(MacroRepeatTime.class, macroRepeatTime.toString()));
            return this;
        }
    }

    private static class MacroRepeatDayRenderer extends DefaultListCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getListCellRendererComponent(
            JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {

            MacroRepeatDay macroRepeatDay = (MacroRepeatDay) value;
            setText(Resources.getString(MacroRepeatDay.class, macroRepeatDay.toString()));
            return this;
        }
    }

    public MacroParameterPanel(final MainModel model) {
        setLayout(new GridBagLayout());

        final JPanel repeatPanel = new JPanel(new GridLayout(2, 1));

        // TODO create a MacroRepeatTimeRenderer to render with resources string

        repeatTime = new JComboBox<MacroRepeatTime>();
        repeatTime.setRenderer(new MacroRepeatTimeRenderer());
        repeatTime.setModel(new DefaultComboBoxModel<MacroRepeatTime>(MacroRepeatTime.values()));

        // TODO create a MacroRepeatDayRenderer to render with resources string

        repeatDay = new JComboBox<MacroRepeatDay>();
        repeatDay.setRenderer(new MacroRepeatDayRenderer());
        repeatDay.setModel(new DefaultComboBoxModel<MacroRepeatDay>(MacroRepeatDay.values()));

        timeSpinner.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                for (StartCondition startCondition : model.getSelectedMacro().getStartConditions()) {
                    if (startCondition instanceof TimeStartCondition) {
                        JSpinner spinner = (JSpinner) e.getSource();
                        Calendar spinnerTime = Calendar.getInstance();

                        spinnerTime.setTime((Date) spinner.getValue());

                        Calendar time = Calendar.getInstance();

                        time.set(Calendar.HOUR_OF_DAY, spinnerTime.get(Calendar.HOUR_OF_DAY));
                        time.set(Calendar.MINUTE, spinnerTime.get(Calendar.MINUTE));
                        ((TimeStartCondition) startCondition).setTime(time);
                    }
                }
            }
        });

        timeButton = new JCheckBox(Resources.getString(getClass(), "time") + ":");

        timeButton.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                LOGGER.debug("The selected item has been changed, silentUpdate: {}.", silentUpdate);

                timeSpinner.setEnabled(e.getStateChange() == ItemEvent.SELECTED);
                disabledRepeatPanel.setEnabled(e.getStateChange() == ItemEvent.SELECTED);
                if (e.getStateChange() == ItemEvent.SELECTED) {
                    Calendar time = Calendar.getInstance();

                    time.setTime((Date) timeSpinner.getValue());
                    if (!silentUpdate) {
                        model.getSelectedMacro().addStartCondition(new TimeStartCondition(time));
                    }
                }
                else {
                    if (!silentUpdate) {
                        model.getSelectedMacro().removeStartCondition(new TimeStartCondition());
                    }
                }

                if (!silentUpdate) {
                    Macro macro = model.getSelectedMacro();
                    macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
                }
            }
        });

        JPanel startConditionPanel = new JPanel(new BorderLayout());
        startConditionPanel
            .setBorder(BorderFactory
                .createTitledBorder(BorderFactory.createEtchedBorder(),
                    Resources.getString(getClass(), "condition") + ":"));

        FlowLayout leftLayout = new FlowLayout();

        leftLayout.setAlignment(FlowLayout.LEFT);
        timeSpinner.setEditor(new JSpinner.DateEditor(timeSpinner, "HH:mm"));

        JPanel timePanel = new JPanel(leftLayout);

        timePanel.add(timeButton);
        timePanel.add(timeSpinner);

        startConditionPanel.add(timePanel, BorderLayout.NORTH);

        repeatTime.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                for (StartCondition startCondition : model.getSelectedMacro().getStartConditions()) {
                    if (startCondition instanceof TimeStartCondition) {
                        MacroRepeatTime repeatTime = (MacroRepeatTime) ((JComboBox<?>) e.getSource()).getSelectedItem();

                        ((TimeStartCondition) startCondition).setRepeatTime(repeatTime);

                        if (!silentUpdate) {
                            Macro macro = model.getSelectedMacro();
                            macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
                        }
                    }
                }
            }
        });
        repeatPanel.add(repeatTime);
        repeatDay.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                for (StartCondition startCondition : model.getSelectedMacro().getStartConditions()) {
                    if (startCondition instanceof TimeStartCondition) {
                        ((TimeStartCondition) startCondition)
                            .setRepeatDay((MacroRepeatDay) ((JComboBox<?>) e.getSource()).getSelectedItem());

                        if (!silentUpdate) {
                            Macro macro = model.getSelectedMacro();
                            macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
                        }
                    }
                }
            }
        });
        repeatPanel.add(repeatDay);
        repeatPanel
            .setBorder(BorderFactory
                .createTitledBorder(BorderFactory.createEtchedBorder(),
                    Resources.getString(getClass(), "repetition") + ":"));

        disabledRepeatPanel = new DisabledPanel(repeatPanel);
        startConditionPanel.add(disabledRepeatPanel, BorderLayout.CENTER);

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;

        add(startConditionPanel, gbc);

        slowdownPanel = new SliderPanel(Resources.getString(getClass(), "delay") + ":", 1, 250, 250, true, 0) {
            private static final long serialVersionUID = 1L;

            @Override
            public void stateChanged(ChangeEvent e) {
                Macro macro = model.getSelectedMacro();

                if (macro != null) {
                    JSlider source = (JSlider) e.getSource();
                    int slowdown = source.getValue();
                    LOGGER.debug("Set the current macro slowdown: {}", slowdown);
                    macro.setSpeed(slowdown);
                    setLabel(macro.getSpeed());

                    macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
                }
            }
        };

        gbc.gridx++;
        gbc.weightx = 1;
        gbc.fill = GridBagConstraints.BOTH;
        add(slowdownPanel, gbc);

        cyclesPanel = new SliderPanel(Resources.getString(getClass(), "cycles") + ":", 1, 251, 1, false, 251) {
            private static final long serialVersionUID = 1L;

            @Override
            public void stateChanged(ChangeEvent e) {
                Macro macro = model.getSelectedMacro();

                if (macro != null) {
                    JSlider source = (JSlider) e.getSource();
                    int cycles = source.getValue();
                    // if more than 250 then we have infinite cycles
                    if (cycles > Macro.REPEAT_RANGE_MAX_VALUE) {
                        cycles = Macro.REPEAT_INFINITE_CYCLES;
                    }
                    macro.setCycles(cycles);
                    setLabel(source.getValue());

                    macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
                }
            }
        };

        gbc.gridx++;
        gbc.weightx = 1;
        gbc.fill = GridBagConstraints.BOTH;
        add(cyclesPanel, gbc);

        model.addMacroSelectionListener(new MacroSelectionListener() {

            @Override
            public void macroChanged() {
                processMacroChanged(model);
            }
        });
    }

    private void processMacroChanged(MainModel model) {
        Macro macro = model.getSelectedMacro();
        LOGGER.info("The selected macro has changed: {}", macro);

        if (this.macro != null) {
            this.macro.removeMacroListener(this);
        }
        // set the macro
        this.macro = macro;

        if (macro != null) {
            this.macro.addMacroListener(this);
        }

        setValueSilently();
    }

    private void setValueSilently() {
        silentUpdate = true;
        try {
            if (macro != null) {
                slowdownPanel.setValueSilently(macro.getSpeed());
                int cycles = macro.getCycles();
                // infinite cycles is handled as 251 in the UI
                if (cycles == Macro.REPEAT_INFINITE_CYCLES) {
                    cycles = Macro.REPEAT_RANGE_MAX_VALUE + 1;
                }
                cyclesPanel.setValueSilently(cycles);

                boolean hasTimeCondition = false;

                for (StartCondition startCondition : macro.getStartConditions()) {
                    if (startCondition instanceof TimeStartCondition) {
                        hasTimeCondition = true;

                        Calendar time = ((TimeStartCondition) startCondition).getTime();

                        if (time != null) {
                            timeSpinner.setValue(time.getTime());
                        }
                        repeatDay.setSelectedIndex(((TimeStartCondition) startCondition).getRepeatDay().ordinal());
                        repeatTime.setSelectedIndex(((TimeStartCondition) startCondition).getRepeatTime().ordinal());
                    }
                }
                timeButton.setSelected(hasTimeCondition);
                timeSpinner.setEnabled(hasTimeCondition);
                disabledRepeatPanel.setEnabled(hasTimeCondition);
            }
        }
        finally {
            silentUpdate = false;
        }
    }

    @Override
    public void functionsAdded(int macroId, int row, Function<? extends BidibStatus>[] functions) {

    }

    @Override
    public void functionRemoved(int macroId, int row) {

    }

    @Override
    public void functionMoved(int macroId, int fromIndex, int toIndex, Function<? extends BidibStatus> fromFunction) {
    }

    @Override
    public void functionsRemoved(int macroId) {

    }

    @Override
    public void startConditionChanged() {
        LOGGER.debug("The start condition has changed.");
        SwingUtils.executeInEDT(() -> setValueSilently());
    }

    @Override
    public void slowdownFactorChanged() {
        LOGGER.debug("The slowdown factor has changed.");
        SwingUtils.executeInEDT(() -> setValueSilently());
    }

    @Override
    public void cyclesChanged() {
        LOGGER.debug("The cycles have changed.");
        SwingUtils.executeInEDT(() -> setValueSilently());
    }
}
