package org.bidib.wizard.mvc.main.controller;

import java.util.List;

import org.apache.commons.lang3.time.StopWatch;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.CvDefinitionRequestListener;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.client.common.view.DefaultBusyFrame;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MainCvDefinitionRequestListener implements CvDefinitionRequestListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(MainCvDefinitionRequestListener.class);

    private final MainModel mainModel;

    private final NodeService nodeService;

    private final String connectionId;

    private StatusBar statusBar;

    private DefaultBusyFrame frame;

    public MainCvDefinitionRequestListener(final MainModel mainModel, final NodeService nodeService,
        final String connectionId, final StatusBar statusBar, final DefaultBusyFrame frame) {
        this.mainModel = mainModel;
        this.nodeService = nodeService;
        this.connectionId = connectionId;
        this.statusBar = statusBar;
        this.frame = frame;
    }

    @Override
    public List<ConfigurationVariable> loadCvValues(List<ConfigurationVariable> configVariables) {
        LOGGER.info("Load CV definition values!");
        StopWatch sw = new StopWatch();
        sw.start();

        boolean cursorChanged = false;
        try {
            cursorChanged = this.frame.setBusy(true);

            NodeInterface selectedNode = this.mainModel.getSelectedNode();

            nodeService.queryConfigVariables(this.connectionId, selectedNode, configVariables);

            LOGGER.debug("Update model with configuration variables: {}", configVariables);
            selectedNode.updateConfigVariableValues(configVariables, true);

            if (cursorChanged) {
                this.frame.setBusy(false);
            }
            sw.stop();

            LOGGER.info("Load CV values has finished! Total loading duration: {}", sw);
            this.statusBar
                .setStatusText(String.format(Resources.getString(MainController.class, "loadCvFinished"), sw),
                    StatusBar.DISPLAY_NORMAL);
            // }
            return configVariables;
        }
        catch (RuntimeException ex) {
            LOGGER.warn("Load CV values has failed!", ex);
            this.statusBar
                .setStatusText(String.format(Resources.getString(MainController.class, "loadCvFailed")),
                    StatusBar.DISPLAY_ERROR);

            throw ex;
        }
    }

    @Override
    public void writeCvValues(List<ConfigurationVariable> cvList) {
        LOGGER.info("Write CV definition values!");
        StopWatch sw = new StopWatch();
        sw.start();

        boolean cursorChanged = false;
        try {
            cursorChanged = this.frame.setBusy(true);

            NodeInterface selectedNode = mainModel.getSelectedNode();

            List<ConfigurationVariable> configVars =
                nodeService.setConfigVariables(this.connectionId, selectedNode, cvList);

            // iterate over the collection of stored variables in the model and update the values.
            // After that notify the tree and delete the new values that are now stored in the node
            selectedNode.updateConfigVariableValues(configVars, false);
        }
        finally {
            if (cursorChanged) {
                this.frame.setBusy(false);
            }
            sw.stop();

            LOGGER.info("Write CV values has finished! Total writing duration: {}", sw);
            this.statusBar
                .setStatusText(String.format(Resources.getString(MainController.class, "writeCvFinished"), sw),
                    StatusBar.DISPLAY_NORMAL);
        }
    }
}
