package org.bidib.wizard.mvc.main.model;

import java.awt.Color;
import java.util.Arrays;
import java.util.function.IntConsumer;

import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimpleHierarchicalPortTableModel;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.BidibStatus;
import org.bidib.wizard.model.status.LightPortStatus;
import org.bidib.wizard.mvc.main.model.listener.LightPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LightPortTableModel
    extends SimpleHierarchicalPortTableModel<LightPortStatus, LightPort, LightPortModelListener> {
    private static final Logger LOGGER = LoggerFactory.getLogger(LightPortTableModel.class);

    private static final long serialVersionUID = 1L;

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_PWM_MIN = 1;

    public static final int COLUMN_PWM_MAX = 2;

    public static final int COLUMN_DIM_MIN = 3;

    public static final int COLUMN_DIM_MAX = 4;

    public static final int COLUMN_RGB = 5;

    public static final int COLUMN_TRANSITION_TIME = 6;

    public static final int COLUMN_DMX_MAPPING = 7;

    public static final int COLUMN_PORT_IDENTIFIER = 8;

    public static final int COLUMN_STATUS = 9;

    public static final int COLUMN_TEST = 10;

    public static final int COLUMN_PORT_INSTANCE = 11;

    public LightPortTableModel() {
        super();
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public Integer getTestColumnIndex() {
        return COLUMN_TEST;
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(LightPortTableModel.class, "label"),
                Resources.getString(LightPortTableModel.class, "portLevelOff"),
                Resources.getString(LightPortTableModel.class, "portLevelOn"),
                Resources.getString(LightPortTableModel.class, "dimmDown"),
                Resources.getString(LightPortTableModel.class, "dimmUp"),
                Resources.getString(LightPortTableModel.class, "rgbValue"),
                Resources.getString(LightPortTableModel.class, "transitionTime"),
                Resources.getString(LightPortTableModel.class, "dmxMapping"),
                Resources.getString(LightPortTableModel.class, "portIdentifier"),
                Resources.getString(LightPortTableModel.class, "status"),
                Resources.getString(LightPortTableModel.class, "test"), null };
    }

    @Override
    public void addRow(LightPort port) {
        if (port != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = port.toString();
            rowData[COLUMN_PWM_MIN] = port.getPwmMin();
            rowData[COLUMN_PWM_MAX] = port.getPwmMax();
            rowData[COLUMN_DIM_MIN] = port.getDimMin();
            rowData[COLUMN_DIM_MAX] = port.getDimMax();
            rowData[COLUMN_RGB] = (port.getRgbValue() != null ? new Color(port.getRgbValue()) : null);
            rowData[COLUMN_TRANSITION_TIME] = port.getTransitionTime();
            rowData[COLUMN_DMX_MAPPING] = port.getDmxMapping();
            rowData[COLUMN_PORT_IDENTIFIER] = port.getPortIdentifier();
            rowData[COLUMN_STATUS] = port.getStatus();

            LightPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;
            rowData[COLUMN_PORT_INSTANCE] = port;

            addRow(rowData);
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        LightPort lightPort = (LightPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        switch (column) {
            case COLUMN_TEST:
                // the test can be changed.
                if (Port.getConfiguredPortType(lightPort) == LcOutputType.LIGHTPORT) {
                    isEditable = true;
                }
                break;
            default:
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            case COLUMN_RGB:
                return Color.class;
            case COLUMN_STATUS:
                return Object.class;
            case COLUMN_TEST:
                return Object.class;
            case COLUMN_PORT_INSTANCE:
            case COLUMN_PORT_IDENTIFIER:
                return Object.class;
            default:
                return Integer.class;
        }
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final LightPort port = (LightPort) super.getValueAt(row, LightPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_PWM_MIN:
                port.setPwmMin((Integer) value);
                super.setValueAt(value, row, column);
                fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_LEVEL_PORT_OFF);
                break;
            case COLUMN_PWM_MAX:
                port.setPwmMax((Integer) value);
                super.setValueAt(value, row, column);
                fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_LEVEL_PORT_ON);
                break;
            case COLUMN_DIM_MIN:
                try {
                    port.setDimMin(Integer.parseInt(value.toString()));
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_DIMM_DOWN);
                }
                catch (NumberFormatException e) {
                }
                break;
            case COLUMN_DIM_MAX:
                try {
                    port.setDimMax(Integer.parseInt(value.toString()));
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_DIMM_UP);
                }
                catch (NumberFormatException e) {
                }
                break;
            case COLUMN_RGB:
                Integer col = null;
                if (value instanceof Color) {
                    col = ((Color) value).getRGB();
                }
                else if (value instanceof Integer) {
                    col = ((Integer) value).intValue();
                }
                port.setRgbValue(col);
                super.setValueAt(value, row, column);
                fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_RGB);
                break;
            case COLUMN_TRANSITION_TIME:
                try {
                    port.setTransitionTime(Integer.parseInt(value.toString()));
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_TRANSITION_TIME);
                }
                catch (NumberFormatException e) {
                }
                break;
            case COLUMN_DMX_MAPPING:
                try {
                    if (value instanceof Integer) {
                        port.setDmxMapping((Integer) value);
                    }
                    else {
                        port.setDmxMapping(Integer.parseInt(value.toString()));
                    }
                    super.setValueAt(value, row, column);
                    fireConfigValuesChanged(port, PortConfigKeys.BIDIB_PCFG_OUTPUT_MAP);
                }
                catch (NumberFormatException e) {
                }
                break;
            case COLUMN_STATUS:
                port.setStatus((LightPortStatus) value);
                super.setValueAt(value, row, column);
                break;
            case COLUMN_TEST:
                LightPortStatus portStatus = (LightPortStatus) value;

                LOGGER.info("Set the test port status: {}", portStatus);

                super.setValueAt(portStatus, row, column);

                // set the value on the temporary port
                final LightPort lightPort = new LightPort();
                lightPort.setId(port.getId());
                lightPort.setStatus(portStatus);
                fireTestButtonPressed(lightPort);

                break;
            default:
                super.setValueAt(value, row, column);
                break;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        final LightPort port = (LightPort) super.getValueAt(row, LightPortTableModel.COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_PORT_IDENTIFIER:
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_DIM_MAX:
                return port.getDimMax();
            case COLUMN_DIM_MIN:
                return port.getDimMin();
            case COLUMN_PWM_MIN:
                return port.getPwmMin();
            case COLUMN_PWM_MAX:
                return port.getPwmMax();
            case COLUMN_RGB:
                return (port.getRgbValue() != null ? new Color(port.getRgbValue()) : null);
            case COLUMN_TRANSITION_TIME:
                return port.getTransitionTime();
            case COLUMN_DMX_MAPPING:
                return port.getDmxMapping();
            case COLUMN_STATUS:
                return port.getStatus();
            case COLUMN_PORT_INSTANCE:
                return port;
            case COLUMN_TEST:
            default:

                break;
        }
        return super.getValueAt(row, column);
    }

    @Override
    public void notifyPortStatusChanged(final LightPort port) {
        // the port status is signaled from the node
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.debug("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);

                // get the opposite status and set it
                LightPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());

                LOGGER.debug("Update port status: {}, oppositeStatus: {}", port.getStatus(), oppositeStatus);

                super.setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(LightPort port) {
        LOGGER.debug("The port config was changed for port: {}", port);

        // update the port status
        notifyPortStatusChanged(port);
    }

    @Override
    public void notifyPortLabelChanged(final LightPort port) {
        LOGGER.info("The port label was changed for port: {}", port.getDebugString());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, LightPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, LightPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final LightPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final LightPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            LightPort current = (LightPort) getValueAt(row, LightPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    /**
     * Set the new port config values of the port.
     * 
     * @param port
     *            the port with the new port config values
     */

    // TODO rename to storePortConfig ??

    public void updatePortConfig(final LightPort port) {

        LOGGER.debug("The config of the port has changed: {}", port.getDebugString());

        PortConfigKeys[] keys = PortConfigKeys.valueOf(port.getKnownPortConfigKeys());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("prepared keys: {}", Arrays.asList(keys));
        }

        fireConfigValuesChanged(port, keys);

        int row = findRow(port);
        if (row > -1) {
            // set the label
            setValueAt(port.getLabel(), row, 0);
        }
    }

    private void fireConfigValuesChanged(LightPort port, PortConfigKeys... portConfigKeys) {

        LOGGER.info("The values of the port have changed: {}", port);

        portListener.configChanged(port, portConfigKeys);
    }

    @Override
    public void changePortType(LcOutputType portType, Port<? extends BidibStatus> port) {

        portListener.changePortType(portType, port);
    }

    @Override
    protected LightPort getRow(int rowIndex) {
        LightPort port = (LightPort) getValueAt(rowIndex, COLUMN_PORT_INSTANCE);
        return port;
    }

    public void fireTestButtonPressed(LightPort port) {
        LOGGER.info("Port status has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }

}
