package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;

import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.context.ApplicationContext;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.AccessorySaveState;
import org.bidib.wizard.api.model.AccessorySwitchTimeModel;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.AccessoryListListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.utils.AccessoryListUtils;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.client.common.view.ComponentUtils;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.AccessoryLabelUtils;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.script.node.types.AccessoryAspectTargetType;
import org.bidib.wizard.common.script.node.types.AccessoryTargetType;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.bidib.wizard.mvc.common.view.table.DefaultTextCellEditor;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.bidib.wizard.mvc.main.controller.AccessoryPanelController;
import org.bidib.wizard.mvc.main.model.AccessoryFactory;
import org.bidib.wizard.mvc.main.model.AccessoryStartupAspectModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SingleColumnTableModel;
import org.bidib.wizard.mvc.main.view.component.AccessoryFileDialog;
import org.bidib.wizard.mvc.main.view.component.DefaultTabSelectionPanel;
import org.bidib.wizard.mvc.main.view.menu.AccessoryListMenu;
import org.bidib.wizard.mvc.main.view.menu.listener.AccessoryListMenuListener;
import org.bidib.wizard.mvc.main.view.panel.listener.AccessoryActionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelChangedListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabSelectionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.panel.renderer.AccessoryListRenderer;
import org.bidib.wizard.mvc.stepcontrol.controller.StepControlControllerInterface;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.PropertyConnector;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.grid.RowStripeTableStyleProvider;
import com.jidesoft.grid.SortableTable;
import com.jidesoft.grid.TableStyleProvider;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

public class AccessoryListPanel
    implements AccessoryListMenuListener, AccessoryActionListener, AccessoryListListener, ChangeLabelSupport,
    TabSelectionListener, TabPanelProvider, TabVisibilityProvider, PendingChangesAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(AccessoryListPanel.class);

    private final List<LabelChangedListener<Accessory>> labelChangedListeners = new ArrayList<>();

    private final TabVisibilityListener tabVisibilityListener;

    private final MainModel mainModel;

    private final AccessoryPanel accessoryPanel;

    private final JScrollPane scrollAccessoryList;

    private final AccessoryList accessoryList;

    private final AccessoryListMenu accessoryListMenu = new AccessoryListMenu();

    private final DisabledPanel disabledBorderPanel;

    private final JPanel contentPanel;

    private final StatusBar statusBar;

    private JPanel buttonPanel;

    private final AccessoryPanelController accessoryPanelController;

    private final SettingsService settingsService;

    private final WizardLabelWrapper wizardLabelWrapper;

    private final ConsoleService consoleService;

    public class AccessoryList extends SortableTable {
        private static final long serialVersionUID = 1L;

        private TableStyleProvider tableStyleProvider;

        private List<ListSelectionListener> listSelectionListeners = new ArrayList<>();

        public AccessoryList(final DefaultTableModel tableModel, final Consumer<Accessory> selectionChangedCallback) {
            super(tableModel);

            getSelectionModel().addListSelectionListener(new ListSelectionListener() {

                @Override
                public void valueChanged(ListSelectionEvent e) {

                    int selectedRow = getSelectedRow();
                    LOGGER.info("Selected accessory changed, selectedRow: {}", selectedRow);

                    if (!e.getValueIsAdjusting()) {
                        final ListSelectionEvent evt =
                            new ListSelectionEvent(tableModel, selectedRow, selectedRow, e.getValueIsAdjusting());

                        for (ListSelectionListener listener : listSelectionListeners) {
                            listener.valueChanged(evt);
                        }

                        selectionChangedCallback.accept(getSelectedItem());

                        // request the focus back after change the selection
                        AccessoryList.this.requestFocus();
                    }
                }
            });
        }

        protected String getEmptyTableText() {
            return "No accessories available.";
        }

        public void adjustRowHeight() {
            setRowHeight(getRowHeight() + 8);
        }

        protected void prepareTableStyleProvider() {
            tableStyleProvider =
                new RowStripeTableStyleProvider(UIManager.getColor("tableRowStripe.background"),
                    UIManager.getColor("tableRowStripe.alternativeBackground"));
        }

        @Override
        public TableStyleProvider getTableStyleProvider() {
            if (tableStyleProvider == null) {
                prepareTableStyleProvider();
            }
            return tableStyleProvider;
        }

        public void setItems(Accessory[] accessories) {
            LOGGER.info("Set the accessories: {}", new Object[] { accessories });

            final DefaultTableModel tableModel = ((DefaultTableModel) getModel());
            if (tableModel.getRowCount() > 0) {
                tableModel.setRowCount(0);
            }

            for (Accessory accessory : accessories) {
                tableModel.addRow(new Object[] { accessory });
            }
        }

        public void refreshView() {
            // repaint the table
            repaint();
        }

        public void addListSelectionListener(ListSelectionListener l) {

            this.listSelectionListeners.add(l);
        }

        public Accessory getSelectedItem() {

            int selectedRow = getSelectedRow();
            if (selectedRow > -1) {
                return (Accessory) getModel().getValueAt(selectedRow, 0);
            }
            return null;
        }
    }

    public AccessoryListPanel(final AccessoryPanelController accessoryPanelController, final MainModel model,
        final AccessoryStartupAspectModel accessoryStartupAspectModel,
        final AccessorySwitchTimeModel accessorySwitchTimeModel, final TabVisibilityListener tabVisibilityListener,
        final SettingsService settingsService, final WizardLabelWrapper wizardLabelWrapper, final StatusBar statusBar,
        final ConsoleService consoleService) {

        this.accessoryPanelController = accessoryPanelController;
        this.mainModel = model;
        this.tabVisibilityListener = tabVisibilityListener;
        this.settingsService = settingsService;
        this.wizardLabelWrapper = wizardLabelWrapper;
        this.consoleService = consoleService;

        // get the status bar
        this.statusBar = statusBar;

        contentPanel = new DefaultTabSelectionPanel(this) {
            private static final long serialVersionUID = 1L;

            @Override
            public boolean equals(Object other) {
                if (other instanceof TabComponentCreator) {
                    TabComponentCreator creator = (TabComponentCreator) other;
                    // TODO if more than a single instance is available this must be changed
                    if (creator.getCreator() instanceof AccessoryListPanel) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public int hashCode() {
                return super.hashCode();
            }

            @Override
            public void tabSelected(boolean selected) {
                LOGGER.debug("Select tab, current component is the AccessoryListPanel.");

                AccessoryListPanel.this.tabSelected(selected);
            }

            @Override
            public String getName() {
                // this is used as tab title
                return Resources.getString(AccessoryListPanel.class, "name");
            }
        };

        contentPanel.setLayout(new BorderLayout());
        contentPanel.setBorder(new EmptyBorder(5, 5, 5, 5));

        accessoryListMenu.addMenuListener(this);

        accessoryPanel =
            new AccessoryPanel(accessoryPanelController, model, accessoryStartupAspectModel, accessorySwitchTimeModel);

        JPanel panel = new JPanel(new BorderLayout());

        disabledBorderPanel = new DisabledPanel(accessoryPanel);
        panel.add(disabledBorderPanel);

        // create the list for the accessories
        final SingleColumnTableModel<Accessory> tableModel = new SingleColumnTableModel<Accessory>(Accessory.class) {
            private static final long serialVersionUID = 1L;

            @Override
            public void setValueAt(Object value, int row, int column) {
                LOGGER.info("Set the new value: {}, index: {}", value, row);

                final NodeInterface node = model.getSelectedNode();

                if (ProductUtils.isStepControl(node.getUniqueId())) {
                    // step control needs special handling
                    if (row < 3) {
                        LOGGER.info("The first 3 accessories are not changeable.");
                        return;
                    }
                }

                super.setValueAt(value, row, column);
            }

            @Override
            public boolean isCellEditable(int row, int column) {
                Accessory selectedAccessory = (Accessory) super.getValueAt(row, column);
                return selectedAccessory.isEditable();
            }

        };

        tableModel.addLabelChangedListener((accessory, label) -> {
            LOGGER.info("The label of the accessory was changed: {}, label: {}", accessory, label);

            long uniqueId = model.getSelectedNode().getUniqueId();

            // the name of the accessory was changed
            accessory.setLabel(label);

            final NodeLabels nodeLabels = this.wizardLabelWrapper.loadLabels(uniqueId);

            AccessoryLabelUtils.replaceAccessoryLabel(nodeLabels, accessory.getId(), label);
            this.wizardLabelWrapper.saveNodeLabels(uniqueId);
        });

        this.accessoryList = new AccessoryList(tableModel, (accessory) -> {
            setEnabled(accessory != null);

            if (buttonPanel != null) {
                List<JButton> buttons = ComponentUtils.harvestComponents(buttonPanel, JButton.class);
                for (JButton button : buttons) {
                    button.setEnabled(accessory != null /* && accessory.isMacroMapped() */);
                }
            }

        });

        accessoryList.adjustRowHeight();

        // do not allow drag columns to other position
        accessoryList.getTableHeader().setReorderingAllowed(false);

        accessoryList.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);

        // disabled sorting
        accessoryList.setSortable(false);
        accessoryList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        // set the renderer
        final TableColumn tc = this.accessoryList.getColumnModel().getColumn(0);
        tc.setPreferredWidth(80);
        tc.setWidth(80);
        final AccessoryListRenderer renderer = new AccessoryListRenderer();
        tc.setCellRenderer(renderer);

        tc.setCellEditor(new DefaultTextCellEditor(new JTextField()));

        this.accessoryList.setPreferredScrollableViewportSize(new Dimension(200, 200));

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(accessoryList));
        accessoryList.getModel().addTableModelListener(new TableModelListener() {

            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(accessoryList.getModel().getRowCount() == 0);
            }
        });

        accessoryList.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                handleMouseEvent(e, accessoryListMenu);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                handleMouseEvent(e, accessoryListMenu);
            }
        });

        overlayTable
            .addOverlayComponent(
                StyledLabelBuilder.createStyledLabel("{" + accessoryList.getEmptyTableText() + ":f:gray}"));

        scrollAccessoryList = new JScrollPane(overlayTable);

        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, scrollAccessoryList, panel);

        splitPane.setContinuousLayout(true);
        splitPane.setOneTouchExpandable(true);
        splitPane.setResizeWeight(0.2);

        contentPanel.add(splitPane, BorderLayout.CENTER);

        setEnabled(false);

        addButtonPanel(panel);
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    private void addButtonPanel(JPanel parent) {

        JButton loadButton = new JButton(Resources.getString(getClass(), "reload"));

        loadButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireReloadAccessory();
            }
        });

        JButton saveButton = new JButton(Resources.getString(getClass(), "save"));

        saveButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSaveAccessory();
            }
        });

        buttonPanel = new ButtonBarBuilder().addGlue().addButton(loadButton, saveButton).build();
        buttonPanel.setBorder(Paddings.TABBED_DIALOG);

        parent.add(buttonPanel, BorderLayout.SOUTH);

        PropertyConnector.connect(accessoryPanel.getSaveButtonEnabledModel(), "value", saveButton, "enabled");
    }

    @Override
    public void exportAccessory() {

        exportAccessory(mainModel.getSelectedAccessory());
    }

    @Override
    public void saveAccessory() {
        fireSaveAccessory();
    }

    @Override
    public void reloadAccessory() {
        fireReloadAccessory();
    }

    private void fireReloadAccessory() {

        reloadAccessory(mainModel.getSelectedAccessory());
    }

    private void fireSaveAccessory() {
        final Accessory selectedAccessory = mainModel.getSelectedAccessory();
        // update the values of the selected accessory with the values in the panel
        accessoryPanel.saveAccessory(selectedAccessory);

        // write the accessory to the node
        saveAccessory(selectedAccessory);
    }

    @Override
    public void importAccessory() {

        importAccessory(mainModel.getSelectedAccessory());
    }

    @Override
    public void deleteAccessory() {
        // and force the panels to reload
        try {
            final Accessory accessory = mainModel.getSelectedAccessory();
            if (accessory != null && accessory.getAccessorySaveState().equals(AccessorySaveState.PENDING_CHANGES)) {
                // show dialog
                int result =
                    JOptionPane
                        .showConfirmDialog(getComponent(),
                            Resources.getString(AccessoryListPanel.class, "accessory_has_pending_changes"),
                            Resources.getString(AccessoryListPanel.class, "pending_changes"),
                            JOptionPane.OK_CANCEL_OPTION);
                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User cancelled discard pending changes.");
                    return;
                }

                // reset pending changes
                mainModel.getSelectedAccessory().setAccessorySaveState(AccessorySaveState.PERMANENTLY_STORED_ON_NODE);
            }

            LOGGER.info("Delete the accessoty by initialize the values.");

            // initialize the accessory
            accessory.initialize();
            accessory.setAccessorySaveState(AccessorySaveState.PENDING_CHANGES);
            accessory.setLabel(null);

            // save the label
            saveChangedLabel(accessory, null);

            mainModel.setSelectedAccessory(accessory);
        }
        catch (Exception ex) {
            LOGGER.warn("Initialize the selected accessory failed.", ex);
        }
    }

    // @Override
    public void setEnabled(boolean enabled) {
        disabledBorderPanel.setEnabled(enabled);
    }

    @Override
    public void changeLabel(TargetType portType) {
        int accessoryNumber = portType.getPortNum();
        String label = portType.getLabel();

        LOGGER.info("Change the label, portType: {}, label: {}", portType, label);

        if (portType instanceof AccessoryTargetType) {

            Accessory accessory =
                AccessoryListUtils.findAccessoryByAccessoryNumber(mainModel.getAccessories(), accessoryNumber);
            if (accessory != null) {
                LOGGER
                    .info("Current accessory: {}, new label: {}, accessoryNumber: {}", accessory, label,
                        accessoryNumber);
                accessory.setLabel(label);

                fireLabelChanged(accessory, label);
            }
        }
        else if (portType instanceof AccessoryAspectTargetType) {
            AccessoryAspectTargetType aspectTargetType = (AccessoryAspectTargetType) portType;

            accessoryPanel.changeLabel(aspectTargetType);
        }
    }

    public void addLabelChangedListener(LabelChangedListener<Accessory> l) {
        labelChangedListeners.add(l);
    }

    public void addListSelectionListener(ListSelectionListener l) {
        accessoryList.addListSelectionListener(l);
    }

    protected void fireLabelChanged(Accessory accessory, String label) {
        labelChanged(accessory, label);

        this.accessoryList.refreshView();

        for (LabelChangedListener<Accessory> l : labelChangedListeners) {
            l.labelChanged(accessory, label);
        }
    }

    protected void handleMouseEvent(final MouseEvent e, final JPopupMenu popupMenu) {
        if (e.isPopupTrigger() && accessoryList.getRowCount() > 0) {

            if (accessoryList.getCellEditor() != null) {
                accessoryList.getCellEditor().stopCellEditing();
            }

            int selectRow = accessoryList.rowAtPoint(e.getPoint());
            accessoryList.setRowSelectionInterval(selectRow, selectRow);

            Accessory accessory = accessoryList.getSelectedItem();

            if (accessory != null && ProductUtils.isStepControl(mainModel.getSelectedNode().getUniqueId())) {

                if (accessory.getId() < 3) {
                    // the step control has 3 fixed items
                    popupMenu.setEnabled(false);
                }
                else {
                    popupMenu.setEnabled(true);
                }
            }
            else {
                popupMenu.setEnabled(true);
            }

            SwingUtilities.invokeLater(() -> popupMenu.show(e.getComponent(), e.getX(), e.getY()));
        }
    }

    @Override
    public void tabSelected(boolean selected) {
        LOGGER.info("Tab is selected: {}", selected);

        if (selected) {
            if (accessoryList.getCellEditor() != null) {
                accessoryList.getCellEditor().stopCellEditing();
            }
        }

        if (selected && mainModel.getSelectedNode() != null) {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            boolean isStepControl = ProductUtils.isStepControl(uniqueId);

            if (isStepControl) {
                LOGGER.info("The current node is a StepControl. Trigger load of CV values.");
                try {
                    StepControlControllerInterface controller =
                        DefaultApplicationContext
                            .getInstance().get(DefaultApplicationContext.KEY_STEPCONTROL_CONTROLLER,
                                StepControlControllerInterface.class);

                    controller.triggerLoadCvValues();
                }
                catch (Exception ex) {
                    LOGGER.warn("Trigger load CV values of StepControl failed.", ex);
                }
            }
        }
    }

    private static final String WORKING_DIR_ACCESSORY_EXCHANGE_KEY = "accessoryExchange";

    @Override
    public void exportAccessory(final Accessory accessory) {

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_ACCESSORY_EXCHANGE_KEY);

        // save the accessory
        AccessoryFileDialog dialog =
            new AccessoryFileDialog(getComponent(),
                // default is legacy filter
                FileDialog.SAVE, storedWorkingDirectory, accessory, false) {
                @Override
                public void approve(String fileName) {
                    try {
                        final NodeInterface selectedNode = mainModel.getSelectedNode();

                        final ApplicationContext context = new DefaultApplicationContext();
                        context.register("uniqueId", selectedNode.getUniqueId());

                        try {
                            NodeLabels labels = wizardLabelWrapper.loadLabels(selectedNode.getUniqueId());
                            context.register("labels", labels);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Load labels from BiDiB failed.", ex);

                            // display warning in console
                            SwingUtilities.invokeLater(() -> {
                                ConsoleController.ensureConsoleVisible();

                                // show an error message in the console
                                consoleService
                                    .addConsoleLine(ConsoleColor.red,
                                        String
                                            .format("Load lables failed for node with uniqueId: %s",
                                                ByteUtils.getUniqueIdAsString(selectedNode.getUniqueId())));
                            });
                        }

                        AccessoryFactory.saveAccessory(fileName, accessory, context);

                        final String workingDir = Paths.get(fileName).getParent().toString();
                        LOGGER.info("Save current workingDir: {}", workingDir);

                        wizardSettings.setWorkingDirectory(WORKING_DIR_ACCESSORY_EXCHANGE_KEY, workingDir);

                        // update the status bar
                        statusBar
                            .setStatusText(String
                                .format(Resources.getString(AccessoryListPanel.class, "exportedAccessory"), fileName),
                                StatusBar.DISPLAY_NORMAL);
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            };
        dialog.showDialog();
    }

    @Override
    public void importAccessory(final Accessory accessory) {

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_ACCESSORY_EXCHANGE_KEY);

        AccessoryFileDialog dialog =
            new AccessoryFileDialog(getComponent(), FileDialog.OPEN, storedWorkingDirectory, accessory, false) {

                @Override
                public void approve(String fileName) {
                    final NodeInterface selectedNode = mainModel.getSelectedNode();

                    final ApplicationContext context = new DefaultApplicationContext();
                    context.register("uniqueId", mainModel.getSelectedNode().getUniqueId());

                    long uniqueId = mainModel.getSelectedNode().getUniqueId();

                    NodeLabels labels = null;
                    try {
                        labels = wizardLabelWrapper.loadLabels(uniqueId);
                        context.register("labels", labels);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Load labels from BiDiB failed.", ex);

                        // display warning in console
                        SwingUtilities.invokeLater(() -> {
                            ConsoleController.ensureConsoleVisible();

                            // show an error message in the console
                            consoleService
                                .addConsoleLine(ConsoleColor.red,
                                    String
                                        .format("Load lables failed for node with uniqueId: %s",
                                            ByteUtils.getUniqueIdAsString(selectedNode.getUniqueId())));
                        });
                    }

                    Accessory accessory = AccessoryFactory.loadAccessory(fileName, selectedNode, context);

                    if (accessory != null) {
                        Accessory selectedAccessory = mainModel.getSelectedAccessory();

                        accessory.setId(selectedAccessory.getId());
                        // set the total number possible macros for this accessory

                        int maximumMacroMappedAspects =
                            Feature
                                .getIntFeatureValue(mainModel.getSelectedNode().getNode().getFeatures(),
                                    BidibLibrary.FEATURE_ACCESSORY_MACROMAPPED);
                        accessory.setMaximumMacroMappedAspects(maximumMacroMappedAspects);
                        if (accessory.getStartupState() == null) {
                            LOGGER
                                .info(
                                    "The imported accessory has no startup state assigned. Use the state from the currently selected accessory.");
                            accessory.setStartupState(selectedAccessory.getStartupState());
                        }

                        mainModel.replaceAccessory(accessory);

                        final String workingDir = Paths.get(fileName).getParent().toString();
                        LOGGER.info("Save current workingDir: {}", workingDir);

                        wizardSettings.setWorkingDirectory(WORKING_DIR_ACCESSORY_EXCHANGE_KEY, workingDir);

                        // update the status bar
                        statusBar
                            .setStatusText(String
                                .format(Resources.getString(AccessoryListPanel.class, "importedAccessory"), fileName),
                                StatusBar.DISPLAY_NORMAL);
                    }

                    if (labels != null) {
                        wizardLabelWrapper.saveNodeLabels(uniqueId);
                    }
                    else {
                        LOGGER.warn("No labels to save available: {}", ByteUtils.getUniqueIdAsString(uniqueId));
                    }
                }
            };
        dialog.showDialog();
    }

    @Override
    public void labelChanged(final Accessory accessory, String label) {

        saveChangedLabel(accessory, label);
    }

    private void saveChangedLabel(final Accessory accessory, String label) {

        if (accessory != null) {
            LOGGER.info("Save the changed label, accessory: {}, new label: {}", accessory, label);

            long uniqueId = mainModel.getSelectedNode().getUniqueId();

            final NodeLabels nodeLabels = wizardLabelWrapper.loadLabels(uniqueId);

            AccessoryLabelUtils.replaceAccessoryLabel(nodeLabels, accessory.getId(), label);

            try {
                wizardLabelWrapper.saveNodeLabels(uniqueId);
            }
            catch (Exception e) {
                LOGGER.warn("Save accessory labels failed.", e);
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public void reloadAccessory(Accessory accessory) {
        if (accessory != null && (AccessorySaveState.PERMANENTLY_STORED_ON_NODE != accessory.getAccessorySaveState())) {

            // show dialog
            int result =
                JOptionPane
                    .showConfirmDialog(getComponent(),
                        Resources.getString(AccessoryListPanel.class, "accessory_has_pending_changes"),
                        Resources.getString(AccessoryListPanel.class, "pending_changes"), JOptionPane.OK_CANCEL_OPTION);
            if (result != JOptionPane.OK_OPTION) {
                LOGGER.info("User canceled discard pending changes.");
                return;
            }

            // reset pending changes
            accessory.setAccessorySaveState(AccessorySaveState.PERMANENTLY_STORED_ON_NODE);
        }

        accessoryPanelController.reloadAccessory(accessory);

        accessory.setAccessorySaveState(AccessorySaveState.PERMANENTLY_STORED_ON_NODE);
    }

    @Override
    public void saveAccessory(Accessory accessory) {
        LOGGER.info("Save the accessory: {}", accessory);

        accessoryPanelController.storeAccessory(accessory);
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof AccessoryListPanel) {
            // TODO if more than a single instance is available this must be changed
            return true;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public boolean isTabVisible() {
        NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasAccessories();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    public void accessoryChanged(final Integer accessoryId) {
        // the selected accessory has changed

        SwingUtils.executeInEDT(() -> internalAccessoryChanged(accessoryId));
    }

    private void internalAccessoryChanged(Integer accessoryId) {

        Accessory accessory = mainModel.getSelectedAccessory();

        setEnabled(accessory != null);

        if (buttonPanel != null) {
            List<JButton> buttons = ComponentUtils.harvestComponents(buttonPanel, JButton.class);
            for (JButton button : buttons) {
                button.setEnabled(accessory != null /* && accessory.isMacroMapped() */);
            }
        }
    }

    @Override
    public void listChanged() {
        LOGGER.info("Accessory list has been changed.");
        SwingUtils.executeInEDT(() -> internalListChanged());
    }

    private void internalListChanged() {
        LOGGER.info("Process internal accessory list has been changed.");

        // clear the selected accessory
        mainModel.setSelectedAccessory(null);

        // set the new accessories
        List<Accessory> accessories = new ArrayList<>();
        accessories.addAll(mainModel.getAccessories());

        // sort by accessory id
        Collections.sort(accessories, new Comparator<Accessory>() {

            @Override
            public int compare(Accessory o1, Accessory o2) {
                return (o1.getId() - o2.getId());
            }
        });

        accessoryList.setItems(accessories.toArray(new Accessory[0]));

        accessoryList.refreshView();

        resetAccessoryScrollPane();
        setEnabled(false);

        tabVisibilityListener.setTabVisible(getComponent(), isTabVisible());
    }

    private void resetAccessoryScrollPane() {
        JScrollBar verticalScrollBar = scrollAccessoryList.getVerticalScrollBar();
        JScrollBar horizontalScrollBar = scrollAccessoryList.getHorizontalScrollBar();
        verticalScrollBar.setValue(verticalScrollBar.getMinimum());
        horizontalScrollBar.setValue(horizontalScrollBar.getMinimum());
    }

    @Override
    public void pendingChangesChanged() {
        SwingUtils.executeInEDT(() -> {
            accessoryList.refreshView();
        });
    }

    @Override
    public boolean hasPendingChanges() {

        NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<Accessory> accessories = node.getAccessories();

            for (Accessory accessory : accessories) {

                if (AccessorySaveState.PENDING_CHANGES == accessory.getAccessorySaveState()) {
                    LOGGER.info("Found pending changes in accessory: {}", accessory);
                    return true;
                }
            }
        }

        return false;
    }

    @Override
    public void savePendingChanges() {
        LOGGER.info("Save the pending changes.");
        // TODO Auto-generated method stub

    }

    @Override
    public void editLabel(final MouseEvent popupEvent) {
        SwingUtilities.invokeLater(() -> {

            int selectedRow = accessoryList.getSelectedRow();
            if (selectedRow > -1) {
                LOGGER.info(">>> Edit row, selectedRow: {}", selectedRow);

                accessoryList.requestFocus();

                accessoryList.editCellAt(selectedRow, 0);

                // select the text in the editor
                final TableCellEditor editor = accessoryList.getCellEditor();
                if (editor instanceof DefaultTextCellEditor) {
                    SwingUtilities.invokeLater(() -> {

                        JTextField textField = (JTextField) ((DefaultTextCellEditor) editor).getComponent();
                        LOGGER.info("Request focus in textField: {}", textField);
                        textField.requestFocusInWindow();
                    });
                }
            }
        });
    }

    @Override
    public void copyLabel() {
        Accessory selectedAccessory = accessoryList.getSelectedItem();
        if (selectedAccessory != null) {
            LOGGER.info("Copy label of selectedRow to clipboard: {}", selectedAccessory);

            final Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            StringSelection stringSelection = new StringSelection(selectedAccessory.getLabel());
            clipboard.setContents(stringSelection, null);
        }
    }

    public void refreshView() {

        this.accessoryList.refreshView();
        this.accessoryPanel.refreshView();
    }

    public void handleStringChanged(int index, String value) {
        this.accessoryPanel.handleStringChanged(index, value);
    }
}
