package org.bidib.wizard.mvc.main.view.panel;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JPanel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.BacklightPortValueListener;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.model.listener.PortTableLayoutListener;
import org.bidib.wizard.client.common.table.AbstractPortHierarchicalTable;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.editor.NumberRangeEditor;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.client.common.view.renderer.DmxChannelRenderer;
import org.bidib.wizard.client.common.view.renderer.PortAwareNumberRenderer;
import org.bidib.wizard.client.common.view.slider.ConfigXAwareSliderRenderer;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.BacklightPortStatus;
import org.bidib.wizard.mvc.common.view.editor.DmxChannelCellEditor;
import org.bidib.wizard.mvc.main.controller.BacklightPortPanelController;
import org.bidib.wizard.mvc.main.model.BacklightPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.BacklightPortModelListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.table.BacklightPortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.TableColumnChooser;

import io.reactivex.rxjava3.subjects.Subject;

public class BacklightPortListPanel
    extends
    SimpleHierarchicalPortListPanel<BacklightPortTableModel, BacklightPortStatus, BacklightPort, BacklightPortValueListener, BacklightPortModelListener>
    implements TabVisibilityProvider, TabPanelProvider, TabComponentCreator, PortTableLayoutListener {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(BacklightPortListPanel.class);

    private final MainModel mainModel;

    private final AtomicBoolean isDmxNode = new AtomicBoolean(true);

    private final TabVisibilityListener tabVisibilityListener;

    // private AtomicBoolean updateModelInProgress = new AtomicBoolean();

    public BacklightPortListPanel(final BacklightPortPanelController controller,
        final BacklightPortTableModel tableModel, final MainModel mainModel,
        final TabVisibilityListener tabVisibilityListener,
        final Subject<PortConfigChangeEvent> portConfigChangeEventSubject) {
        super(tableModel, Resources.getString(BacklightPortListPanel.class, "emptyTable"), portConfigChangeEventSubject,
            mainModel);

        this.mainModel = mainModel;
        this.tabVisibilityListener = tabVisibilityListener;
    }

    @Override
    protected AbstractPortHierarchicalTable<BacklightPort> createPortTable(
        BacklightPortTableModel tableModel, String emptyTableText) {

        return new BacklightPortTable(tableModel, emptyTableText);
    }

    @Override
    protected void createTable(
        final BacklightPortTableModel tableModel, String emptyTableText,
        final Subject<PortConfigChangeEvent> portConfigChangeEventSubject,
        final NodeSelectionProvider nodeSelectionProvider) {

        super.createTable(tableModel, emptyTableText, portConfigChangeEventSubject, nodeSelectionProvider);

        // create the portlist menu listener to handle the port menu
        final PortListMenuListener portListMenuListener =
            createPortListMenuListener(this.table, nodeSelectionProvider, null);
        table.setPortListMenuListener(portListMenuListener);

        // force repaint of slider renderers after resize panel
        addComponentListener(new ComponentAdapter() {

            @Override
            public void componentResized(ComponentEvent e) {
                LOGGER.trace("Component is resized: {}", e.getComponent());
                table.repaint();
            }
        });
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof BacklightPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    protected void processListChanged() {
        super.processListChanged();

        tabVisibilityListener.setTabVisible(this, isTabVisible());

        final NodeInterface node = mainModel.getSelectedNode();
        portTableLayoutChanged(node);
    }

    @Override
    public void portTableLayoutChanged(final NodeInterface node) {

        // check if the DMX channel column is visible to change the column label
        if (TableColumnChooser.isVisibleColumn(table, BacklightPortTableModel.COLUMN_DMX_MAPPING)) {
            LOGGER.info("The DMX column is visible for node: {}", node);
            if (node != null) {
                boolean newNodeIsDmxNode = ProductUtils.isOneDMX(node.getUniqueId());

                if (newNodeIsDmxNode != isDmxNode.get()) {
                    LOGGER.info("The DMX support of node has changed, newNodeIsDmxNode: {}", newNodeIsDmxNode);

                    this.isDmxNode.set(newNodeIsDmxNode);

                    TableColumn tcDmxMapping =
                        table.getColumnModel().getColumn(BacklightPortTableModel.COLUMN_DMX_MAPPING);

                    if (newNodeIsDmxNode) {
                        // show the DMX channel renderer and editor
                        tcDmxMapping.setCellRenderer(new DmxChannelRenderer());
                        tcDmxMapping.setCellEditor(new DmxChannelCellEditor(1, 0x100));

                        tcDmxMapping.setHeaderValue(Resources.getString(BacklightPortTableModel.class, "dmxMapping"));
                    }
                    else {
                        // show the channel renderer and editor
                        tcDmxMapping.setCellRenderer(new PortAwareNumberRenderer());
                        tcDmxMapping.setCellEditor(new NumberRangeEditor(0, 0xFF));

                        tcDmxMapping.setHeaderValue(Resources.getString(BacklightPortTableModel.class, "channel"));
                    }

                    // repaint the table header to make the new label appear
                    JTableHeader th = table.getTableHeader();
                    th.repaint();
                }
                else {
                    LOGGER.info("The DMX column is visible but the isDmxNode property has not changed: {}", isDmxNode);
                }
            }
        }
        else {
            LOGGER.info("The DMX column is not visible.");
        }

        final List<BacklightPort> ports = new ArrayList<>();
        ports.addAll(getPorts());

        if (CollectionUtils.isNotEmpty(ports)) {
            // get the max dimm value
            try {
                BacklightPort backlightPort = ports.get(0);
                if (backlightPort != null) {
                    int dimmStretchMin = backlightPort.getDimStretchMin();
                    int dimmStretchMax = backlightPort.getDimStretchMax();
                    LOGGER
                        .info("Set the values for dimmStretchMin: {}, dimmStretchMax: {}", dimmStretchMin,
                            dimmStretchMax);

                    final ConfigXAwareSliderRenderer sliderRendererDimmMin =
                        (ConfigXAwareSliderRenderer) table
                            .getColumnModel().getColumn(BacklightPortTableModel.COLUMN_DIM_SLOPE_DOWN)
                            .getCellRenderer();
                    final ConfigXAwareSliderRenderer sliderRendererDimmMax =
                        (ConfigXAwareSliderRenderer) table
                            .getColumnModel().getColumn(BacklightPortTableModel.COLUMN_DIM_SLOPE_UP).getCellRenderer();
                    sliderRendererDimmMin.setMaxValue(dimmStretchMin);
                    sliderRendererDimmMax.setMaxValue(dimmStretchMax);
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Get the dimm stretch values failed.", ex);
            }
        }
        // tabVisibilityListener.setTabVisible(this, isTabVisible());
    }

    @Override
    public Class<?> getPortClass() {
        return BacklightPort.class;
    }

    @Override
    public boolean isTabVisible() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasBacklightPorts();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    protected List<BacklightPort> getPorts() {
        final NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            return node.getBacklightPorts();
        }
        return Collections.emptyList();
    }

    public void refreshView() {
        listChanged();
    }
}
