package org.bidib.wizard.mvc.preferences.controller;

import java.beans.PropertyChangeListener;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.UIManager;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.InvalidLibraryException;
import org.bidib.jbidibc.netbidib.pairingstore.PairingStore;
import org.bidib.jbidibc.rxtx.PortIdentifierUtils;
import org.bidib.jbidibc.scm.ScmPortIdentifierUtils;
import org.bidib.wizard.api.LookupService;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.CommPort;
import org.bidib.wizard.api.model.common.NetBidibServiceInfo;
import org.bidib.wizard.api.model.event.ConsoleMessageEvent;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.client.common.preferences.view.panel.SettingsPanelInterface;
import org.bidib.wizard.common.model.settings.ExperimentalSettingsInterface;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.common.model.settings.Misc2SettingsInterface;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.common.model.settings.NetBidibSettingsInterface;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.model.settings.types.LookAndFeel;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.model.settings.ExperimentalSettings;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.core.model.settings.Misc2Settings;
import org.bidib.wizard.core.model.settings.MiscSettings;
import org.bidib.wizard.core.model.settings.NetBidibSettings;
import org.bidib.wizard.core.model.settings.WizardSettings;
import org.bidib.wizard.core.utils.AopUtils;
import org.bidib.wizard.mvc.preferences.model.PreferencesModel;
import org.bidib.wizard.mvc.preferences.view.PreferencesView;
import org.bidib.wizard.mvc.preferences.view.listener.PreferencesViewListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.togglz.core.manager.FeatureManager;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.formdev.flatlaf.FlatDarkLaf;
import com.formdev.flatlaf.FlatLaf;
import com.formdev.flatlaf.FlatLightLaf;
import com.formdev.flatlaf.extras.FlatAnimatedLafChange;

public class PreferencesController {
    private static final Logger LOGGER = LoggerFactory.getLogger(PreferencesController.class);

    private final JFrame parent;

    private PreferencesModel model;

    @Autowired
    private GlobalSettingsInterface globalSettings;

    @Autowired
    private WizardSettingsInterface wizardSettings;

    @Autowired
    private MiscSettingsInterface miscSettings;

    @Autowired
    private Misc2SettingsInterface misc2Settings;

    @Autowired
    private ExperimentalSettingsInterface experimentalSettings;

    @Autowired
    private NetBidibSettingsInterface netBidibSettings;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private PairingStore pairingStore;

    @Autowired
    private ObjectMapper serializingObjectMapper;

    @Autowired
    private LookupService lookupService;

    @Autowired
    private FeatureManager featureManager;

    @Autowired
    private ApplicationEventPublisher errorEventPublisher;

    private final List<SettingsPanelInterface> settingsPanelInterfaces;

    public PreferencesController(JFrame parent, final List<SettingsPanelInterface> settingsPanelInterfaces) {
        this.parent = parent;
        this.settingsPanelInterfaces = settingsPanelInterfaces;
    }

    private void loadCommPorts() {
        if (globalSettings.isSerialPortsEnabled()) {
            LOGGER.info("Load the comm ports, model: {}", model);
            final Set<CommPort> commPorts = new HashSet<>();

            try {
                final Set<CommPort> detectedComPorts = lookupService.getDetectedComPorts();

                // use PortIdentifierUtils because we must load the RXTX libraries
                List<String> portIdentifiers = null;

                switch (miscSettings.getSelectedSerialPortProvider()) {
                    case "SCM":
                        portIdentifiers = ScmPortIdentifierUtils.getPortIdentifiers();
                        break;

                    case "SPSW":
                        portIdentifiers = org.bidib.jbidibc.purejavacomm.PortIdentifierUtils.getPortIdentifiers();
                        break;
                    case "JSerialComm":
                        portIdentifiers = org.bidib.jbidibc.jserialcomm.PortIdentifierUtils.getPortIdentifiers();
                        break;
                    case "PureJavaComm":
                        portIdentifiers = org.bidib.jbidibc.purejavacomm.PortIdentifierUtils.getPortIdentifiers();
                        break;
                    case "FTDI":
                        portIdentifiers = org.bidib.jbidibc.ftdi.serial.PortIdentifierUtils.getPortIdentifiers();
                        break;

                    default:
                        portIdentifiers = PortIdentifierUtils.getPortIdentifiers();
                        break;
                }

                if (portIdentifiers != null) {
                    for (String id : portIdentifiers) {
                        LOGGER.info("Add new CommPort with id: {}", id);

                        // lookup if more information is in the detectedComPorts of the preferences
                        CommPort detectedComPort = null;
                        for (CommPort commPort : detectedComPorts) {
                            if (id.equals(commPort.getName())) {
                                detectedComPort = commPort;
                                LOGGER.info("Found detected comm port: {}", detectedComPort);
                                break;
                            }
                        }

                        if (detectedComPort != null) {
                            commPorts.add(detectedComPort);
                        }
                        else {
                            commPorts.add(new CommPort(id));
                        }
                    }
                }

                if (StringUtils.isNotBlank(globalSettings.getPreviousSelectedComPort())) {
                    LOGGER.info("Add the previous selected COM Port: {}", globalSettings.getPreviousSelectedComPort());
                    commPorts.add(new CommPort(globalSettings.getPreviousSelectedComPort().trim()));
                }
            }
            catch (InvalidLibraryException ex) {
                LOGGER
                    .warn(
                        "Fetch port identifiers failed. This can be caused because the ext/lib directory of the Java installation contains an old RXTXComm.jar!",
                        ex);

                JOptionPane
                    .showMessageDialog(parent,
                        Resources
                            .getString(PreferencesController.class, "fetch-port-identifiers-failed",
                                new Object[] { new File(SystemUtils.getJavaHome(), "lib/ext").getPath() }),
                        Resources.getString(PreferencesController.class, "title-error"), JOptionPane.ERROR_MESSAGE);
            }
            catch (Exception ex) {
                LOGGER.warn("Fetch port identifiers failed.", ex);
            }
            model.setAvailableCommPorts(commPorts);
        }
        else {
            LOGGER.info("No comm ports loaded because serial enabled flag is not set.");
        }
    }

    private void loadNetBidibServices() {

        try {
            final Set<NetBidibServiceInfo> detectedNetBidibServices = lookupService.getDetectedNetBidibServices();

            model.setDetectedNetBidibServices(detectedNetBidibServices);
        }
        catch (Exception ex) {
            LOGGER.warn("Fetch netBidib services failed.", ex);

            model.setDetectedNetBidibServices(Collections.emptySet());
        }
    }

    public void start() {
        LOGGER.debug("Start the preferences controller.");

        // The preferences model should provide a copy of the settings and save the changed values after save is
        // pressed

        try {
            final GlobalSettings globalSettings =
                serializingObjectMapper
                    .treeToValue(serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.globalSettings)),
                        GlobalSettings.class);

            final WizardSettingsInterface wizardSettings =
                serializingObjectMapper
                    .treeToValue(serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.wizardSettings)),
                        WizardSettings.class);

            final MiscSettingsInterface miscSettings =
                serializingObjectMapper
                    .treeToValue(serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.miscSettings)),
                        MiscSettings.class);

            final Misc2SettingsInterface misc2Settings =
                serializingObjectMapper
                    .treeToValue(serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.misc2Settings)),
                        Misc2Settings.class);

            final ExperimentalSettingsInterface experimentalSettings =
                serializingObjectMapper
                    .treeToValue(
                        serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.experimentalSettings)),
                        ExperimentalSettings.class);

            final NetBidibSettingsInterface netBidibSettings =
                serializingObjectMapper
                    .treeToValue(serializingObjectMapper.valueToTree(AopUtils.getTargetObject(this.netBidibSettings)),
                        NetBidibSettings.class);

            model =
                new PreferencesModel(globalSettings, wizardSettings, miscSettings, misc2Settings, experimentalSettings,
                    netBidibSettings, pairingStore);

            if (globalSettings.isSerialPortsEnabled()) {
                LOGGER.info("Load the comm ports of the system.");
                loadCommPorts();
            }

            if (globalSettings.isNetBidibEnabled()) {
                LOGGER.info("Load the detected netBidib services of the system.");
                loadNetBidibServices();
            }

            final PreferencesView view =
                new PreferencesView(parent, model, this.settingsPanelInterfaces, this.featureManager);

            final PropertyChangeListener pcl = evt -> {
                LOGGER.info("The look and feel has been changed. New value: {}", evt.getNewValue());

                LookAndFeel lookAndFeel = experimentalSettings.getLookAndFeel();

                FlatAnimatedLafChange.showSnapshot();

                try {
                    UIManager
                        .setLookAndFeel(lookAndFeel == LookAndFeel.FlatLight ? new FlatLightLaf() : new FlatDarkLaf());
                }
                catch (Exception ex) {
                    LOGGER.warn("Set L&F failed.", ex);
                }

                // update all components
                FlatLaf.updateUI();

                FlatAnimatedLafChange.hideSnapshotWithAnimation();
            };

            try {
                this.experimentalSettings.addPropertyChangeListener(ExperimentalSettings.PROPERTY_LOOK_AND_FEEL, pcl);

                view.addPreferencesViewListener(new PreferencesViewListener() {

                    @Override
                    public void save() {
                        LOGGER.info("Save the preferences values.");

                        // save the mode in the preferences

                        try {
                            BeanUtils.copyProperties(PreferencesController.this.globalSettings, globalSettings);

                            BeanUtils.copyProperties(PreferencesController.this.wizardSettings, wizardSettings);
                            BeanUtils.copyProperties(PreferencesController.this.miscSettings, miscSettings);
                            BeanUtils.copyProperties(PreferencesController.this.misc2Settings, misc2Settings);
                            BeanUtils
                                .copyProperties(PreferencesController.this.experimentalSettings, experimentalSettings);
                            BeanUtils.copyProperties(PreferencesController.this.netBidibSettings, netBidibSettings);
                        }
                        catch (IllegalAccessException | InvocationTargetException ex) {
                            LOGGER.warn("Copy properties to to settings objects failed.", ex);
                        }

                        settingsService.storeSettings();

                        // TODO store the pairing store
                        try {
                            LOGGER.info("Store the pairing store content.");

                            final File currentPairingStoreLocation =
                                new File(PreferencesController.this.netBidibSettings.getPairingStoreLocation());
                            File pairingStorePath =
                                PreferencesController.this.pairingStore.getPairingStore().getParentFile();

                            LOGGER
                                .debug(
                                    "Check if location of paring store has changed, current pairingStore location: {}, path of pairingStore: {}",
                                    currentPairingStoreLocation, pairingStorePath);

                            if (currentPairingStoreLocation.equals(pairingStorePath)) {

                                PreferencesController.this.pairingStore.store();
                            }
                            else {

                                PreferencesController.this
                                    .handleError(new InvalidConfigurationException(
                                        "Restart Wizard because location of pairing store has changed!"));
                            }
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Store pairing store content failed.", ex);
                        }
                    }

                    @Override
                    public void cancel() {
                    }
                });

                view.showView();

            }
            finally {
                LOGGER.info("The preferences view was closed.");

                PreferencesController.this.experimentalSettings
                    .removePropertyChangeListener(ExperimentalSettings.PROPERTY_LOOK_AND_FEEL, pcl);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Prepare and open preferences view failed.", ex);
        }
    }

    private void handleError(RuntimeException ex) {
        LOGGER.warn("Handle error from underlying components: {}", ex.getMessage());

        final ConsoleMessageEvent errorEvent;
        if (ex instanceof InvalidConfigurationException) {
            errorEvent = new ConsoleMessageEvent(ConsoleColor.red, ex.getMessage());
        }
        else {
            errorEvent = new ConsoleMessageEvent(ConsoleColor.red, ex.getMessage());
        }

        errorEventPublisher.publishEvent(errorEvent);
    }
}
