package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Frame;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.LabelFactory;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.enums.DetachedState;
import org.bidib.jbidibc.messages.enums.IdentifyState;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.LabelAware;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.dialog.EscapeDialog;
import org.bidib.wizard.client.common.event.MainControllerEvent;
import org.bidib.wizard.client.common.nodetree.JideNodeTree;
import org.bidib.wizard.client.common.nodetree.JideNodeTreeTableModel;
import org.bidib.wizard.client.common.uils.SwingUtils;
import org.bidib.wizard.client.common.view.BusyFrame;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.menu.listener.NodeListMenuListener;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.service.SettingsService;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.dialog.NodeLabelView;
import org.bidib.wizard.mvc.common.exception.NodeSelectionChangeException;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.menu.NodeListMenu;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelAction;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelActionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelChangedListener;
import org.bidib.wizard.mvc.main.view.panel.listener.NodeListActionListener;
import org.bidib.wizard.mvc.worklist.controller.WorkListItemProvider;
import org.bidib.wizard.mvc.worklist.model.WorkItemModel;
import org.bidib.wizard.nodescript.client.view.listener.NodeTreeScriptingListener;
import org.bushe.swing.event.EventBus;
import org.bushe.swing.event.annotation.AnnotationProcessor;
import org.bushe.swing.event.annotation.EventSubscriber;
import org.oxbow.swingbits.dialog.task.TaskDialogs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;

import com.jgoodies.common.base.Objects;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.SearchableUtils;
import com.vlsolutions.swing.docking.DockKey;
import com.vlsolutions.swing.docking.Dockable;

public class NodeListPanel implements NodeListMenuListener, Dockable, NodeTreeScriptingListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(NodeListPanel.class);

    private final List<NodeListActionListener> nodeListListeners = new LinkedList<NodeListActionListener>();

    private final List<LabelChangedListener<NodeInterface>> labelChangedListeners = new LinkedList<>();

    private final JideNodeTree nodeTree;

    private LabelAction<NodeInterface> labelAction;

    private final NodeListMenu nodeListMenu;

    private final JPopupMenu popupMenu;

    private JPanel contentPanel;

    private final MainModel mainModel;

    private final SettingsService settingsService;

    private final WizardLabelWrapper wizardLabelWrapper;

    private final ApplicationEventPublisher applicationEventPublisher;

    private final WorkListItemProvider workListItemProvider;

    private final StatusBar statusBar;

    public NodeListPanel(final MainModel mainModel, final SettingsService settingsService,
        final WizardLabelWrapper wizardLabelWrapper, final ApplicationEventPublisher applicationEventPublisher,
        final WorkListItemProvider workListItemProvider, final StatusBar statusBar) {

        this.mainModel = mainModel;
        this.settingsService = settingsService;
        this.wizardLabelWrapper = wizardLabelWrapper;
        this.applicationEventPublisher = applicationEventPublisher;
        this.workListItemProvider = workListItemProvider;
        this.statusBar = statusBar;

        final JideNodeTreeTableModel jideNodeTreeTableModel = new JideNodeTreeTableModel();

        this.nodeTree = new JideNodeTree(jideNodeTreeTableModel, settingsService, wizardLabelWrapper);

        this.labelAction = new LabelAction<NodeInterface>() {
            @Override
            public void editLabel(
                NodeInterface node, Point location, LabelActionListener<NodeInterface> actionListener) {

                Frame frame = JOptionPane.getFrameForComponent(null);

                final NodeLabels nodeLabels = wizardLabelWrapper.loadLabels(node.getUniqueId());
                NodeLabelView view = new NodeLabelView(frame, node, nodeLabels) {
                    private static final long serialVersionUID = 1L;

                    @Override
                    public void labelChanged(String value) {
                        LOGGER.info("The label of the node was changed to value: '{}'", value);
                        if (node instanceof LabelAware) {
                            ((LabelAware) node).setLabel(value);
                        }
                        actionListener.fireLabelChanged(node, value);
                    }
                };
                if (location.x > -1 && location.y > -1) {
                    view.setLocation(location.x + 20, location.y + 20);
                }
                else {
                    view.setLocationRelativeTo(null);
                }

                view.setVisible(true);
            }
        };

        // register the nodeTree in the context
        LOGGER.info("Register the nodeTree in applicationContext.");
        DefaultApplicationContext.getInstance().register(DefaultApplicationContext.KEY_NODE_TREE, this.nodeTree);

        nodeListMenu = new NodeListMenu(mainModel, settingsService, wizardLabelWrapper);
        popupMenu = nodeListMenu.getPopupMenu();

        DockKeys.DOCKKEY_NODE_LIST_PANEL.setName(Resources.getString(getClass(), "title"));
        // turn off autohide and close features
        DockKeys.DOCKKEY_NODE_LIST_PANEL.setCloseEnabled(false);
        DockKeys.DOCKKEY_NODE_LIST_PANEL.setAutoHideEnabled(false);

        this.nodeTree.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                // get the selected node
                NodeInterface node = NodeListPanel.this.nodeTree.selectElement(e.getPoint());
                LOGGER.info("Node was selected: {}", node);

                if (node != null) {

                    if (e.getClickCount() == 1 && e.isPopupTrigger()) {
                        // update the menu items based on the features of the node
                        nodeListMenu.setAddressMessagesEnabled(node.isAddressMessagesEnabled());
                        nodeListMenu.setDccStartEnabled(node.isDccStartEnabled());
                        nodeListMenu.setExternalStartEnabled(node.isExternalStartEnabled());
                        nodeListMenu.setFeedbackMessagesEnabled(node.isFeedbackMessagesEnabled());
                        nodeListMenu.setKeyMessagesEnabled(node.isKeyMessagesEnabled());
                        nodeListMenu.setClearErrorEnabled(node.isNodeHasError());
                        handleMouseEvent(e, popupMenu);
                    }
                    else if (e.getClickCount() == 1 && e.isControlDown()) {

                        e.consume();

                        node
                            .setIdentifyState(
                                node.getIdentifyState() == IdentifyState.ON ? IdentifyState.OFF : IdentifyState.ON);

                        LOGGER
                            .info("Clicked on node label and control is pressed. Set the identify state: {}",
                                node.getIdentifyState());

                        for (NodeListActionListener l : nodeListListeners) {
                            l.identify(node);
                        }

                    }
                    else if (e.getClickCount() == 2) {
                        // select the node
                        LOGGER.info("Select the node: {}", node);

                        e.consume();

                        try {
                            setSelectedNode(node);
                        }
                        catch (NodeSelectionChangeException ex) {
                            LOGGER.warn("The change of node selection was denied.", ex);
                        }
                    }
                }
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                LOGGER.debug("Mouse released.");
                if (e.isPopupTrigger()) {
                    mousePressed(e);
                }
            }
        });

        this.nodeTree.setItems(new NodeInterface[0]);
        // this.nodeTree.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void listChanged() {
                LOGGER.debug("List changed, use the AWT-thread to set the nodes in the label list.");
                if (SwingUtilities.isEventDispatchThread()) {
                    internalListChanged(mainModel);
                }
                else {
                    SwingUtilities.invokeLater(() -> internalListChanged(mainModel));
                }
            }

            @Override
            public void nodeStateChanged(final NodeInterface node) {
                // LOGGER.debug("Node state changed, use the AWT-thread to refresh the labellist.");
                // if (SwingUtilities.isEventDispatchThread()) {
                // NodeListPanel.this.nodeTree.refreshView();
                // }
                // else {
                // SwingUtilities.invokeLater(() -> NodeListPanel.this.nodeTree.refreshView());
                // }
            }

            @Override
            public void listNodeRemoved(final NodeInterface node) {

                if (Objects.equals(getSelectedItem(), node)) {
                    LOGGER.warn("The selected node is removed. Set the selected node to null.");
                    SwingUtils.executeInEDT(() -> setSelectedNode(null));
                }
            }
        });
        nodeListMenu.addMenuListener(this);

        // add quick search field
        SearchableUtils.installSearchable(this.nodeTree);

        // add the eventbus processing
        AnnotationProcessor.process(this);
    }

    private JPanel createPanel() {
        LOGGER.info("Create the content panel.");

        final JPanel contentPanel = new JPanel(new BorderLayout());

        final JScrollPane scrollPane = new JScrollPane(this.nodeTree);

        contentPanel.add(scrollPane);

        return contentPanel;
    }

    @Override
    public Component getComponent() {
        if (contentPanel == null) {
            contentPanel = createPanel();
        }
        return contentPanel;
    }

    @Override
    public DockKey getDockKey() {
        return DockKeys.DOCKKEY_NODE_LIST_PANEL;
    }

    @EventSubscriber(eventClass = LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        // this.nodeTree.refreshView();
    }

    private void internalListChanged(final MainModel model) {
        NodeInterface selectedValue = this.nodeTree.getSelectedItem();
        LOGGER.info("The list has changed, current selected value: {}", selectedValue);

        // set the current items in the label list
        NodeInterface[] nodes = new NodeInterface[0];
        if (model.getNodeProvider() != null) {
            nodes = model.getNodeProvider().getNodes().toArray(new NodeInterface[0]);
        }
        this.nodeTree.setItems(nodes);
        LOGGER.debug("The list has changed, set the new nodes has finished.");
    }

    @Override
    public void addressMessagesEnabled(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setAddressMessagesEnabled(isSelected);

            // TODO change this that the node fires a propertyChangeEvent and remove this listener
            for (NodeListActionListener l : nodeListListeners) {
                l.enableAddressMessages(node);
            }
        }
    }

    @Override
    public void displayNodeDetails() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            try {
                setSelectedNode(node);
            }
            catch (NodeSelectionChangeException ex) {
                LOGGER.warn("The change of node selection was denied.", ex);
            }
        }
    }

    @Override
    public void editLabel(final MouseEvent popupEvent) {
        NodeInterface selectedItem = getSelectedItem();
        if (selectedItem != null) {

            labelAction
                .editLabel(selectedItem, selectedIndexToLocation(),
                    (NodeInterface object, String label) -> NodeListPanel.this.fireLabelChanged(object, label));
        }
    }

    protected void handleMouseEvent(MouseEvent e, JPopupMenu popupMenu) {
        if (e.isPopupTrigger() && this.nodeTree.getItemSize() > 0) {
            this.nodeTree.selectElement(e.getPoint());
            popupMenu.show(e.getComponent(), e.getX(), e.getY());
        }
    }

    protected Point selectedIndexToLocation() {
        int selectedIndex = this.nodeTree.getSelectedIndex();
        return selectedIndexToLocation(selectedIndex);
    }

    private Point selectedIndexToLocation(int selectedIndex) {
        Point result = new Point();

        if (selectedIndex >= 0) {
            final Point listPosition = this.nodeTree.getLocationOnScreen();
            final Point itemPosition = this.nodeTree.indexToLocation(selectedIndex);
            result.setLocation(listPosition.x + itemPosition.x, listPosition.y + itemPosition.y);
        }
        return result;
    }

    public void addListSelectionListener(ListSelectionListener l) {
        this.nodeTree.addListSelectionListener(l);
    }

    public void addNodeListListener(NodeListActionListener l) {
        addLabelChangedListener(l);
        nodeListListeners.add(l);
    }

    public void addLabelChangedListener(LabelChangedListener<NodeInterface> l) {
        labelChangedListeners.add(l);
    }

    @Override
    public void dccStartEnabled(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setDccStartEnabled(isSelected);
            for (NodeListActionListener l : nodeListListeners) {
                l.enableDccStart(node);
            }
        }
    }

    @Override
    public void importNode() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.importNode(node);
            }
        }
    }

    @Override
    public void exportNode() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.exportNode(node);
            }
        }
    }

    @Override
    public void externalStartEnabled(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setExternalStartEnabled(isSelected);
            for (NodeListActionListener l : nodeListListeners) {
                l.enableExternalStart(node);
            }
        }
    }

    @Override
    public void feedbackMessagesEnabled(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setFeedbackMessagesEnabled(isSelected);
            for (NodeListActionListener l : nodeListListeners) {
                l.enableFeedbackMessages(node);
            }
        }
    }

    @Override
    public void feedbackMirrorDisabled(boolean disable) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setFeedbackMirrorDisabled(disable);

            // this will set the secureAckEnabled flag on BidibNode to false
            for (NodeListActionListener l : nodeListListeners) {
                l.disableFeedbackMirror(node, disable);
            }
        }
    }

    @Override
    public void firmwareUpdate() {

        for (NodeListActionListener l : nodeListListeners) {
            l.firmwareUpdate(this.nodeTree.getSelectedItem());
        }
    }

    @Override
    public void identify(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setIdentifyState(isSelected ? IdentifyState.ON : IdentifyState.OFF);
            for (NodeListActionListener l : nodeListListeners) {
                l.identify(node);
            }
        }
    }

    @Override
    public void dccAdvView() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.dccAdvView(node);
            }
        }
    }

    @Override
    public void keyMessagesEnabled(boolean isSelected) {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            node.setInputMessagesEnabled(isSelected);
            for (NodeListActionListener l : nodeListListeners) {
                l.enableKeyMessages(node);
            }
        }
    }

    @Override
    public void features() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Open the features dialog on node: {}", node);
        if (node != null) {
            Point itemPosition = selectedIndexToLocation();

            for (NodeListActionListener l : nodeListListeners) {
                l.features(node);
            }
        }
    }

    @Override
    public void reset() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Open the reset dialog on node: {}", node);
        if (node != null) {
            // show the reset dialog for the selected node
            Frame parent = JOptionPane.getFrameForComponent(this.nodeTree);
            new ResetDialog(parent, node, true);
        }
    }

    @Override
    public void reloadNode() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Open the reload dialog on node: {}", node);
        if (node != null) {
            // show the reload dialog for the selected node
            Frame parent = JOptionPane.getFrameForComponent(this.nodeTree);
            new ReloadNodeDialog(parent, node, true);
        }
    }

    @Override
    public void ping() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.info("Send ping to node: {}", node);

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.ping(node, new byte[] { (byte) 0x01 });
            }
        }
    }

    @Override
    public void enable() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.info("Send enable to node: {}", node);

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.enableNode(node);
            }
        }
    }

    @Override
    public void disable() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.info("Send disable to node: {}", node);

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.disableNode(node);
            }
        }
    }

    @Override
    public void readUniqueId() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.info("Read the uniqueId from the node: {}", node);
        long uniqueId = 0;

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                uniqueId = l.readUniqueId(node);
            }
        }

        StringBuilder message = new StringBuilder(Resources.getString(getClass(), "current-uniqueId"));
        message.append(ByteUtils.getUniqueIdAsString(uniqueId));
        Frame parent = JOptionPane.getFrameForComponent(this.nodeTree);
        JOptionPane.showMessageDialog(parent, message);
    }

    @Override
    public void detachAttachNode() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.info("Logoff from the node: {}", node);

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {

                // if the node is detached we have to attach it
                boolean detached = node.getDetachedState() == DetachedState.DETACHED;
                LOGGER.info("Current detached state: {}", detached);
                l.detachAttachNode(node, !detached);
            }
        }
    }

    @Override
    public void dmxModeler() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Open the dmxModeler on node: {}", node);

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.dmxModeler(node);
            }
        }
    }

    protected void fireLabelChanged(final NodeInterface node, String label) {
        for (LabelChangedListener<NodeInterface> l : labelChangedListeners) {
            l.labelChanged(node, label);
        }

        // this.nodeTree.refreshView();
    }

    private final class ResetDialog extends EscapeDialog {
        private static final long serialVersionUID = 1L;

        public ResetDialog(final Frame parent, final NodeInterface node, boolean modal) {
            super(parent, Resources.getString(ResetDialog.class, "title"), modal);
            getContentPane().setLayout(new BorderLayout());

            JPanel mainPanel = new JPanel(new BorderLayout());

            mainPanel
                .add(new JLabel(Resources.getString(getClass(), "message"), UIManager.getIcon("OptionPane.warningIcon"),
                    SwingConstants.LEADING), BorderLayout.CENTER);
            mainPanel.setBorder(new EmptyBorder(5, 5, 5, 5));

            // buttons
            JButton reset = new JButton(Resources.getString(getClass(), "reset"));

            reset.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setVisible(false);
                    fireReset(node);
                }
            });

            JButton cancel = new JButton(Resources.getString(getClass(), "cancel"));

            cancel.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setVisible(false);
                    fireCancel();
                }
            });

            JPanel buttons =
                new ButtonBarBuilder().addGlue().addButton(reset, cancel).border(Paddings.BUTTON_BAR_PAD).build();

            mainPanel.add(buttons, BorderLayout.SOUTH);

            getContentPane().add(mainPanel);
            pack();

            setLocationRelativeTo(parent);
            setMinimumSize(getSize());
            setVisible(true);
        }

        private void fireReset(NodeInterface node) {
            LOGGER.info("Reset the current node: {}", node);

            for (NodeListActionListener l : nodeListListeners) {
                l.reset(node);
            }
        }

        private void fireCancel() {

        }
    }

    private final class ReloadNodeDialog extends EscapeDialog {
        private static final long serialVersionUID = 1L;

        public ReloadNodeDialog(final Frame parent, final NodeInterface node, boolean modal) {
            super(parent, Resources.getString(ReloadNodeDialog.class, "title"), modal);
            getContentPane().setLayout(new BorderLayout());

            JPanel mainPanel = new JPanel(new BorderLayout());

            mainPanel
                .add(new JLabel(Resources.getString(ReloadNodeDialog.class, "message"),
                    UIManager.getIcon("OptionPane.warningIcon"), SwingConstants.LEADING), BorderLayout.CENTER);
            mainPanel.setBorder(new EmptyBorder(5, 5, 5, 5));

            // buttons
            JButton reloadNode = new JButton(Resources.getString(ReloadNodeDialog.class, "reload"));

            reloadNode.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setVisible(false);
                    fireReload(node);
                }
            });

            JButton cancel = new JButton(Resources.getString(getClass(), "cancel"));

            cancel.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setVisible(false);
                    fireCancel();
                }
            });

            JPanel buttons =
                new ButtonBarBuilder().addGlue().addButton(reloadNode, cancel).border(Paddings.BUTTON_BAR_PAD).build();

            mainPanel.add(buttons, BorderLayout.SOUTH);

            getContentPane().add(mainPanel);
            pack();

            setLocationRelativeTo(parent);
            setMinimumSize(getSize());
            setVisible(true);
        }

        private void fireReload(NodeInterface node) {
            LOGGER.info("Reload the current node: {}", node);

            for (NodeListActionListener l : nodeListListeners) {
                l.reloadNode(node);
            }
        }

        private void fireCancel() {

        }
    }

    @Override
    public void loco() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null && org.bidib.jbidibc.messages.utils.NodeUtils.hasCommandStationFunctions(node.getUniqueId())) {
            LOGGER.info("Open the loco dialog.");
            for (NodeListActionListener l : nodeListListeners) {
                l.loco(node);
            }
        }
    }

    @Override
    public void locoList() {
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null && org.bidib.jbidibc.messages.utils.NodeUtils.hasCommandStationFunctions(node.getUniqueId())) {
            LOGGER.info("Open the loco table.");
            for (NodeListActionListener l : nodeListListeners) {
                l.locoList(node);
            }
        }
    }

    @Override
    public void dccAccessory() {
        Point itemPosition = selectedIndexToLocation();
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.dccAccessory(node, itemPosition.x, itemPosition.y);
            }
        }
    }

    @Override
    public void locoCv() {
        Point itemPosition = selectedIndexToLocation();
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.locoCv(node, itemPosition.x, itemPosition.y);
            }
        }
    }

    @Override
    public void locoCvPt() {
        Point itemPosition = selectedIndexToLocation();
        NodeInterface node = this.nodeTree.getSelectedItem();

        if (node != null) {
            for (NodeListActionListener l : nodeListListeners) {
                l.locoCvPt(node, itemPosition.x, itemPosition.y);
            }
        }
    }

    @Override
    public void showDetails() {
        Point itemPosition = selectedIndexToLocation();

        for (NodeListActionListener l : nodeListListeners) {
            l.nodeDetails(this.nodeTree.getSelectedItem(), itemPosition.x, itemPosition.y);
        }
    }

    @Override
    public void bulkSwitchNode() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Open the bulk switch node dialog on node: {}", node);
        if (node != null) {
            Point itemPosition = selectedIndexToLocation();
            for (NodeListActionListener l : nodeListListeners) {
                l.bulkSwitchDialog(node, itemPosition.x, itemPosition.y);
            }
        }
    }

    @Override
    public void clearErrors() {
        NodeInterface node = this.nodeTree.getSelectedItem();
        LOGGER.debug("Clear errors on node: {}", node);
        if (node != null) {
            node.setNodeHasError(false);
        }
    }

    @Override
    public void savePendingChanges() {
        LOGGER.info("Save the pending changes.");

        this.applicationEventPublisher
            .publishEvent(new MainControllerEvent(MainControllerEvent.Action.savePendingChanges));
    }

    @Override
    public void setSelectedNode(final NodeInterface node) {
        LOGGER.info("Set the selected node: {}", node);

        this.nodeTree.setSelectedItem(node);
    }

    @Override
    public NodeInterface getSelectedItem() {
        return this.nodeTree.getSelectedItem();
    }

    @Override
    public void generateDocumentation() {

        LOGGER.info("Generate documentation.");
        NodeInterface node = getSelectedItem();
        if (node != null) {

            for (NodeListActionListener l : nodeListListeners) {
                l.generateDocumentation(node);
            }
        }
    }

    private static final String BIDIB_DEFAULT_NAMES_EXTENSION = "xml";

    private static final String WORKING_DIR_DEFAULT_LABELS_KEY = "defaultLabels";

    @Override
    public void generateDefaultLabels() {

        LOGGER.info("Generate default labels.");
        try {
            NodeInterface node = getSelectedItem();
            if (node != null) {

                long uniqueId = node.getUniqueId();

                final NodeLabels nodeLabels = this.wizardLabelWrapper.prepareDefaultLabels(uniqueId);
                if (nodeLabels != null) {
                    // show dialog
                    String lang = XmlLocaleUtils.getXmlLocaleVendorCV();
                    final String shortLang = StringUtils.substringBefore(lang, "-");

                    // "bidib-default-names-13-138-de.xml"
                    StringBuilder sb = new StringBuilder("bidib-default-names-");
                    sb.append(org.bidib.jbidibc.messages.utils.NodeUtils.getVendorId(uniqueId)).append("-");
                    sb
                        .append(org.bidib.jbidibc.messages.utils.NodeUtils
                            .getPid(uniqueId, node.getNode().getRelevantPidBits()))
                        .append("-").append(shortLang).append(".xml");

                    String defaultFileName = sb.toString();
                    LOGGER.info("Prepared default filename: {}", defaultFileName);

                    final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
                    String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_DEFAULT_LABELS_KEY);

                    String bidibNodeFilesDescription =
                        Resources.getString(NodeListPanel.class, "bidibDefaultLabelFilesDescription");
                    final FileFilter bidibNodesFilter =
                        new FileNameExtensionFilter(bidibNodeFilesDescription, BIDIB_DEFAULT_NAMES_EXTENSION);

                    final FileDialog dialog =
                        new FileDialog(this.nodeTree, FileDialog.SAVE, storedWorkingDirectory, defaultFileName,
                            bidibNodesFilter) {
                            @Override
                            public void approve(final String selectedFile) {
                                File file = new File(selectedFile);

                                String fileName = file.getPath();
                                fileName = makeSureFileNameHasExtension(fileName, BIDIB_DEFAULT_NAMES_EXTENSION);

                                LOGGER.info("Save default labels to file: {}", fileName);

                                BusyFrame busyFrame =
                                    DefaultApplicationContext
                                        .getInstance().get(DefaultApplicationContext.KEY_MAIN_FRAME, BusyFrame.class);

                                try {
                                    busyFrame.setBusy(true);

                                    // prepare the instance of labelFactory
                                    LabelFactory labelFactory = new LabelFactory();
                                    labelFactory.saveNodeLabel(nodeLabels, file, false);

                                    NodeListPanel.this.statusBar
                                        .setStatusText(Resources
                                            .getString(NodeListPanel.class, "generate-default-labels-passed", fileName),
                                            StatusBar.DISPLAY_NORMAL);

                                    final String workingDir = Paths.get(selectedFile).getParent().toString();
                                    LOGGER.info("Save current workingDir: {}", workingDir);

                                    wizardSettings.setWorkingDirectory(WORKING_DIR_DEFAULT_LABELS_KEY, workingDir);
                                }
                                catch (Exception ex) {
                                    LOGGER.warn("Generate default labels failed.", ex);

                                    TaskDialogs
                                        .build(JOptionPane.getFrameForComponent(NodeListPanel.this.nodeTree),
                                            Resources
                                                .getString(NodeListPanel.class,
                                                    "generate-default-labels-failed.instruction"),
                                            Resources.getString(NodeListPanel.class, "generate-default-labels-failed"))
                                        .title(
                                            Resources.getString(NodeListPanel.class, "generate-default-labels.title"))
                                        .exception(ex);
                                }
                                finally {
                                    busyFrame.setBusy(false);
                                }
                            }
                        };
                    dialog.showDialog();
                }
                else {
                    JOptionPane
                        .showMessageDialog(getComponent(), "No labels available for selected node.", "Default Labels",
                            JOptionPane.ERROR_MESSAGE);
                }
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Generate default labels failed.", ex);
        }
    }

    @Override
    public void applyDefaultLabels() {
        LOGGER.info("Apply the default labels of the node.");

        NodeInterface node = getSelectedItem();
        if (node != null) {

            final WorkItemModel workItemModel =
                this.workListItemProvider.getDefaultLabelsWorkItemModel(mainModel.getConnectionId(), node);
            if (workItemModel != null) {
                this.workListItemProvider.applyAction(workItemModel);
            }
        }
    }

    public void performShutdown() {
        // TODO Auto-generated method stub

    }

    @Override
    public void deleteNodeLabels() {
        LOGGER.info("Delete the node labels.");
        NodeInterface node = getSelectedItem();
        if (node != null) {

            for (NodeListActionListener l : nodeListListeners) {
                l.deleteNodeLabels(node);
            }

            // force a reload of the changed labels
            final LabelsChangedEvent labelsChangedEvent = new LabelsChangedEvent(node.getUniqueId());
            LOGGER.info("Publish the labelsChangedEvent: {}", labelsChangedEvent);
            EventBus.publish(labelsChangedEvent);

            this.applicationEventPublisher.publishEvent(labelsChangedEvent);
        }

    }
}
