package org.bidib.wizard.mvc.main.view.table;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.table.TableCellRenderer;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.FeedbackConfidenceData;
import org.bidib.jbidibc.messages.FeedbackDynStateData;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.model.status.FeedbackPortStatus;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.swing.MultilineLabel;

public class FeedbackPortTableCellRenderer implements TableCellRenderer {
    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPortTableCellRenderer.class);

    private JPanel panel;

    private DisabledPanel disabledInnerPanel;

    private final Color backgroundFree;

    private final Color backgroundOccupied;

    private final Color backgroundInvalid;

    private final Color backgroundInvalidOccupied;

    public FeedbackPortTableCellRenderer() {
        panel = new JPanel(new GridBagLayout());
        panel.setOpaque(true);

        disabledInnerPanel = createInnerPanel();

        this.backgroundFree = UIManager.getColor("Feedback.background.free");
        this.backgroundOccupied = UIManager.getColor("Feedback.background.occupied");
        this.backgroundInvalid = UIManager.getColor("Feedback.background.invalid");
        this.backgroundInvalidOccupied = UIManager.getColor("Feedback.background.invalid-occupied");
    }

    @Override
    public Component getTableCellRendererComponent(
        JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

        // clear the panel
        panel.removeAll();

        if (value instanceof FeedbackPort) {
            FeedbackPort port = (FeedbackPort) value;

            FeedbackPortStatus status = port.getStatus();

            final List<FeedbackAddressData> addresses = new ArrayList<>(port.getAddresses());
            final Set<FeedbackDynStateData> dynStates = port.getDynStates();

            // prepare the port label
            String label = null;

            if (StringUtils.isNotBlank(port.getLabel())) {
                label = String.format("%1$02d : %2$s", port.getId(), port.getLabel());
            }
            else if (port.getId() > -1) {
                label = String.format("%1$02d", port.getId());
            }
            else {
                label = " ";
            }
            portLabel.setText(label);

            final FeedbackConfidenceData confidence = port.getConfidence();
            if (confidence != null) {

                StringBuilder sb = new StringBuilder();
                sb
                    .append(confidence.isInvalid() ? "1" : "0").append(",").append(confidence.isFreeze() ? "1" : "0")
                    .append(",").append(confidence.isNoSignal() ? "1" : "0");
                confidenceLabel.setText(sb.toString());

                sb.setLength(0);

                sb.append("<html><b>CONFIDENCE</b><br>VOID: ");
                sb.append(confidence.isInvalid() ? "true" : "false").append("<br>");
                sb.append("FREEZE: ").append(confidence.isFreeze() ? "true" : "false").append("<br>");
                sb.append("NO DCC SIGNAL: ").append(confidence.isNoSignal() ? "true" : "false").append("</html>");
                // LOGGER.info("Set the tooltip text: {}", sb);
                panel.setToolTipText(sb.toString());
            }

            addressPanel.removeAll();
            if (addresses != null) {
                for (FeedbackAddressData address : addresses) {
                    addressPanel.add(new MultilineLabel(getAddressData(address, dynStates)));
                }
            }

            // background color
            if (status == FeedbackPortStatus.OCCUPIED) {
                innerPanel.setBackground(this.backgroundOccupied);
            }
            else {
                innerPanel.setBackground(this.backgroundFree);
            }

            boolean valid = false;
            if (confidence != null) {
                valid = !confidence.isFreeze() && !confidence.isNoSignal() && !confidence.isInvalid();
            }
            else {
                LOGGER.warn("No confidence available: {}", port);
            }

            // the disabled panel is enabled if the value is invalid
            if (!valid) {
                if (status == FeedbackPortStatus.OCCUPIED) {
                    innerPanel.setBackground(this.backgroundInvalidOccupied);
                }
                else {
                    innerPanel.setBackground(this.backgroundInvalid);
                }
            }
            disabledInnerPanel.setEnabled(valid);

            GridBagConstraints c = new GridBagConstraints();
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1;
            c.weighty = 1;
            panel.add(disabledInnerPanel, c);
        }
        return panel;
    }

    private JPanel innerPanel;

    private JLabel portLabel;

    private JLabel confidenceLabel;

    private JPanel addressPanel;

    private JLabel dynStateLabel;

    private JLabel timestampLabel;

    private DisabledPanel createInnerPanel() {

        innerPanel = new JPanel(new GridBagLayout());

        final GridBagConstraints c = new GridBagConstraints();

        c.anchor = GridBagConstraints.CENTER;
        c.gridx = 0;
        c.gridy = 0;
        c.weightx = 1;

        // prepare the port label
        portLabel = new JLabel();
        Font f = portLabel.getFont();

        portLabel.setFont(f.deriveFont(f.getStyle() | Font.BOLD));
        innerPanel.add(portLabel, c);

        c.gridy++;
        confidenceLabel = new JLabel();
        innerPanel.add(confidenceLabel, c);

        addressPanel = new JPanel();
        addressPanel.setOpaque(false);
        addressPanel.setLayout(new BoxLayout(addressPanel, BoxLayout.PAGE_AXIS));
        c.gridy++;
        innerPanel.add(addressPanel, c);

        dynStateLabel = new JLabel();
        c.gridy++;
        innerPanel.add(dynStateLabel, c);

        timestampLabel = new JLabel();
        c.gridy++;
        innerPanel.add(timestampLabel, c);

        c.fill = GridBagConstraints.BOTH;
        c.gridy++;
        c.weighty = 1;
        innerPanel.add(Box.createVerticalGlue(), c);

        // gray out depending on confidence
        final DisabledPanel disabledInnerPanel = new DisabledPanel(innerPanel);

        c.gridy = 0;
        panel.add(disabledInnerPanel, c);

        return disabledInnerPanel;
    }

    private String getAddressData(FeedbackAddressData address, final Collection<FeedbackDynStateData> dynStates) {
        StringBuilder result = new StringBuilder();

        if (address != null) {
            if (result.length() > 0) {
                result.append(',');
            }
            int addr = address.getAddress();
            result.append(addr);
            switch (address.getType()) {
                case LOCOMOTIVE_LEFT:
                    result.append("\u2191");
                    break;
                case LOCOMOTIVE_RIGHT:
                    result.append("\u2193");
                    break;
                default:
                    break;
            }
            result.append('(');
            result.append(address.getSpeed());
            result.append(')');

            if (CollectionUtils.isNotEmpty(dynStates)) {
                // search dyn state values
                for (FeedbackDynStateData dynState : dynStates) {
                    // append the dynstate
                    if (dynState.getDecoderAddress() == addr) {
                        dynState.appendDynState(result);
                    }
                }
            }
        }
        else {
            result.append("[]");
        }
        return result.toString();
    }
}
