package org.bidib.wizard.mvc.booster.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.enums.BoosterControl;
import org.bidib.jbidibc.messages.utils.MessageUtils;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.BoosterStatusListener;
import org.bidib.wizard.api.model.listener.CommandStationStatusListener;
import org.bidib.wizard.api.service.console.ConsoleColor;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.view.statusbar.StatusBarPublisher;
import org.bidib.wizard.model.status.BoosterStatus;
import org.bidib.wizard.model.status.CommandStationStatus;
import org.bidib.wizard.mvc.console.controller.ConsoleController;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;
import com.jgoodies.common.collect.ArrayListModel;

public class BoosterTableModel extends Model {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(BoosterTableModel.class);

    public static final String PROPERTY_BOOSTERS = "boosters";

    private ArrayListModel<BoosterModel> boosterList = new ArrayListModel<>();

    private final PropertyChangeListener boosterListener;

    private final CommandStationStatusListener commandStationStatusListener;

    private final BoosterStatusListener boosterStatusListener;

    private final ConsoleService consoleService;

    public BoosterTableModel(final StatusBarPublisher<String, Integer> publisher, final ConsoleService consoleService) {
        this.consoleService = consoleService;

        final Consumer<Integer> overcurrent = new Consumer<Integer>() {

            @Override
            public void accept(Integer current) {
                LOGGER.debug("OverCurrent detected: {}", current);

                StringBuffer sb = new StringBuffer("Received high current with value: ");
                sb.append(current);
                sb.append("mA");
                LOGGER.warn(sb.toString());

                publisher.publish(sb.toString(), 30);
            }
        };

        // create the propertyChangeListener to refresh the booster list
        boosterListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                switch (evt.getPropertyName()) {
                    case BoosterModel.PROPERTY_BOOSTER_LABEL:
                        LOGGER.debug("The booster label has been changed.");
                        BoosterModel booster = (BoosterModel) evt.getSource();
                        int index = boosterList.indexOf(booster);
                        if (index > -1) {
                            SwingUtilities.invokeLater(() -> boosterList.fireContentsChanged(index));
                        }
                        break;
                    default:
                        break;
                }
            }
        };

        commandStationStatusListener = new CommandStationStatusListener() {

            @Override
            public void statusChanged(final NodeInterface node, CommandStationStatus commandStationStatus) {

                LOGGER.info("Set the command station status: {}", commandStationStatus);
                if (SwingUtilities.isEventDispatchThread()) {
                    setCommandStationStatus(node, commandStationStatus);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            setCommandStationStatus(node, commandStationStatus);
                        }
                    });
                }

            }
        };

        boosterStatusListener = new BoosterStatusListener() {

            @Override
            public void voltageChanged(NodeInterface node, Integer voltage) {

                if (SwingUtilities.isEventDispatchThread()) {
                    setBoosterVoltage(node, voltage);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            setBoosterVoltage(node, voltage);
                        }
                    });
                }
            }

            @Override
            public void temperatureChanged(NodeInterface node, Integer temperature) {

                if (SwingUtilities.isEventDispatchThread()) {
                    setBoosterTemperature(node, temperature);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            setBoosterTemperature(node, temperature);
                        }
                    });
                }
            }

            @Override
            public void stateChanged(NodeInterface node, BoosterStatus status) {

                LOGGER.debug("Set the booster state: {}, node: {}", status, node);
                if (SwingUtilities.isEventDispatchThread()) {
                    changeBoosterState(node, status);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            changeBoosterState(node, status);
                        }
                    });
                }
            }

            private void changeBoosterState(final NodeInterface node, final BoosterStatus status) {

                setBoosterState(node, status);

                try {
                    if (BoosterStatus.OFF_SHORT == status || BoosterStatus.OFF_HOT == status) {
                        // display warning in console
                        ConsoleController.ensureConsoleVisible();

                        String nodeIdentifier = MessageUtils.formatNodeAddressAndName(node.getNode());

                        // add line
                        consoleService.addConsoleLine(ConsoleColor.red,
                                String
                                    .format("The booster node %s has signalled an booster state error: %s",
                                        nodeIdentifier, status));
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Show booster state error in console failed.", ex);
                }
            }

            @Override
            public void maximumCurrentChanged(NodeInterface node, Integer maximumCurrent) {

                // if (SwingUtilities.isEventDispatchThread()) {
                // setBoosterMaxCurrent(node, maximumCurrent);
                // }
                // else {
                // SwingUtilities.invokeLater(new Runnable() {
                //
                // @Override
                // public void run() {
                // setBoosterMaxCurrent(node, maximumCurrent);
                // }
                // });
                // }
            }

            @Override
            public void currentChanged(NodeInterface node, Integer current, long timestamp) {
                SwingUtilities.invokeLater(() -> setBoosterCurrent(node, current, overcurrent));
            }

            @Override
            public void controlChanged(NodeInterface node, BoosterControl control) {
                LOGGER.info("The booster control has changed: {}", control);
                // TODO Auto-generated method stub

            }

        };

    }

    public void addBooster(final NodeInterface node, final WizardLabelWrapper wizardLabelWrapper) {

        synchronized (boosterList) {
            BoosterModel booster = new BoosterModel(node, wizardLabelWrapper, boosterListener);
            if (!boosterList.contains(booster)) {
                LOGGER.info("Add booster to booster list: {}", node);
                booster.registerNode();

                String nodeLabel = booster.prepareNodeLabel();
                booster.setNodeLabel(nodeLabel);

                List<BoosterModel> oldValue = new LinkedList<>(boosterList);
                boosterList.add(booster);

                firePropertyChange(PROPERTY_BOOSTERS, oldValue, boosterList);

                if (node.getBoosterNode() != null) {
                    node.getBoosterNode().addBoosterStatusListener(boosterStatusListener);

                    if (node.getBoosterNode().getBoosterStatus() != null) {
                        booster.setState(node.getBoosterNode().getBoosterStatus().getBoosterState());
                    }
                }

                if (node.getCommandStationNode() != null) {
                    node.getCommandStationNode().addCommandStationStatusListener(commandStationStatusListener);

                    if (node.getCommandStationNode().getCommandStationState() != null) {
                        booster.setCommandStationState(node.getCommandStationNode().getCommandStationState());
                    }
                }

            }
            else {
                LOGGER.warn("Node is already in booster list: {}", node);
            }
        }
    }

    public void removeBooster(final NodeInterface node) {

        synchronized (boosterList) {
            LOGGER.info("Remove booster from booster list: {}", node);

            List<BoosterModel> oldValue = new LinkedList<>(boosterList);
            int index = boosterList.indexOf(new BoosterModel(node, null, null));
            if (index > -1) {
                BoosterModel removed = boosterList.remove(index);
                if (removed != null) {
                    try {
                        if (removed.getBooster().getBoosterNode() != null) {
                            removed.getBooster().getBoosterNode().removeBoosterStatusListener(boosterStatusListener);
                        }

                        if (node.getCommandStationNode() != null) {
                            node
                                .getCommandStationNode()
                                .removeCommandStationStatusListener(commandStationStatusListener);
                        }

                        removed.freeNode();
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Free booster node failed.", ex);
                    }
                }

                firePropertyChange(PROPERTY_BOOSTERS, oldValue, boosterList);
            }
        }
    }

    public ArrayListModel<BoosterModel> getBoosterListModel() {
        return boosterList;
    }

    public List<BoosterModel> getBoosters() {
        return Collections.unmodifiableList(boosterList);
    }

    public void setBoosterState(NodeInterface node, BoosterStatus state) {
        int fireChangeIndex = -1;

        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.trace("Found booster to update: {}", booster);
                    booster.setState(state != null ? state.getBoosterState() : null);

                    int index = boosterList.indexOf(booster);
                    fireChangeIndex = index;
                    break;
                }
            }
        }

        if (fireChangeIndex > -1) {
            boosterList.fireContentsChanged(fireChangeIndex);
        }
    }

    public void setCommandStationStatus(NodeInterface node, CommandStationStatus status) {
        int fireChangeIndex = -1;
        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.trace("Found command station to update: {}", booster);

                    if (status == null) {
                        LOGGER.warn("The command station status is not available. Use OFF as state.");
                        status = CommandStationStatus.OFF;
                    }

                    booster.setCommandStationState(status.getCommandStationState());

                    int index = boosterList.indexOf(booster);
                    fireChangeIndex = index;
                    break;
                }
            }
        }
        if (fireChangeIndex > -1) {
            boosterList.fireContentsChanged(fireChangeIndex);
        }
    }

    public void setBoosterCurrent(NodeInterface node, Integer current, Consumer<Integer> overcurrent) {
        int fireChangeIndex = -1;
        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.trace("Found booster to update: {}", booster);
                    booster.setCurrent(current);

                    if (current != null && booster.getMaxCurrent() != null && overcurrent != null
                        && current > booster.getMaxCurrent()) {
                        overcurrent.accept(current);
                    }

                    int index = boosterList.indexOf(booster);
                    fireChangeIndex = index;
                    break;
                }
            }
        }
        if (fireChangeIndex > -1) {
            boosterList.fireContentsChanged(fireChangeIndex);
        }
    }

    public void triggerFetchBoosterMaxCurrent(NodeInterface node) {

        BoosterModel boosterToFetch = null;
        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.info("Fetch the max current value for booster: {}", booster);
                    boosterToFetch = booster;
                    break;
                }
            }
        }

        if (boosterToFetch != null) {
            boosterToFetch.fetchMaxBoosterCurrent();
        }
    }

    public void setBoosterVoltage(NodeInterface node, Integer voltage) {
        int fireChangeIndex = -1;
        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.trace("Found booster to update: {}", booster);
                    booster.setVoltage(voltage);

                    int index = boosterList.indexOf(booster);
                    fireChangeIndex = index;
                    break;
                }
            }
        }
        if (fireChangeIndex > -1) {
            boosterList.fireContentsChanged(fireChangeIndex);
        }
    }

    public void setBoosterTemperature(NodeInterface node, int temperature) {
        int fireChangeIndex = -1;
        synchronized (boosterList) {
            for (BoosterModel booster : boosterList) {
                if (booster.getBooster().getUniqueId() == node.getUniqueId()) {
                    LOGGER.trace("Found booster to update: {}", booster);
                    booster.setTemperature(temperature);

                    int index = boosterList.indexOf(booster);
                    fireChangeIndex = index;
                    break;
                }
            }
        }
        if (fireChangeIndex > -1) {
            boosterList.fireContentsChanged(fireChangeIndex);
        }
    }
}
