package org.bidib.wizard.mvc.main.view.statusbar;

import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.Timer;

import org.apache.commons.lang3.StringUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.PreferencesPortType;
import org.bidib.wizard.api.model.listener.DefaultStatusListener;
import org.bidib.wizard.api.notification.ConnectionAction;
import org.bidib.wizard.api.notification.TimeEvent;
import org.bidib.wizard.common.model.settings.ConnectionConfiguration;
import org.bidib.wizard.common.view.statusbar.StatusBar;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.connection.DefaultBidibConnection;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.core.model.settings.MiscSettings;
import org.bidib.wizard.core.service.ConnectionService;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.core.utils.AopUtils;
import org.bidib.wizard.model.ports.event.LineStatusEvent;
import org.bidib.wizard.model.ports.event.StatusEvent;
import org.bidib.wizard.mvc.main.model.StatusModel;
import org.bidib.wizard.mvc.main.view.panel.listener.AbstractStatusListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.jidesoft.status.LabelStatusBarItem;
import com.jidesoft.status.MemoryStatusBarItem;
import com.jidesoft.status.ProgressStatusBarItem;
import com.jidesoft.status.ResizeStatusBarItem;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.JideBoxLayout;
import com.jidesoft.utils.PortingUtils;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;

public class DefaultJideStatusBar implements StatusBar {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultJideStatusBar.class);

    private JLabel statusLabel;

    private DigitalClockStatusBarItem modelClock;

    private DefaultOverlayable overlayLedPanel;

    private ConnectionStatusBarItem ledPanel;

    private StatusButtonStatusBarItem statusButtonPanel;

    private Timer cleanTimer;

    private LabelStatusBarItem connectionProviderLabel;

    private com.jidesoft.status.StatusBar statusBar;

    private ProgressStatusBarItem progress;

    private final StatusModel statusModel;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private ConnectionService connectionService;

    private CompositeDisposable compDisposable = new CompositeDisposable();

    /**
     * Constructor.
     */
    public DefaultJideStatusBar(final StatusModel statusModel) {
        this.statusModel = statusModel;
    }

    public void initialize() {

        this.statusBar = new com.jidesoft.status.StatusBar();

        this.progress = new ProgressStatusBarItem() {
            private static final long serialVersionUID = 1L;

            @Override
            protected JLabel createProgressLabel() {

                JLabel progressLabel = super.createProgressLabel();
                statusLabel = progressLabel;
                return progressLabel;
            }
        };
        progress.setCancelCallback(new ProgressStatusBarItem.CancelCallback() {
            @Override
            public void cancelPerformed() {
                progress.setStatus(Resources.getString(StatusBar.class, "cancelled"));
                progress.showStatus();
            }
        });
        statusBar.add(this.progress, JideBoxLayout.VARY);

        JLabel simulationLabel = new JLabel("Simulation", SwingConstants.CENTER);
        Font oldFont = simulationLabel.getFont();
        simulationLabel.setFont(oldFont.deriveFont(Font.BOLD, oldFont.getSize2D() + 2));
        simulationLabel.setForeground(new Color(190, 15, 20));
        PortingUtils.removeFocus(simulationLabel);

        this.ledPanel = new ConnectionStatusBarItem(this.statusModel);
        this.ledPanel.setPreferredWidth(170);

        this.overlayLedPanel = new DefaultOverlayable(this.ledPanel, simulationLabel);
        this.statusBar.add(this.overlayLedPanel, JideBoxLayout.FLEXIBLE);

        this.connectionProviderLabel = new LabelStatusBarItem();
        this.connectionProviderLabel.setPreferredWidth(70);
        this.statusBar.add(connectionProviderLabel, JideBoxLayout.FLEXIBLE);

        this.statusButtonPanel = new StatusButtonStatusBarItem(this.statusModel);
        this.statusButtonPanel.setPreferredWidth(40);
        this.statusBar.add(this.statusButtonPanel, JideBoxLayout.FIX);

        updateSelectedPort(this.connectionProviderLabel);
        setTimeStatusButtonEnabled();

        LOGGER.info("Create the digital clock.");
        this.modelClock = new DigitalClockStatusBarItem(settingsService);

        this.modelClock.setPreferredWidth(100);
        this.statusBar.add(this.modelClock, JideBoxLayout.FLEXIBLE);
        this.modelClock.setToolTipText(Resources.getString(StatusBar.class, "modeltime"));

        final MemoryStatusBarItem memoryItem = new MemoryStatusBarItem();
        this.statusBar.add(memoryItem, JideBoxLayout.FLEXIBLE);

        this.statusBar.add(new ResizeStatusBarItem(), JideBoxLayout.FIX);

        this.modelClock.setEnabled(settingsService.getGlobalSettings().isModelTimeEnabled());

        this.statusModel.setRunning(settingsService.getGlobalSettings().isModelTimeEnabled());

        this.statusModel.addModelClockStatusListener(modelClock);

        this.statusModel.addStatusListener(new DefaultStatusListener() {
            @Override
            public void runningChanged(boolean running) {
                if (running) {
                    modelClock.start();
                }
                else {
                    modelClock.stop();
                }
            }
        });

        this.statusButtonPanel.addStatusListener(new AbstractStatusListener() {
            @Override
            public void switchedOff() {

                // TODO use the globalSettings.setModelTimeEnabled instead of local running flag

                settingsService.getGlobalSettings().setModelTimeEnabled(false);

                statusModel.setModelClockStartEnabled(false);
                statusModel.setRunning(false);
            }

            @Override
            public void switchedOn() {

                // TODO use the globalSettings.setModelTimeEnabled instead of local running flag

                settingsService.getGlobalSettings().setModelTimeEnabled(true);

                statusModel.setModelClockStartEnabled(true);
                statusModel.setRunning(true);
            }
        });

        try {
            final GlobalSettings gs = AopUtils.getTargetObject(settingsService.getGlobalSettings());

            gs.addPropertyChangeListener(GlobalSettings.PROPERTY_SELECTED_PORTTYPE, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    LOGGER.info("The selected port type has changed, update the selected port label.");
                    updateSelectedPort(connectionProviderLabel);
                }
            });

            gs
                .addPropertyChangeListener(MiscSettings.PROPERTY_SELECTED_SERIALPORT_PROVIDER,
                    new PropertyChangeListener() {

                        @Override
                        public void propertyChange(PropertyChangeEvent evt) {

                            updateSelectedPort(connectionProviderLabel);
                        }
                    });

            gs.addPropertyChangeListener(GlobalSettings.PROPERTY_MODEL_TIME_ENABLED, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    boolean isModelTimeEnabled = settingsService.getGlobalSettings().isModelTimeEnabled();
                    statusModel.setRunning(isModelTimeEnabled);
                    modelClock.setEnabled(isModelTimeEnabled);
                }
            });
        }
        catch (Exception ex) {
            LOGGER.warn("Add property change listeners failed.", ex);
        }
        updateSelectedPort(connectionProviderLabel);

        Disposable dispTimeEvents = connectionService.subscribeTimeEvents(te -> {
            LOGGER.debug("Received a time event: {}", te);

            if (ConnectionRegistry.CONNECTION_ID_MAIN.equals(te.getConnectionId())) {

                setModelTime(te);
            }
        }, error -> {
            LOGGER.warn("The time event signalled a failure: {}", error);
        });
        compDisposable.add(dispTimeEvents);

        Disposable dispStatusEvents = connectionService.subscribeStatusEvents(se -> {
            LOGGER.debug("Received a connection status event: {}", se);

            if (ConnectionRegistry.CONNECTION_ID_MAIN.equals(se.getConnectionId())) {

                setConnectionStatus(se);
            }
        }, error -> {
            LOGGER.warn("The status event signalled a failure: {}", error);
        });
        compDisposable.add(dispStatusEvents);

        Disposable dispConnectionActions = connectionService.subscribeConnectionActions(ca -> {
            if (ConnectionRegistry.CONNECTION_ID_MAIN.equals(ca.getConnectionId())) {

                processConnectionAction(ca);
            }
        }, error -> {
            LOGGER.warn("The connection actions signalled a failure: {}", error);
        });
        compDisposable.add(dispConnectionActions);

        this.statusModel.addPropertyChangeListener(StatusModel.PROPERTY_RUNNING, evt -> {

            boolean isRunning = statusModel.isRunning();

            LOGGER.info("The running property has changed, isRunning: {}", isRunning);

            boolean isConnected = statusModel.isCd();
            boolean isModelTimeEnabled = settingsService.getGlobalSettings().isModelTimeEnabled();
            if (isConnected && isModelTimeEnabled) {

                boolean running = statusModel.isRunning();
                LOGGER.info("Send the model clock, running: {}", running);

                connectionService.setModelTimeEnabled(ConnectionRegistry.CONNECTION_ID_MAIN, running);
            }
            else {
                connectionService.setModelTimeEnabled(ConnectionRegistry.CONNECTION_ID_MAIN, false);
            }
        });

    }

    private void setTimeStatusButtonEnabled() {
        // use the tunneled connection as enabled criteria

        PreferencesPortType selectedPortType =
            ConnectionConfiguration
                .toPreferencesPortType(settingsService.getGlobalSettings().getConnectionConfigurations(),
                    ConnectionRegistry.CONNECTION_ID_MAIN);

        boolean isTunneled = PreferencesPortType.checkSelectedPortTypeIsTunnel(selectedPortType);
        statusButtonPanel.setEnabled(!isTunneled);
    }

    private void updateSelectedPort(final LabelStatusBarItem connectionProviderLabel) {
        updateSelectedPort(connectionProviderLabel, null);
    }

    private void updateSelectedPort(final LabelStatusBarItem connectionProviderLabel, final Integer baudrate) {

        try {
            PreferencesPortType selectedPortType =
                ConnectionConfiguration
                    .toPreferencesPortType(settingsService.getGlobalSettings().getConnectionConfigurations(),
                        ConnectionRegistry.CONNECTION_ID_MAIN);
            LOGGER.info("The selectedPortType to use: {}, baudrate: {}", selectedPortType, baudrate);
            String connectionProvider = evaluateConnectionProviderLabel(selectedPortType);
            connectionProviderLabel.setText(connectionProvider);

            String connectionName = selectedPortType != null ? selectedPortType.getConnectionName() : null;
            if (StringUtils.isNotBlank(connectionProvider)) {
                if (baudrate != null) {
                    connectionProviderLabel
                        .setToolTipText(connectionProvider + " - " + connectionName + " - baud: "
                            + (baudrate.intValue() == 1_000_000 ? "1M" : baudrate));
                }
                else {
                    connectionProviderLabel.setToolTipText(connectionProvider + " - " + connectionName);
                }
            }

            boolean simulationMode = PreferencesPortType.isSimulation(selectedPortType);

            this.ledPanel.setEnabled(!simulationMode);
            this.overlayLedPanel.setOverlayVisible(simulationMode);

            boolean isTunneled = PreferencesPortType.checkSelectedPortTypeIsTunnel(selectedPortType);
            statusButtonPanel.setEnabled(!isTunneled);
        }
        catch (Exception ex) {
            LOGGER.warn("Change display connection provider failed.", ex);
        }
    }

    @Override
    public JComponent getComponent() {
        if (statusBar == null) {
            initialize();
        }
        return statusBar;
    }

    private void setConnectionStatus(final StatusEvent se) {

        if (se instanceof LineStatusEvent) {
            LineStatusEvent lse = (LineStatusEvent) se;
            statusModel.setCts(lse.isCTS());
        }

    }

    /**
     * Display a message on the status bar.
     * 
     * @param message
     *            The message to be displayed.
     */
    @Override
    public void setStatusText(String message) {
        LOGGER.debug("Set status message: {}", message);

        setStatusText(message, -1);
    }

    private void processConnectionAction(final ConnectionAction ca) {
        LOGGER
            .info("Process the connection action, connectionId: {}, messageKey: {}, context: {}", ca.getConnectionId(),
                ca.getMessageKey(), ca.getContext());

        final String messageKey = ca.getMessageKey();
        switch (messageKey) {
            case "bidib-connect-with-baudrate":

                final Integer baudrate = ca.getContext().get("baudrate", Integer.class, null);
                SwingUtilities.invokeLater(() -> {
                    String statusText = Resources.getString(DefaultBidibConnection.class, messageKey, baudrate);
                    setStatusText(statusText, StatusBar.DISPLAY_NORMAL);
                    updateSelectedPort(connectionProviderLabel, baudrate);
                });

                break;

            case "bidib-reconnect-with-next-baudrate":
                final Integer nextBaudrate = ca.getContext().get("baudrate", Integer.class, null);
                SwingUtilities.invokeLater(() -> {
                    String statusText = Resources.getString(DefaultBidibConnection.class, messageKey, nextBaudrate);
                    setStatusText(statusText, StatusBar.DISPLAY_NORMAL);
                    // updateSelectedPort(connectionProviderLabel, nextBaudrate);
                });

                break;
            default:
                break;
        }
    }

    /**
     * Display a message on the status bar then clear it after specified time.
     * 
     * @param message
     *            The message to be displayed.
     * @param seconds
     *            Time wait for clearing the message (in seconds). Any value lesser than 1 disable this functionality.
     */
    @Override
    public synchronized void setStatusText(final String message, final int seconds) {
        LOGGER.info("Set the status text: {}, seconds: {}", message, seconds);

        // stop the clean timer if any assigned
        if (cleanTimer != null) {
            LOGGER.info("Stop the clean timer.");
            cleanTimer.stop();
            cleanTimer = null;
        }

        if (SwingUtilities.isEventDispatchThread()) {
            updateStatusText(message, seconds);
        }
        else {
            try {
                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {
                        updateStatusText(message, seconds);
                    }
                });
            }
            catch (Exception ex) {
                LOGGER.warn("Update status text failed.", ex);
            }
        }
    }

    private void updateStatusText(String message, int seconds) {

        switch (seconds) {
            case DISPLAY_ERROR:
                statusLabel.setForeground(Color.RED);
                break;
            case DISPLAY_NORMAL:
                statusLabel.setForeground(Color.BLACK);
                break;
            default:
                statusLabel.setForeground(Color.RED);
                break;
        }
        if (StringUtils.isNotBlank(message)) {
            progress.setStatus(message);
        }
        else {
            progress.setStatus(null);
        }
        if (seconds > 0) {
            // start the clean timer
            cleanTimer = new Timer(seconds * 1000, new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    LOGGER.info("Reset the statusbar.");
                    statusLabel.setForeground(Color.BLACK);
                    progress.setStatus(null);
                }
            });
            cleanTimer.setRepeats(false);
            cleanTimer.start();
        }
    }

    @Override
    public void setModelTime(TimeEvent timeEvent) {
        modelClock.setModelTime(timeEvent);
    }

    private String evaluateConnectionProviderLabel(final PreferencesPortType selectedPortType) {
        String serialProviderLabel = null;

        if (selectedPortType != null && selectedPortType.getConnectionPortType() != null) {
            switch (selectedPortType.getConnectionPortType()) {
                case SerialOverTcp:
                    serialProviderLabel = "SerOverTcp";
                    break;
                case SerialPort:
                case SerialSymLink:
                    serialProviderLabel = settingsService.getMiscSettings().getSelectedSerialPortProvider();
                    break;
                case NetBidibClient:
                    serialProviderLabel = "NetBidib";
                    break;
                case Speedometer:
                    serialProviderLabel = "Speedometer";
                    break;
                default:
                    serialProviderLabel = "Sim";
                    break;
            }
        }
        return serialProviderLabel;
    }

    @Override
    public void setProgress(String message, boolean finished) {

        this.progress.setProgressStatus(message);
        if (!finished) {
            this.progress.setProgressStatus(message);
            // this.progress.setIndeterminate(true);
            this.progress.setProgress(10);
        }
        else {
            this.progress.setDefaultStatus(message);
            this.progress.setProgress(100);

            // progress.setStatus(message);
            progress.showStatus();
        }
    }

}
