package org.bidib.wizard.mvc.main.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.FeedbackAddressDataWithDynState;
import org.bidib.jbidibc.messages.FeedbackDynStateData;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.mvc.main.controller.GlobalDetectorStatusChangeProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.EventList;

public class GlobalDetectorModel extends Model implements GlobalDetectorStatusChangeProvider {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalDetectorModel.class);

    private NodeInterface selectedNode;

    private PropertyChangeListener pclFeedbackAddresses;

    private PropertyChangeListener pclFeedbackDynState;

    private EventList<FeedbackAddressDataWithDynState> addressDataEventList = new BasicEventList<>();

    public GlobalDetectorModel() {

        pclFeedbackAddresses = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("Property of feedback port has changed, evt: {}", evt);

                try {

                    NodeInterface node = (NodeInterface) evt.getSource();

                    FeedbackPort globalDetectorPort = node.getGlobalDetectorFeedbackPort();

                    addressDataEventList.getReadWriteLock().writeLock().lock();
                    try {

                        addressDataEventList.clear();

                        List<FeedbackAddressData> addresses = globalDetectorPort.getAddresses();
                        if (CollectionUtils.isNotEmpty(addresses)) {
                            LOGGER.info("Add new addresses: {}", addresses);

                            for (FeedbackAddressData address : addresses) {
                                FeedbackAddressDataWithDynState addressDataWithDynState =
                                    new FeedbackAddressDataWithDynState(address.getAddress(), address.getType(),
                                        address.getSpeed());

                                addressDataEventList.add(addressDataWithDynState);
                            }

                            updateDynStateData(globalDetectorPort, addressDataEventList);
                        }
                        else {
                            LOGGER.info("No addresses available.");
                        }
                    }
                    finally {
                        addressDataEventList.getReadWriteLock().writeLock().unlock();
                    }

                }
                catch (Exception ex) {
                    LOGGER.warn("Get the changed data from the global detector port failed.", ex);
                }
            }

        };

        pclFeedbackDynState = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                LOGGER.info("Property of feedback port has changed, evt: {}", evt);

                try {
                    FeedbackPort globalDetectorPort = (FeedbackPort) evt.getSource();

                    addressDataEventList.getReadWriteLock().writeLock().lock();
                    try {

                        updateDynStateData(globalDetectorPort, addressDataEventList);
                    }
                    finally {
                        addressDataEventList.getReadWriteLock().writeLock().unlock();
                    }

                }
                catch (Exception ex) {
                    LOGGER.warn("Get the changed data from the global detector port failed.", ex);
                }
            }
        };
    }

    private void updateDynStateData(
        final FeedbackPort globalDetectorPort, final EventList<FeedbackAddressDataWithDynState> addressDataEventList) {

        Set<FeedbackDynStateData> dynStates = globalDetectorPort.getDynStates();
        if (CollectionUtils.isNotEmpty(dynStates)) {
            LOGGER.info("Add new dynStates: {}", dynStates);

            for (int index = 0; index < addressDataEventList.size(); index++) {

                FeedbackAddressDataWithDynState addressDataWithDynState = addressDataEventList.get(index);
                int decoderAddress = addressDataWithDynState.getAddress();
                List<FeedbackDynStateData> decoderDynStates =
                    dynStates
                        .stream().filter(dyn -> dyn.getDecoderAddress() == decoderAddress).collect(Collectors.toList());
                addressDataWithDynState.setDynStates(decoderDynStates);

                // force update
                addressDataEventList.set(index, addressDataWithDynState);
            }
        }
        else {
            LOGGER.info("No dynStates available.");
        }

    }

    @Override
    public NodeInterface getSelectedNode() {
        return selectedNode;
    }

    /**
     * @param selectedNode
     *            the selectedNode to set
     */
    public void setSelectedNode(NodeInterface selectedNode) {

        if (this.selectedNode != null) {
            // unregister node listener
            LOGGER.info("Remove pcl from previous selected node: {}", selectedNode);
            this.selectedNode
                .removePropertyChangeListener(NodeInterface.PROPERTY_GLOBAL_DETECTOR_FEEDBACKPORT,
                    pclFeedbackAddresses);
            this.selectedNode.removePropertyChangeListener(FeedbackPort.PROPERTY_DYNSTATES, pclFeedbackDynState);
        }

        this.selectedNode = selectedNode;

        if (this.selectedNode != null) {
            // register node listener
            LOGGER.info("Add pcl to currently selected node: {}", selectedNode);
            this.selectedNode
                .addPropertyChangeListener(NodeInterface.PROPERTY_GLOBAL_DETECTOR_FEEDBACKPORT, pclFeedbackAddresses);
            this.selectedNode.addPropertyChangeListener(FeedbackPort.PROPERTY_DYNSTATES, pclFeedbackDynState);

            addressDataEventList.getReadWriteLock().writeLock().lock();

            try {
                // load the data
                addressDataEventList.clear();

                List<FeedbackAddressData> addresses = this.selectedNode.getGlobalDetectorFeedbackPort().getAddresses();
                if (CollectionUtils.isNotEmpty(addresses)) {
                    LOGGER.info("Add new addresses: {}", addresses);
                    for (FeedbackAddressData address : addresses) {
                        FeedbackAddressDataWithDynState addressDataWithDynState =
                            new FeedbackAddressDataWithDynState(address.getAddress(), address.getType(),
                                address.getSpeed());

                        addressDataEventList.add(addressDataWithDynState);
                    }
                }
                else {
                    LOGGER.info("No addresses available.");
                }

                updateDynStateData(this.selectedNode.getGlobalDetectorFeedbackPort(), addressDataEventList);
            }
            finally {
                addressDataEventList.getReadWriteLock().writeLock().unlock();
            }
        }
        else {
            LOGGER.info("No node selected, clear the addressDataEventList.");

            addressDataEventList.getReadWriteLock().writeLock().lock();
            try {
                // load the data
                addressDataEventList.clear();
            }
            finally {
                addressDataEventList.getReadWriteLock().writeLock().unlock();
            }
        }
    }

    /**
     * @return the addressDataEventList
     */
    public EventList<FeedbackAddressDataWithDynState> getAddressDataEventList() {
        return addressDataEventList;
    }

}
