package org.bidib.wizard.mvc.main.model;

import java.util.function.IntConsumer;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.model.ports.MotorPort;
import org.bidib.wizard.model.status.MotorPortStatus;
import org.bidib.wizard.mvc.main.model.listener.MotorPortModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MotorPortTableModel extends SimplePortTableModel<MotorPortStatus, MotorPort, MotorPortModelListener> {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(MotorPortTableModel.class);

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_VALUE = 1;

    public static final int COLUMN_STATUS = 2;

    public static final int COLUMN_TEST = 3;

    public static final int COLUMN_PORT_INSTANCE = 4;

    public MotorPortTableModel() {
        super();
    }

    @Override
    protected void initialize() {
        columnNames =
            new String[] { Resources.getString(MotorPortTableModel.class, "label"),
                Resources.getString(MotorPortTableModel.class, "value"),
                Resources.getString(MotorPortTableModel.class, "status"),
                Resources.getString(MotorPortTableModel.class, "test"), null };
    }

    @Override
    public int getColumnPortInstance() {
        return COLUMN_PORT_INSTANCE;
    }

    @Override
    public void notifyPortStatusChanged(MotorPort port) {
        LOGGER.info("Update the port status: {}", port);
        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, COLUMN_PORT_INSTANCE))) {

                LOGGER.info("The port state has changed: {}", port.getStatus());
                super.setValueAt(port.getStatus(), row, COLUMN_STATUS);
                MotorPortStatus oppositeStatus = PortUtils.getOppositeStatus(port.getStatus());
                LOGGER.info("Set the opposite state: {}", oppositeStatus);
                super.setValueAt(oppositeStatus, row, COLUMN_TEST);
                break;
            }
        }
    }

    @Override
    public void notifyPortConfigChanged(MotorPort port) {
        // TODO Auto-generated method stub

    }

    @Override
    public void addRow(MotorPort motorPort) {
        if (motorPort != null) {
            Object[] rowData = new Object[columnNames.length];

            rowData[COLUMN_LABEL] = motorPort.getLabel();
            rowData[COLUMN_VALUE] = motorPort.getValue();
            rowData[COLUMN_STATUS] = motorPort.getStatus();
            MotorPortStatus oppositeStatus = PortUtils.getOppositeStatus(motorPort.getStatus());
            rowData[COLUMN_TEST] = oppositeStatus;

            rowData[COLUMN_PORT_INSTANCE] = motorPort;
            addRow(rowData);
        }
        else {
            LOGGER.warn("Performed addRow without servo port instance.");
        }
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        MotorPort motorPort = (MotorPort) getValueAt(row, COLUMN_PORT_INSTANCE);
        switch (column) {
            default:
                if (motorPort.isEnabled()) {
                    isEditable = true;
                }
                break;
        }
        return isEditable;
    }

    @Override
    public Class<?> getColumnClass(int column) {
        switch (column) {
            case COLUMN_LABEL:
                return String.class;
            default:
                return Integer.class;
        }
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        switch (column) {
            case COLUMN_LABEL:
                column = COLUMN_PORT_INSTANCE;
                break;
            case COLUMN_VALUE:
                MotorPort port = (MotorPort) super.getValueAt(row, COLUMN_PORT_INSTANCE);

                if (port != null) {
                    // return the value for the target position
                    result = port.getValue();
                    LOGGER.info("Get the port value: {}, row: {}", result, row);
                }
                break;
            default:
                column = COLUMN_PORT_INSTANCE;
                break;
        }
        if (result == null) {
            result = super.getValueAt(row, column);
        }
        return result;
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        final MotorPort port = (MotorPort) getValueAt(row, COLUMN_PORT_INSTANCE);

        switch (column) {
            case COLUMN_LABEL:
                port.setLabel((String) value);
                super.setValueAt(port.toString(), row, column);
                fireLabelChanged(port, port.getLabel());
                break;
            case COLUMN_VALUE:
                // set the value for the target position
                int intValue = ((Integer) value);
                LOGGER.info("Set the value, new value: {}, port.value: {}", value, port.getValue());

                port.setValue(intValue);
                super.setValueAt(intValue, row, column);

                // set the value on the temporary port
                final MotorPort motorPort = new MotorPort();
                motorPort.setId(port.getId());
                motorPort.setValue(intValue);

                fireTestButtonPressed(motorPort);

                break;
        }
    }

    @Override
    public void notifyPortLabelChanged(final MotorPort port) {
        LOGGER.info("The port label was changed for port: {}", port.getDebugString());

        for (int row = 0; row < getRowCount(); row++) {
            if (port.equals(getValueAt(row, MotorPortTableModel.COLUMN_PORT_INSTANCE))) {
                super.setValueAt(port.toString(), row, MotorPortTableModel.COLUMN_LABEL);

                break;
            }
        }
    }

    @Override
    public void refreshRow(final MotorPort port, final IntConsumer rowConsumer) {
        int row = findRow(port);
        rowConsumer.accept(row);
    }

    private int findRow(final MotorPort port) {
        for (int row = 0; row < getRowCount(); row++) {
            MotorPort current = (MotorPort) getValueAt(row, MotorPortTableModel.COLUMN_PORT_INSTANCE);

            if (port.equals(current)) {
                return row;
            }
        }
        return -1;
    }

    // @Override
    // protected void fireLabelChanged(MotorPort motorPort, String label) {
    // portListener.labelChanged(motorPort, label);
    // }

    public void fireTestButtonPressed(final MotorPort port) {
        LOGGER.info("The value has changed for port: {}", port);

        portListener.testButtonPressed(port, port.getStatus());
    }

}
