package org.bidib.wizard.mvc.main.view.panel;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.Collections;
import java.util.List;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JTable;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.enums.FeatureEnum;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.FeedbackPortListener;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.dialog.LabelDialog;
import org.bidib.wizard.model.ports.FeedbackPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.model.status.FeedbackPortStatus;
import org.bidib.wizard.mvc.main.controller.FeedbackPortPanelController;
import org.bidib.wizard.mvc.main.controller.listener.FeedbackPortPanelListener;
import org.bidib.wizard.mvc.main.model.FeedbackPortModel;
import org.bidib.wizard.mvc.main.model.FeedbackPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.client.common.model.SimplePortTableModel;
import org.bidib.wizard.client.common.model.listener.PortModelListener;
import org.bidib.wizard.client.common.view.menu.PortListMenu;
import org.bidib.wizard.client.common.view.menu.listener.PortListMenuListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.client.common.table.DefaultPortListMenuListener;
import org.bidib.wizard.mvc.main.view.table.FeedbackPortTableCellRenderer;
import org.bidib.wizard.mvc.main.view.table.FlagEditor;
import org.bidib.wizard.client.common.table.PortTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FeedbackPortListPanel
    extends SimplePortListPanel<FeedbackPortStatus, FeedbackPort, FeedbackPortListener, PortModelListener<FeedbackPort>>
    implements TabVisibilityProvider, TabPanelProvider, TabComponentCreator {
    private static final Logger LOGGER = LoggerFactory.getLogger(FeedbackPortListPanel.class);

    private static final long serialVersionUID = 1L;

    private final FeedbackPortPanelListener controller;

    private final TabVisibilityListener tabVisibilityListener;

    public FeedbackPortListPanel(final FeedbackPortPanelController controller, final FeedbackPortTableModel tableModel,
        final FeedbackPortModel feedbackPortModel, MainModel model, final TabVisibilityListener tabVisibilityListener) {
        super(tableModel, Resources.getString(FeedbackPortListPanel.class, "emptyTable"));

        this.controller = controller;
        this.tabVisibilityListener = tabVisibilityListener;

        LOGGER.debug("Create new FeedbackPortListPanel.");

        table.setDefaultCellRenderer(new FeedbackPortTableCellRenderer());
        table.setDefaultEditor(FeedbackPort.class, new FlagEditor() {
            private static final long serialVersionUID = 1L;

            @Override
            public Component getTableCellEditorComponent(
                JTable table, Object value, boolean isSelected, int row, int column) {
                if (value instanceof FeedbackPort) {
                    textField.setText(value != null ? ((FeedbackPort) value).getLabel() : null);
                    textField.setToolTipText(null);
                }
                else {
                    textField.setText(value != null ? value.toString() : null);
                    textField.setToolTipText(null);
                }
                return textField;
            }
        });

        table.setRowHeight(100);
        table.setTableHeader(null);

        feedbackPortModel.addPortListListener(this);
    }

    @Override
    protected PortTable createPortTable(
        final SimplePortTableModel<FeedbackPortStatus, FeedbackPort, PortModelListener<FeedbackPort>> tableModel,
        String emptyTableText) {
        return new PortTable(tableModel, emptyTableText) {
            private static final long serialVersionUID = 1L;

            @Override
            public void adjustRowHeight() {
                // we don't need to adjust the row height
            }

            @Override
            protected void prepareTableStyleProvider() {
                // we don't need the table style provider
            }

            @Override
            public void clearTable() {
            }

            @Override
            protected PortListMenu createMenu() {
                return new FeedbackPortListMenu(popupEvent, "No port selected.");
            }

            @Override
            protected PortListMenuListener createMenuListener() {

                // for feedback ports the port mapping is not visible
                setPortMappingVisible(false);

                // create the port list menu
                LOGGER.info("Create the menu listener.");
                return new FeedbackPortListMenuListener() {
                    @Override
                    public void editLabel(final MouseEvent popupEvent) {
                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        LOGGER.info("Edit label on row: {}, column: {}", row, column);
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof Port<?>) {
                                // val = ((Port<?>) val).toString();
                                val = ((Port<?>) val).getLabel();
                            }

                            // show the port name editor
                            new LabelDialog((String) val, popupEvent.getXOnScreen(), popupEvent.getYOnScreen()) {
                                @Override
                                public void labelChanged(String label) {
                                    LOGGER
                                        .info("Set the new label for row: {}, column: {}, label: {}", row, column,
                                            label);
                                    setValueAt(label, row, column);
                                }
                            };
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }

                    @Override
                    public void openLocoDialog() {
                        LOGGER.info("Open the loco controller.");

                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof FeedbackPort) {

                                FeedbackPort feedbackPort = (FeedbackPort) val;
                                if (CollectionUtils.isNotEmpty(feedbackPort.getAddresses())) {
                                    FeedbackAddressData addressData = feedbackPort.getAddresses().get(0);
                                    fireOpenLocoDialog(addressData);
                                }
                                else {
                                    LOGGER.info("No address available.");
                                    fireOpenLocoDialog(null);
                                }
                            }
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }

                    @Override
                    public void openPomDialog() {
                        LOGGER.info("Open the POM controller.");

                        final int row = getRow(popupEvent.getPoint());
                        final int column = getColumn(popupEvent.getPoint());
                        if (row > -1) {
                            Object val = getValueAt(row, column);
                            if (val instanceof FeedbackPort) {

                                FeedbackPort feedbackPort = (FeedbackPort) val;
                                if (CollectionUtils.isNotEmpty(feedbackPort.getAddresses())) {
                                    FeedbackAddressData addressData = feedbackPort.getAddresses().get(0);
                                    fireOpenPomDialog(addressData);
                                }
                                else {
                                    LOGGER.info("No address available.");
                                    fireOpenPomDialog(null);
                                }
                            }
                        }
                        else {
                            LOGGER.warn("The row is not available!");
                        }
                    }
                };
            }

            @Override
            protected void showPortListMenu(MouseEvent e, PortListMenu portListMenu, int row, int column) {

                Object value = getValueAt(row, column);
                if (row >= 0 && column >= 0 && (value instanceof Port<?> || value instanceof String)) {
                    if (row >= 0 && getSelectedRowCount() == 0) {
                        setRowSelectionInterval(row, row);
                    }

                    if (value instanceof FeedbackPort) {
                        final FeedbackPort feedbackPort = (FeedbackPort) value;
                        // if the port has mapping support enabled the activate the menu
                        if (!feedbackPort.isEnabled()) {
                            portListMenu.setMapPortEnabled(feedbackPort.isRemappingEnabled(), isPortMappingVisible());
                        }
                        else {
                            portListMenu.setMapPortEnabled(false, isPortMappingVisible());
                        }

                        // prepare the port label
                        String label = null;

                        if (StringUtils.isNotBlank(feedbackPort.getLabel())) {
                            label = String.format("%1$02d : %2$s", feedbackPort.getId(), feedbackPort.getLabel());
                        }
                        else if (feedbackPort.getId() > -1) {
                            label = String.format("%1$02d", feedbackPort.getId());
                        }
                        else {
                            label = " ";
                        }

                        ((FeedbackPortListMenu) portListMenu).setLabel("Port: " + label);

                    }
                    else {
                        portListMenu.setMapPortEnabled(false, false);
                        ((FeedbackPortListMenu) portListMenu).setLabel("No port selected.");
                    }
                    portListMenu.setInsertPortsEnabled(false, false);

                    portListMenu.setPopupEvent(popupEvent);

                    grabFocus();
                    portListMenu.show(e.getComponent(), e.getX(), e.getY());
                }
            }

        };
    }

    @Override
    public JPanel getComponent() {
        return this;
    }

    @Override
    public Object getCreator() {
        return this;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TabComponentCreator) {
            TabComponentCreator creator = (TabComponentCreator) other;
            // TODO if more than a single instance is available this must be changed
            if (creator.getCreator() instanceof FeedbackPortListPanel) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public void listChanged() {
        LOGGER.info("List has changed, remove all rows and add rows again.");
        tableModel.setRowCount(0);

        List<FeedbackPort> ports = getPorts();

        ((FeedbackPortTableModel) tableModel).addRows(ports);

        tabVisibilityListener.setTabVisible(this, isTabVisible());
    }

    @Override
    protected List<FeedbackPort> getPorts() {
        final NodeInterface node = controller.getSelectedNode();
        if (node != null) {
            return node.getFeedbackPorts();
        }
        return Collections.emptyList();
    }

    @Override
    public boolean isTabVisible() {

        final NodeInterface node = controller.getSelectedNode();
        if (node != null) {
            int feedbackSize =
                Feature.getIntFeatureValue(node.getNode().getFeatures(), FeatureEnum.FEATURE_BM_SIZE.getNumber());
            boolean isTabVisible = feedbackSize > 0;
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    public Class<?> getPortClass() {
        return FeedbackPort.class;
    }

    private class FeedbackPortListMenu extends PortListMenu {

        private static final long serialVersionUID = 1L;

        private JMenuItem openLocoDialog;

        private JMenuItem openPomDialog;

        public FeedbackPortListMenu(final MouseEvent popupEvent, String label) {
            super(popupEvent, label);

            openLocoDialog = new JMenuItem(Resources.getString(FeedbackPortListMenu.class, "openLocoDialog") + " ...");

            openLocoDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenLocoDialog();
                }

            });
            add(openLocoDialog);

            openPomDialog = new JMenuItem(Resources.getString(FeedbackPortListMenu.class, "openPomDialog") + " ...");

            openPomDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenPomDialog();
                }

            });
            add(openPomDialog);
        }

        private void fireOpenLocoDialog() {
            for (PortListMenuListener l : menuListeners) {
                if (l instanceof FeedbackPortListMenuListener) {
                    ((FeedbackPortListMenuListener) l).openLocoDialog();
                }
            }
        }

        private void fireOpenPomDialog() {
            for (PortListMenuListener l : menuListeners) {
                if (l instanceof FeedbackPortListMenuListener) {
                    ((FeedbackPortListMenuListener) l).openPomDialog();
                }
            }
        }
    }

    private void fireOpenLocoDialog(final FeedbackAddressData addressData) {

        controller.openLocoDialog(addressData);
    }

    private void fireOpenPomDialog(final FeedbackAddressData addressData) {

        controller.openPomDialog(addressData);
    }

    private abstract class FeedbackPortListMenuListener extends DefaultPortListMenuListener {

        public abstract void openLocoDialog();

        public abstract void openPomDialog();
    }

    public void refreshView() {
        listChanged();
    }
}
