package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedList;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import org.bidib.jbidibc.messages.FeedbackAddressData;
import org.bidib.jbidibc.messages.FeedbackAddressDataWithDynState;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.client.common.table.AbstractEmptyTable;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.client.common.view.slider.SliderRenderer;
import org.bidib.wizard.mvc.main.controller.listener.GlobalDetectorPanelListener;
import org.bidib.wizard.mvc.main.model.GlobalDetectorModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

import ca.odell.glazedlists.EventList;
import ca.odell.glazedlists.SortedList;
import ca.odell.glazedlists.gui.TableFormat;
import ca.odell.glazedlists.swing.AdvancedTableModel;
import ca.odell.glazedlists.swing.GlazedListsSwing;

public class GlobalDetectorPanel implements TabVisibilityProvider, TabPanelProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalDetectorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:50dlu:grow";

    private final GlobalDetectorPanelListener controller;

    private final GlobalDetectorModel globalDetectorModel;

    private final TabVisibilityListener tabVisibilityListener;

    private JPanel contentPanel;

    private final AdvancedTableModel<FeedbackAddressDataWithDynState> decodersTableModel;

    public GlobalDetectorPanel(final GlobalDetectorPanelListener controller,
        final GlobalDetectorModel globalDetectorModel, MainModel model,
        final TabVisibilityListener tabVisibilityListener) {

        this.controller = controller;
        this.globalDetectorModel = globalDetectorModel;
        this.tabVisibilityListener = tabVisibilityListener;
        LOGGER.debug("Create new GlobalDetectorPanel.");

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        EventList<FeedbackAddressDataWithDynState> addressDataEventList = globalDetectorModel.getAddressDataEventList();
        SortedList<FeedbackAddressDataWithDynState> sortedAddressData =
            new SortedList<FeedbackAddressDataWithDynState>(addressDataEventList, new AddressDataComparator());

        decodersTableModel =
            GlazedListsSwing
                .eventTableModelWithThreadProxyList(sortedAddressData, new FeedbackAddressDataTableFormat());

        final DetectorTable feedbackAddressDataTable = new DetectorTable(decodersTableModel);
        feedbackAddressDataTable.adjustRowHeight();
        // feedbackAddressDataJTable.setRowAutoResizes(false);
        // feedbackAddressDataJTable.setRowHeight(20);

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(feedbackAddressDataTable));
        feedbackAddressDataTable.getModel().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(feedbackAddressDataTable.getModel().getRowCount() == 0);
            }
        });

        overlayTable
            .addOverlayComponent(
                StyledLabelBuilder.createStyledLabel("{" + feedbackAddressDataTable.getEmptyTableText() + ":f:gray}"));

        dialogBuilder.add(overlayTable).xy(1, 1);

        contentPanel = dialogBuilder.build();

        // the name is displayed on the tab
        contentPanel.setName(getName());
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    public String getName() {
        return Resources.getString(getClass(), "name");
    }

    @Override
    public boolean isTabVisible() {
        NodeInterface node = globalDetectorModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.isGlobalDetectorAvailable();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    public void listChanged() {
        LOGGER.debug("List has changed, remove all rows and add rows again.");

        boolean isTabVisible = isTabVisible();
        tabVisibilityListener.setTabVisible(contentPanel, isTabVisible);

        // if (isTabVisible) {
        // LOGGER.info("Update the displayed data.");
        //
        // }
    }

    private static class AddressDataComparator implements Comparator<FeedbackAddressData> {

        @Override
        public int compare(FeedbackAddressData pos1, FeedbackAddressData pos2) {
            return pos1.getAddress() - pos2.getAddress();
        }
    }

    private static class FeedbackAddressDataTableFormat implements TableFormat<FeedbackAddressDataWithDynState> {

        private final String[] columnNames =
            { Resources.getString(FeedbackAddressDataTableFormat.class, "column.decoder"),
                Resources.getString(FeedbackAddressDataTableFormat.class, "column.speed"),
                Resources.getString(FeedbackAddressDataTableFormat.class, "column.dynstate") };

        @Override
        public int getColumnCount() {
            return columnNames.length;
        }

        @Override
        public String getColumnName(int column) {

            switch (column) {
                case 0:
                case 1:
                case 2:
                    return columnNames[column];
                default:
                    throw new IllegalStateException();
            }
        }

        @Override
        public Object getColumnValue(FeedbackAddressDataWithDynState feedbackAddressData, int column) {

            switch (column) {
                case 0:
                    return feedbackAddressData.getAddress();
                case 1:
                    return feedbackAddressData.getSpeed();
                case 2:
                    return feedbackAddressData.getDynStates();
                default:
                    throw new IllegalStateException();
            }
        }
    }

    private final class DetectorTable extends AbstractEmptyTable {

        private static final long serialVersionUID = 1L;

        private final DecoderListMenu menu;

        public DetectorTable(final AdvancedTableModel<FeedbackAddressDataWithDynState> decodersTableModel) {
            super(decodersTableModel, Resources.getString(GlobalDetectorPanel.class, "emptyTable"));

            menu = createMenu();

            addMouseListener(new MouseAdapter() {
                @Override
                public void mousePressed(MouseEvent e) {
                    handleMouseEvent(e, menu);
                }

                @Override
                public void mouseReleased(MouseEvent e) {
                    handleMouseEvent(e, menu);
                }
            });

            addMenuListener(menu);
        }

        @Override
        public void adjustRowHeight() {
            // set the correct row height
            SliderRenderer sliderEditor =
                new SliderRenderer(0, 255, 10);
            // sliderEditor.createComponent(0);

            int rowHeight =
                sliderEditor.getTableCellRendererComponent(this, 1, false, false, 0, 0).getPreferredSize().height + 6;
            LOGGER.info("Set row height: {}", rowHeight);

            setRowHeight(rowHeight);
        }

        protected DecoderListMenu createMenu() {
            // create the port list menu
            return new DecoderListMenu("No decoder selected.");
        }

        protected GlobalDetectorListMenuListener createMenuListener() {
            // create the port list menu
            return new GlobalDetectorListMenuListener() {
                @Override
                public void openLocoDialog() {
                    LOGGER.info("Open the loco controller.");

                    final int row = getRow(popupEvent.getPoint());
                    if (row > -1) {

                        int actualRow = getActualRowAt(row);
                        FeedbackAddressDataWithDynState feedbackAddressData =
                            decodersTableModel.getElementAt(actualRow);

                        int addressData = feedbackAddressData.getAddress();
                        if (addressData > 0) {
                            fireOpenLocoDialog(feedbackAddressData);
                        }
                        else {
                            LOGGER.info("No address available.");
                            fireOpenLocoDialog(null);
                        }
                    }
                    else {
                        LOGGER.warn("The row is not available!");
                    }
                }

                @Override
                public void openPomDialog() {
                    LOGGER.info("Open the POM controller.");

                    final int row = getRow(popupEvent.getPoint());
                    if (row > -1) {
                        int actualRow = getActualRowAt(row);
                        FeedbackAddressDataWithDynState feedbackAddressData =
                            decodersTableModel.getElementAt(actualRow);

                        int addressData = feedbackAddressData.getAddress();
                        if (addressData > 0) {
                            fireOpenPomDialog(feedbackAddressData);
                        }
                        else {
                            LOGGER.info("No address available.");
                            fireOpenPomDialog(null);
                        }
                    }
                    else {
                        LOGGER.warn("The row is not available!");
                    }
                }
            };
        }

        protected void addMenuListener(DecoderListMenu portListMenu) {
            portListMenu.addMenuListener(createMenuListener());
        }

        private void handleMouseEvent(MouseEvent e, DecoderListMenu portListMenu) {
            if (e.isPopupTrigger() && getRowCount() > 0) {
                popupEvent = e;

                int row = getRow(e.getPoint());
                int column = getColumn(e.getPoint());
                showPortListMenu(e, portListMenu, row, column);
            }
        }

        /**
         * Show the port list menu.
         * 
         * @param e
         *            the mouse event
         * @param portListMenu
         *            the portListMenu to use
         * @param row
         *            the row
         * @param column
         *            the column
         */
        protected void showPortListMenu(MouseEvent e, DecoderListMenu portListMenu, int row, int column) {
            // always get the value of column 0
            column = 0;
            Object value = getValueAt(row, column);

            int actualRow = getActualRowAt(row);
            value = decodersTableModel.getElementAt(actualRow);

            LOGGER.info("Show port list menu, row: {}, actualRow: {}, current decoder: {}", row, actualRow, value);

            if (row >= 0 && column >= 0 && (value instanceof FeedbackAddressDataWithDynState)) {
                if (row >= 0 && getSelectedRowCount() == 0) {
                    setRowSelectionInterval(row, row);
                }

                // prepare the port label
                FeedbackAddressDataWithDynState addressDataWithDynState = (FeedbackAddressDataWithDynState) value;
                String label =
                    Resources.getString(GlobalDetectorPanel.class, "decoder") + addressDataWithDynState.getAddress();

                portListMenu.setLabel(label);

                grabFocus();
                portListMenu.show(e.getComponent(), e.getX(), e.getY());
            }
        }

    }

    private static class DecoderListMenu extends BasicPopupMenu {
        private static final long serialVersionUID = 1L;

        protected final Collection<GlobalDetectorListMenuListener> menuListeners = new LinkedList<>();

        private JMenuItem openLocoDialog;

        private JMenuItem openPomDialog;

        public DecoderListMenu(String label) {
            super(label);

            openLocoDialog = new JMenuItem(Resources.getString(DecoderListMenu.class, "openLocoDialog") + " ...");

            openLocoDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenLocoDialog();
                }

            });
            add(openLocoDialog);

            openPomDialog = new JMenuItem(Resources.getString(DecoderListMenu.class, "openPomDialog") + " ...");

            openPomDialog.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    fireOpenPomDialog();
                }

            });
            add(openPomDialog);

        }

        public void addMenuListener(GlobalDetectorListMenuListener l) {
            menuListeners.add(l);
        }

        private void fireOpenLocoDialog() {
            for (GlobalDetectorListMenuListener l : menuListeners) {
                if (l instanceof GlobalDetectorListMenuListener) {
                    l.openLocoDialog();
                }
            }
        }

        private void fireOpenPomDialog() {
            for (GlobalDetectorListMenuListener l : menuListeners) {
                if (l instanceof GlobalDetectorListMenuListener) {
                    l.openPomDialog();
                }
            }
        }

    }

    private void fireOpenLocoDialog(final FeedbackAddressDataWithDynState addressData) {

        controller.openLocoDialog(addressData);
    }

    private void fireOpenPomDialog(final FeedbackAddressDataWithDynState addressData) {

        controller.openPomDialog(addressData);
    }

    private static class GlobalDetectorListMenuListener {

        public void openLocoDialog() {
        }

        public void openPomDialog() {
        }
    }
}
