package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.enums.LcMacroState;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.exception.ProtocolNoAnswerException;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroSaveState;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.MacroListListener;
import org.bidib.wizard.api.utils.MacroUtils;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.common.script.node.types.TargetType;
import org.bidib.wizard.common.utils.MacroListUtils;
import org.bidib.wizard.common.view.statusbar.StatusBar;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.labels.DefaultWizardLabelFactory;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.common.view.panel.DisabledPanel;
import org.bidib.wizard.mvc.common.view.table.DefaultTextCellEditor;
import org.bidib.wizard.mvc.main.controller.MacroPanelController;
import org.bidib.wizard.mvc.main.controller.listener.MacroPanelListener;
import org.bidib.wizard.mvc.main.model.MacroFactory;
import org.bidib.wizard.mvc.main.model.MacroFactory.ExportFormat;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.SingleColumnTableModel;
import org.bidib.wizard.mvc.main.model.listener.MacroSelectionListener;
import org.bidib.wizard.mvc.main.view.component.DefaultTabSelectionPanel;
import org.bidib.wizard.mvc.main.view.component.MacroFileDialog;
import org.bidib.wizard.mvc.main.view.menu.MacroListMenu;
import org.bidib.wizard.mvc.main.view.menu.listener.MacroListMenuListener;
import org.bidib.wizard.mvc.main.view.panel.listener.LabelChangedListener;
import org.bidib.wizard.mvc.main.view.panel.listener.MacroActionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabComponentCreator;
import org.bidib.wizard.mvc.main.view.panel.listener.TabSelectionListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityProvider;
import org.bidib.wizard.mvc.main.view.panel.renderer.MacroListRenderer;
import org.bidib.wizard.nodescript.script.node.ChangeLabelSupport;
import org.oxbow.swingbits.dialog.task.TaskDialogs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jidesoft.grid.RowStripeTableStyleProvider;
import com.jidesoft.grid.SortableTable;
import com.jidesoft.grid.TableStyleProvider;
import com.jidesoft.swing.DefaultOverlayable;
import com.jidesoft.swing.StyledLabelBuilder;

public class MacroListPanel
    implements MacroListMenuListener, MacroListListener, MacroActionListener, MacroSelectionListener,
    ChangeLabelSupport, TabVisibilityProvider, TabPanelProvider, PendingChangesAware, TabSelectionListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(MacroListPanel.class);

    private static final String EMPTY_LABEL = " ";

    private final MainModel mainModel;

    private final Collection<LabelChangedListener<Macro>> labelChangedListeners = new LinkedList<>();

    private final TabVisibilityListener tabVisibilityListener;

    private MacroPanelListener macroPanelListener;

    private final MacroPanel macroPanel;

    private final MacroListMenu macroListMenu = new MacroListMenu();

    private final DisabledPanel disabledBorderPanel;

    private JScrollPane scrollMacroList;

    private final JPanel contentPanel;

    private final StatusBar statusBar;

    private final SettingsService settingsService;

    private final WizardLabelWrapper wizardLabelWrapper;

    private final MacroList macroList;

    private final MacroPanelController macroPanelController;

    public class MacroList extends SortableTable {

        private static final long serialVersionUID = 1L;

        private TableStyleProvider tableStyleProvider;

        private List<ListSelectionListener> listSelectionListeners = new ArrayList<>();

        public MacroList(final DefaultTableModel tableModel) {
            super(tableModel);

            getSelectionModel().addListSelectionListener(new ListSelectionListener() {

                @Override
                public void valueChanged(ListSelectionEvent e) {

                    int selectedRow = getSelectedRow();

                    if (!e.getValueIsAdjusting()) {
                        final ListSelectionEvent evt =
                            new ListSelectionEvent(tableModel, selectedRow, selectedRow, e.getValueIsAdjusting());

                        for (ListSelectionListener listener : listSelectionListeners) {
                            listener.valueChanged(evt);
                        }
                    }
                }
            });

        }

        protected String getEmptyTableText() {
            return "No macros available.";
        }

        public void adjustRowHeight() {
            setRowHeight(getRowHeight() + 8);
        }

        protected void prepareTableStyleProvider() {
            tableStyleProvider =
                new RowStripeTableStyleProvider(UIManager.getColor("tableRowStripe.background"),
                    UIManager.getColor("tableRowStripe.alternativeBackground"));
        }

        @Override
        public TableStyleProvider getTableStyleProvider() {
            if (tableStyleProvider == null) {
                prepareTableStyleProvider();
            }
            return tableStyleProvider;
        }

        public void setItems(Macro[] macros) {
            LOGGER.info("Set the macros: {}", new Object[] { macros });

            final DefaultTableModel tableModel = ((DefaultTableModel) getModel());
            if (tableModel.getColumnCount() > 0) {
                tableModel.setRowCount(0);
            }
            for (Macro macro : macros) {
                tableModel.addRow(new Object[] { macro });
            }
        }

        public void refreshView() {
            // repaint the table
            repaint();
        }

        public void addListSelectionListener(ListSelectionListener l) {
            LOGGER.info("Add list selection listener to macroList: {}", l);
            this.listSelectionListeners.add(l);
        }

        public Macro getSelectedItem() {

            int selectedRow = getSelectedRow();
            if (selectedRow > -1) {
                return (Macro) getModel().getValueAt(selectedRow, 0);
            }
            return null;
        }
    }

    public MacroListPanel(final MacroPanelController macroPanelController, final MainModel model,
        final TabVisibilityListener tabVisibilityListener, final SettingsService settingsService,
        final WizardLabelWrapper wizardLabelWrapper, final DefaultWizardLabelFactory bidibLabelFactory) {

        this.macroPanelController = macroPanelController;
        this.mainModel = model;
        this.tabVisibilityListener = tabVisibilityListener;
        this.wizardLabelWrapper = wizardLabelWrapper;
        this.settingsService = settingsService;

        // get the status bar
        statusBar = DefaultApplicationContext.getInstance().get("statusBar", StatusBar.class);

        contentPanel = new DefaultTabSelectionPanel(this) {

            private static final long serialVersionUID = 1L;

            @Override
            public boolean equals(Object other) {
                if (other instanceof TabComponentCreator) {
                    TabComponentCreator creator = (TabComponentCreator) other;
                    // TODO if more than a single instance is available this must be changed
                    if (creator.getCreator() instanceof MacroListPanel) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public int hashCode() {
                return super.hashCode();
            }

            @Override
            public void tabSelected(boolean selected) {

            }

            @Override
            public String getName() {
                // this is used as tab title
                return Resources.getString(MacroListPanel.class, "name");
            }
        };
        contentPanel.setLayout(new BorderLayout());
        contentPanel.setBorder(new EmptyBorder(5, 5, 5, 5));

        macroListMenu.addMenuListener(this);

        this.macroPanel = new MacroPanel(this.macroPanelController, this.mainModel, this.settingsService, this);

        final JPanel panel = new JPanel(new BorderLayout());

        disabledBorderPanel = new DisabledPanel(this.macroPanel);
        panel.add(disabledBorderPanel);

        // create the list for the macros
        final SingleColumnTableModel<Macro> tableModel = new SingleColumnTableModel<>(Macro.class);
        tableModel.addLabelChangedListener((macro, label) -> {
            LOGGER.info("The label of the macro was changed: {}, label: {}", macro, label);

            long uniqueId = model.getSelectedNode().getUniqueId();

            // the name of the macro was changed

            macro.setLabel(label);

            final NodeLabels nodeLabels = this.wizardLabelWrapper.loadLabels(uniqueId);
            BidibLabelUtils.replaceMacroLabel(nodeLabels, macro.getId(), label);
            this.wizardLabelWrapper.saveNodeLabels(uniqueId);
        });

        this.macroList = new MacroList(tableModel);

        macroList.adjustRowHeight();

        // do not allow drag columns to other position
        macroList.getTableHeader().setReorderingAllowed(false);

        macroList.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);

        // disabled sorting
        macroList.setSortable(false);
        macroList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        // set the renderer
        final TableColumn tc = this.macroList.getColumnModel().getColumn(0);
        tc.setPreferredWidth(80);
        tc.setWidth(80);
        final MacroListRenderer renderer = new MacroListRenderer();
        tc.setCellRenderer(renderer);

        tc.setCellEditor(new DefaultTextCellEditor(new JTextField()));

        this.macroList.setPreferredScrollableViewportSize(new Dimension(200, 200));

        final DefaultOverlayable overlayTable = new DefaultOverlayable(new JScrollPane(macroList));
        macroList.getModel().addTableModelListener(new TableModelListener() {

            @Override
            public void tableChanged(TableModelEvent e) {
                overlayTable.setOverlayVisible(macroList.getModel().getRowCount() == 0);
            }
        });

        macroList.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                handleMouseEvent(e, macroListMenu);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                handleMouseEvent(e, macroListMenu);
            }
        });

        overlayTable
            .addOverlayComponent(
                StyledLabelBuilder.createStyledLabel("{" + macroList.getEmptyTableText() + ":f:gray}"));

        scrollMacroList = new JScrollPane(overlayTable);

        // JScrollPane scrollMacroContent = new JScrollPane(panel);

        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, scrollMacroList, panel);

        splitPane.setContinuousLayout(true);
        splitPane.setOneTouchExpandable(true);
        splitPane.setResizeWeight(0.2);

        contentPanel.add(splitPane, BorderLayout.CENTER);

        setEnabled(false);
    }

    /**
     * @param macroPanelListener
     *            the macroPanelListener
     */
    public void setMacroPanelListener(MacroPanelListener macroPanelListener) {
        this.macroPanelListener = macroPanelListener;
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    public void setEnabled(boolean enabled) {
        disabledBorderPanel.setEnabled(enabled);
    }

    private void fireExportMacro() {
        Macro macro = mainModel.getSelectedMacro();
        if (macro.isValid()) {

            exportMacro(macro);
        }
        else {
            LOGGER.warn("Macro is not valid for export.");
        }
    }

    private void fireReloadMacro() {
        reloadMacro(mainModel.getSelectedMacro());
    }

    private void fireSaveMacro() {
        Macro macro = mainModel.getSelectedMacro();
        if (macro.isValid()) {
            saveMacro(macro);
        }
        else {
            // show warning dialog
            JOptionPane
                .showMessageDialog(contentPanel, Resources.getString(getClass(), "macro_content_not_valid"),
                    Resources.getString(getClass(), "save"), JOptionPane.WARNING_MESSAGE);
        }
    }

    private void fireTestMacro(boolean transferBeforeStart) {
        Macro macro = mainModel.getSelectedMacro();
        if (macro.isValid()) {

            if (!transferBeforeStart && macro.getMacroSaveState().equals(MacroSaveState.PENDING_CHANGES)) {
                // show a hint to transfer the macro to the node
                int result =
                    JOptionPane
                        .showConfirmDialog(contentPanel,
                            Resources.getString(getClass(), "macro_transfer_pending_changes_before_test"),
                            Resources.getString(getClass(), "start"), JOptionPane.YES_NO_OPTION,
                            JOptionPane.WARNING_MESSAGE);

                if (result == JOptionPane.YES_OPTION) {
                    transferBeforeStart = true;
                }
            }

            startMacro(macro, transferBeforeStart);
        }
        else {
            // show warning dialog
            JOptionPane
                .showMessageDialog(contentPanel, Resources.getString(getClass(), "macro_content_not_valid"),
                    Resources.getString(getClass(), "start"), JOptionPane.WARNING_MESSAGE);
        }
    }

    @Override
    public void remoteStartMacro() {
        Macro macro = mainModel.getSelectedMacro();

        startMacro(macro, false);
    }

    private void fireStopMacro() {

        stopMacro(mainModel.getSelectedMacro());
    }

    private void fireInitializeMacro() {

        initializeMacro(mainModel.getSelectedMacro());
    }

    @Override
    public void transferMacro() {
        Macro macro = mainModel.getSelectedMacro();
        if (macro.isValid()) {

            transferMacro(mainModel.getSelectedMacro());
        }
        else {
            // show warning dialog
            JOptionPane
                .showMessageDialog(contentPanel, Resources.getString(getClass(), "macro_content_not_valid"),
                    Resources.getString(getClass(), "transfer"), JOptionPane.WARNING_MESSAGE);
        }
    }

    @Override
    public void exportMacro() {
        fireExportMacro();
    }

    @Override
    public void startMacro() {
        boolean powerUser = settingsService.getWizardSettings().isPowerUser();
        fireTestMacro(powerUser);
    }

    @Override
    public void stopMacro() {
        fireStopMacro();
    }

    @Override
    public void initializeMacro() {
        fireInitializeMacro();
    }

    @Override
    public void saveMacro() {
        fireSaveMacro();
    }

    @Override
    public void reloadMacro() {
        fireReloadMacro();
    }

    @Override
    public void testMacro(boolean transferBeforeStart) {
        fireTestMacro(transferBeforeStart);
    }

    @Override
    public void listChanged() {
        LOGGER.info("Macro list has been changed.");
        if (SwingUtilities.isEventDispatchThread()) {
            internalListChanged();
        }
        else {
            SwingUtilities.invokeLater(() -> internalListChanged());
        }
    }

    private void internalListChanged() {
        LOGGER.info("Macro list has been changed. Reload the macros.");

        List<Macro> macros = new ArrayList<>();
        macros.addAll(mainModel.getMacros());

        // sort by macro id
        Collections.sort(macros, new Comparator<Macro>() {

            @Override
            public int compare(Macro o1, Macro o2) {
                return (o1.getId() - o2.getId());
            }
        });

        macroList.setItems(macros.toArray(new Macro[0]));

        resetMacroScrollPane();
        setEnabled(false);

        tabVisibilityListener.setTabVisible(getComponent(), isTabVisible());
    }

    private void resetMacroScrollPane() {
        JScrollBar verticalScrollBar = scrollMacroList.getVerticalScrollBar();
        JScrollBar horizontalScrollBar = scrollMacroList.getHorizontalScrollBar();
        verticalScrollBar.setValue(verticalScrollBar.getMinimum());
        horizontalScrollBar.setValue(horizontalScrollBar.getMinimum());
    }

    @Override
    public void macroChanged() {
        LOGGER.info("Macro selection has changed.");

        Macro macro = mainModel.getSelectedMacro();
        LOGGER.info("The selected macro has been changed, macro: {}", macro);

        if (macro != null) {
            labelChanged(macro, macro.toString());
        }
        else {
            labelChanged(macro, EMPTY_LABEL);
        }

        setEnabled(macro != null);

        // transfer the focus back to the macro list
        SwingUtilities.invokeLater(() -> macroList.requestFocusInWindow());
    }

    @Override
    public void pendingChangesChanged() {

        if (SwingUtilities.isEventDispatchThread()) {
            macroList.refreshView();
        }
        else {
            SwingUtilities.invokeLater(() -> macroList.refreshView());
        }
    }

    @Override
    public void changeLabel(TargetType portType) {
        int portNum = portType.getPortNum();
        String label = portType.getLabel();

        Macro macro = MacroListUtils.findMacroByMacroNumber(mainModel.getMacros(), portNum);
        if (macro != null) {
            LOGGER.info("Current macro: {}, new label: {}", macro, label);
            macro.setLabel(label);
            if (macro.equals(mainModel.getSelectedMacro())) {
                labelChanged(macro, label);
            }
            fireLabelChanged(macro, label);
        }
    }

    @Override
    public boolean isTabVisible() {
        NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            boolean isTabVisible = node.hasMacros();
            LOGGER.debug("Check if tab is visible: {}", isTabVisible);
            return isTabVisible;
        }
        return false;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof MacroListPanel) {
            // TODO if more than a single instance is available this must be changed

            return true;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    private static final String WORKING_DIR_MACRO_EXCHANGE_KEY = "macroExchange";

    @Override
    public void exportMacro(final Macro macro) {

        if (CollectionUtils.isNotEmpty(macro.getFunctions()) && MacroUtils.hasEmptySteps(macro)) {
            // step with no function in macro, ask the user to export empty macro
            // show dialog
            int result =
                JOptionPane
                    .showConfirmDialog(getComponent(),
                        Resources.getString(MacroListPanel.class, "ask_continue_export_macro_with_empty_step"),
                        Resources.getString(MacroListPanel.class, "export_macro.title"), JOptionPane.OK_CANCEL_OPTION,
                        JOptionPane.WARNING_MESSAGE);
            if (result != JOptionPane.OK_OPTION) {
                LOGGER.info("User canceled export empty macro.");
                return;
            }

            LOGGER.info("Remove empty macro steps.");
            MacroUtils.removeEmptySteps(macro);

        }

        if (CollectionUtils.isEmpty(macro.getFunctions())) {
            // no functions in macro, ask the user to export empty macro
            // show dialog
            int result =
                JOptionPane
                    .showConfirmDialog(getComponent(),
                        Resources.getString(MacroListPanel.class, "ask_export_empty_macro"),
                        Resources.getString(MacroListPanel.class, "export_macro.title"), JOptionPane.OK_CANCEL_OPTION);
            if (result != JOptionPane.OK_OPTION) {
                LOGGER.info("User canceled export empty macro.");
                return;
            }
        }

        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_MACRO_EXCHANGE_KEY);
        FileDialog dialog = new MacroFileDialog(getComponent(), FileDialog.SAVE, storedWorkingDirectory, macro) {

            @Override
            public void approve(String selectedFile) {
                try {
                    boolean flatPortModel = false;
                    if (mainModel.getSelectedNode() != null
                        && mainModel.getSelectedNode().getNode().isPortFlatModelAvailable()) {
                        flatPortModel = true;
                    }
                    LOGGER.info("Set the flat port model flag: {}", flatPortModel);
                    macro.setFlatPortModel(flatPortModel);

                    ExportFormat exportFormat = ExportFormat.jaxb;
                    MacroFactory.saveMacro(selectedFile, macro, exportFormat);

                    // update the status bar
                    statusBar
                        .setStatusText(
                            String.format(Resources.getString(MacroListPanel.class, "exportedMacro"), selectedFile),
                            StatusBar.DISPLAY_NORMAL);

                    final String workingDir = Paths.get(selectedFile).toString();
                    LOGGER.info("Save current workingDir: {}", workingDir);

                    wizardSettings.setWorkingDirectory(WORKING_DIR_MACRO_EXCHANGE_KEY, workingDir);
                }
                catch (Exception ex) {
                    LOGGER.warn("Export macro failed.", ex);
                    // throw new RuntimeException(ex);

                    TaskDialogs
                        .build(JOptionPane.getFrameForComponent(getComponent()),
                            Resources.getString(MacroListPanel.class, "export_macro_failed.instruction"),
                            Resources.getString(MacroListPanel.class, "export_macro_failed", macro.toString()))
                        .title(Resources.getString(MacroListPanel.class, "export_macro.title")).exception(ex);
                }
            }

        };
        dialog.showDialog();
    }

    @Override
    public void importMacro() {

        if (mainModel.getSelectedMacro() != null
            && mainModel.getSelectedMacro().getMacroSaveState().equals(MacroSaveState.PENDING_CHANGES)) {

            // show dialog
            int result =
                JOptionPane
                    .showConfirmDialog(getComponent(),
                        Resources.getString(MacroListPanel.class, "macro_has_pending_changes"),
                        Resources.getString(MacroListPanel.class, "pending_changes"), JOptionPane.OK_CANCEL_OPTION);
            if (result != JOptionPane.OK_OPTION) {
                LOGGER.info("User canceled discard pending changes.");
                return;
            }

            // reset pending changes
            mainModel.getSelectedMacro().setMacroSaveState(MacroSaveState.SAVED_ON_NODE);
        }

        final NodeInterface selectedNode = mainModel.getSelectedNode();
        final WizardSettingsInterface wizardSettings = settingsService.getWizardSettings();
        String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_MACRO_EXCHANGE_KEY);

        FileDialog dialog = new MacroFileDialog(getComponent(), FileDialog.OPEN, storedWorkingDirectory, null) {

            @Override
            public void approve(String selectedFile) {
                ExportFormat exportFormat = ExportFormat.jaxb;

                // a new macro instance is created
                Macro macro = MacroFactory.loadMacro(selectedFile, exportFormat, selectedNode);

                LOGGER.info("Loaded macro, flatPortModel: {}", macro.isFlatPortModel());
                // set the new macro id
                macro.setId(mainModel.getSelectedMacro().getId());
                // set pending changes flag
                macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);

                // replace the imported macro
                replaceMacro(macro, false);

                final String workingDir = Paths.get(selectedFile).toString();
                LOGGER.info("Save current workingDir: {}", workingDir);

                wizardSettings.setWorkingDirectory(WORKING_DIR_MACRO_EXCHANGE_KEY, workingDir);
            }
        };
        dialog.showDialog();
    }

    // @Override
    public void replaceMacro(Macro macro, boolean saveOnNode) {
        LOGGER.info("Replace the macro: {}", macro);
        if (macro != null) {
            // set the total number possible functions for this macro

            final NodeInterface node = mainModel.getSelectedNode();
            macro.setFunctionSize(node.getMaxMacroSteps());

            if (mainModel.getSelectedMacro() != null
                && mainModel.getSelectedMacro().getMacroSaveState().equals(MacroSaveState.PENDING_CHANGES)) {
                // show dialog
                int result =
                    JOptionPane
                        .showConfirmDialog(getComponent(),
                            Resources.getString(MacroListPanel.class, "macro_has_pending_changes"),
                            Resources.getString(MacroListPanel.class, "pending_changes"), JOptionPane.OK_CANCEL_OPTION);
                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User canceled discard pending changes.");
                    return;
                }
            }

            // replace the macro in the main model
            mainModel.replaceMacro(macro);
        }

        if (saveOnNode) {
            LOGGER.info("Transfer the macro to node and save permanently: {}", macro);
            try {
                macroPanelListener.storeMacroOnNode(macro);

                LOGGER.info("Transfer and save macro passed.");
                // reset pending changes
                // macro.setMacroSaveState(false);
            }
            catch (RuntimeException ex) {
                LOGGER.warn("Transfer macro content or save macro failed.", ex);

                mainModel.setNodeHasError(mainModel.getSelectedNode(), true);

                if (ex.getCause() instanceof ProtocolNoAnswerException) {
                    TaskDialogs
                        .build(JOptionPane.getFrameForComponent(getComponent()),
                            Resources.getString(MacroListPanel.class, "save-macro-to-node-failed.instruction"),
                            Resources.getString(MacroListPanel.class, "save-macro-to-node-failed", macro.toString()))
                        .title(Resources.getString(MacroListPanel.class, "save-macro.title")).exception(ex);
                }
            }
        }
    }

    @Override
    public void labelChanged(final Macro macro, String label) {
        LOGGER.info("Label of macro has changed, macro: {}, label: {}", macro, label);

        saveChangedLabel(macro, label);
    }

    private void saveChangedLabel(final Macro macro, String label) {

        if (macro != null) {
            long uniqueId = mainModel.getSelectedNode().getNode().getUniqueId();

            try {
                final NodeLabels nodeLabels = this.wizardLabelWrapper.loadLabels(uniqueId);
                BidibLabelUtils.replaceMacroLabel(nodeLabels, macro.getId(), label);
                this.wizardLabelWrapper.saveNodeLabels(uniqueId);
            }
            catch (Exception e) {
                LOGGER.warn("Save macro labels failed.", e);
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public void reloadMacro(Macro macro) {

        if (macro != null && macro.getMacroSaveState().equals(MacroSaveState.PENDING_CHANGES)) {

            // show dialog
            int result =
                JOptionPane
                    .showConfirmDialog(getComponent(),
                        Resources.getString(MacroListPanel.class, "macro_has_pending_changes"),
                        Resources.getString(MacroListPanel.class, "pending_changes"), JOptionPane.OK_CANCEL_OPTION);
            if (result != JOptionPane.OK_OPTION) {
                LOGGER.info("User canceled discard pending changes.");
                return;
            }

            // reset pending changes
            macro.setMacroSaveState(MacroSaveState.SAVED_ON_NODE);
        }

        try {
            LcMacroState lcMacroState = macroPanelListener.reloadMacro(macro);
            LOGGER.info("Reload macro returned: {}", lcMacroState);

            mainModel.getSelectedMacro().setMacroSaveState(MacroSaveState.PERMANENTLY_STORED_ON_NODE);
            macro.setContainsError(false);
        }
        catch (InvalidConfigurationException ex) {
            LOGGER.warn("Restore macro content failed.", ex);

            mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
        }
    }

    @Override
    public void saveMacro(Macro macro) {
        LOGGER.info("Save macro: {}", macro);
        try {
            // the macro save state is reset by the saveMacro call

            LcMacroState lcMacroState = macroPanelListener.saveMacro(macro);
            LOGGER.info("Save macro returned: {}", lcMacroState);

            macro.setContainsError(false);
        }
        catch (RuntimeException ex) {
            LOGGER.warn("Save macro content failed.", ex);

            if (ex.getCause() instanceof ProtocolNoAnswerException) {
                TaskDialogs
                    .build(JOptionPane.getFrameForComponent(getComponent()),
                        Resources.getString(MacroListPanel.class, "save-macro-to-node-failed.instruction"),
                        Resources.getString(MacroListPanel.class, "save-macro-to-node-failed", macro.toString()))
                    .title(Resources.getString(MacroListPanel.class, "save-macro.title")).exception(ex);
            }

            mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
        }
    }

    @Override
    public void startMacro(Macro macro, boolean transferBeforeStart) {
        LOGGER.info("Start macro: {}, transferBeforeStart: {}", macro, transferBeforeStart);
        LcMacroState lcMacroState = macroPanelListener.startMacro(macro, transferBeforeStart);
        LOGGER.info("Start macro returned: {}", lcMacroState);
    }

    @Override
    public void stopMacro(Macro macro) {
        LOGGER.info("Stop macro: {}", macro);
        LcMacroState lcMacroState = macroPanelListener.stopMacro(macro);
        LOGGER.info("Stop macro returned: {}", lcMacroState);
    }

    @Override
    public void transferMacro(Macro macro) {
        LOGGER.info("Transfer macro: {}", macro);
        try {
            macroPanelListener.transferMacro(macro);
            LOGGER.info("Transfer macro passed.");
        }
        catch (InvalidConfigurationException ex) {
            LOGGER.warn("Transfer macro content failed.", ex);

            mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
        }
    }

    @Override
    public void initializeMacro(Macro macro) {
        // and force the panels to reload
        try {
            if (mainModel.getSelectedMacro() != null && mainModel.getSelectedMacro().equals(macro)
                && mainModel.getSelectedMacro().getMacroSaveState().equals(MacroSaveState.PENDING_CHANGES)) {
                // show dialog
                int result =
                    JOptionPane
                        .showConfirmDialog(getComponent(),
                            Resources.getString(MacroListPanel.class, "macro_has_pending_changes"),
                            Resources.getString(MacroListPanel.class, "pending_changes"), JOptionPane.OK_CANCEL_OPTION);
                if (result != JOptionPane.OK_OPTION) {
                    LOGGER.info("User canceled discard pending changes.");
                    return;
                }

                // reset pending changes
                mainModel.getSelectedMacro().setMacroSaveState(MacroSaveState.SAVED_ON_NODE);
            }
            // initialize the macro
            macro.initialize();
            macro.setMacroSaveState(MacroSaveState.PENDING_CHANGES);
            macro.setLabel(null);
            // save the label
            saveChangedLabel(macro, null);

            mainModel.setSelectedMacro(macro);
        }
        catch (Exception ex) {
            LOGGER.warn("Set the new selcted macro failed.", ex);
        }
    }

    @Override
    public boolean hasPendingChanges() {

        NodeInterface node = mainModel.getSelectedNode();
        if (node != null) {
            List<Macro> macros = node.getMacros();

            for (Macro macro : macros) {

                if (MacroSaveState.PENDING_CHANGES == macro.getMacroSaveState()) {
                    LOGGER.info("Found pending changes in macro: {}", macro);
                    return true;
                }
            }
        }

        return false;
    }

    public void addLabelChangedListener(LabelChangedListener<Macro> l) {
        labelChangedListeners.add(l);
    }

    public void addListSelectionListener(ListSelectionListener l) {
        macroList.addListSelectionListener(l);
    }

    protected void fireLabelChanged(Macro macro, String label) {
        labelChanged(macro, label);

        macroList.refreshView();

        for (LabelChangedListener<Macro> l : labelChangedListeners) {
            l.labelChanged(macro, label);
        }
    }

    @Override
    public void tabSelected(boolean selected) {
        LOGGER.info("Tab is selected: {}", selected);

        if (selected) {
            if (macroList.getCellEditor() != null) {
                macroList.getCellEditor().stopCellEditing();
            }
        }
    }

    @Override
    public void editLabel(final MouseEvent popupEvent) {
        SwingUtilities.invokeLater(() -> {

            int selectedRow = macroList.getSelectedRow();
            if (selectedRow > -1) {
                LOGGER.info(">>> Edit row, selectedRow: {}", selectedRow);

                macroList.requestFocus();

                macroList.editCellAt(selectedRow, 0);

                // select the text in the editor
                final TableCellEditor editor = macroList.getCellEditor();
                if (editor != null) {
                    SwingUtilities.invokeLater(() -> {

                        JTextField textField = (JTextField) ((DefaultTextCellEditor) editor).getComponent();

                        textField.requestFocusInWindow();
                    });
                }
            }
        });
    }

    protected void handleMouseEvent(MouseEvent e, JPopupMenu popupMenu) {
        if (e.isPopupTrigger() && macroList.getRowCount() > 0) {

            if (macroList.getCellEditor() != null) {
                macroList.getCellEditor().stopCellEditing();
            }

            int selectRow = macroList.rowAtPoint(e.getPoint());

            macroList.setRowSelectionInterval(selectRow, selectRow);

            SwingUtilities.invokeLater(() -> popupMenu.show(e.getComponent(), e.getX(), e.getY()));
        }
    }

    public void refreshView() {

        macroList.refreshView();
    }

}
