package org.bidib.wizard.mvc.main.view.table;

import java.awt.Color;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.Int16PortConfigValue;
import org.bidib.jbidibc.messages.port.RgbPortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.component.SliderAndValuePanel;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.ColorRGBToIntegerConverter;
import org.bidib.wizard.client.common.converter.StringToDmxChannelConverter;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardBindings;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.renderer.BidibStatusListRenderer;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.LightPortStatus;
import org.bidib.wizard.mvc.main.model.LightPortTableModel;
import org.bidib.wizard.mvc.main.view.table.converter.PortStatusToStringConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;
import com.jidesoft.combobox.ColorExComboBox;
import com.jidesoft.converter.ColorConverter;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class LightPortEditorPanel extends AbstractPortEditorPanel<LightPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(LightPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, max(60dlu;pref), 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_STATUS_EVENTS_MILLIS = 80;

    private static final int TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS = 100;

    private ValueModel portLevelOffConverterModel;

    private ValueModel portLevelOnConverterModel;

    private ValueModel dimmDownConverterModel;

    private ValueModel dimmUpConverterModel;

    private ValueModel rgbValueModel;

    private ValueModel transitionTimeConverterModel;

    private ValueModel dmxMappingConverterModel;

    private JComboBox<LightPortStatus> comboPortStatus;

    private PublishSubject<LightPortStatus> statusEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> localPortConfigChangeEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject;

    public LightPortEditorPanel(LightPort port, Consumer<LightPort> saveCallback,
        final Consumer<LightPort> valueCallback, final Consumer<LightPort> refreshCallback,
        final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject,
        final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, nodeSelectionProvider);

        this.portConfigChangeEventSubject = portConfigChangeEventSubject;
    }

    @Override
    protected LightPort clonePort(final LightPort port) {
        // create a clone of the input port
        final LightPort clone =
            LightPort
                .builder() //
                .withDimMax(port.getDimMax()) //
                .withDimMin(port.getDimMin()) //
                .withPwmMax(port.getPwmMax()) //
                .withPwmMin(port.getPwmMin()) //
                .withRgbValue(port.getRgbValue()) //
                .withTransitionTime(port.getTransitionTime()) //
                .withDmxMapping(port.getDmxMapping()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();

        // security check
        if (clone.getDimMax() == 0) {
            clone.setDimMax(1);
        }
        if (clone.getDimMin() == 0) {
            clone.setDimMin(1);
        }

        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final LightPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(LightPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(LightPortTableModel.class, "label") + ":").xy(1, row);
        final JTextField portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        dialogBuilder.add(portName).xyw(3, row, 7);

        row += 2;

        // if port level off is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_PWM_MIN);

            // port level off
            this.portLevelOffConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            JTextField portLevelOffText = new JTextField();
            final IntegerInputValidationDocument portLevelOffDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            portLevelOffDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            portLevelOffText.setDocument(portLevelOffDocument);
            portLevelOffText.setColumns(2);
            portLevelOffText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(portLevelOffText, portLevelOffConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "portLevelOff")).xy(1, row);
            dialogBuilder.add(portLevelOffText).xy(3, row);

            portLevelOffText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(portLevelOffText, true);
            ValidationComponentUtils.setMessageKeys(portLevelOffText, "validation.portLevelOff_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, 255, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, valueValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int portLevelOffValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The portLevelOff value has been changed: {}", portLevelOffValue);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_LEVEL_PORT_OFF,
                            new BytePortConfigValue(ByteUtils.getLowByte(portLevelOffValue))));
                }
            });

            row += 2;
        }

        // if port level on is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_PWM_MAX);

            // PWM max
            this.portLevelOnConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            JTextField portLevelOnText = new JTextField();
            final IntegerInputValidationDocument portLevelOnDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            portLevelOnDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            portLevelOnText.setDocument(portLevelOnDocument);
            portLevelOnText.setColumns(2);
            portLevelOnText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(portLevelOnText, portLevelOnConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "portLevelOn")).xy(1, row);
            dialogBuilder.add(portLevelOnText).xy(3, row);

            portLevelOnText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(portLevelOnText, true);
            ValidationComponentUtils.setMessageKeys(portLevelOnText, "validation.portLevelOn_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(0, 255, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, valueValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int portLevelOnValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The portLevelOn value has been changed: {}", portLevelOnValue);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_LEVEL_PORT_ON,
                            new BytePortConfigValue(ByteUtils.getLowByte(portLevelOnValue))));
                }
            });

            row += 2;
        }

        // if dimm min is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DIM_MIN);

            // dimm min
            this.dimmDownConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8) ? 65535 : 255;

            JTextField dimmDownText = new JTextField();
            final IntegerInputValidationDocument dimmDownDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmDownDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(1, maxRange));
            dimmDownText.setDocument(dimmDownDocument);
            dimmDownText.setColumns(2);
            dimmDownText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmDownText, dimmDownConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dimmDown")).xy(1, row);
            dialogBuilder.add(dimmDownText).xy(3, row);

            dimmDownText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmDownText, true);
            ValidationComponentUtils.setMessageKeys(dimmDownText, "validation.dimmDown_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(1, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, valueValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8);

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int dimmDownValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The dimmDown value has been changed: {}", dimmDownValue);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port,
                            dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_DOWN,
                            dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmDownValue))
                                : new BytePortConfigValue(ByteUtils.getLowByte(dimmDownValue))));
                }
            });

            row += 2;
        }

        // if dimm max is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)
            || isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {

            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DIM_MAX);

            // dimm max
            this.dimmUpConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            int maxRange = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8) ? 65535 : 255;

            JTextField dimmUpText = new JTextField();
            final IntegerInputValidationDocument dimmUpDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            dimmUpDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(1, maxRange));
            dimmUpText.setDocument(dimmUpDocument);
            dimmUpText.setColumns(2);
            dimmUpText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dimmUpText, dimmUpConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dimmUp")).xy(1, row);
            dialogBuilder.add(dimmUpText).xy(3, row);

            dimmUpText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(dimmUpText, true);
            ValidationComponentUtils.setMessageKeys(dimmUpText, "validation.dimmUp_key");

            // use the clone port to provide the values
            final SliderAndValuePanel valueSlider = new SliderAndValuePanel(1, maxRange, 10);
            valueSlider.createComponent();
            WizardBindings.bind(valueSlider, valueValueModel);

            dialogBuilder.add(valueSlider).xyw(5, row, 5);

            final boolean dimm_8_8 = isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8);

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int dimmUpValue = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The dimmUp value has been changed: {}", dimmUpValue);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port,
                            dimm_8_8 ? BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8 : BidibLibrary.BIDIB_PCFG_DIMM_UP,
                            dimm_8_8 ? new Int16PortConfigValue(ByteUtils.getWORD(dimmUpValue))
                                : new BytePortConfigValue(ByteUtils.getLowByte(dimmUpValue))));
                }
            });

            row += 2;
        }

        // if RGB value is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_RGB)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_RGB_VALUE);

            this.rgbValueModel = new ConverterValueModel(valueValueModel, new ColorRGBToIntegerConverter());

            final ColorExComboBox rgbColorComboBox = new ColorExComboBox();
            rgbColorComboBox.setConverterContext(ColorConverter.CONTEXT_RGB);
            rgbColorComboBox.setColorValueVisible(false);

            rgbColorComboBox.setSelectedColor((Color) rgbValueModel.getValue());

            rgbColorComboBox.addItemListener(new ItemListener() {
                @Override
                public void itemStateChanged(ItemEvent e) {
                    Color selectedColor = rgbColorComboBox.getSelectedColor();

                    rgbValueModel.setValue(selectedColor);
                }
            });

            this.rgbValueModel.addValueChangeListener((evt) -> {
                LOGGER.info("Set the color: {}", evt.getNewValue());

                rgbColorComboBox.setSelectedColor((Color) evt.getNewValue());
            });

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "rgbValue")).xy(1, row);
            dialogBuilder.add(rgbColorComboBox).xy(3, row);

            ValidationComponentUtils.setMandatory(rgbColorComboBox, true);
            ValidationComponentUtils.setMessageKeys(rgbColorComboBox, "validation.rgbValue_key");

            row += 2;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int rgbValue = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The rgb value has been changed: {}", rgbValue);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_RGB,
                            new RgbPortConfigValue(ByteUtils.getRGB(rgbValue))));
                }
            });
        }

        // if transition time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TRANSITION_TIME)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_TRANSITION_TIME);

            this.transitionTimeConverterModel =
                new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            JTextField transitionTimeText = new JTextField();
            final IntegerInputValidationDocument transitionTimeDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            transitionTimeDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            transitionTimeText.setDocument(transitionTimeDocument);
            transitionTimeText.setColumns(2);
            transitionTimeText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(transitionTimeText, transitionTimeConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "transitionTime")).xy(1, row);
            dialogBuilder.add(transitionTimeText).xy(3, row);

            ValidationComponentUtils.setMandatory(transitionTimeText, true);
            ValidationComponentUtils.setMessageKeys(transitionTimeText, "validation.transitionTime_key");

            row += 2;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int transitionTime = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The transitionTime value has been changed: {}", transitionTime);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_TRANSITION_TIME,
                            new Int16PortConfigValue(transitionTime)));
                }
            });
        }

        // if DMX mapping is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)) {
            final BufferedValueModel valueValueModel =
                getPresentationModel().getBufferedModel(LightPort.PROPERTY_DMX_MAPPING);

            this.dmxMappingConverterModel = new ConverterValueModel(valueValueModel, new StringToDmxChannelConverter());

            JTextField dmxMappingText = new JTextField();
            final IntegerInputValidationDocument dmxMappingDocument =
                new IntegerInputValidationDocument(3, InputValidationDocument.NUMERIC);
            dmxMappingDocument
                .setRangeValidationCallback(new DefaultRangeValidationCallback(MIN_DMX_MAPPING_VALUE, 255));
            dmxMappingText.setDocument(dmxMappingDocument);
            dmxMappingText.setColumns(2);
            dmxMappingText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(dmxMappingText, dmxMappingConverterModel, false);

            dialogBuilder.add(Resources.getString(LightPortTableModel.class, "dmxMapping")).xy(1, row);
            dialogBuilder.add(dmxMappingText).xy(3, row);

            ValidationComponentUtils.setMandatory(dmxMappingText, true);
            ValidationComponentUtils.setMessageKeys(dmxMappingText, "validation.dmxMapping_key");

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int outputMap = ((Integer) valueValueModel.getValue()).intValue();

                    LOGGER.info("The outputMap value has been changed: {}", outputMap);

                    this.localPortConfigChangeEventSubject
                        .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_OUTPUT_MAP,
                            new BytePortConfigValue(ByteUtils.getLowByte(outputMap))));
                }
            });

            row += 2;
        }

        // change the status
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(LightPort.PROPERTY_STATUS);

        final ConverterValueModel portStatusConverterModel =
            new ConverterValueModel(selectionHolderPortStatus,
                new PortStatusToStringConverter(LightPortStatus.class, "status."));

        final JLabel statusLabel = WizardComponentFactory.createLabel(portStatusConverterModel);

        final SelectionInList<LightPortStatus> portStatusSelection =
            new SelectionInList<>(LightPortStatus.ON.getValues());

        final ValueModel portStatusHolder = new ValueHolder(selectionHolderPortStatus.getValue());
        ComboBoxAdapter<LightPortStatus> comboBoxAdapterPortStatus =
            new ComboBoxAdapter<>(portStatusSelection, portStatusHolder);
        this.comboPortStatus = new JComboBox<>();
        this.comboPortStatus.setRenderer(new BidibStatusListRenderer<>(LightPortStatus.class));
        this.comboPortStatus.setModel(comboBoxAdapterPortStatus);

        LightPortStatus oppositeStatus =
            PortUtils.getOppositeStatus((LightPortStatus) selectionHolderPortStatus.getValue());
        LOGGER.debug("Set the opposite status in the port status selection: {}", oppositeStatus);
        this.comboPortStatus.setSelectedItem(oppositeStatus);

        final JButton btnPortStatus = new JButton(Resources.getString(LightPortTableModel.class, "test"));

        btnPortStatus.addActionListener(evt -> {
            LightPortStatus portStatus = (LightPortStatus) portStatusHolder.getValue();
            LOGGER.info("The port status has been changed: {}", portStatus);

            statusEventSubject.onNext(portStatus);
        });

        // throttle the status events
        final Disposable disp =
            this.statusEventSubject
                .throttleLatest(TIME_BETWEEN_STATUS_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(portStatus -> sendStatusToPort(port, portStatus));
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(LightPortTableModel.class, "status") + ":").xy(1, row);
        dialogBuilder.add(statusLabel).xy(3, row);
        dialogBuilder.add(this.comboPortStatus).xy(5, row);
        dialogBuilder.add(btnPortStatus).xy(7, row);

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 5);

        // combine the port config events
        final Disposable dispLocalPortConfigChange =
            this.localPortConfigChangeEventSubject
                .buffer(TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance())
                .subscribe(evts -> {
                    // combine the values
                    PortConfigChangeEvent event = null;
                    for (PortConfigChangeEvent evt : evts) {
                        LOGGER.info("Process event: {}", evt);
                        if (event == null) {
                            event = evt;
                            continue;
                        }
                        event.getPortConfig().putAll(evt.getPortConfig());
                    }

                    if (event != null) {
                        LOGGER.info("Publish the config change event to the node: {}", event);
                        portConfigChangeEventSubject.onNext(event);
                    }
                });
        getCompDisp().add(dispLocalPortConfigChange);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        if (this.portLevelOffConverterModel != null) {
            this.portLevelOffConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.portLevelOnConverterModel != null) {
            this.portLevelOnConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmDownConverterModel != null) {
            this.dimmDownConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dimmUpConverterModel != null) {
            this.dimmUpConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.rgbValueModel != null) {
            this.rgbValueModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.transitionTimeConverterModel != null) {
            this.transitionTimeConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.dmxMappingConverterModel != null) {
            this.dmxMappingConverterModel.addValueChangeListener(evt -> triggerValidation());
        }

        enableComponents();

        triggerValidation();

        return getPanel();
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case LightPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
                        updatePortStatusFromOriginalPort();
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        if (getOriginalPort().getPortType() == LcOutputType.LIGHTPORT) {
                            LOGGER.info("Current port type is LIGHTPORT.");
                            updatePortStatusFromOriginalPort();
                            enableComponents();
                        }
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    private void updatePortStatusFromOriginalPort() {
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(LightPort.PROPERTY_STATUS);
        LightPortStatus status = getOriginalPort().getStatus();
        LOGGER.info("Current status of original port: {}", status);
        selectionHolderPortStatus.setValue(status);

        if (LightPortEditorPanel.this.comboPortStatus != null) {
            LightPortStatus oppositeStatus =
                PortUtils.getOppositeStatus((LightPortStatus) selectionHolderPortStatus.getValue());
            LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
            LightPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
        }
    }

    @Override
    protected ValidationResult validate(final LightPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.portLevelOffConverterModel != null && this.portLevelOffConverterModel.getValue() == null) {
            support.addError("portLevelOff_key", "not_empty_for_write");
        }
        if (this.portLevelOnConverterModel != null && this.portLevelOnConverterModel.getValue() == null) {
            support.addError("portLevelOn_key", "not_empty_for_write");
        }
        if (this.dimmDownConverterModel != null && this.dimmDownConverterModel.getValue() == null) {
            support.addError("dimmDown_key", "not_empty_for_write");
        }
        if (this.dimmUpConverterModel != null && this.dimmUpConverterModel.getValue() == null) {
            support.addError("dimmUp_key", "not_empty_for_write");
        }
        if (this.rgbValueModel != null && this.rgbValueModel.getValue() == null) {
            support.addError("rgbValue_key", "not_empty_for_write");
        }
        if (this.transitionTimeConverterModel != null && this.transitionTimeConverterModel.getValue() == null) {
            support.addError("transitionTime_key", "not_empty_for_write");
        }
        if (this.dmxMappingConverterModel != null && this.dmxMappingConverterModel.getValue() == null) {
            support.addError("dmxMapping_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    private void sendStatusToPort(final LightPort port, LightPortStatus portStatus) {
        LOGGER.info("Send the new portStatus to the port: {}", portStatus);

        triggerValidation();

        final LightPort lightPort = LightPort.builder().withStatus(portStatus).withId(port.getId()).build();

        getValueCallback().accept(lightPort);
    }

}
