package org.bidib.wizard.mvc.main.view.table;

import java.beans.PropertyChangeEvent;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;

import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.LoadTypeEnum;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.client.common.controller.NodeSelectionProvider;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.rxjava2.SwingScheduler;
import org.bidib.wizard.client.common.table.AbstractPortEditorPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.renderer.BidibStatusListRenderer;
import org.bidib.wizard.client.common.view.validation.DefaultRangeValidationCallback;
import org.bidib.wizard.client.common.view.validation.IconFeedbackPanel;
import org.bidib.wizard.client.common.view.validation.IntegerInputValidationDocument;
import org.bidib.wizard.client.common.view.validation.PropertyValidationI18NSupport;
import org.bidib.wizard.model.ports.GenericPort;
import org.bidib.wizard.model.ports.SwitchPairPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.model.status.SwitchPortStatus;
import org.bidib.wizard.mvc.main.model.SwitchPairPortTableModel;
import org.bidib.wizard.mvc.main.view.panel.renderer.LoadTypeCellRenderer;
import org.bidib.wizard.mvc.main.view.table.converter.PortStatusToStringConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueHolder;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.common.collect.ArrayListModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jgoodies.validation.ValidationResult;
import com.jgoodies.validation.util.PropertyValidationSupport;
import com.jgoodies.validation.view.ValidationComponentUtils;

import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class SwitchPairPortEditorPanel extends AbstractPortEditorPanel<SwitchPairPort> {

    private static final Logger LOGGER = LoggerFactory.getLogger(SwitchPairPortEditorPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private static final int TIME_BETWEEN_STATUS_EVENTS_MILLIS = 80;

    private static final int TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS = 100;

    private ValueModel switchOffTimeConverterModel;

    private ValueModel selectionHolderLoadType;

    private JTextField portName;

    private JTextField switchOffTimeText;

    private JComboBox<LoadTypeEnum> comboLoadType;

    private JComboBox<SwitchPortStatus> comboPortStatus;

    private JButton btnPortStatus;

    private PublishSubject<SwitchPortStatus> statusEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> localPortConfigChangeEventSubject = PublishSubject.create();

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject;

    public SwitchPairPortEditorPanel(SwitchPairPort port, Consumer<SwitchPairPort> saveCallback,
                                     final Consumer<SwitchPairPort> valueCallback, final Consumer<SwitchPairPort> refreshCallback,
                                     final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject, final NodeSelectionProvider nodeSelectionProvider) {
        super(port, saveCallback, valueCallback, refreshCallback, nodeSelectionProvider);

        this.portConfigChangeEventSubject = portConfigChangeEventSubject;
    }

    @Override
    protected SwitchPairPort clonePort(final SwitchPairPort port) {
        // create a clone of the input port
        final SwitchPairPort clone =
            SwitchPairPort
                .builder() //
                .withLoadType(port.getLoadType()) //
                .withSwitchOffTime(port.getSwitchOffTime()) //
                .withStatus(port.getStatus()) //
                .withRemappingEnabled(port.isRemappingEnabled()) //
                .withKnownPortConfigKeys(port.getKnownPortConfigKeys()) //
                .withId(port.getId()) //
                .withLabel(port.getLabel()) //
                .withEnabled(port.isEnabled()) //
                .withIsInactive(port.isInactive()) //
                .withPortIdentifier(port.getPortIdentifier()).build();
        return clone;
    }

    @Override
    protected JPanel doCreateComponent(final SwitchPairPort port) {

        int row = 1;
        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new PortEditorPanelDebugContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new PortEditorPanelContainer(this);
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // port name
        final BufferedValueModel bufferedPortNameModel =
            getPresentationModel().getBufferedModel(SwitchPairPort.PROPERTY_LABEL);

        dialogBuilder.add(Resources.getString(SwitchPairPortTableModel.class, "label") + ":").xy(1, row);
        this.portName = WizardComponentFactory.createTextField(bufferedPortNameModel, false);
        this.portName.setEnabled(port.isEnabled());
        dialogBuilder.add(this.portName).xyw(3, row, 7);

        row += 2;

        // if switchOff time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_TICKS)) {

            // switchoff time
            final BufferedValueModel valueValueModel = getPresentationModel().getBufferedModel(SwitchPairPort.PROPERTYNAME_SWITCHOFFTIME);
            this.switchOffTimeConverterModel = new ConverterValueModel(valueValueModel, new StringToIntegerConverter());

            this.switchOffTimeText = new JTextField();
            final IntegerInputValidationDocument switchOffTimeDocument =
                new IntegerInputValidationDocument(5, InputValidationDocument.NUMERIC);
            switchOffTimeDocument.setRangeValidationCallback(new DefaultRangeValidationCallback(0, 255));
            this.switchOffTimeText.setDocument(switchOffTimeDocument);
            this.switchOffTimeText.setColumns(2);
            this.switchOffTimeText.setHorizontalAlignment(JTextField.RIGHT);

            // bind manually because we changed the document of the textfield
            Bindings.bind(this.switchOffTimeText, switchOffTimeConverterModel, false);

            dialogBuilder.add(Resources.getString(SwitchPairPortTableModel.class, "switchOffTime") + ":").xy(1, row);
            dialogBuilder.add(this.switchOffTimeText).xy(3, row);

            this.switchOffTimeText.setEnabled(port.isEnabled());

            ValidationComponentUtils.setMandatory(this.switchOffTimeText, true);
            ValidationComponentUtils.setMessageKeys(this.switchOffTimeText, "validation.switchOffTime_key");

            row += 2;

            valueValueModel.addValueChangeListener(evt -> {
                if (valueValueModel.getValue() instanceof Integer) {
                    int switchOffTimeValue = ((Integer) valueValueModel.getValue()).intValue();
                    LOGGER.info("The switchOffTime value has been changed: {}", switchOffTimeValue);

                    this.localPortConfigChangeEventSubject
                            .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_TICKS,
                                    new BytePortConfigValue(ByteUtils.getLowByte(switchOffTimeValue))));
                }
            });
        }

        // if loadType time is not available we must not show it here
        if (isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_LOAD_TYPE)) {
            // loadType
            this.selectionHolderLoadType = getPresentationModel().getBufferedModel(SwitchPairPort.PROPERTYNAME_LOADTYPE);

            final ArrayListModel<LoadTypeEnum> loadTypeList = new ArrayListModel<>();
            for (LoadTypeEnum value : LoadTypeEnum.values()) {
                loadTypeList.add(value);
            }

            final SelectionInList<LoadTypeEnum> loadTypeSelection =
                new SelectionInList<>((ListModel<LoadTypeEnum>) loadTypeList);

            final ComboBoxAdapter<LoadTypeEnum> comboBoxAdapterLoadType =
                new ComboBoxAdapter<LoadTypeEnum>(loadTypeSelection, selectionHolderLoadType);
            this.comboLoadType = new JComboBox<>();
            this.comboLoadType.setModel(comboBoxAdapterLoadType);
            this.comboLoadType.setRenderer(new LoadTypeCellRenderer());

            this.comboLoadType.setEnabled(port.isEnabled());

            dialogBuilder.add(Resources.getString(SwitchPairPortTableModel.class, "loadType") + ":").xy(1, row);
            dialogBuilder.add(this.comboLoadType).xy(3, row);

            row += 2;

            selectionHolderLoadType.addValueChangeListener(evt -> {
                if (selectionHolderLoadType.getValue() instanceof Integer) {
                    int loadTypeValue = ((Integer) selectionHolderLoadType.getValue()).intValue();
                    LOGGER.info("The loadType value has been changed: {}", loadTypeValue);

                    this.localPortConfigChangeEventSubject
                            .onNext(new PortConfigChangeEvent(null, 0L, port, BidibLibrary.BIDIB_PCFG_LOAD_TYPE,
                                    new BytePortConfigValue(ByteUtils.getLowByte(loadTypeValue))));
                }
            });
        }

        // change the status
        final ValueModel selectionHolderPortStatus = getPresentationModel().getModel(SwitchPairPort.PROPERTY_STATUS);

        final ConverterValueModel portStatusConverterModel =
            new ConverterValueModel(selectionHolderPortStatus, new PortStatusToStringConverter(SwitchPairPortTable.STATUS_RESOURCE_KEY, "status."));

        final JLabel statusLabel = WizardComponentFactory.createLabel(portStatusConverterModel);

        final SelectionInList<SwitchPortStatus> portStatusSelection =
            new SelectionInList<>(SwitchPortStatus.ON.getValues());

        // create a holder for the status
        final ValueModel portStatusHolder = new ValueHolder(selectionHolderPortStatus.getValue());
        ComboBoxAdapter<SwitchPortStatus> comboBoxAdapterPortStatus =
            new ComboBoxAdapter<>(portStatusSelection, portStatusHolder);
        this.comboPortStatus = new JComboBox<>();
        this.comboPortStatus
            .setRenderer(new BidibStatusListRenderer<>(SwitchPairPortTable.STATUS_RESOURCE_KEY, "status."));
        this.comboPortStatus.setModel(comboBoxAdapterPortStatus);

        SwitchPortStatus oppositeStatus =
            PortUtils.getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
        LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
        this.comboPortStatus.setSelectedItem(oppositeStatus);

        this.btnPortStatus = new JButton(Resources.getString(SwitchPairPortTableModel.class, "test"));

        this.btnPortStatus.addActionListener(evt -> {
            SwitchPortStatus portStatus = (SwitchPortStatus) portStatusHolder.getValue();
            LOGGER.info("The port status has been changed: {}", portStatus);

            statusEventSubject.onNext(portStatus);
        });

        final Disposable disp =
            this.statusEventSubject
                .throttleLatest(TIME_BETWEEN_STATUS_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance(),
                    true)
                .subscribe(portStatus -> sendStatusToPort(port, portStatus));
        getCompDisp().add(disp);

        dialogBuilder.add(Resources.getString(SwitchPairPortTableModel.class, "status") + ":").xy(1, row);
        dialogBuilder.add(statusLabel).xy(3, row);
        dialogBuilder.add(this.comboPortStatus).xy(5, row);
        dialogBuilder.add(this.btnPortStatus).xy(7, row);

        this.comboPortStatus.setEnabled(port.isEnabled());
        this.btnPortStatus.setEnabled(port.isEnabled());

        row += 2;

        // add buttons
        final JPanel buttonPanel = createButtonPanel();
        dialogBuilder.add(buttonPanel).xyw(1, row, 9);

        // combine the port config events
        final Disposable dispLocalPortConfigChange =
                this.localPortConfigChangeEventSubject
                        .buffer(TIME_BETWEEN_PORT_CONFIG_EVENTS_MILLIS, TimeUnit.MILLISECONDS, SwingScheduler.getInstance())
                        .subscribe(evts -> {
                            // combine the values
                            PortConfigChangeEvent event = null;
                            for (PortConfigChangeEvent evt : evts) {
                                LOGGER.info("Process event: {}", evt);
                                if (event == null) {
                                    event = evt;
                                    continue;
                                }
                                event.getPortConfig().putAll(evt.getPortConfig());
                            }

                            if (event != null) {
                                LOGGER.info("Publish the config change event to the node: {}", event);
                                portConfigChangeEventSubject.onNext(event);
                            }
                        });
        getCompDisp().add(dispLocalPortConfigChange);

        // check if we have validation enabled
        if (getValidationResultModel() != null) {
            LOGGER.debug("Create iconfeedback panel.");
            JComponent cvIconPanel = new IconFeedbackPanel(getValidationResultModel(), dialogBuilder.build());
            JPanel panel = new PortEditorPanelContainer(this);
            FormBuilder feedbackBuilder = FormBuilder.create().columns("p:g").rows("fill:p:grow").panel(panel);

            feedbackBuilder.add(cvIconPanel).xy(1, 1);

            setPanel(feedbackBuilder.build());
        }
        else {
            setPanel(dialogBuilder.build());
        }

        bufferedPortNameModel.addValueChangeListener(evt -> triggerValidation());
        if (this.switchOffTimeConverterModel != null) {
            this.switchOffTimeConverterModel.addValueChangeListener(evt -> triggerValidation());
        }
        if (this.selectionHolderLoadType != null) {
            this.selectionHolderLoadType.addValueChangeListener(evt -> triggerValidation());
        }

        enableComponents();

        triggerValidation();

        return getPanel();
    }

    @Override
    protected void propertyChanged(final PropertyChangeEvent evt) {
        LOGGER
            .info("The port property has been changed, propertyName: {}, new value: {}", evt.getPropertyName(),
                evt.getNewValue());

        super.propertyChanged(evt);

        SwingUtilities.invokeLater(() -> {
            try {
                // the status of the port was changed
                switch (evt.getPropertyName()) {
                    case SwitchPairPort.PROPERTY_STATUS:
                    case GenericPort.PROPERTY_PORT_STATUS:
//                        final ValueModel selectionHolderPortStatus =
//                            getPresentationModel().getModel(SwitchPairPort.PROPERTY_STATUS);
//                        SwitchPortStatus status = getOriginalPort().getStatus();
//                        LOGGER.info("Current status of original port: {}", status);
//                        selectionHolderPortStatus.setValue(status);
//
//                        if (SwitchPairPortEditorPanel.this.comboPortStatus != null) {
//                            SwitchPortStatus oppositeStatus =
//                                PortUtils.getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
//                            LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
//                            SwitchPairPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
//                        }
                        updatePortStatusFromOriginalPort();
                        break;
                    case GenericPort.PROPERTY_PORT_CONFIG_CHANGED:
                        break;
                    case GenericPort.PROPERTY_PORT_TYPE_CHANGED:
                        LOGGER.info("The port type has changed: {}", evt.getNewValue());
                        if (getOriginalPort().getPortType() == LcOutputType.SWITCHPAIRPORT) {
                            LOGGER.info("Current port type is SWITCHPAIRPORT.");
                            updatePortStatusFromOriginalPort();
                            enableComponents();
                        }
                        break;

                    default:
                        break;
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Update the status failed.", ex);
            }
        });
    }

    private void updatePortStatusFromOriginalPort() {
        final ValueModel selectionHolderPortStatus =
                getPresentationModel().getModel(SwitchPairPort.PROPERTY_STATUS);
        SwitchPortStatus status = getOriginalPort().getStatus();
        LOGGER.info("Current status of original port: {}", status);
        selectionHolderPortStatus.setValue(status);

        if (SwitchPairPortEditorPanel.this.comboPortStatus != null) {
            SwitchPortStatus oppositeStatus =
                    PortUtils.getOppositeStatus((SwitchPortStatus) selectionHolderPortStatus.getValue());
            LOGGER.info("Set the opposite status in the port status selection: {}", oppositeStatus);
            SwitchPairPortEditorPanel.this.comboPortStatus.setSelectedItem(oppositeStatus);
        }
    }

    @Override
    protected ValidationResult validate(final SwitchPairPort port) {
        PropertyValidationSupport support = new PropertyValidationI18NSupport(getPresentationModel(), "validation");

        if (this.switchOffTimeConverterModel != null && this.switchOffTimeConverterModel.getValue() == null) {
            support.addError("switchOffTime_key", "not_empty_for_write");
        }

        if (this.selectionHolderLoadType != null && this.selectionHolderLoadType.getValue() == null) {
            support.addError("loadType_key", "not_empty_for_write");
        }

        ValidationResult validationResult = support.getResult();
        return validationResult;
    }

    private void sendStatusToPort(final SwitchPairPort port, SwitchPortStatus portStatus) {
        LOGGER.info("Send the new portStatus to the port: {}", portStatus);

        triggerValidation();

        final SwitchPairPort switchPairPort =
            SwitchPairPort.builder().withStatus(portStatus).withId(port.getId()).build();

        getValueCallback().accept(switchPairPort);
    }

    @Override
    protected void doEnableComponents(final SwitchPairPort port) {

        boolean enabled = port.isEnabled();
        this.portName.setEnabled(enabled);

        if (this.comboLoadType != null) {
            this.comboLoadType.setEnabled(enabled);
        }

        if (switchOffTimeText != null) {
            switchOffTimeText.setEnabled(enabled);
        }

        this.comboPortStatus.setEnabled(enabled);
        this.btnPortStatus.setEnabled(enabled);
    }
}
