package org.bidib.wizard.mvc.firmware.model;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.SwingUtilities;

import org.bidib.jbidibc.exchange.firmware.FilenameType;
import org.bidib.jbidibc.exchange.firmware.FirmwareDefinitionType;
import org.bidib.jbidibc.exchange.firmware.FirmwareNode;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.core.model.firmware.NodeUpdateStatus;
import org.bidib.wizard.core.model.firmware.UpdateStatus;
import org.bidib.wizard.mvc.firmware.model.listener.FirmwareModelListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

public class FirmwareModel extends Model {
    private static final Logger LOGGER = LoggerFactory.getLogger(FirmwareModel.class);

    private static final long serialVersionUID = 1L;

    public static final String PROPERTYNAME_IN_PROGRESS = "inProgress";

    public static final String PROPERTYNAME_IDLE = "idle";

    public static final String PROPERTYNAME_IDLE_AND_VALID = "idleAndValid";

    public static final String PROPERTYNAME_UPDATE_STATUS = "updateStatus";

    public static final String PROPERTYNAME_NODE_UPDATE_STATUS = "nodeUpdateStatus";

    public static final String PROPERTYNAME_CV_DEFINITION_FILES = "cvDefinitionFiles";

    public static final String PROPERTYNAME_UPDATE_VERSION = "updateVersion";

    public static final String PROPERTYNAME_DEFAULT_LABELS_FILES = "defaultLabelsFiles";

    public static final String PROPERTYNAME_VENDOR_IMAGE_FILES = "vendorImageFiles";

    public static final String PROPERTYNAME_PREVENT_ASK_OVERRIDE_FILE = "preventAskOverrideFile";

    public static final String PROPERTYNAME_FIRMWARE_DEFINITIONS = "firmwareDefinitions";

    public static final String PROPERTYNAME_SELECTED_FIRMWARE = "selectedFirmware";

    private final Collection<FirmwareModelListener> listeners = new LinkedList<FirmwareModelListener>();

    private boolean inProgress = false;

    private int progressValue = 0;

    private NodeInterface node;

    private String nodeName;

    private String productName;

    private String uniqueId;

    private String nodeCurrentVersion;

    private boolean errorDetected = false;

    private String firmwareArchivePath;

    private String cvDefinitionArchivePath;

    private String vendorImageArchivePath;

    private List<FirmwareNode> firmwareFiles;

    private List<String> cvDefinitionFiles;

    private List<String> vendorImageFiles;

    private List<FirmwareDefinitionType> firmwareDefinitions = new LinkedList<>();

    private FirmwareDefinitionType selectedFirmware;

    private UpdateStatus updateStatus = UpdateStatus.NONE;

    private NodeUpdateStatus nodeUpdateStatus = NodeUpdateStatus.NONE;

    private boolean isBootloaderRootNode;

    private String updateVersion;

    private List<FilenameType> defaultLabelsFiles;

    private boolean preventAskOverrideFile;

    public void addFirmwareModelListener(FirmwareModelListener l) {
        listeners.add(l);
    }

    public void removeFirmwareModelListener(FirmwareModelListener l) {
        listeners.remove(l);
    }

    public boolean isInProgress() {
        return inProgress;
    }

    public void setInProgress(boolean inProgress) {
        final boolean oldValue = this.inProgress;
        final boolean oldIdleAndValid = isIdleAndValid();

        this.inProgress = inProgress;

        final boolean newIdleAndValid = isIdleAndValid();

        SwingUtilities.invokeLater(() -> {
            firePropertyChange(PROPERTYNAME_IN_PROGRESS, oldValue, inProgress);

            firePropertyChange(PROPERTYNAME_IDLE, !oldValue, !inProgress);

            firePropertyChange(PROPERTYNAME_IDLE_AND_VALID, oldIdleAndValid, newIdleAndValid);
        });
    }

    public void setIdle(boolean idle) {
        final boolean oldValue = !this.inProgress;
        final boolean oldIdleAndValid = isIdleAndValid();

        this.inProgress = !idle;

        final boolean newIdleAndValid = isIdleAndValid();

        SwingUtilities.invokeLater(() -> {
            firePropertyChange(PROPERTYNAME_IDLE, oldValue, inProgress);

            firePropertyChange(PROPERTYNAME_IDLE_AND_VALID, oldIdleAndValid, newIdleAndValid);
        });
    }

    public boolean isIdle() {
        return !inProgress;
    }

    public boolean isIdleAndValid() {
        boolean isIdleAndValid = !inProgress && !UpdateStatus.NODE_LOST.equals(updateStatus);
        LOGGER.info("isIdleAndValid: {}, inProgress: {}, updateStatus: {}", isIdleAndValid, inProgress, updateStatus);
        return isIdleAndValid;
    }

    public int getProgressValue() {
        return progressValue;
    }

    public void setProgressValue(int progressValue) {
        if (this.progressValue != progressValue) {
            this.progressValue = progressValue;
            fireProgressChanged(progressValue);
        }
    }

    public void addProcessingStatus(String processingStatus, final int style, Object... args) {

        fireProcessingStatusChanged(processingStatus, style, args);
    }

    /**
     * @return the node
     */
    public NodeInterface getNode() {
        return node;
    }

    /**
     * @param node
     *            the node to set
     */
    public void setNode(final NodeInterface node) {
        this.node = node;
    }

    /**
     * @return the nodeName
     */
    public String getNodeName() {
        return nodeName;
    }

    /**
     * @param nodeName
     *            the nodeName to set
     */
    public void setNodeName(String nodeName) {
        this.nodeName = nodeName;
    }

    /**
     * @return the productName
     */
    public String getProductName() {
        return productName;
    }

    /**
     * @param productName
     *            the productName to set
     */
    public void setProductName(String productName) {
        this.productName = productName;
    }

    /**
     * @return the unique id
     */
    public String getUniqueId() {
        return uniqueId;
    }

    /**
     * @param uniqueId
     *            the unique id to set
     */
    public void setUniqueId(String uniqueId) {
        this.uniqueId = uniqueId;
    }

    /**
     * @return the firmware version
     */
    public String getNodeCurrentVersion() {
        return nodeCurrentVersion;
    }

    /**
     * @param version
     *            the current firmware version on the node to set
     */
    public void setNodeCurrentVersion(String version) {
        this.nodeCurrentVersion = version;
    }

    /**
     * @return the errorDetected
     */
    public boolean isErrorDetected() {
        return errorDetected;
    }

    /**
     * Set the firmware file to use
     * 
     * @param firmwareFiles
     *            the files
     * @param firmwareArchivePath
     *            the path to the archive
     */
    public void setFirmwareFiles(String firmwareArchivePath, List<FirmwareNode> firmwareFiles) {
        this.firmwareArchivePath = firmwareArchivePath;
        this.firmwareFiles = firmwareFiles;

        fireFirmwareFileChanged();
    }

    /**
     * @return the list of firmwareFiles
     */
    public List<FirmwareNode> getFirmwareFiles() {
        List<FirmwareNode> files = new LinkedList<>();
        files.addAll(firmwareFiles);
        return Collections.unmodifiableList(files);
    }

    /**
     * @return the cvDefinitionFiles
     */
    public List<String> getCvDefinitionFiles() {
        if (cvDefinitionFiles == null) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(cvDefinitionFiles);
    }

    /**
     * @param cvDefinitionArchivePath
     *            the cv definition archive path
     * @param cvDefinitionFiles
     *            the cvDefinitionFiles to set
     */
    public void setCvDefinitionFiles(String cvDefinitionArchivePath, List<String> cvDefinitionFiles) {
        this.cvDefinitionArchivePath = cvDefinitionArchivePath;
        List<String> oldValue = this.cvDefinitionFiles;

        if (cvDefinitionFiles == null) {
            cvDefinitionFiles = Collections.emptyList();
        }

        this.cvDefinitionFiles = cvDefinitionFiles;

        firePropertyChange(PROPERTYNAME_CV_DEFINITION_FILES, oldValue, cvDefinitionFiles);
    }

    /**
     * @return the vendorImageFiles
     */
    public List<String> getVendorImageFiles() {
        if (vendorImageFiles == null) {
            return Collections.emptyList();
        }

        return Collections.unmodifiableList(vendorImageFiles);
    }

    /**
     * @param vendorImageArchivePath
     *            the vendor image archive path
     * @param vendorImageFiles
     *            the vendorImageFiles to set
     */
    public void setVendorImageFiles(String vendorImageArchivePath, List<String> vendorImageFiles) {

        this.vendorImageArchivePath = vendorImageArchivePath;

        List<String> oldValue = this.vendorImageFiles;

        if (cvDefinitionFiles == null) {
            cvDefinitionFiles = Collections.emptyList();
        }

        this.vendorImageFiles = vendorImageFiles;

        firePropertyChange(PROPERTYNAME_VENDOR_IMAGE_FILES, oldValue, this.vendorImageFiles);
    }

    /**
     * @return the cvDefinitionArchivePath
     */
    public String getCvDefinitionArchivePath() {
        return cvDefinitionArchivePath;
    }

    /**
     * @return the vendorImageArchivePath
     */
    public String getVendorImageArchivePath() {
        return vendorImageArchivePath;
    }

    /**
     * @return the firmwareArchivePath
     */
    public String getFirmwareArchivePath() {
        return firmwareArchivePath;
    }

    /**
     * @return the updateStatus
     */
    public UpdateStatus getUpdateStatus() {
        return updateStatus;
    }

    /**
     * @param updateStatus
     *            the updateStatus to set
     */
    public void setUpdateStatus(UpdateStatus updateStatus) {
        final UpdateStatus oldValue = this.updateStatus;
        final boolean oldIdleAndValid = isIdleAndValid();

        this.updateStatus = updateStatus;

        final boolean newIdleAndValid = isIdleAndValid();

        SwingUtilities.invokeLater(() -> {
            firePropertyChange(PROPERTYNAME_UPDATE_STATUS, oldValue, updateStatus);

            firePropertyChange(PROPERTYNAME_IDLE_AND_VALID, oldIdleAndValid, newIdleAndValid);
        });
    }

    /**
     * @return the nodeUpdateStatus
     */
    public NodeUpdateStatus getNodeUpdateStatus() {
        return nodeUpdateStatus;
    }

    /**
     * @param nodeUpdateStatus
     *            the nodeUpdateStatus to set
     */
    public void setNodeUpdateStatus(NodeUpdateStatus nodeUpdateStatus) {
        final NodeUpdateStatus oldValue = this.nodeUpdateStatus;
        this.nodeUpdateStatus = nodeUpdateStatus;

        SwingUtilities
            .invokeLater(() -> firePropertyChange(PROPERTYNAME_NODE_UPDATE_STATUS, oldValue, this.nodeUpdateStatus));
    }

    /**
     * @return the isBootloaderRootNode
     */
    public boolean isBootloaderRootNode() {
        return isBootloaderRootNode;
    }

    /**
     * @param isBootloaderRootNode
     *            the isBootloaderRootNode to set
     */
    public void setBootloaderRootNode(boolean isBootloaderRootNode) {
        this.isBootloaderRootNode = isBootloaderRootNode;
    }

    /**
     * @return the updateVersion
     */
    public String getUpdateVersion() {
        return updateVersion;
    }

    /**
     * @param updateVersion
     *            the updateVersion to set
     */
    public void setUpdateVersion(final String updateVersion) {
        final String oldValue = this.updateVersion;

        this.updateVersion = updateVersion;

        SwingUtilities.invokeLater(() -> {
            firePropertyChange(PROPERTYNAME_UPDATE_VERSION, oldValue, updateVersion);
        });
    }

    /**
     * @return the defaultLabelsFiles
     */
    public List<FilenameType> getDefaultLabelsFiles() {
        return defaultLabelsFiles;
    }

    /**
     * @param defaultLabelsFiles
     *            the defaultLabelsFiles to set
     */
    public void setDefaultLabelsFiles(String cvDefinitionArchivePath, List<FilenameType> defaultLabelsFiles) {
        this.cvDefinitionArchivePath = cvDefinitionArchivePath;

        List<FilenameType> oldValue = this.defaultLabelsFiles;

        this.defaultLabelsFiles = defaultLabelsFiles;

        firePropertyChange(PROPERTYNAME_DEFAULT_LABELS_FILES, oldValue, defaultLabelsFiles);
    }

    private void fireProcessingStatusChanged(String processingStatus, final int style, Object... args) {
        for (FirmwareModelListener l : listeners) {
            l.processingStatusChanged(processingStatus, style, args);
        }
    }

    private void fireProgressChanged(int progressValue) {
        for (FirmwareModelListener l : listeners) {
            l.progressValueChanged(progressValue);
        }
    }

    private void fireFirmwareFileChanged() {
        for (FirmwareModelListener l : listeners) {
            l.firmwareFileChanged();
        }
    }

    public void setPreventAskOverrideFile(boolean preventAskOverrideFile) {
        boolean oldValue = this.preventAskOverrideFile;
        this.preventAskOverrideFile = preventAskOverrideFile;

        firePropertyChange(PROPERTYNAME_PREVENT_ASK_OVERRIDE_FILE, oldValue, preventAskOverrideFile);
    }

    /**
     * @return the preventAskOverrideFile
     */
    public boolean isPreventAskOverrideFile() {
        return preventAskOverrideFile;
    }

    /**
     * @return the firmwareDefinitions
     */
    public List<FirmwareDefinitionType> getFirmwareDefinitions() {
        return Collections.unmodifiableList(firmwareDefinitions);
    }

    /**
     * @param firmwareDefinitions
     *            the firmwareDefinitions to set
     */
    public void setFirmwareDefinitions(final List<FirmwareDefinitionType> firmwareDefinitions) {
        final List<FirmwareDefinitionType> oldValue = Collections.unmodifiableList(this.firmwareDefinitions);

        this.firmwareDefinitions.clear();
        this.firmwareDefinitions.addAll(firmwareDefinitions);

        firePropertyChange(PROPERTYNAME_FIRMWARE_DEFINITIONS, oldValue, this.firmwareDefinitions);
    }

    /**
     * @return the selectedFirmware
     */
    public FirmwareDefinitionType getSelectedFirmware() {
        return selectedFirmware;
    }

    /**
     * @param selectedFirmware
     *            the selectedFirmware to set
     */
    public void setSelectedFirmware(FirmwareDefinitionType selectedFirmware) {
        FirmwareDefinitionType oldValue = this.selectedFirmware;
        this.selectedFirmware = selectedFirmware;

        firePropertyChange(PROPERTYNAME_SELECTED_FIRMWARE, oldValue, this.selectedFirmware);
    }

}
