package org.bidib.wizard.mvc.main.view.menu;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.swing.Box;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.event.MenuListener;
import javax.xml.parsers.ParserConfigurationException;

import org.bidib.api.json.types.ConnectionPhase;
import org.bidib.jbidibc.messages.enums.DetachedState;
import org.bidib.jbidibc.messages.enums.IdentifyState;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.CommandStationNodeInterface;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.NodeListener;
import org.bidib.wizard.common.context.DefaultApplicationContext;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.common.model.settings.ConnectionConfiguration;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.common.model.settings.LocalHostSettingsInterface;
import org.bidib.wizard.common.model.settings.MiscSettingsInterface;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.main.model.ConnectionPhaseModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.menu.listener.MainMenuListener;
import org.bidib.wizard.mvc.main.view.panel.NodeListPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import com.vlsolutions.swing.docking.DockKey;
import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingContext;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.event.DockableStateChangeEvent;
import com.vlsolutions.swing.docking.event.DockableStateChangeListener;

public class MainMenuBar extends JMenuBar implements NodeListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(MainMenuBar.class);

    private static final long serialVersionUID = 1L;

    private final NodeListMenu nodeListMenu;

    private final WindowListMenu windowListMenu;

    private NodeInterface selectedNode;

    private DockingFramesRegistry dockingFramesRegistry;

    private JMenuItem connectItem;

    private JMenuItem disconnectItem;

    private JMenuItem listenNetBidibItem;

    private JMenuItem dccAdvItem;

    private JMenuItem debugInterfaceItem;

    private JMenuItem netDebugItem;

    private JMenuItem debugConsoleItem;

    private JMenuItem tracerClientItem;

    private JMenuItem nodesClientButton;

    private JMenuItem localHostClientItem;

    private JMenuItem scriptClientButton;

    private JMenuItem discoverySingleListServicesButton;

    private final DockingDesktop desktop;

    private final DockingContext dockingContext;

    private final GlobalSettingsInterface globalSettings;

    private final MiscSettingsInterface miscSettings;

    private final LocalHostSettingsInterface localHostSettings;

    private final ConnectionPhaseModel connectionPhaseModel;

    private final MainMenuListener mainMenuListener;

    public MainMenuBar(final MainModel mainModel, final DockingDesktop desktop, final DockingContext dockingContext,
        final SettingsService settingsService, final ConnectionPhaseModel connectionPhaseModel,
        final MainMenuListener mainMenuListener, final WizardLabelWrapper wizardLabelWrapper) {
        this.desktop = desktop;
        this.dockingContext = dockingContext;
        this.connectionPhaseModel = connectionPhaseModel;
        this.mainMenuListener = mainMenuListener;

        this.globalSettings = settingsService.getGlobalSettings();
        this.miscSettings = settingsService.getMiscSettings();
        this.localHostSettings = settingsService.getLocalHostSettings();

        addDockableStateChangeListener(desktop);
        windowListMenu = new WindowListMenu(dockingFramesRegistry);
        nodeListMenu = new NodeListMenu(mainModel, settingsService, wizardLabelWrapper);

        JMenu fileMenu = new JMenu(Resources.getString(getClass(), "file"));
        connectItem = new JMenuItem(Resources.getString(getClass(), "connect"));

        connectItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, ActionEvent.ALT_MASK));
        connectItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireConnect();
            }
        });
        fileMenu.add(connectItem);

        disconnectItem = new JMenuItem(Resources.getString(getClass(), "disconnect"));

        disconnectItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.ALT_MASK));
        disconnectItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireDisconnect();
            }
        });
        disconnectItem.setEnabled(false);
        fileMenu.add(disconnectItem);

        // listen for netBidib
        listenNetBidibItem = new JMenuItem(Resources.getString(getClass(), "listenNetBidib"));

        listenNetBidibItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L, ActionEvent.ALT_MASK));
        listenNetBidibItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireListenNetBidib();
            }
        });

        // enable if netBidib is selected
        listenNetBidibItem.setEnabled(false);
        fileMenu.add(listenNetBidibItem);

        fileMenu.addSeparator();

        enableListenNetBidibItem();

        final JMenuItem exitItem = new JMenuItem(Resources.getString(getClass(), "quit"));

        exitItem
            .setAccelerator(
                KeyStroke.getKeyStroke(KeyEvent.VK_Q, Toolkit.getDefaultToolkit().getMenuShortcutKeyMaskEx()));
        exitItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireExit();
            }
        });
        fileMenu.add(exitItem);
        add(fileMenu);

        JMenu editMenu = new JMenu(Resources.getString(getClass(), "edit"));
        JMenuItem preferencesItem = new JMenuItem(Resources.getString(getClass(), "preferences") + " ...");

        preferencesItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                firePreferences();
            }
        });
        editMenu.add(preferencesItem);
        add(editMenu);

        final JMenu nodeMenu = nodeListMenu.getJMenu();

        nodeMenu.addMenuListener(new MenuListener() {
            @Override
            public void menuCanceled(javax.swing.event.MenuEvent e) {
            }

            @Override
            public void menuDeselected(javax.swing.event.MenuEvent e) {
            }

            @Override
            public void menuSelected(javax.swing.event.MenuEvent e) {

                if (e.getSource() instanceof JMenu) {
                    if (!((JMenu) e.getSource()).isEnabled()) {
                        LOGGER.debug("The menu is not enabled, skip processing.");
                        return;
                    }
                }
                // update the menu items
                try {
                    NodeListPanel nodeListPanel =
                        DefaultApplicationContext.getInstance().get("nodeListPanel", NodeListPanel.class);
                    final NodeInterface node = nodeListPanel.getSelectedItem();
                    if (node != null) {
                        nodeListMenu.updateMenuItems(node);
                    }
                }
                catch (Exception ex) {
                    LOGGER.warn("Update menu items of nodeListMenu failed.", ex);
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null),
                            Resources.getString(MainMenuBar.class, "menuitemserror.message"),
                            Resources.getString(MainMenuBar.class, "menuitemserror.title"), JOptionPane.ERROR_MESSAGE);
                }
            }
        });
        nodeMenu.setEnabled(false);
        nodeMenu.setText(Resources.getString(getClass(), "node"));
        add(nodeMenu);

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeChanged(final NodeInterface node) {

                if (SwingUtilities.isEventDispatchThread()) {
                    internalNodeChanged(mainModel, nodeMenu);
                }
                else {
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            internalNodeChanged(mainModel, nodeMenu);
                        }
                    });
                }
            }
        });

        final JMenu framesMenu = windowListMenu.getJMenu();

        final JMenuItem saveWorkspaceItem = new JMenuItem(Resources.getString(getClass(), "saveWorkspace"));

        saveWorkspaceItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireSaveWorkspace();
            }
        });

        saveWorkspaceItem.setEnabled(true);
        windowListMenu.addTopItem(saveWorkspaceItem);

        final JMenuItem loadWorkspaceItem = new JMenuItem(Resources.getString(getClass(), "loadWorkspace"));

        loadWorkspaceItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireLoadWorkspace();
            }
        });

        loadWorkspaceItem.setEnabled(true);
        windowListMenu.addTopItem(loadWorkspaceItem);

        final JMenuItem loadDefaultWorkspaceItem =
            new JMenuItem(Resources.getString(getClass(), "loadDefaultWorkspace"));

        loadDefaultWorkspaceItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireLoadDefaultWorkspace();
            }
        });

        windowListMenu.addTopItem(loadDefaultWorkspaceItem);

        // -----------------------

        final JMenuItem boosterTableItem = new JMenuItem(Resources.getString(getClass(), "boosterTable"));

        boosterTableItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireBoosterTable();
            }
        });

        // always add booster table at first position
        String dockKeyKey = "BoosterTableView";
        windowListMenu.addStickyItem(dockKeyKey, boosterTableItem);

        final JMenuItem feedbackPositionTableItem =
            new JMenuItem(Resources.getString(getClass(), "feedbackPositionTable"));

        feedbackPositionTableItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireFeedbackPositionTable(mainModel);
            }
        });

        // always add booster table at first position
        dockKeyKey = "FeedbackPositionTableView";
        windowListMenu.addStickyItem(dockKeyKey, feedbackPositionTableItem);

        final JMenuItem consoleItem = new JMenuItem(Resources.getString(getClass(), "console"));

        consoleItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireConsole();
            }
        });

        // always add booster table at first position
        dockKeyKey = "ConsoleView";
        windowListMenu.addStickyItem(dockKeyKey, consoleItem);

        framesMenu.addMenuListener(new MenuListener() {

            @Override
            public void menuSelected(javax.swing.event.MenuEvent e) {
                windowListMenu.prepareMenu();
            }

            @Override
            public void menuDeselected(javax.swing.event.MenuEvent e) {
            }

            @Override
            public void menuCanceled(javax.swing.event.MenuEvent e) {
            }
        });
        // framesMenu.setEnabled(false);
        framesMenu.setText(Resources.getString(getClass(), "window"));
        add(framesMenu);

        JMenu toolsMenu = new JMenu(Resources.getString(getClass(), "tools"));

        // node script
        JMenuItem nodeScriptItem = new JMenuItem(Resources.getString(getClass(), "nodeScript"));

        nodeScriptItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, ActionEvent.ALT_MASK));
        nodeScriptItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireNodeScript(mainModel);
            }
        });
        toolsMenu.add(nodeScriptItem);

        // firmware repo
        JMenuItem firmwareRepoItem = new JMenuItem(Resources.getString(getClass(), "firmwareRepo"));
        firmwareRepoItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireShowFirmwareRepo();
            }
        });
        toolsMenu.add(firmwareRepoItem);

        // nodes client
        nodesClientButton = new JMenuItem(Resources.getString(getClass(), "nodesClient"));
        nodesClientButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                // create the nodes client controller
                fireShowNodesClient();
            }
        });
        nodesClientButton.setEnabled(false);
        toolsMenu.add(nodesClientButton);

        // the developer tools
        final JMenu developerToolsMenu = new JMenu(Resources.getString(getClass(), "developerTools"));
        toolsMenu.add(developerToolsMenu);

        JMenuItem logPanelItem = new JMenuItem(Resources.getString(getClass(), "logPanel") + " ...");

        logPanelItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireLogPanel();
            }
        });
        logPanelItem.setEnabled(true);
        developerToolsMenu.add(logPanelItem);

        dccAdvItem = new JMenuItem(Resources.getString(getClass(), "dccAdv"));

        dccAdvItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.ALT_MASK));
        dccAdvItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireAddAdv(mainModel);
            }
        });

        // the DCC-A menu item is enabled depending on the feature of the command station!
        dccAdvItem.setEnabled(false);
        toolsMenu.add(dccAdvItem);

        // the debug interface
        debugInterfaceItem = new JMenuItem(Resources.getString(getClass(), "debugInterface"));

        debugInterfaceItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I, ActionEvent.ALT_MASK));
        debugInterfaceItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireDebugInterface();
            }
        });

        // the debug interface menu item is always enabled
        debugInterfaceItem.setEnabled(true);
        developerToolsMenu.add(debugInterfaceItem);

        // the debug interface
        netDebugItem = new JMenuItem(Resources.getString(getClass(), "netDebug"));

        netDebugItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.ALT_MASK));
        netDebugItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireNetDebug();
            }
        });

        // the netDebug interface menu item is always enabled
        netDebugItem.setEnabled(true);
        developerToolsMenu.add(netDebugItem);

        // the debug console
        debugConsoleItem = new JMenuItem(Resources.getString(getClass(), "debugConsole"));

        debugConsoleItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.ALT_MASK));
        debugConsoleItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireDebugConsole(mainModel);
            }
        });

        // the debug console menu item is always enabled
        debugConsoleItem.setEnabled(true);
        developerToolsMenu.add(debugConsoleItem);

        // the tracer client
        tracerClientItem = new JMenuItem(Resources.getString(getClass(), "tracerClient"));

        tracerClientItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireShowTracerClient();
            }
        });

        // the tracer client menu item is always enabled
        tracerClientItem.setEnabled(true);
        developerToolsMenu.add(tracerClientItem);

        // the ping table
        final JMenuItem pingTableItem = new JMenuItem(Resources.getString(getClass(), "pingTable"));

        pingTableItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                firePingTable();
            }
        });

        developerToolsMenu.add(pingTableItem);

        // the localhost client
        localHostClientItem = new JMenuItem(Resources.getString(getClass(), "localHostClient"));

        localHostClientItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireShowLocalHostClient();
            }
        });

        localHostClientItem
            .setEnabled(localHostSettings.isLocalHostClientEnabled() && localHostSettings.isLocalHostServiceEnabled());
        localHostSettings.addPropertyChangeListener(evt -> {
            switch (evt.getPropertyName()) {
                case LocalHostSettingsInterface.PROPERTY_LOCALHOST_CLIENT_ENABLED:
                case LocalHostSettingsInterface.PROPERTY_LOCALHOST_SERVICE_ENABLED:
                    localHostClientItem
                        .setEnabled(localHostSettings.isLocalHostClientEnabled()
                            && localHostSettings.isLocalHostServiceEnabled());
                    break;
                default:
                    break;
            }
        });

        developerToolsMenu.add(localHostClientItem);

        // script client
        scriptClientButton = new JMenuItem(Resources.getString(getClass(), "scriptClient"));
        scriptClientButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                // create the script client controller
                fireShowScriptClient();
            }
        });

        // TODO disable
        // scriptClientButton.setEnabled(false);
        developerToolsMenu.add(scriptClientButton);

        // discovery single list services
        discoverySingleListServicesButton =
            new JMenuItem(Resources.getString(getClass(), "discoverySingleListServices"));
        discoverySingleListServicesButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireDiscoverySingleListServicesButton();
            }
        });

        // TODO disable
        // discoverySingleListServicesButton.setEnabled(false);
        developerToolsMenu.add(discoverySingleListServicesButton);

        // add tools sub menu to menubar
        add(toolsMenu);

        add(Box.createHorizontalGlue());

        JMenu helpMenu = new JMenu(Resources.getString(getClass(), "help"));
        JMenuItem aboutItem = new JMenuItem(Resources.getString(getClass(), "about") + " ...");

        aboutItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireAbout();
            }
        });
        helpMenu.add(aboutItem);

        JMenuItem logFilesItem = new JMenuItem(Resources.getString(getClass(), "logFiles") + " ...");

        logFilesItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                fireCollectLogFiles();
            }
        });
        helpMenu.add(logFilesItem);
        add(helpMenu);
    }

    public void enableListenNetBidibItem() {
        String connectionId = ConnectionRegistry.CONNECTION_ID_MAIN;
        ConnectionConfiguration connectionConfig =
            this.globalSettings
                .getConnectionConfigurations().stream().filter(cfg -> cfg.getId().equals(connectionId)).findFirst()
                .orElse(null);
        if (connectionConfig != null && "NetBidibClient".equals(connectionConfig.getConnectionProvider())) {
            ConnectionPhase connectionPhase = connectionPhaseModel.getConnectionPhase();
            if (connectionPhase == null || ConnectionPhase.DISCONNECTED == connectionPhase) {

                // TODO enable the listenNetBidibButton
                // listenNetBidibItem.setEnabled(true);
            }
            else {
                listenNetBidibItem.setEnabled(false);
            }
        }
        else {
            listenNetBidibItem.setEnabled(false);
        }
    }

    protected void fireLoadDefaultWorkspace() {

        try (InputStream in = MainMenuBar.class.getResourceAsStream("/workspace/default-workspace.xml")) {

            dockingContext.readXML(in);
        }
        catch (IOException | ParserConfigurationException | SAXException ioe) {
            LOGGER.warn("Load default workspace failed.", ioe);
            // process exception here
        }

    }

    protected void fireLoadWorkspace() {

        String workspacePath = this.miscSettings.getWorkspacePath();

        File loadFile = null;
        try {
            loadFile = new File(workspacePath, "workspace.xml");
            LOGGER.info("Load workspace from file: {}", loadFile);

            if (!loadFile.getParentFile().exists()) {
                throw new IllegalArgumentException("The requested file is not available: " + loadFile);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Load workspace from file failed.", ex);

            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(this),
                    Resources.getString(MainMenuBar.class, "invalid-load-workspace-path.message"),
                    Resources.getString(MainMenuBar.class, "invalid-workspace-path.title"), JOptionPane.ERROR_MESSAGE);
            return;
        }

        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(loadFile))) {

            dockingContext.readXML(in);
        }
        catch (IOException | ParserConfigurationException | SAXException ioe) {
            LOGGER.warn("Load workspace failed.", ioe);

            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(this),
                    Resources.getString(MainMenuBar.class, "invalid-load-workspace-content.message"),
                    Resources.getString(MainMenuBar.class, "invalid-workspace-content.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    protected void fireSaveWorkspace() {

        String workspacePath = this.miscSettings.getWorkspacePath();
        File saveFile = null;
        try {
            saveFile = new File(workspacePath, "workspace.xml");
            LOGGER.info("Save workspace to file: {}", saveFile);

            if (!saveFile.getParentFile().exists()) {
                // try to create directory for workspace
                if (!saveFile.getParentFile().mkdirs()) {
                    LOGGER.warn("Create the directory to store the workspace failed: {}", saveFile.getParentFile());
                    throw new IllegalArgumentException(
                        "The requested directory is not available: " + saveFile.getParentFile());
                }
            }

        }
        catch (Exception ex) {
            LOGGER.warn("Save workspace to file failed.", ex);

            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(this),
                    Resources.getString(MainMenuBar.class, "invalid-save-workspace-path.message"),
                    Resources.getString(MainMenuBar.class, "invalid-workspace-path.title"), JOptionPane.ERROR_MESSAGE);
            return;
        }

        try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(saveFile))) {

            desktop.writeXML(out);
        }
        catch (IOException ioe) {
            LOGGER.warn("Save workspace failed.", ioe);

            JOptionPane
                .showMessageDialog(JOptionPane.getFrameForComponent(this),
                    Resources
                        .getString(MainMenuBar.class, "invalid-save-workspace-content.message",
                            new Object[] { saveFile.getPath() }),
                    Resources.getString(MainMenuBar.class, "invalid-workspace-content.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    private void internalNodeChanged(MainModel model, JMenu nodeMenu) {
        if (selectedNode != null) {
            selectedNode.removeNodeListener(MainMenuBar.this);
        }

        final NodeInterface newNode = model.getSelectedNode();

        nodeMenu.setEnabled(newNode != null);
        if (newNode != null) {
            // add as listener to the selected node
            newNode.addNodeListener(MainMenuBar.this);
        }

        LOGGER.debug("Set the selected node: {}", newNode);
        setSelectedNode(newNode);
    }

    private void addDockableStateChangeListener(final DockingDesktop desktop) {

        this.dockingFramesRegistry = new DockingFramesRegistry();

        DockableStateChangeListener listener = new DockableStateChangeListener() {
            @Override
            public void dockableStateChanged(DockableStateChangeEvent event) {
                DockableState newState = event.getNewState();
                LOGGER.info("The state of the dockable has changed: {}", event);
                if (newState.isDocked()) {
                    // a new frame is docked
                    DockKey dockKey = newState.getDockable().getDockKey();
                    Dockable dockable = dockingFramesRegistry.getDockable(dockKey);
                    if (dockable != null) {
                        LOGGER.debug("Dockable already registered: {}", dockKey);
                    }
                    else {
                        dockable = newState.getDockable();
                        LOGGER.info("Register new dockable: {}", dockable);
                        dockingFramesRegistry.addDockable(dockKey, dockable);
                    }

                }
                if (newState.isClosed()) {
                    DockKey dockKey = newState.getDockable().getDockKey();
                    dockingFramesRegistry.removeDockable(dockKey);

                    // the dockable has been closed
                    if (newState.getDockable() instanceof DockableStateChangeListener) {
                        desktop.removeDockableStateChangeListener((DockableStateChangeListener) newState.getDockable());
                    }
                    desktop.unregisterDockable(newState.getDockable()); // forget about it !
                }
            }
        };
        desktop.addDockableStateChangeListener(listener);
    }

    @Override
    public void addressMessagesEnabledChanged(final NodeInterface node, Boolean isAddressMessagesEnabled) {
        nodeListMenu.setAddressMessagesEnabled(isAddressMessagesEnabled);
    }

    @Override
    public void dccStartEnabledChanged(final NodeInterface node, Boolean isDccStartEnabled) {
        nodeListMenu.setDccStartEnabled(isDccStartEnabled);
    }

    @Override
    public void externalStartEnabledChanged(final NodeInterface node, Boolean isExternalStartEnabled) {
        nodeListMenu.setExternalStartEnabled(isExternalStartEnabled);
    }

    @Override
    public void dccAdvAvailableChanged(final NodeInterface node, Boolean dccAdvAvailable) {
        // LOGGER.trace("Set the dccAdv item available: {}", dccAdvAvailable);
        nodeListMenu.setDccAdvEnabled(dccAdvAvailable);
        dccAdvItem.setEnabled(dccAdvAvailable != null && dccAdvAvailable.booleanValue());
    }

    @Override
    public void feedbackMessagesEnabledChanged(final NodeInterface node, Boolean isFeedbackMessagesEnabled) {
        nodeListMenu.setFeedbackMessagesEnabled(isFeedbackMessagesEnabled);
    }

    @Override
    public void feedbackMirrorDisabledChanged(final NodeInterface node, Boolean isFeedbackMirrorDisabled) {
        nodeListMenu.setFeedbackMirrorDisabled(isFeedbackMirrorDisabled);
    }

    private void fireBoosterTable() {
        this.mainMenuListener.boosterTable();
    }

    private void fireFeedbackPositionTable(final MainModel mainModel) {
        this.mainMenuListener.feedbackPositionTable();
    }

    private void firePingTable() {
        this.mainMenuListener.pingTable();
    }

    private void fireLogPanel() {
        this.mainMenuListener.logPanel();
    }

    private void fireConsole() {
        this.mainMenuListener.console();
    }

    private void fireCollectLogFiles() {
        this.mainMenuListener.collectLogFiles();
    }

    private void fireAbout() {
        this.mainMenuListener.about();
    }

    private void fireExit() {
        this.mainMenuListener.exit();
    }

    private void fireConnect() {
        this.mainMenuListener.connect();
    }

    private void fireDisconnect() {
        this.mainMenuListener.disconnect();
    }

    private void fireListenNetBidib() {
        this.mainMenuListener.listenNetBidib();
    }

    private void firePreferences() {
        this.mainMenuListener.preferences();
    }

    private void fireNodeScript(final MainModel mainModel) {
        this.mainMenuListener.nodeScriptView();
    }

    private void fireAddAdv(final MainModel mainModel) {
        this.mainMenuListener.dccAdv();
    }

    private void fireDebugInterface() {
        this.mainMenuListener.debugInterface();
    }

    private void fireNetDebug() {
        this.mainMenuListener.netDebug();
    }

    private void fireDebugConsole(final MainModel mainModel) {
        this.mainMenuListener.debugConsole();
    }

    private void fireShowFirmwareRepo() {
        this.mainMenuListener.showFirmwareRepo();
    }

    private void fireShowNodesClient() {
        this.mainMenuListener.showNodesClient();
    }

    private void fireShowScriptClient() {
        this.mainMenuListener.showScriptClient();
    }

    private void fireShowTracerClient() {
        this.mainMenuListener.showTracerClient();
    }

    private void fireShowLocalHostClient() {
        this.mainMenuListener.showLocalHostClient();
    }

    private void fireDiscoverySingleListServicesButton() {
        this.mainMenuListener.discoverySingleListServices();
    }

    @Override
    public void identifyStateChanged(final NodeInterface node, IdentifyState identifyState) {
        // identify state of the selected node has changed
        nodeListMenu.setIdentifyStateSelected(identifyState);
    }

    @Override
    public void detachedStateChanged(final NodeInterface node, DetachedState detachedState) {
        // detached state of the selected node has changed
        nodeListMenu.setDetachedStateSelected(detachedState);
    }

    @Override
    public void keyMessagesEnabledChanged(final NodeInterface node, Boolean isKeyMessagesEnabled) {
        nodeListMenu.setKeyMessagesEnabled(isKeyMessagesEnabled);
    }

    @Override
    public void labelsChanged(final NodeInterface node) {
    }

    private void setSelectedNode(final NodeInterface node) {
        this.selectedNode = node;

        if (selectedNode != null && selectedNode.getCommandStationNode() != null) {

            final CommandStationNodeInterface csNode = selectedNode.getCommandStationNode();
            dccAdvAvailableChanged(selectedNode,
                csNode.isDccAdvAvailable() != null ? csNode.isDccAdvAvailable() : false);
        }
        else {
            dccAdvAvailableChanged(selectedNode, false);
        }
    }

    public void opening() {
        LOGGER.info("Port is opening.");
        connectItem.setEnabled(false);
        // disconnectItem.setEnabled(true);
        listenNetBidibItem.setEnabled(false);
    }

    public void opened(String port) {
        LOGGER.info("The port was opened: {}.", port);

        connectItem.setEnabled(false);
        disconnectItem.setEnabled(true);
        listenNetBidibItem.setEnabled(false);

        nodesClientButton.setEnabled(true);
    }

    public void closed(String port) {
        LOGGER.info("The port was closed: {}.", port);

        connectItem.setEnabled(true);
        disconnectItem.setEnabled(false);
        // listenNetBidibItem.setEnabled(true);

        nodesClientButton.setEnabled(false);

        enableListenNetBidibItem();
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
    }
}
