package org.bidib.wizard.mvc.main.view.panel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.bidib.jbidibc.core.schema.BidibFactory;
import org.bidib.jbidibc.core.schema.DecoderVendorFactory;
import org.bidib.jbidibc.core.schema.bidib.products.ProductType;
import org.bidib.jbidibc.core.schema.bidib2.FeatureCode;
import org.bidib.jbidibc.core.schema.bidib2.common.DocumentationType;
import org.bidib.jbidibc.core.schema.decodervendor.VendorType;
import org.bidib.jbidibc.exchange.bidib.ProductsUtils;
import org.bidib.jbidibc.messages.Feature;
import org.bidib.jbidibc.messages.SoftwareVersion;
import org.bidib.jbidibc.messages.StringData;
import org.bidib.jbidibc.messages.enums.FeatureEnum;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.ConversionUtils;
import org.bidib.jbidibc.messages.utils.NodeUtils;
import org.bidib.wizard.api.LookupService;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.utils.XmlLocaleUtils;
import org.bidib.wizard.client.common.converter.SoftwareVersionConverter;
import org.bidib.wizard.client.common.converter.UniqueIdConverter;
import org.bidib.wizard.client.common.table.AbstractEmptyTable;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.client.common.view.BasicPopupMenu;
import org.bidib.wizard.client.common.view.BidibNodeNameUtils;
import org.bidib.wizard.client.common.view.TabPanelProvider;
import org.bidib.wizard.common.utils.ImageUtils;
import org.bidib.wizard.config.FirmwareControllerFactory;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.firmwarerepo.core.FirmwareRepoService;
import org.bidib.wizard.mvc.firmware.controller.FirmwareController;
import org.bidib.wizard.mvc.main.model.InfoModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.PresentationModel;
import com.jgoodies.binding.adapter.AbstractTableAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;
import com.jidesoft.swing.JideButton;
import com.jidesoft.swing.StyledLabel;

public class InfoPanel implements TabPanelProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(InfoPanel.class);

    private static final String ENCODED_DETAILS_COLUMN_SPECS =
        "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref, 3dlu, fill:10dlu:grow, 3dlu, pref";

    private final MainModel mainModel;

    private final DetailsPanel detailsPanel;

    private final JPanel contentPanel;

    private List<VendorType> vendors;

    private final PropertyChangeListener nodeListener;

    private NodeInterface selectedNode;

    private JLabel imageContainer;

    private List<FeatureCode> featureCodes;

    private final SettingsService settingsService;

    private final LookupService lookupService;

    private final FirmwareControllerFactory firmwareControllerFactory;

    private final FirmwareRepoService firmwareRepoService;

    public static class InfoFormPanel extends JPanel {

        private static final long serialVersionUID = 1L;

        public InfoFormPanel() {
            super(new BorderLayout());
        }

        @Override
        public String getName() {
            // this is used as tab title
            return Resources.getString(InfoPanel.class, "name");
        }
    }

    public InfoPanel(final MainModel model, final SettingsService settingsService, final LookupService lookupService,
        final FirmwareControllerFactory firmwareControllerFactory, final FirmwareRepoService firmwareRepoService) {
        this.mainModel = model;
        this.settingsService = settingsService;
        this.lookupService = lookupService;
        this.firmwareControllerFactory = firmwareControllerFactory;
        this.firmwareRepoService = firmwareRepoService;

        nodeListener = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                switch (evt.getPropertyName()) {
                    case org.bidib.jbidibc.messages.Node.PROPERTY_FEATURES:
                        LOGGER.info("The features have changed, update the node info.");
                        // the features have changed
                        SwingUtilities.invokeLater(() -> updateNodeInfo(selectedNode));
                        break;
                    case org.bidib.jbidibc.messages.Node.PROPERTY_USERNAME:
                        LOGGER.info("The username has changed, update the node info.");
                        // the features have changed
                        SwingUtilities.invokeLater(() -> updateNodeInfo(selectedNode));
                        break;
                }
            }
        };

        // load the feature codes
        featureCodes = BidibFactory.getFeatureCodes();

        vendors = DecoderVendorFactory.getDecoderVendors();

        this.detailsPanel = createDetailsPanel();

        this.contentPanel = this.detailsPanel.getContentPanel();
    }

    @Override
    public JPanel getComponent() {
        return contentPanel;
    }

    private DetailsPanel createDetailsPanel() {

        DetailsPanel detailsPanel = new DetailsPanel(new InfoModel());

        final Consumer<SoftwareVersion> firmwareUpdateConsumer = sw -> {
            final NodeInterface node = InfoPanel.this.selectedNode;

            JFrame frame = (JFrame) JOptionPane.getFrameForComponent(InfoPanel.this.getComponent());

            final SoftwareVersion newFirmware = node.getUpdateFirmwareVersion();
            LOGGER.info("Show the firmware update dialog for node: {}, newFirmware: {}", node, newFirmware);

            FirmwareController firmwareController =
                this.firmwareControllerFactory.createFirmwareController(firmwareRepoService, node, frame);
            firmwareController.start(newFirmware);
        };

        detailsPanel.preparePanel(firmwareUpdateConsumer);
        return detailsPanel;
    }

    public void nodeChanged() {
        LOGGER.info("The selected node has changed.");
        updateComponentState();
    }

    private void updateComponentState() {

        final NodeInterface node = mainModel.getSelectedNode();

        if (selectedNode != null && !selectedNode.equals(node)) {
            // the selected node has changed.
            selectedNode.getNode().removePropertyChangeListener(nodeListener);
        }

        final InfoModel infoModel = detailsPanel.getPresentationModel().getBean();
        if (node != null) {
            infoModel.setFeatures(null);

            selectedNode = node;

            // register the change listener for features
            selectedNode.getNode().addPropertyChangeListener(nodeListener);

            updateNodeInfo(node);
        }
        else {
            infoModel.setFeatures(null);
            selectedNode = null;
        }
    }

    private void updateNodeInfo(final NodeInterface node) {
        if (node == null) {
            // no node available
            this.detailsPanel.getFirmwareUpdateButton().setEnabled(false);
            return;
        }

        final org.bidib.jbidibc.messages.Node coreNode = node.getNode();

        LOGGER
            .info("Update the node info panel for node: {}, coreNode: {}, coreNode identity: {}", node, coreNode,
                coreNode.getIdentity());

        long uniqueId = node.getUniqueId();
        int vendorId = NodeUtils.getVendorId(node.getUniqueId());
        int productId = NodeUtils.getPid(node.getUniqueId(), coreNode.getRelevantPidBits());

        LOGGER
            .info("Search for vendorId: {} (0x{}), productId: {} (0x{})", vendorId, ByteUtils.intToHex(vendorId),
                productId, ByteUtils.intToHex(productId));

        Optional<ProductType> product = this.lookupService.getProduct(vendorId, productId);

        // prepare information of node
        final StringBuilder description = new StringBuilder();
        String url = null;
        if (product.isPresent()) {
            description.append(product.get().getName());
            // get the documentation in the correct language
            String language = XmlLocaleUtils.getXmlLocale();
            DocumentationType doc = ProductsUtils.getDocumentationOfLanguage(product.get(), language);
            if (doc != null) {
                if (StringUtils.isNotBlank(doc.getDescription())) {
                    description.append(" - ").append(doc.getDescription());
                }

                url = doc.getUrl();
            }
        }
        else {
            LOGGER.info("No products available to get the description for node: {}", node);
        }

        if (description.length() == 0) {
            description.append(Resources.getString(getClass(), "product-unknown"));
        }

        String productName = coreNode.getStoredString(StringData.INDEX_PRODUCTNAME);
        if (StringUtils.isBlank(productName)) {
            productName = "";
        }

        final PresentationModel<InfoModel> presentationModel = this.detailsPanel.getPresentationModel();

        final InfoModel infoModel = presentationModel.getBean();
        infoModel.setDescription(description.toString());
        infoModel.setProductName(productName);
        infoModel.setVendor(DecoderVendorFactory.getDecoderVendorName(vendors, vendorId));
        infoModel.setHomepage(url);
        infoModel.setUniqueId(uniqueId);
        infoModel.setUserName(BidibNodeNameUtils.getNodeName(node));
        infoModel.setAddress(NodeUtils.formatAddressLong(coreNode.getAddr()));
        infoModel.setFirmware(coreNode.getSoftwareVersion());
        infoModel.setProtocol(coreNode.getProtocolVersion() != null ? coreNode.getProtocolVersion().toString() : null);

        infoModel.setFeatures(Collections.unmodifiableList(getSortedFeatures(node)));

        this.detailsPanel.resortFeaturesTable();

        this.detailsPanel.getFirmwareUpdateButton().setEnabled(node.isUpdatable());

        // try to load image
        try {
            LOGGER.debug("Size of container: {}", imageContainer.getSize());

            ImageIcon icon = loadImage(vendorId, productId, 200, 200);
            imageContainer.setIcon(icon);
        }
        catch (Exception ex) {
            LOGGER.warn("Load image of node failed, vid: {}, pid: {}.", vendorId, productId, ex);
            imageContainer.setIcon(null);
        }
    }

    private List<Feature> getSortedFeatures(final NodeInterface node) {
        final List<Feature> features = new ArrayList<>();
        features.addAll(node.getNode().getFeatures());
        Collections.sort(features, (o1, o2) -> {
            return o1.getType() - o2.getType();
        });

        return features;
    }

    private static String getFeatureUnit(List<FeatureCode> featureCodes, Feature feature) {

        if (feature != null && feature.getFeatureEnum() != null) {

            switch (feature.getFeatureEnum()) {
                case FEATURE_BST_AMPERE:
                    // calculate the current value
                    return String.format("%d mA", ConversionUtils.convertCurrent(feature.getValue()));
                default:
                    break;
            }

            FeatureCode featureCode = IterableUtils.find(featureCodes, new Predicate<FeatureCode>() {

                @Override
                public boolean evaluate(FeatureCode featureCode) {
                    return featureCode.getId() == feature.getType();
                }
            });
            if (featureCode != null) {
                String featureUnit = featureCode.getUnit();
                if (StringUtils.isNotEmpty(featureUnit)) {
                    return featureUnit;
                }
            }
        }
        return " ";
    }

    private ImageIcon loadImage(int vendorId, long productId, int width, int height) {

        String labelPath = settingsService.getMiscSettings().getBidibConfigDir();
        String searchPathUser = "/data/images";
        File fileUser = new File(labelPath, searchPathUser);
        searchPathUser = fileUser.getPath();

        ImageIcon icon = null;
        File imageFile = new File(fileUser, "bidib-" + vendorId + "-" + productId + ".png");
        if (imageFile.exists()) {
            icon = ImageUtils.loadImageIcon(imageFile, width, height);
        }

        if (icon == null) {
            icon =
                ImageUtils
                    .loadImageIcon(getClass(), "/images/bidib-" + vendorId + "-" + productId + ".png", width, height);
        }

        return icon;
    }

    private class DetailsPanel {

        private PresentationModel<InfoModel> presentationModel;

        private JButton firmwareUpdateButton;

        private JPanel contentPanel;

        private AbstractEmptyTable featuresTable;

        public DetailsPanel(final InfoModel infoModel) {

            this.presentationModel = new PresentationModel<>(infoModel);
        }

        public PresentationModel<InfoModel> getPresentationModel() {
            return this.presentationModel;
        }

        public JButton getFirmwareUpdateButton() {
            return this.firmwareUpdateButton;
        }

        protected JPanel getContentPanel() {
            return this.contentPanel;
        }

        protected void resortFeaturesTable() {
            this.featuresTable.resort();
        }

        public void preparePanel(final Consumer<SoftwareVersion> firmwareUpdateConsumer) {

            final JPanel contentPanel = new JPanel() {
                private static final long serialVersionUID = 1L;

                @Override
                public String getName() {
                    // this is used as tab title
                    return Resources.getString(InfoPanel.class, "name");
                }
            };

            final FormBuilder builder =
                FormBuilder.create().columns(ENCODED_DETAILS_COLUMN_SPECS).rows("pref, 3dlu, pref").panel(contentPanel);
            builder.border(Paddings.TABBED_DIALOG);

            final ValueModel descriptionModel = presentationModel.getModel(InfoModel.PROPERTY_DESCRIPTION);
            builder.add(Resources.getString(InfoPanel.class, "description")).xy(1, 1);
            final JLabel description = WizardComponentFactory.createLabel(descriptionModel);
            builder.add(description).xyw(3, 1, 7);

            final ValueModel productNameModel = presentationModel.getModel(InfoModel.PROPERTY_PRODUCTNAME);
            builder.add(Resources.getString(InfoPanel.class, "productname")).xy(1, 3);
            final JLabel productName = WizardComponentFactory.createLabel(productNameModel);
            builder.add(productName).xyw(3, 3, 7);

            builder.appendRows("3dlu, pref");

            final ValueModel vendorModel = presentationModel.getModel(InfoModel.PROPERTY_VENDOR);
            builder.add(Resources.getString(InfoPanel.class, "vendor")).xy(1, 5);
            final JLabel vendorInfo = WizardComponentFactory.createLabel(vendorModel);
            builder.add(vendorInfo).xyw(3, 5, 7);

            builder.appendRows("3dlu, pref");

            final ValueModel homepageModel = presentationModel.getModel(InfoModel.PROPERTY_HOMEPAGE);
            builder.add(Resources.getString(InfoPanel.class, "homepage")).xy(1, 7);
            final JideButton homepage = WizardComponentFactory.createHyperlinkButton(homepageModel);
            homepage.setAlwaysShowHyperlink(true);
            homepage.setAction(new AbstractAction() {
                private static final long serialVersionUID = 1L;

                @Override
                public void actionPerformed(ActionEvent e) {
                    String homepageText = homepage.getText();
                    if (StringUtils.isNotBlank(homepageText)) {
                        try {
                            URI uri = new URI(homepageText);
                            Desktop.getDesktop().browse(uri);
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Open homepage failed: {}", homepageText, ex);
                        }
                    }
                }
            });
            builder.add(homepage).xyw(3, 7, 7);

            builder.appendRows("3dlu, pref");

            final ValueModel uniqueIdModel = presentationModel.getModel(InfoModel.PROPERTY_UNIQUEID);

            //
            final ValueModel uniqueIdRawModel =
                new ConverterValueModel(uniqueIdModel, new UniqueIdConverter(
                    uniqueId -> ByteUtils.convertUniqueIdToString(ByteUtils.convertLongToUniqueId(uniqueId))));
            final ValueModel uniqueIdPrettyModel =
                new ConverterValueModel(uniqueIdModel,
                    new UniqueIdConverter(uniqueId -> ByteUtils.getUniqueIdAsString(uniqueId)));

            builder.add(Resources.getString(InfoPanel.class, "uniqueId")).xy(1, 9);
            final StyledLabel uniqueIdRaw =
                WizardComponentFactory.createStyledLabel(uniqueIdRawModel, "{%1$s:f:#FF0000}");
            builder.add(uniqueIdRaw).xyw(3, 9, 3);

            final JPopupMenu uniqueIdRawPopupMenu =
                new UniqueIdPopupMenu(() -> InfoPanel.this.detailsPanel.getPresentationModel().getBean(),
                    uniqueId -> ByteUtils.convertUniqueIdToString(ByteUtils.convertLongToUniqueId(uniqueId)));
            MouseListener popupListenerRaw = new PopupListener(uniqueIdRawPopupMenu);
            uniqueIdRaw.addMouseListener(popupListenerRaw);

            final Color prettyColor = UIManager.getColor("InfoPanel.uniqueId.formatted.foreground");
            final String prettyColorHex = Integer.toString(prettyColor.getRGB(), 16);
            final StyledLabel uniqueIdPrettyPrint =
                WizardComponentFactory.createStyledLabel(uniqueIdPrettyModel, ", {%1$s:f:#" + prettyColorHex + "}");
            builder.add(uniqueIdPrettyPrint).xy(7, 9);

            final JPopupMenu uniqueIdPrettyPopupMenu =
                new UniqueIdPopupMenu(() -> InfoPanel.this.detailsPanel.getPresentationModel().getBean(),
                    uniqueId -> ByteUtils.getUniqueIdAsString(uniqueId));
            MouseListener popupListenerPretty = new PopupListener(uniqueIdPrettyPopupMenu);
            uniqueIdPrettyPrint.addMouseListener(popupListenerPretty);

            builder.appendRows("3dlu, pref");

            final ValueModel userNameModel = presentationModel.getModel(InfoModel.PROPERTY_USERNAME);
            builder.add(Resources.getString(InfoPanel.class, "username")).xy(1, 11);
            final StyledLabel userName = WizardComponentFactory.createStyledLabel(userNameModel, "{%1$s:f:#00AA55}");
            builder.add(userName).xyw(3, 11, 7);

            builder.appendRows("3dlu, pref");

            final ValueModel addressModel = presentationModel.getModel(InfoModel.PROPERTY_ADDRESS);
            builder.add(Resources.getString(InfoPanel.class, "address")).xy(1, 13);
            final JLabel address = WizardComponentFactory.createLabel(addressModel);
            builder.add(address).xyw(3, 13, 7);

            builder.appendRows("3dlu, pref");

            final ValueModel firmwareModel =
                new ConverterValueModel(presentationModel.getModel(InfoModel.PROPERTY_FIRMWARE),
                    new SoftwareVersionConverter());
            builder.add(Resources.getString(InfoPanel.class, "firmware")).xy(1, 15);
            final JLabel firmware = WizardComponentFactory.createLabel(firmwareModel);
            builder.add(firmware).xy(3, 15);

            this.firmwareUpdateButton = new JButton("Update");
            this.firmwareUpdateButton.setEnabled(false);
            this.firmwareUpdateButton.addActionListener(evt -> {
                firmwareUpdateConsumer.accept(presentationModel.getBean().getFirmware());
            });
            builder.add(this.firmwareUpdateButton).xy(5, 15);

            builder.appendRows("3dlu, pref");

            final ValueModel protocolModel = presentationModel.getModel(InfoModel.PROPERTY_PROTOCOL);
            builder.add(Resources.getString(InfoPanel.class, "protocol")).xy(1, 17);
            final JLabel protocol = WizardComponentFactory.createLabel(protocolModel);
            builder.add(protocol).xyw(3, 17, 7);

            // -------
            imageContainer = new JLabel();
            imageContainer.setPreferredSize(new Dimension(200, 200));
            imageContainer.setMaximumSize(new Dimension(200, 200));

            JPanel imagePanel = new JPanel(new BorderLayout());
            imagePanel.setOpaque(false);
            imagePanel.add(imageContainer, BorderLayout.NORTH);

            builder.add(imagePanel).xywh(11, 1, 1, 15);
            // -------

            builder.appendRows("3dlu, fill:40dlu:grow");

            final ValueModel featureModel = presentationModel.getModel(InfoModel.PROPERTY_FEATURES);

            final SelectionInList<Feature> featureList = new SelectionInList<>((List<Feature>) featureModel.getValue());
            this.featuresTable =
                WizardComponentFactory
                    .createTable(new FeatureTableAdapter(featureList, featureCodes),
                        Resources.getString(InfoPanel.class, "no-features-available"));
            featuresTable.getColumnModel().getColumn(0).setPreferredWidth(60);
            featuresTable.getColumnModel().getColumn(0).setMaxWidth(80);
            featuresTable.getColumnModel().getColumn(1).setPreferredWidth(60);
            featuresTable.getColumnModel().getColumn(1).setMaxWidth(80);
            featuresTable.getColumnModel().getColumn(2).setPreferredWidth(60);
            featuresTable.getColumnModel().getColumn(2).setMaxWidth(80);

            JScrollPane scroll = new JScrollPane();
            scroll.setViewportView(featuresTable);

            builder.add(scroll).xyw(1, 19, 11);

            // build the panel content
            this.contentPanel = builder.build();
        }
    }

    private static class UniqueIdPopupMenu extends BasicPopupMenu {
        private static final long serialVersionUID = 1L;

        private JMenuItem copyToClipboardItem;

        private final Supplier<InfoModel> infoModelSupplier;

        private final Function<Long, String> formatter;

        public UniqueIdPopupMenu(final Supplier<InfoModel> infoModelSupplier, final Function<Long, String> formatter) {
            this.infoModelSupplier = infoModelSupplier;
            this.formatter = formatter;

            this.copyToClipboardItem = new JMenuItem(Resources.getString(InfoPanel.class, "copy-to-clipboard"));
            this.copyToClipboardItem.addActionListener(e -> {

                final InfoModel infoModel = this.infoModelSupplier.get();
                Long uniqueId = infoModel.getUniqueId();
                // add to clipboard
                String uniqueIdRaw = this.formatter.apply(uniqueId);
                StringSelection selection = new StringSelection(uniqueIdRaw);
                Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                clipboard.setContents(selection, selection);

            });
            add(this.copyToClipboardItem);
        }
    }

    private static class PopupListener extends MouseAdapter {

        private final JPopupMenu popup;

        public PopupListener(JPopupMenu popup) {
            this.popup = popup;
        }

        @Override
        public void mousePressed(MouseEvent e) {
            maybeShowPopup(e);
        }

        @Override
        public void mouseReleased(MouseEvent e) {
            maybeShowPopup(e);
        }

        private void maybeShowPopup(MouseEvent e) {
            if (e.isPopupTrigger()) {
                popup.show(e.getComponent(), e.getX(), e.getY());
            }
        }
    }

    private static class FeatureTableAdapter extends AbstractTableAdapter<Feature> {

        private static final long serialVersionUID = 1L;

        private static final String[] COLUMNNAMES =
            new String[] { Resources.getString(FeatureTableAdapter.class, "id"),
                Resources.getString(FeatureTableAdapter.class, "value"),
                Resources.getString(FeatureTableAdapter.class, "unit"),
                Resources.getString(FeatureTableAdapter.class, "name") };

        private final List<FeatureCode> featureCodes;

        public FeatureTableAdapter(final SelectionInList<Feature> featureList, List<FeatureCode> featureCodes) {
            super(featureList, COLUMNNAMES);
            this.featureCodes = featureCodes;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            final Feature feature = getRow(rowIndex);
            switch (columnIndex) {
                case 0:
                    final FeatureEnum featureEnum = feature.getFeatureEnum();
                    return featureEnum != null ? featureEnum.getNumber() : feature.getType();
                case 1:
                    return feature.getValue();
                case 2:
                    return getFeatureUnit(featureCodes, feature);
                case 3:
                    return feature.getFeatureName();
                default:
                    break;
            }
            return null;
        }
    }
}
