package org.bidib.wizard.mvc.preferences.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.time.DayOfWeek;
import java.time.LocalTime;
import java.time.format.TextStyle;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Locale;
import java.util.function.Consumer;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;

import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.converter.StringToIntegerConverter;
import org.bidib.wizard.client.common.preferences.view.panel.AbstractSettingsPanel;
import org.bidib.wizard.client.common.text.InputValidationDocument;
import org.bidib.wizard.client.common.text.WizardComponentFactory;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.mvc.common.view.converter.StringToLocalTimeConverter;
import org.bidib.wizard.mvc.preferences.model.PreferencesModel;
import org.bidib.wizard.mvc.preferences.view.listener.PreferencesViewListener;

import com.jgoodies.binding.adapter.Bindings;
import com.jgoodies.binding.adapter.ComboBoxAdapter;
import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.binding.value.ConverterValueModel;
import com.jgoodies.binding.value.ValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;

/**
 * This panel displays the time preferences.
 * 
 */
public class TimePanel extends AbstractSettingsPanel<GlobalSettingsInterface> {
    private final Collection<PreferencesViewListener> listeners = new LinkedList<PreferencesViewListener>();

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "pref, 3dlu, pref, 3dlu, pref, 3dlu, pref";

    private final GlobalSettingsInterface globalSettings;

    private JPanel contentPanel;

    private final PreferencesModel model;

    public TimePanel(final PreferencesModel model, final GlobalSettingsInterface globalSettings, final Consumer<Boolean> bufferingCallback) {
        super(bufferingCallback);
        this.model = model;
        this.globalSettings = globalSettings;
    }

    public JPanel createPanel() {

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        // use value model here and only update settings on focus lost
        preparePresentationModel(this.globalSettings);

        // time factor
        final ValueModel timeFactorConverterModel =
            new ConverterValueModel(presentationModel.getBufferedModel(GlobalSettings.PROPERTY_TIME_FACTOR),
                new StringToIntegerConverter(Integer.valueOf(60)));

        dialogBuilder.add(Resources.getString(getClass(), "timeFactor") + ":").xy(1, 1);
        final JTextField timeFactor = new JTextField();
        timeFactor.setDocument(new InputValidationDocument(4, InputValidationDocument.NUMERIC));

        // bind manually because we changed the document of the textfield
        Bindings.bind(timeFactor, timeFactorConverterModel, false);

        dialogBuilder.add(timeFactor).xy(3, 1);

        // serial port provider
        final BufferedValueModel selectionHolderDayOfWeek =
                presentationModel.getBufferedModel(GlobalSettings.PROPERTY_DAY_OF_WEEK);

        SelectionInList<DayOfWeek> dayOfWeekSelection =
                new SelectionInList<>((ListModel<DayOfWeek>) model.getDayOfWeekList());

        ComboBoxAdapter<DayOfWeek> comboBoxAdapterDayOfWeek =
                new ComboBoxAdapter<>(dayOfWeekSelection, selectionHolderDayOfWeek);
        final JComboBox<DayOfWeek> comboDayOfWeek = new JComboBox<>();
        comboDayOfWeek.setRenderer(new DayOfWeekRenderer());
        comboDayOfWeek.setModel(comboBoxAdapterDayOfWeek);

        dialogBuilder.add(Resources.getString(TimePanel.class, "dayOfWeek") + ":").xy(1, 3);
        dialogBuilder.add(comboDayOfWeek).xy(3, 3);

        // start time
        final ValueModel startTimeModel =
            new ConverterValueModel(presentationModel.getBufferedModel(GlobalSettings.PROPERTY_START_TIME),
                new StringToLocalTimeConverter());

        final InputValidationDocument startTimeValidatingDocument =
            new InputValidationDocument(4, InputValidationDocument.NUMERIC);
        JSpinner startTime =
            WizardComponentFactory
                .createSpinnerTemporalEditor(startTimeModel, false, LocalTime.of(0, 0), LocalTime.of(23, 59),
                    startTimeValidatingDocument);

        dialogBuilder.add(Resources.getString(getClass(), "startTime")).xy(1, 5);
        dialogBuilder.add(startTime).xy(3, 5);

        // discovery
        final BufferedValueModel modelTimeEnabledModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_MODEL_TIME_ENABLED);

        JCheckBox modelTimeEnabled =
            WizardComponentFactory
                .createCheckBox(modelTimeEnabledModel, Resources.getString(getClass(), "modelTimeEnabled"));
        dialogBuilder.add(modelTimeEnabled).xyw(1, 7, 5);

        contentPanel = dialogBuilder.build();
        contentPanel.setOpaque(false);
        return contentPanel;
    }

    class DayOfWeekRenderer extends JLabel implements ListCellRenderer<DayOfWeek> {

        public DayOfWeekRenderer() {
            setOpaque(true);
        }

        public Component getListCellRendererComponent(
                JList list,
                DayOfWeek value,
                int index,
                boolean isSelected,
                boolean cellHasFocus) {

            if (isSelected) {
                setBackground(list.getSelectionBackground());
                setForeground(list.getSelectionForeground());
            } else {
                setBackground(list.getBackground());
                setForeground(list.getForeground());
            }

            setText(value.getDisplayName(TextStyle.FULL, Locale.getDefault()));

            return this;
        }
    }

}
