package org.bidib.wizard.dialog;

import java.awt.BorderLayout;
import java.awt.Component;
import java.util.function.Consumer;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.WindowConstants;

import org.bidib.jbidibc.messages.helpers.Context;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.netbidib.NetBidibContextKeys;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.dialog.EscapeDialog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.forms.FormsSetup;
import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.factories.Paddings;

public class PairingDialog extends EscapeDialog {

    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(PairingDialog.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 10dlu, fill:50dlu:grow";

    public enum PairingDialogType {
        PairingRequired, PairingRequested;
    }

    private int result = JOptionPane.CANCEL_OPTION;

    private final Consumer<Boolean> pairingCallback;

    private Timer pairingCountDown;

    private int remainingPairingTime = 30;

    private final JProgressBar progressBar;

    public PairingDialog(final Component parent, final PairingDialogType pairingDialogType, final Context context,
        boolean modal, final Consumer<Boolean> pairingCallback) {
        super(JOptionPane.getFrameForComponent(parent), Resources.getString(PairingDialog.class, "dialogTitle"), modal);
        this.pairingCallback = pairingCallback;

        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        getContentPane().setLayout(new BorderLayout());

        FormBuilder builder =
            FormBuilder
                .create().columns(ENCODED_DIALOG_COLUMN_SPECS)
                .rows("p, 10dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p");
        builder.border(Paddings.DIALOG);

        LOGGER.info("Provided context: {}", context);

        final Integer pairingTimeout =
            context.get(NetBidibContextKeys.KEY_PAIRING_TIMEOUT, Integer.class, Integer.valueOf(30));
        this.remainingPairingTime = pairingTimeout.intValue();

        String dialogMessageTemplate =
            PairingDialogType.PairingRequired == pairingDialogType ? "dialogMessageParingRequired"
                : "dialogMessageParingRequested";

        final JLabel messageLabel =
            FormsSetup
                .getComponentFactoryDefault()
                .createTitle(Resources.getString(PairingDialog.class, dialogMessageTemplate));

        progressBar = new JProgressBar(0, this.remainingPairingTime);

        this.pairingCountDown = new Timer(1000, evt -> {
            this.remainingPairingTime--;

            // messageLabel
            // .setText(Resources.getString(PairingDialog.class, dialogMessageTemplate, this.remainingPairingTime));
            progressBar.setValue(this.remainingPairingTime);
            progressBar.setString(String.format("%ds", this.remainingPairingTime));

            if (this.remainingPairingTime == 0) {
                // normally the timer in the NetBidibClient will elapse and close the connection
                try {
                    LOGGER.info("The pairing timer has elapsed.");
                    fireCancel();
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop timer failed.", ex);
                }
            }
        });
        this.pairingCountDown.setRepeats(true);

        int row = 1;

        // add content
        builder.add(messageLabel).xyw(1, row, 3);

        row += 2;
        builder.add(Resources.getString(getClass(), "remaining-time")).xyw(1, row, 3);
        row += 2;
        builder.add(progressBar).xyw(1, row, 3);
        progressBar.setStringPainted(true);
        progressBar.setValue(this.remainingPairingTime);
        progressBar.setString(String.format("%ds", this.remainingPairingTime));

        row += 2;

        // add the data from the context

        builder.add(Resources.getString(PairingDialog.class, "productName")).xy(1, row);
        builder.add(context.get(NetBidibContextKeys.KEY_DESCRIPTOR_PROD_STRING, String.class, "")).xy(3, row);
        row += 2;
        builder.add(Resources.getString(PairingDialog.class, "userName")).xy(1, row);
        builder.add(context.get(NetBidibContextKeys.KEY_DESCRIPTOR_USER_STRING, String.class, "")).xy(3, row);
        row += 2;
        builder.add(Resources.getString(PairingDialog.class, "uniqueId")).xy(1, row);
        Long uniqueId = context.get(NetBidibContextKeys.KEY_DESCRIPTOR_UID, Long.class, null);
        String uniqueIdString = ByteUtils.getUniqueIdAsString(uniqueId);
        builder.add(uniqueIdString != null ? uniqueIdString : "").xy(3, row);
        row += 2;
        builder.add(Resources.getString(PairingDialog.class, "requestorName")).xy(1, row);
        builder.add(context.get(NetBidibContextKeys.KEY_REQUESTOR_NAME, String.class, "")).xy(3, row);

        JButton pairButton = new JButton(Resources.getString(PairingDialog.class, "pair"));
        pairButton.addActionListener(evt -> firePair());

        JButton cancel = new JButton(Resources.getString(PairingDialog.class, "cancel"));
        if (pairingCallback == null) {
            cancel.setEnabled(false);
        }

        cancel.addActionListener(evt -> fireCancel());

        JPanel buttons = null;
        if (PairingDialogType.PairingRequired == pairingDialogType) {
            buttons = new ButtonBarBuilder().addGlue().addButton(cancel).build();
        }
        else {

            buttons = new ButtonBarBuilder().addGlue().addButton(pairButton, cancel).build();
        }
        row += 2;
        builder.add(buttons).xyw(1, row, 3);

        getContentPane().add(builder.build());

        pack();
        setLocationRelativeTo(parent);

        setMinimumSize(getSize());
    }

    @Override
    public void setVisible(boolean visible) {
        LOGGER.info("Set the pairing dialog visible: {}", visible);

        if (visible == false) {
            if (this.pairingCountDown != null) {
                try {
                    this.pairingCountDown.stop();
                }
                catch (Exception ex) {
                    LOGGER.warn("Stop pairing countdown timer failed.", ex);
                }
                this.pairingCountDown = null;
            }
        }
        else {
            if (this.pairingCountDown != null) {
                this.pairingCountDown.start();
            }
        }

        super.setVisible(visible);
    }

    private void firePair() {
        LOGGER.info("Accept the pairing process.");
        if (pairingCallback != null) {
            try {
                pairingCallback.accept(Boolean.TRUE);
            }
            catch (Exception ex) {
                LOGGER.warn("Call cancelCallback failed.", ex);
            }
        }

        fireClose();
    }

    private void fireCancel() {
        LOGGER.info("Cancel the pairing process.");

        if (pairingCallback != null) {
            try {
                pairingCallback.accept(Boolean.FALSE);
            }
            catch (Exception ex) {
                LOGGER.warn("Call cancelCallback failed.", ex);
            }
        }

        fireClose();
    }

    private void fireClose() {
        LOGGER.info("Close the dialog.");

        if (this.pairingCountDown != null) {
            try {
                LOGGER.info("Stop the pairing countdown timer.");
                this.pairingCountDown.stop();
            }
            catch (Exception ex) {
                LOGGER.warn("Stop pairing countdown timer failed.", ex);
            }
            this.pairingCountDown = null;
        }

        if (SwingUtilities.isEventDispatchThread()) {
            setVisible(false);
        }
        else {
            SwingUtilities.invokeLater(() -> {
                setVisible(false);
            });
        }
    }

    public int getResult() {
        return result;
    }

}
