package org.bidib.wizard.mvc.dmx.controller;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

import org.apache.commons.collections.CollectionUtils;
import org.bidib.jbidibc.core.node.ConfigurationVariable;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.NodeListProvider;
import org.bidib.wizard.api.model.listener.NodeListListener;
import org.bidib.wizard.api.service.node.NodeService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.client.common.view.DockUtils;
import org.bidib.wizard.client.common.view.cvdef.CvDefinitionTreeHelper;
import org.bidib.wizard.client.common.view.cvdef.NodeNode;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.DmxChannel;
import org.bidib.wizard.model.ports.DmxLightPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.Port;
import org.bidib.wizard.mvc.dmx.controller.listener.DmxModelerControllerListener;
import org.bidib.wizard.mvc.dmx.model.DmxScenery;
import org.bidib.wizard.mvc.dmx.model.DmxSceneryModel;
import org.bidib.wizard.mvc.dmx.view.DmxModelerView;
import org.bidib.wizard.mvc.dmx.view.DmxSceneryView;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockingConstants;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.RelativeDockablePosition;

public class DmxModelerController implements DmxModelerControllerListener, NodeListListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(DmxModelerController.class);

    private final JFrame parent;

    private DockingDesktop desktop;

    private NodeListProvider nodeListProvider;

    private DmxSceneryModel dmxSceneryModel;

    private DmxModelerView view;

    private final NodeInterface node;

    @Autowired
    private NodeService nodeService;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    public DmxModelerController(final NodeInterface node, final JFrame parent) {
        this.parent = parent;
        this.node = node;

        dmxSceneryModel = new DmxSceneryModel();

    }

    public void start(final DockingDesktop desktop, final NodeListProvider mainModel) {
        this.nodeListProvider = mainModel;
        this.desktop = desktop;

        LOGGER.info("Start controller.");

        // check if the booster table view is already opened
        String searchKey = DockKeys.DMX_SCENERY_VIEW;
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable view = desktop.getContext().getDockableByKey(searchKey);
        if (view != null) {
            LOGGER.info("Select the existing DMX scenery view.");
            DockUtils.selectWindow(view);
            return;
        }

        createDockable();
    }

    public Dockable createDockable() {

        List<DmxChannel> dmxChannels = new LinkedList<>();
        for (int channelId = 0; channelId < 64; channelId++) {
            // + 1 because DMX channel id starts from 1
            dmxChannels.add(new DmxChannel(channelId + 1));
        }
        dmxSceneryModel.setDmxChannels(dmxChannels);

        List<Macro> macros = new LinkedList<>();
        macros.add(null);
        macros.addAll(this.node.getMacros());
        dmxSceneryModel.setMacros(macros);

        // get some CV values from OneDMX
        if (ProductUtils.isOneDMX(this.node.getUniqueId())) {
            LOGGER.info("The current node is a OneDMX.");

            // do not get all CV values from the OneDMX because the other values are loaded via MSG_LC_CONFIG messages

            // get the initial state for the DMX channels
            if (CollectionUtils.isNotEmpty(dmxChannels)) {
                Map<String, ConfigurationVariable> dmxChannelMap = null;
                dmxChannelMap =
                    new CvDefinitionTreeHelper()
                        .getNodes(this.node, "Config\\DMX channels (init)", NodeNode.COLUMN_KEY, false, null);

                if (dmxChannelMap == null) {
                    dmxChannelMap =
                        new CvDefinitionTreeHelper()
                            .getNodes(this.node, "Config\\DMX-Channels", NodeNode.COLUMN_KEY, false, null);
                }
                LOGGER.info("Fetched dmxChannelMap: {}", dmxChannelMap);

                final List<ConfigurationVariable> cvList =
                    new LinkedList<ConfigurationVariable>(dmxChannelMap.values());

                final List<ConfigurationVariable> cvValueList =
                    nodeService.queryConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, this.node, cvList);
                LOGGER.info("Fetched cvList: {}", cvValueList);

                for (DmxChannel dmxChannel : dmxChannels) {
                    int channelId = dmxChannel.getChannelId();

                    // -1 because the DMX channelId starts from 1
                    ConfigurationVariable cv = cvValueList.get(channelId - 1);
                    if (!cv.isTimeout()) {
                        int initialState = Integer.parseInt(cv.getValue());
                        if (initialState > -1 && initialState < 256) {
                            dmxChannel.setInitialState(initialState);
                        }
                    }
                }
            }

            // TODO this is old ...
            boolean oldCVHandlingEnabled = false;
            if (oldCVHandlingEnabled) {
                // get the DMX target channel of the lightports
                Map<String, ConfigurationVariable> lightPortMap = null;
                lightPortMap =
                    new CvDefinitionTreeHelper()
                        .getNodes(this.node, "Config\\Light ports", NodeNode.COLUMN_KEY, false, null);

                LOGGER.trace("returned lightPortMap: {}", lightPortMap);
                if (lightPortMap != null && lightPortMap.size() > 0) {
                    List<ConfigurationVariable> cvList = new LinkedList<ConfigurationVariable>(lightPortMap.values());

                    cvList = nodeService.queryConfigVariables(ConnectionRegistry.CONNECTION_ID_MAIN, this.node, cvList);
                    LOGGER.info("Fetched cvList: {}", cvList);

                    // the list of configured light ports
                    List<LightPort> lightPorts = this.node.getLightPorts();

                    // the list of DMX wrapped light ports
                    List<DmxLightPort> dmxLightPorts = new LinkedList<>();

                    // assign the mapped DMX channel
                    for (int index = 0; index < lightPorts.size(); index++) {
                        ConfigurationVariable cv = cvList.get(index);
                        LOGGER.info("Current cv: {}", cv);

                        LightPort lightPort = lightPorts.get(index);

                        // create a new wrapper around the lightport
                        DmxLightPort dmxLightPort = new DmxLightPort(lightPort);
                        try {
                            int channelId = Integer.parseInt(cv.getValue());
                            if (channelId > -1 /* && channelId < dmxChannels.size() */) {
                                // channelId because DMX channels starts from 0 in CVs
                                // DmxChannel dmxTargetChannel = dmxChannels.get(channelId);

                                DmxChannel dmxTargetChannel =
                                    dmxChannels
                                        .stream().filter(ch -> ch.getChannelId() == channelId).findFirst().orElse(null);
                                dmxLightPort.setDmxTargetChannel(dmxTargetChannel);
                            }
                            else {
                                LOGGER.debug("Invalid channel id for dmxChannel: {}", channelId);
                            }
                        }
                        catch (IndexOutOfBoundsException | NumberFormatException ex) {
                            LOGGER.warn("Set DMX target channel failed.", ex);
                        }
                        dmxLightPorts.add(dmxLightPort);
                    }

                    dmxSceneryModel.setLightPorts(dmxLightPorts);
                }
                else {
                    LOGGER.warn("No lightports configured.");
                }
            }
            else {

                // the list of configured light ports
                List<LightPort> lightPorts = this.node.getLightPorts();

                // the list of DMX wrapped light ports
                List<DmxLightPort> dmxLightPorts = new LinkedList<>();

                // assign the mapped DMX channel
                for (LightPort lightPort : lightPorts) {

                    DmxLightPort dmxLightPort = new DmxLightPort(lightPort);
                    int channelId = lightPort.getDmxMapping() != null ? lightPort.getDmxMapping() : -1;
                    if (channelId > -1) {

                        DmxChannel dmxTargetChannel =
                            dmxChannels
                                .stream().filter(ch -> ch.getChannelId() == (channelId + 1)).findFirst().orElse(null);

                        LOGGER.info("Prepared dmxTargetChannel: {}", dmxTargetChannel);

                        dmxLightPort.setDmxTargetChannel(dmxTargetChannel);
                    }
                    else {
                        LOGGER.debug("Invalid channel id for dmxChannel: {}", channelId);
                    }

                    dmxLightPorts.add(dmxLightPort);
                }

                dmxSceneryModel.setLightPorts(dmxLightPorts);

            }
        }

        dmxSceneryModel.setBacklightPorts(new LinkedList<>(this.node.getBacklightPorts()));

        LOGGER.info("Open the DMX scenery panel.");

        DmxSceneryView dmxSceneryView =
            new DmxSceneryView(desktop, this.node, this.dmxSceneryModel, this.settingsService, this.wizardLabelWrapper);
        dmxSceneryView.setDmxModelerControllerListener(this);
        dmxSceneryView.createPanel();

        Dockable nodeListPanel = desktop.getContext().getDockableByKey(DockKeys.NODE_LIST_PANEL);
        desktop.split(nodeListPanel, dmxSceneryView, DockingConstants.SPLIT_BOTTOM, 0.2);

        nodeListProvider.addNodeListListener(this);

        return dmxSceneryView;
    }

    @Override
    public void openView(final DmxScenery dmxScenery) {
        // check if the scenery is already opened
        String searchKey = DmxModelerView.prepareKey(dmxScenery);
        LOGGER.info("Search for view with key: {}", searchKey);
        Dockable dmxModelerView = desktop.getContext().getDockableByKey(searchKey);
        if (dmxModelerView == null) {
            LOGGER.info("Create new DmxModelerView.");

            // TODO check if it would be better to provide the DmxSceneryModel instead of the node
            // because the DmxSceneryModel has the macros of the node
            view = new DmxModelerView(desktop, dmxScenery, dmxSceneryModel);

            Dockable tabPanel = desktop.getContext().getDockableByKey(DockKeys.TAB_PANEL);
            if (tabPanel != null) {
                desktop.createTab(tabPanel, view, 1, true);
            }
            else {
                desktop.addDockable(view, RelativeDockablePosition.RIGHT);
            }

            // load the scenery points
            view.loadSceneryPoints();
        }
        else {
            LOGGER.info("Select the existing modeler view.");
            DockUtils.selectWindow(dmxModelerView);
        }
    }

    @Override
    public void closeView(final DmxSceneryModel dmxSceneryModel) {
        // check if the scenery is opened
        for (DmxScenery dmxScenery : dmxSceneryModel.getSceneries()) {
            String searchKey = DmxModelerView.prepareKey(dmxScenery);
            LOGGER.info("Search for view with key: {}", searchKey);
            Dockable dmxModelerView = desktop.getContext().getDockableByKey(searchKey);
            if (dmxModelerView != null) {
                LOGGER.info("Close the dmxModelerView: {}", dmxModelerView);

                desktop.close(dmxModelerView);
            }
        }
    }

    @Override
    public void listChanged() {
        LOGGER.info("The node list has changed.");
    }

    @Override
    public void nodeChanged(final NodeInterface node) {
        LOGGER.debug("The node has changed, current node in model: {}", node);
        if (SwingUtilities.isEventDispatchThread()) {
            internalNodeChanged(node);
        }
        else {
            SwingUtilities.invokeLater(() -> internalNodeChanged(node));
        }
    }

    private void internalNodeChanged(final NodeInterface node) {

        LOGGER.debug("handle node has changed, node: {}", this.node);

        if (this.node != null && this.node.equals(node)) {
            LOGGER.debug("The node in the model has not changed.");
            return;
        }

    }

    private void cleanupSelectedNode() {

        final List<DmxLightPort> lightPorts = dmxSceneryModel.getLightPorts();
        if (CollectionUtils.isNotEmpty(lightPorts)) {
            for (DmxLightPort lightPort : lightPorts) {
                lightPort.freeLightPort();
            }
        }

        for (DmxScenery dmxScenery : dmxSceneryModel.getSceneries()) {
            // check if the scenery is opened
            String searchKey = DmxModelerView.prepareKey(dmxScenery);
            LOGGER.info("Search for view with key: {}", searchKey);
            Dockable dmxModelerViewDockable = desktop.getContext().getDockableByKey(searchKey);
            if (dmxModelerViewDockable != null && dmxModelerViewDockable instanceof DmxModelerView) {
                LOGGER.info("Close the DmxModelerView: {}", dmxModelerViewDockable);
                desktop.close(dmxModelerViewDockable);
            }
        }

        // check if we must close the scenery panel
        Dockable dmxSceneryDockable = desktop.getContext().getDockableByKey("DmxSceneryView");
        if (dmxSceneryDockable != null) {
            LOGGER.info("Close the DMX scenery panel: {}", dmxSceneryDockable);

            desktop.close(dmxSceneryDockable);
        }

        // unregister node list listener
        nodeListProvider.removeNodeListListener(this);
    }

    @Override
    public void nodeStateChanged(final NodeInterface node) {
    }

    @Override
    public void listNodeAdded(NodeInterface node) {
    }

    @Override
    public void listNodeRemoved(NodeInterface node) {

        if (this.node != null && this.node.equals(node)) {
            LOGGER.info("The selected node was removed from the node list.");

            cleanupSelectedNode();
        }

    }

    @Override
    public void nodeWillChange(final NodeInterface node) {

    }

    @Override
    public void portConfigChanged(Port<?> port) {
        LOGGER.info("The port config has changed, port: {}", port);
        try {
            if (port instanceof DmxLightPort) {
                LightPort lightPort = ((DmxLightPort) port).getLightPort();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), lightPort);
            }
            else if (port instanceof BacklightPort) {
                BacklightPort backlightPort = (BacklightPort) port;

                LOGGER.info("Current backlightPort: {}", backlightPort);

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), backlightPort);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Set the port parameters failed.", ex);
            // model.setNodeHasError(model.getSelectedNode(), true);
        }
    }
}
