package org.bidib.wizard.mvc.dmx.model;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.bidib.jbidibc.exchange.dmxscenery.BacklightPortType;
import org.bidib.jbidibc.exchange.dmxscenery.DmxChannelType;
import org.bidib.jbidibc.exchange.dmxscenery.DmxSceneryPointType;
import org.bidib.jbidibc.exchange.dmxscenery.DmxSceneryPointsType;
import org.bidib.jbidibc.exchange.dmxscenery.DmxSceneryType;
import org.bidib.jbidibc.exchange.dmxscenery.LightPortType;
import org.bidib.jbidibc.exchange.dmxscenery.LineColorUtils;
import org.bidib.jbidibc.exchange.dmxscenery.PortType;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.jbidibc.messages.utils.CollectionUtils;
import org.bidib.wizard.api.model.function.MacroFunction;
import org.bidib.wizard.api.utils.PortUtils;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.DmxChannel;
import org.bidib.wizard.model.ports.DmxLightPort;
import org.bidib.wizard.model.ports.LightPort;
import org.bidib.wizard.model.ports.Port;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.beans.Model;

/**
 * 
 * The <code>DmxScenery</code> holds the data of a single scenery. The single scenery contains points of one or more DMX
 * channels.
 */
public class DmxScenery extends Model {
    private static final Logger LOGGER = LoggerFactory.getLogger(DmxScenery.class);

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_NAME = "name";

    public static final String PROPERTY_MACRO_NUMBER = "macroNumber";

    public static final String PROPERTY_DIMM_STRETCH = "dimmStretch";

    public static final String PROPERTY_USED_CHANNELS = "usedChannels";

    public static final String PROPERTY_SCENERY_POINTS = "sceneryPoints";

    private final String id;

    private String name;

    private List<DmxChannel> usedChannels = new LinkedList<>();

    private List<DmxSceneryPoint> sceneryPoints = new LinkedList<>();

    private Integer macroNumber;

    /**
     * Creates a new instance of DmxScenery with the provided scenery id.
     * 
     * @param id
     *            the scenery id
     */
    public DmxScenery(final String id) {
        this.id = id;
    }

    /**
     * @return the scenery id
     */
    public String getId() {
        return id;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        String oldValue = this.name;

        this.name = name;

        firePropertyChange(PROPERTY_NAME, oldValue, name);
    }

    /**
     * @return the usedChannels
     */
    public List<DmxChannel> getUsedChannels() {
        return Collections.unmodifiableList(usedChannels);
    }

    /**
     * @param usedChannels
     *            the usedChannels to set
     */
    public void setUsedChannels(List<DmxChannel> usedChannels) {
        List<DmxChannel> oldValue = new LinkedList<>(this.usedChannels);

        this.usedChannels.clear();
        this.usedChannels.addAll(usedChannels);

        firePropertyChange(PROPERTY_USED_CHANNELS, oldValue, this.usedChannels);
    }

    /**
     * @param usedChannels
     *            the usedChannels to add
     */
    public void addUsedChannels(List<DmxChannel> usedChannels) {
        List<DmxChannel> oldValue = new LinkedList<>(this.usedChannels);

        this.usedChannels.addAll(usedChannels);

        firePropertyChange(PROPERTY_USED_CHANNELS, oldValue, this.usedChannels);
    }

    /**
     * @param usedChannel
     *            the usedChannel to add
     */
    public void addUsedChannel(DmxChannel usedChannel) {
        List<DmxChannel> oldValue = new LinkedList<>(this.usedChannels);

        this.usedChannels.add(usedChannel);

        firePropertyChange(PROPERTY_USED_CHANNELS, oldValue, this.usedChannels);
    }

    /**
     * @param usedChannel
     *            the usedChannel to remove
     */
    public void removeUsedChannel(DmxChannel usedChannel) {
        List<DmxChannel> oldValue = new LinkedList<>(this.usedChannels);

        this.usedChannels.remove(usedChannel);

        firePropertyChange(PROPERTY_USED_CHANNELS, oldValue, this.usedChannels);
    }

    /**
     * @return the sceneryPoints
     */
    public List<DmxSceneryPoint> getSceneryPoints() {
        return Collections.unmodifiableList(sceneryPoints);
    }

    /**
     * @param sceneryPoints
     *            the sceneryPoints to set
     */
    public void setSceneryPoints(List<DmxSceneryPoint> sceneryPoints) {
        List<DmxSceneryPoint> oldValue = new LinkedList<>(this.sceneryPoints);

        this.sceneryPoints.clear();
        this.sceneryPoints.addAll(sceneryPoints);

        firePropertyChange(PROPERTY_SCENERY_POINTS, oldValue, this.sceneryPoints);
    }

    public void addSceneryPoint(DmxSceneryPoint dmxSceneryPoint) {
        List<DmxSceneryPoint> oldValue = new LinkedList<>(this.sceneryPoints);

        this.sceneryPoints.add(dmxSceneryPoint);

        firePropertyChange(PROPERTY_SCENERY_POINTS, oldValue, this.sceneryPoints);
    }

    public boolean removeSceneryPoint(DmxSceneryPoint dmxSceneryPoint) {
        List<DmxSceneryPoint> oldValue = new LinkedList<>(this.sceneryPoints);

        boolean removed = this.sceneryPoints.remove(dmxSceneryPoint);

        firePropertyChange(PROPERTY_SCENERY_POINTS, oldValue, this.sceneryPoints);
        return removed;
    }

    /**
     * @return the macroNumber
     */
    public Integer getMacroNumber() {
        return macroNumber;
    }

    /**
     * @param macroNumber
     *            the macroNumber to set
     */
    public void setMacroNumber(Integer macroNumber) {
        Integer oldValue = this.macroNumber;

        this.macroNumber = macroNumber;

        firePropertyChange(PROPERTY_MACRO_NUMBER, oldValue, this.macroNumber);
    }

    @Override
    public String toString() {
        return name;
    }

    public DmxScenery withDmxScenery(DmxEnvironmentProvider dmxEnvironmentProvider, DmxSceneryType dmxScenery) {
        setName(dmxScenery.getSceneryName());
        setMacroNumber(dmxScenery.getMacroNumber());
        if (CollectionUtils.hasElements(dmxScenery.getDmxChannel())) {
            List<DmxChannel> dmxChannels = new LinkedList<>();
            for (DmxChannelType dmxChannelType : dmxScenery.getDmxChannel()) {
                // check if the DMX channel is available
                int channelNumber = dmxChannelType.getChannelNumber();

                int index = dmxEnvironmentProvider.getDmxChannels().indexOf(new DmxChannel(channelNumber));
                if (index > -1) {
                    DmxChannel dmxChannel = dmxEnvironmentProvider.getDmxChannels().get(index);

                    // TODO we must reconfigure the OneDMX if a port is not assigned to a channel

                    // configure the assigned channels
                    withDmxChannel(dmxChannel, dmxEnvironmentProvider, dmxChannelType);
                    dmxChannels.add(dmxChannel);
                }
                else {
                    LOGGER.warn("No DMX channel found in system with channel number: {}", channelNumber);
                }
            }

            // set the channels
            addUsedChannels(dmxChannels);
        }

        if (dmxScenery.getDmxSceneryPoints() != null) {
            // handle the scenery points
            for (DmxSceneryPointType dmxSceneryPointType : dmxScenery.getDmxSceneryPoints().getDmxSceneryPoint()) {
                DmxSceneryPoint dmxSceneryPoint =
                    new DmxSceneryPoint()
                        .withTimeOffset(dmxSceneryPointType.getTimeOffset())
                        .withBrightness(dmxSceneryPointType.getBrightness())
                        .withUniqueId(dmxSceneryPointType.getUniqueId())
                        .withDmxChannelNumber(dmxSceneryPointType.getDmxChannelNumber());

                // set the macro number
                if (dmxSceneryPointType.getMacroNumber() != null) {
                    MacroFunction macro = new MacroFunction();
                    macro.setMacroId(dmxSceneryPointType.getMacroNumber());
                    dmxSceneryPoint.setMacro(macro);
                }

                // set the port and action
                if (dmxSceneryPointType.getPort() != null) {
                    dmxSceneryPoint.withPort(fromPortType(dmxEnvironmentProvider, dmxSceneryPointType.getPort()));
                    dmxSceneryPoint
                        .withAction(PortUtils
                            .actionFromPortType(dmxSceneryPointType.getPort(), dmxSceneryPointType.getBidibAction()));

                }

                sceneryPoints.add(dmxSceneryPoint);
            }
        }

        return this;
    }

    public DmxChannel withDmxChannel(
        final DmxChannel dmxChannel, DmxEnvironmentProvider dmxEnvironmentProvider, DmxChannelType dmxChannelType) {
        dmxChannel.setLineColor(LineColorUtils.getColor(dmxChannelType.getLineColor()));

        // TODO check if the DMX target channel of all imported ports are configured correct
        int dmxTargetChannel = dmxChannelType.getChannelNumber();

        // add the ports
        if (CollectionUtils.hasElements(dmxChannelType.getPort())) {
            List<Port<?>> ports = new LinkedList<>();
            for (PortType portType : dmxChannelType.getPort()) {
                LOGGER.info("Current port: {}", portType);

                if (portType instanceof BacklightPortType) {
                    int portNum = portType.getPortNumber();
                    for (BacklightPort port : dmxEnvironmentProvider.getBacklightPorts()) {
                        if (port.getId() == portNum) {
                            // found matching port
                            ports.add(port);
                            break;
                        }
                    }
                }
                else if (portType instanceof LightPortType) {
                    int portNum = portType.getPortNumber();
                    for (DmxLightPort port : dmxEnvironmentProvider.getLightPorts()) {
                        if (port.getId() == portNum) {
                            // found matching port
                            ports.add(port);
                            break;
                        }
                    }
                }
                else {
                    LOGGER.warn("Unsupported port type detected: {}", portType);
                }
            }

            // TODO if we have autoAdd of ports enabled all ports that are mapped to the DMX channel are added
            // automatically
            boolean autoAddPorts = false;
            if (autoAddPorts) {
                // check if the OneDMX has more ports assigned to the current channel
                for (DmxLightPort dmxLightPort : dmxEnvironmentProvider.getLightPorts()) {
                    // check if the DMX target channel matches the id of the current DMX channel
                    if (dmxLightPort.getDmxTargetChannel() != null
                        && dmxLightPort.getDmxTargetChannel().getChannelId() == dmxTargetChannel) {
                        // check if we must assign this port
                        if (!ports.contains(dmxLightPort)) {
                            LOGGER.info("Add new dmxLightPort that is configured in the OneDMX: {}", dmxLightPort);
                            ports.add(dmxLightPort);
                        }
                    }
                }

                for (BacklightPort backlightPort : dmxEnvironmentProvider.getBacklightPorts()) {
                    // check if the DMX target channel matches the id of the current DMX channel
                    if (backlightPort.getDmxMapping() == dmxTargetChannel) {
                        // check if we must assign this port
                        if (!ports.contains(backlightPort)) {
                            LOGGER.info("Add new backlightPort that is configured in the OneDMX: {}", backlightPort);
                            ports.add(backlightPort);
                        }
                    }
                }
            }

            LOGGER.info("Set the assigned ports: {}", ports);
            dmxChannel.setAssignedPorts(ports);
        }

        return dmxChannel;
    }

    private Port<?> fromPortType(DmxEnvironmentProvider dmxEnvironmentProvider, PortType portType) {

        if (portType != null) {
            int portNumber = portType.getPortNumber();

            // search the ports ...
            if (portType instanceof LightPortType) {
                // return get the lightport
                for (DmxLightPort lightPort : dmxEnvironmentProvider.getLightPorts()) {
                    if (lightPort.getId() == portNumber) {
                        return lightPort;
                    }
                }
            }
            else if (portType instanceof BacklightPortType) {
                // return get the backlightport
                for (BacklightPort backlightPort : dmxEnvironmentProvider.getBacklightPorts()) {
                    if (backlightPort.getId() == portNumber) {
                        return backlightPort;
                    }
                }
            }
        }
        LOGGER.warn("No port found for portType: {}", portType);
        return null;
    }

    private PortType fromPort(Port<?> port) {
        PortType portType = null;
        if (port instanceof DmxLightPort) {
            LightPort lightPort = ((DmxLightPort) port).getLightPort();
            portType = new LightPortType().withPortNumber(lightPort.getId());
        }
        else if (port instanceof BacklightPort) {
            portType = new BacklightPortType().withPortNumber(port.getId());
        }
        else {
            LOGGER.warn("Unsupported port detected: {}", port);
        }
        return portType;
    }

    public DmxSceneryType fromDmxScenery() {
        DmxSceneryType dmxSceneryType =
            new DmxSceneryType().withSceneryName(getName()).withMacroNumber(getMacroNumber());
        // store the channels
        for (DmxChannel dmxChannel : getUsedChannels()) {
            DmxChannelType dmxChannelType =
                new DmxChannelType()
                    .withChannelName(dmxChannel.toString()).withChannelNumber(dmxChannel.getChannelId())
                    .withLineColor(LineColorUtils.getColorType(dmxChannel.getLineColor()));
            // add the ports
            for (Port<?> port : dmxChannel.getAssignedPorts()) {
                PortType portType = fromPort(port);
                if (portType != null) {
                    dmxChannelType.getPort().add(portType);
                }
            }
            dmxSceneryType.getDmxChannel().add(dmxChannelType);
        }

        if (CollectionUtils.hasElements(sceneryPoints)) {
            DmxSceneryPointsType dmxSceneryPoints = new DmxSceneryPointsType();
            for (DmxSceneryPoint dmxSceneryPoint : sceneryPoints) {
                // check if the port is available
                PortType portType = fromPort(dmxSceneryPoint.getPort());
                DmxSceneryPointType dmxSceneryPointType =
                    new DmxSceneryPointType()
                        .withTimeOffset(dmxSceneryPoint.getTimeOffset())
                        .withDmxChannelNumber(dmxSceneryPoint.getDmxChannelNumber())
                        .withBrightness(dmxSceneryPoint.getBrightness()).withUniqueId(dmxSceneryPoint.getUniqueId());
                if (portType != null) {
                    dmxSceneryPointType.withPort(portType);

                    if (dmxSceneryPoint.getAction() != null) {
                        dmxSceneryPointType
                            .withBidibAction(ByteUtils.getInt(dmxSceneryPoint.getAction().getType().getType()));
                    }

                }
                else if (dmxSceneryPoint.getMacro() != null) {
                    dmxSceneryPointType.withMacroNumber(dmxSceneryPoint.getMacro().getMacroId());
                }
                else {
                    LOGGER.warn("Incomplete scenery point without port assigned is stored.");
                }
                dmxSceneryPoints.getDmxSceneryPoint().add(dmxSceneryPointType);
            }
            dmxSceneryType.setDmxSceneryPoints(dmxSceneryPoints);
        }
        return dmxSceneryType;
    }

}
