package org.bidib.wizard.mvc.main.controller;

import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;

import org.bidib.jbidibc.core.schema.bidiblabels.AccessoryLabel;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.AccessorySwitchTimeModel;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.service.console.ConsoleService;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.labels.AccessoryLabelUtils;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.mvc.main.model.AccessoryStartupAspectModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.AccessoryRequestListener;
import org.bidib.wizard.mvc.main.view.component.DefaultBusyFrame;
import org.bidib.wizard.mvc.main.view.panel.AccessoryListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.bushe.swing.event.annotation.AnnotationProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

public class AccessoryPanelController implements AccessoryRequestListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(AccessoryPanelController.class);

    private final MainModel mainModel;

    private final AccessoryStartupAspectModel accessoryStartupAspectModel;

    private final AccessorySwitchTimeModel accessorySwitchTimeModel;

    private AccessoryListPanel accessoryListPanel;

    @Autowired
    private SwitchingNodeService switchingService;

    @Autowired
    private SettingsService settingsService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    @Autowired
    private StatusBar statusBar;

    @Autowired
    private ConsoleService consoleService;

    public AccessoryPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        // create the startup aspect model
        accessoryStartupAspectModel = new AccessoryStartupAspectModel();

        // create the switch time model
        accessorySwitchTimeModel = new AccessorySwitchTimeModel();

        // add the eventbus processing
        AnnotationProcessor.process(this);

    }

    public AccessoryListPanel createAccessoryListPanel(final TabVisibilityListener tabVisibilityListener) {

        accessoryListPanel =
            new AccessoryListPanel(this, mainModel, accessoryStartupAspectModel, accessorySwitchTimeModel,
                tabVisibilityListener, settingsService, wizardLabelWrapper, statusBar, consoleService);

        // add a list selection listener for the accessory list
        accessoryListPanel.addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(final ListSelectionEvent e) {
                if (!e.getValueIsAdjusting()) {

                    try {

                        // the selected accessory has changed
                        DefaultBusyFrame.setWaitCursor(accessoryListPanel.getComponent());

                        final DefaultTableModel accessoryListModel = (DefaultTableModel) e.getSource();

                        int selectedRow = e.getFirstIndex();

                        if (selectedRow > -1) {
                            Object value = accessoryListModel.getValueAt(selectedRow, 0);
                            LOGGER.info("The selected accessory has changed: {}, selectedRow: {}", value, selectedRow);

                            if (!(value instanceof Accessory)) {
                                LOGGER.info("Discard change of label.");
                                return;
                            }
                            Accessory accessory = (Accessory) value;

                            if (accessory != null) {
                                mainModel.setSelectedAccessory(accessory);
                            }
                        }
                    }
                    finally {
                        DefaultBusyFrame.setDefaultCursor(accessoryListPanel.getComponent());
                    }
                }
            }
        });

        return accessoryListPanel;
    }

    public List<Macro> getMacros() {
        return mainModel.getMacros();
    }

    public NodeInterface getSelectedNode() {
        return mainModel.getSelectedNode();
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    public AccessoryLabel getAccessoryAspectsLabels(int accessoryId) {

        NodeLabels nodeLabels = getNodeLabels();
        if (nodeLabels == null) {
            LOGGER.warn("No node labels avaialble.");
            return null;
        }

        return AccessoryLabelUtils.getAccessoryLabel(nodeLabels, accessoryId);
    }

    public void setAccessoryAspectLabel(int accessoryId, int aspectId, String label) {

        NodeLabels nodeLabels = getNodeLabels();
        if (nodeLabels == null) {
            LOGGER.info("No node labels avaialble.");
            return;
        }

        boolean save = true;

        setAccessoryAspectLabel(accessoryId, aspectId, label, save);
    }

    public void setAccessoryAspectLabel(int accessoryId, int aspectId, String label, boolean save) {

        NodeLabels nodeLabels = getNodeLabels();
        AccessoryLabelUtils.replaceAspectLabel(nodeLabels, accessoryId, aspectId, label);

        if (save) {
            saveLabels();
        }
    }

    private void saveLabels() {
        try {
            long uniqueId = getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save accessory labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void activateAspect(final Accessory accessory, int aspectNumber) {
        LOGGER.info("Start accessory with aspect: {}, accessory: {}", aspectNumber, accessory);

        try {
            DefaultBusyFrame.setWaitCursor(accessoryListPanel.getComponent());

            switchingService
                .setAccessoryAspect(ConnectionRegistry.CONNECTION_ID_MAIN, getSelectedNode().getSwitchingNode(),
                    accessory, aspectNumber);
        }
        finally {
            DefaultBusyFrame.setDefaultCursor(accessoryListPanel.getComponent());
        }
    }

    @Override
    public void storeAccessory(final Accessory accessory) {
        LOGGER.info("Store the accessory: {}", accessory);

        try {
            DefaultBusyFrame.setWaitCursor(accessoryListPanel.getComponent());

            // create a clone of the accessory
            final Accessory accessoryClone = Accessory.cloneAccessoryData(accessory);

            switchingService
                .saveAccessory(ConnectionRegistry.CONNECTION_ID_MAIN, getSelectedNode().getSwitchingNode(),
                    accessoryClone);
        }
        finally {
            DefaultBusyFrame.setDefaultCursor(accessoryListPanel.getComponent());
        }
    }

    public void reloadAccessory(Accessory accessory) {
        LOGGER.info("Reload the accessory: {}", accessory);

        try {
            DefaultBusyFrame.setWaitCursor(accessoryListPanel.getComponent());

            // create a clone of the accessory
            final Accessory accessoryClone = Accessory.cloneAccessoryData(accessory);

            switchingService
                .reloadAccessory(ConnectionRegistry.CONNECTION_ID_MAIN, getSelectedNode().getSwitchingNode(),
                    accessoryClone);
        }
        finally {
            DefaultBusyFrame.setDefaultCursor(accessoryListPanel.getComponent());
        }
    }

    public void saveAccessoryLabel(final Accessory accessory, String label) {

        LOGGER.info("Save the accessory label, accessory: {}, label: {}", accessory, label);

        accessory.setLabel(label);

        NodeLabels nodeLabels = getNodeLabels();

        AccessoryLabelUtils.replaceAccessoryLabel(nodeLabels, accessory.getId(), label);

        saveLabels();
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.accessoryListPanel != null) {
            SwingUtilities.invokeLater(() -> this.accessoryListPanel.refreshView());
        }
    }

}
