package org.bidib.wizard.mvc.main.controller;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import org.apache.commons.collections4.MapUtils;
import org.bidib.jbidibc.core.schema.bidiblabels.NodeLabels;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.enums.LcOutputType;
import org.bidib.jbidibc.messages.enums.PortConfigKeys;
import org.bidib.jbidibc.messages.exception.InvalidConfigurationException;
import org.bidib.jbidibc.messages.port.BytePortConfigValue;
import org.bidib.jbidibc.messages.port.Int16PortConfigValue;
import org.bidib.jbidibc.messages.port.PortConfigValue;
import org.bidib.jbidibc.messages.utils.ByteUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.BacklightPortValueListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.api.utils.PortListUtils;
import org.bidib.wizard.common.labels.LabelsChangedEvent;
import org.bidib.wizard.common.labels.WizardLabelFactory;
import org.bidib.wizard.common.labels.WizardLabelWrapper;
import org.bidib.wizard.core.labels.BidibLabelUtils;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.model.ports.BacklightPort;
import org.bidib.wizard.model.ports.PortTypeAware;
import org.bidib.wizard.model.ports.ServoPort;
import org.bidib.wizard.model.ports.event.PortConfigChangeEvent;
import org.bidib.wizard.mvc.main.model.BacklightPortTableModel;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.model.listener.BacklightPortModelListener;
import org.bidib.wizard.mvc.main.view.exchange.NodeExchangeHelper;
import org.bidib.wizard.mvc.main.view.panel.BacklightPortListPanel;
import org.bidib.wizard.mvc.main.view.panel.listener.TabVisibilityListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;

import io.reactivex.rxjava3.disposables.CompositeDisposable;
import io.reactivex.rxjava3.disposables.Disposable;
import io.reactivex.rxjava3.subjects.PublishSubject;

public class BacklightPortPanelController {

    private static final Logger LOGGER = LoggerFactory.getLogger(BacklightPortPanelController.class);

    private final MainModel mainModel;

    @Autowired
    private SwitchingNodeService switchingNodeService;

    @Autowired
    private WizardLabelWrapper wizardLabelWrapper;

    private BacklightPortListPanel backlightPortListPanel;

    private final PublishSubject<PortConfigChangeEvent> portConfigChangeEventSubject = PublishSubject.create();

    private CompositeDisposable compDisp;

    public BacklightPortPanelController(final MainModel mainModel) {
        this.mainModel = mainModel;

        this.compDisp = new CompositeDisposable();
    }

    public BacklightPortListPanel createPanel(final TabVisibilityListener tabVisibilityListener) {

        final BacklightPortTableModel tableModel = new BacklightPortTableModel();

        // this port listener signals the changes from the tablemodel that must be written to the node
        tableModel.setPortListener(new BacklightPortModelListener() {

            @Override
            public void labelChanged(final BacklightPort port, String label) {
                port.setLabel(label);

                try {
                    final NodeLabels nodeLabels = getNodeLabels();
                    BidibLabelUtils
                        .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.backlightPort, port.getId(),
                            port.getLabel());
                    saveLabels();
                }
                catch (InvalidConfigurationException ex) {
                    LOGGER.warn("Save backlight port labels failed.", ex);

                    String labelPath = ex.getReason();
                    JOptionPane
                        .showMessageDialog(JOptionPane.getFrameForComponent(null), Resources
                            .getString(NodeExchangeHelper.class, "labelfileerror.message", new Object[] { labelPath }),
                            Resources.getString(NodeExchangeHelper.class, "labelfileerror.title"),
                            JOptionPane.ERROR_MESSAGE);
                }
            }

            @Override
            public void configChanged(final BacklightPort port, final PortConfigKeys... portConfigKeys) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                try {
                    final NodeInterface node = mainModel.getSelectedNode();
                    switchingNodeService
                        .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), port);
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the backlightport config failed.", ex);

                    mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                }
            }

            @Override
            public void changePortType(LcOutputType portType, BacklightPort port) {
                LOGGER.info("The port type will change to: {}, port: {}", portType, port);

                final Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                        mainModel.getSelectedNode().getSwitchingNode(), port, portType, values);
            }

            @Override
            public void testButtonPressed(final BacklightPort port, int requestedValue) {
                LOGGER.info("Test pressed for port: {}, requestedValue: {}", port, requestedValue);

                try {
                    // create a new instance of the port
                    BacklightPort backlightPort = new BacklightPort();
                    backlightPort.setId(port.getId());
                    backlightPort.setValue(requestedValue);

                    final NodeInterface node = mainModel.getSelectedNode();
                    switchingNodeService
                        .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, node.getSwitchingNode(), backlightPort);
                }
                catch (Exception ex) {
                    LOGGER.warn("Set the backlightport status failed.", ex);

                    mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                }
            }
        });

        this.backlightPortListPanel =
            new BacklightPortListPanel(this, tableModel, mainModel, tabVisibilityListener,
                portConfigChangeEventSubject);

        // this port listener signals the changed values from the node
        backlightPortListPanel.setPortListener(new BacklightPortValueListener() {

            @Override
            public Class<?> getPortClass() {
                return BacklightPort.class;
            }

            @Override
            public void labelChanged(BacklightPort port, String label) {
                LOGGER.info("The label has been changed by nodeScript, port: {}, label: {}", port, label);

                final NodeLabels nodeLabels = getNodeLabels();
                BidibLabelUtils
                    .replaceLabel(nodeLabels, WizardLabelFactory.LabelTypes.backlightPort, port.getId(),
                        port.getLabel());
                saveLabels();

                backlightPortListPanel.repaint();
            }

            @Override
            public void valueChanged(final NodeInterface node, final BacklightPort port) {
                LOGGER.info("Value has changed for backlight port: {}", port);

                SwingUtilities.invokeLater(() -> tableModel.notifyPortStatusChanged(port));
            }

            @Override
            public void configChanged(final NodeInterface node, final BacklightPort port) {
                LOGGER.info("The configuration of the port has changed: {}", port);

                // update port config
                SwingUtilities.invokeLater(() -> tableModel.notifyPortConfigChanged(port));
            }

            @Override
            public void valuesChanged(final BacklightPort port, final PortConfigKeys... portConfigKeys) {
                LOGGER.info("The port value are changed for port: {}, portConfigKeys: {}", port, portConfigKeys);

                Map<Byte, PortConfigValue<?>> values = new LinkedHashMap<>();

                // we must get the configured port
                final List<ServoPort> servoPorts = mainModel.getSelectedNode().getServoPorts();
                final ServoPort servoPort = PortListUtils.findPortByPortNumber(servoPorts, port.getId());

                for (PortConfigKeys key : portConfigKeys) {

                    if (!servoPort.isPortConfigKeySupported(key)) {
                        LOGGER.info("Unsupported port config key detected: {}", key);
                        continue;
                    }

                    switch (key) {
                        case BIDIB_PCFG_DIMM_DOWN:
                            int dimMin = port.getDimSlopeDown();

                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_DOWN_8_8,
                                        new Int16PortConfigValue(ByteUtils.getWORD(dimMin)));
                            }
                            else if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_DOWN)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_DOWN,
                                        new BytePortConfigValue(ByteUtils.getLowByte(dimMin)));
                            }
                            break;
                        case BIDIB_PCFG_DIMM_UP:
                            int dimMax = port.getDimSlopeUp();

                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_UP_8_8,
                                        new Int16PortConfigValue(ByteUtils.getWORD(dimMax)));
                            }
                            else if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_DIMM_UP)) {
                                values
                                    .put(BidibLibrary.BIDIB_PCFG_DIMM_UP,
                                        new BytePortConfigValue(ByteUtils.getLowByte(dimMax)));
                            }
                            break;
                        case BIDIB_PCFG_OUTPUT_MAP:
                            Integer dmxMapping = port.getDmxMapping();
                            if (port.isPortConfigKeySupported(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP)
                                && dmxMapping != null) {
                                LOGGER
                                    .info("The backlightport has dmxMapping support: {}, dmxMapping: {}", port,
                                        dmxMapping);

                                values
                                    .put(BidibLibrary.BIDIB_PCFG_OUTPUT_MAP,
                                        new BytePortConfigValue(ByteUtils.getLowByte(dmxMapping)));
                            }
                            break;

                        default:
                            LOGGER.warn("Unsupported port config key detected: {}", key);
                            break;
                    }
                }

                if (MapUtils.isNotEmpty(values)) {
                    try {
                        LOGGER.info("Set the port params: {}", values);
                        // don't set the port type param to not send BIDIB_PCFG_RECONFIG
                        switchingNodeService
                            .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN,
                                mainModel.getSelectedNode().getSwitchingNode(), port, null, values);
                    }
                    catch (Exception ex) {
                        LOGGER.warn("Set the backlightport parameters failed.", ex);

                        mainModel.setNodeHasError(mainModel.getSelectedNode(), true);
                    }
                }
                else {
                    LOGGER.info("No config values to save available.");
                }
            }

            @Override
            public void testButtonPressed(final NodeInterface node, final BacklightPort port, int requestedValue) {

            }
        });

        mainModel.addNodeListListener(new DefaultNodeListListener() {
            @Override
            public void nodeChanged(NodeInterface node) {
                super.nodeChanged(node);

                LOGGER.info("The selected node has been changed: {}", node);

                compDisp.dispose();
                compDisp.clear();

                compDisp = new CompositeDisposable();

                if (node != null) {
                    addBacklightPortModelListener(node);
                }
            }
        });

        NodeInterface selectedNode = mainModel.getSelectedNode();
        if (selectedNode != null) {
            addBacklightPortModelListener(selectedNode);
        }

        return this.backlightPortListPanel;
    }

    private NodeLabels getNodeLabels() {
        final WizardLabelFactory wizardLabelFactory = wizardLabelWrapper.getWizardLabelFactory();

        NodeLabels nodeLabels = wizardLabelFactory.loadLabels(mainModel.getSelectedNode().getUniqueId());
        return nodeLabels;
    }

    private void saveLabels() {
        try {
            long uniqueId = mainModel.getSelectedNode().getUniqueId();
            wizardLabelWrapper.saveNodeLabels(uniqueId);
        }
        catch (Exception e) {
            LOGGER.warn("Save backlight labels failed.", e);
            throw new RuntimeException(e);
        }
    }

    @EventListener(LabelsChangedEvent.class)
    public void labelsChangedEvent(LabelsChangedEvent labelsChangedEvent) {
        LOGGER.info("The labels have changed, node: {}", labelsChangedEvent);

        if (this.backlightPortListPanel != null) {
            SwingUtilities.invokeLater(() -> this.backlightPortListPanel.refreshView());
        }
    }

    private void addBacklightPortModelListener(final NodeInterface selectedNode) {

        LOGGER.info("Add backlight port model listener for node: {}", selectedNode);

        final Disposable disp = this.portConfigChangeEventSubject.subscribe(evt -> {
            LOGGER.info("Received event: {}", evt);

            final PortTypeAware port = evt.getPort();

            // update port config
            try {
                final BacklightPort backlightPort = new BacklightPort();
                backlightPort.setId(port.getPortNumber());

                LOGGER.info("Prepared backlight port: {}", backlightPort);

                switchingNodeService
                    .setPortConfig(ConnectionRegistry.CONNECTION_ID_MAIN, selectedNode.getSwitchingNode(),
                        backlightPort, null, evt.getPortConfig());
            }
            catch (Exception ex) {
                LOGGER.warn("Set the backlight port config failed.", ex);
                selectedNode.setNodeHasError(true);
                selectedNode.setReasonData("Set the backlight port config failed.");
            }
        });

        compDisp.add(disp);
    }
}
