package org.bidib.wizard.mvc.main.model;

import java.util.Collection;
import java.util.LinkedList;

import javax.swing.table.DefaultTableModel;

import org.apache.commons.lang3.StringUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.AccessoryAspect;
import org.bidib.wizard.api.model.AccessoryAspectMacro;
import org.bidib.wizard.api.model.Macro;
import org.bidib.wizard.api.model.MacroRef;
import org.bidib.wizard.client.common.view.listener.ButtonListener;
import org.bidib.wizard.mvc.main.model.listener.AccessoryPortListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is the table model for the accessory aspects table.
 * 
 */
public class AccessoryTableModel extends DefaultTableModel implements ButtonListener {
    private static final long serialVersionUID = 1L;

    private static final Logger LOGGER = LoggerFactory.getLogger(AccessoryTableModel.class);

    public static final int COLUMN_LABEL = 0;

    public static final int COLUMN_MACRO_REF = 1;

    public static final int COLUMN_ACCESSORY_ASPECT_INSTANCE = 2;

    private static final String[] COLUMN_NAMES =
        new String[] { Resources.getString(AccessoryTableModel.class, "aspect"),
            Resources.getString(AccessoryTableModel.class, "macro"),
            Resources.getString(AccessoryTableModel.class, "test") };

    private final Collection<AccessoryPortListener> portListeners = new LinkedList<>();

    private final MainModel model;

    private Accessory selectedAccessory;

    public AccessoryTableModel(final MainModel model) {
        this.model = model;

        setColumnIdentifiers(COLUMN_NAMES);
    }

    public void addPortListener(AccessoryPortListener l) {
        portListeners.add(l);
    }

    public void setSelectedAccessory(Accessory selectedAccessory) {
        this.selectedAccessory = selectedAccessory;
    }

    public Accessory getSelectedAccessory() {
        return selectedAccessory;
    }

    public void addRow(AccessoryAspect accessoryAspect) {

        if (accessoryAspect != null) {
            Object[] rowData = new Object[COLUMN_NAMES.length];

            rowData[COLUMN_LABEL] = accessoryAspect.toString();
            if (accessoryAspect instanceof AccessoryAspectMacro) {
                // add the macroRef instance
                rowData[COLUMN_MACRO_REF] = ((AccessoryAspectMacro) accessoryAspect).getMacroRef();
            }
            rowData[COLUMN_ACCESSORY_ASPECT_INSTANCE] = accessoryAspect;

            LOGGER.info("Add new row: {}", rowData);
            addRow(rowData);
        }

    }

    private void fireTestButtonPressed(int aspectId) {
        for (AccessoryPortListener l : portListeners) {
            l.testButtonPressed(selectedAccessory.getId(), aspectId);
        }
    }

    private void fireAspectLabelChanged(int aspectIndex) {
        for (AccessoryPortListener l : portListeners) {
            l.labelChanged(selectedAccessory.getId(), aspectIndex);
        }
    }

    /**
     * Get the row instance.
     * 
     * @param row
     *            the row
     * @return the row instance
     */
    public AccessoryAspect getAccessoryAspectInstance(int row) {
        AccessoryAspect result = (AccessoryAspect) super.getValueAt(row, COLUMN_ACCESSORY_ASPECT_INSTANCE);
        return result;
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        boolean isEditable = false;
        switch (column) {
            case COLUMN_MACRO_REF /* macro */:
                Accessory accessory = selectedAccessory;
                if (accessory != null) {
                    isEditable = accessory.isMacroMapped();

                    if (isEditable) {
                        try {
                            AccessoryAspect accessoryAspect = getAccessoryAspectInstance(row);
                            if (accessoryAspect instanceof AccessoryAspectMacro) {
                                // the current accessory aspect is macro mapped
                                isEditable = !((AccessoryAspectMacro) accessoryAspect).isImmutableAccessory();
                                LOGGER.info("The current aspect is immutable: {}", !isEditable);
                            }
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Check if aspect is immutable failed.", ex);
                        }
                    }
                }
                break;
            case COLUMN_ACCESSORY_ASPECT_INSTANCE:
                isEditable = true;
                break;
            default:
                isEditable = true;
                break;
        }

        return isEditable;
    }

    @Override
    public void setValueAt(Object value, int row, int column) {
        LOGGER.info("setValueAt, row: {}, column: {}, value: {}", row, column, value);
        switch (column) {
            case COLUMN_LABEL:
                // if the label is empty we trim to null
                String label = StringUtils.trimToNull((String) value);
                AccessoryAspect accessoryAspect = getAccessoryAspectInstance(row);
                if (accessoryAspect != null) {
                    accessoryAspect.setLabel(label);
                }

                LOGGER.info("Set label value in model: {}", label);
                super.setValueAt(label, row, column);

                // keep the label in the AccessoryAspectsLabels
                fireAspectLabelChanged(row);

                break;
            case COLUMN_MACRO_REF:
                if (value instanceof Macro) {
                    // set the new macro directly in the selected accessory of the main model
                    int macroId = ((Macro) value).getId();
                    LOGGER.info("Set the macroId: {}", macroId);
                    model.getSelectedAccessory().replaceAspectMacro(row, macroId);

                    accessoryAspect = getAccessoryAspectInstance(row);
                    if (accessoryAspect instanceof AccessoryAspectMacro) {
                        MacroRef macroRef = ((AccessoryAspectMacro) accessoryAspect).getMacroRef();
                        macroRef.setId(macroId);
                        super.setValueAt(macroRef, row, column);
                    }
                    else {
                        fireTableCellUpdated(row, column);
                    }
                }
                else {
                    LOGGER.warn("Current value is not a macro: {}", value);
                    MacroRef macroRef = (MacroRef) value;
                    if (macroRef.getId() != null) {
                        int macroId = macroRef.getId();
                        LOGGER.info("Set the macroId: {}", macroId);
                        model.getSelectedAccessory().replaceAspectMacro(row, macroId);
                    }

                    super.setValueAt(value, row, column);
                }
                break;
            case COLUMN_ACCESSORY_ASPECT_INSTANCE:
                // fireTestButtonPressed(row);
                break;
            default:
                super.setValueAt(value, row, column);
        }
    }

    @Override
    public Object getValueAt(int row, int column) {

        Object value = null;

        switch (column) {
            case COLUMN_LABEL:

                AccessoryAspect accessoryAspect =
                    (AccessoryAspect) super.getValueAt(row, COLUMN_ACCESSORY_ASPECT_INSTANCE);
                String label = null;
                if (accessoryAspect != null) {
                    label = accessoryAspect.getLabel();
                }
                // if (StringUtils.isBlank(label)) {
                // label = Resources.getString(getClass(), "aspect") + "_" + row;
                // }
                value = label;
                break;
            case COLUMN_MACRO_REF:
            case COLUMN_ACCESSORY_ASPECT_INSTANCE:
                value = super.getValueAt(row, column);
                break;
        }

        LOGGER.debug("getValueAt, row: {}, column: {}, value: {}", row, column, value);

        return value;
    }

    @Override
    public void buttonPressed(int row, int column, Object value) {
        LOGGER.info("The button was pressed, row: {}, column: {}", row, column);
        if (column == COLUMN_ACCESSORY_ASPECT_INSTANCE) {
            fireTestButtonPressed(row);
        }
    }
}
