package org.bidib.wizard.mvc.preferences.view.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileFilter;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.api.model.common.CommPort;
import org.bidib.wizard.api.model.common.NetBidibServiceInfo;
import org.bidib.wizard.api.model.common.PreferencesPortType;
import org.bidib.wizard.api.model.common.PreferencesPortType.ConnectionPortType;
import org.bidib.wizard.client.common.preferences.view.panel.AbstractSettingsPanel;
import org.bidib.wizard.common.model.settings.ConnectionConfiguration;
import org.bidib.wizard.common.model.settings.GlobalSettingsInterface;
import org.bidib.wizard.common.model.settings.WizardSettingsInterface;
import org.bidib.wizard.core.dialog.FileDialog;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.model.settings.GlobalSettings;
import org.bidib.wizard.mvc.preferences.model.PreferencesModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.value.BufferedValueModel;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jgoodies.forms.factories.Paddings;

public class ConnectionPanel extends AbstractSettingsPanel<GlobalSettingsInterface> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionPanel.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS =
        "top:pref, 8dlu, top:pref, 8dlu, top:pref, 8dlu, top:pref, 8dlu, pref";

    private final PreferencesModel model;

    private final GlobalSettingsInterface globalSettings;

    private CommPortRadioPanel commPortRadioPanel;

    private SpeedometerPanel speedometerPanel;

    private NetBidibPanel netBidibPanel;

    private SerialOverTcpPortRadioPanel serialOverTcpPortRadioPanel;

    private JLabel labelSerial;

    private JLabel labelSerialOverTcp;

    private JLabel labelNetBidib;

    private JLabel labelSpeedometer;

    private JPanel contentPanel;

    private final WizardSettingsInterface wizardSettings;

    private BufferedValueModel selectedPortTypeModel;

    private BufferedValueModel previousSelectedNetBidibHostModel;

    private BufferedValueModel previousSelectedSpeedometerPortModel;

    private BufferedValueModel previousSelectedSerialOverTcpHostModel;

    private BufferedValueModel previousSelectedSerialSymLinkModel;

    private BufferedValueModel previousSelectedComPortModel;

    private BufferedValueModel previousSelectedSimulationFileNameModel;

    public ConnectionPanel(final PreferencesModel model, final GlobalSettingsInterface globalSettings,
        final WizardSettingsInterface wizardSettings, final Consumer<Boolean> bufferingCallback) {
        super(bufferingCallback);
        this.model = model;
        this.globalSettings = globalSettings;
        this.wizardSettings = wizardSettings;
    }

    public JPanel createPanel() {

        labelSerial = new JLabel(Resources.getString(MiscSettingsPanel.class, "serialPort") + ":");
        labelSerialOverTcp = new JLabel(Resources.getString(MiscSettingsPanel.class, "serialOverTcpPort") + ":");
        labelNetBidib = new JLabel(Resources.getString(MiscSettingsPanel.class, "netBidib") + ":");
        labelSpeedometer = new JLabel(Resources.getString(MiscSettingsPanel.class, "speedometer") + ":");

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        dialogBuilder.border(Paddings.TABBED_DIALOG);

        preparePresentationModel(this.globalSettings);

        dialogBuilder.add(labelSerial).xy(1, 1);

        final JComboBox<CommPort> commPortsCombo = new JComboBox<>();

        ButtonGroup group = new ButtonGroup();
        commPortRadioPanel = new CommPortRadioPanel(group, commPortsCombo);
        dialogBuilder.add(commPortRadioPanel).xy(3, 1);

        // serial over tcp panel
        dialogBuilder.add(labelSerialOverTcp).xy(1, 3);

        serialOverTcpPortRadioPanel = new SerialOverTcpPortRadioPanel(group);
        dialogBuilder.add(serialOverTcpPortRadioPanel).xy(3, 3);

        // netbidib panel
        dialogBuilder.add(labelNetBidib).xy(1, 5);

        final JComboBox<NetBidibServiceInfo> netBidibServiceInfoCombo = new JComboBox<>();

        netBidibPanel = new NetBidibPanel(group, netBidibServiceInfoCombo);
        dialogBuilder.add(netBidibPanel).xy(3, 5);

        // speedometer panel
        dialogBuilder.add(labelSpeedometer).xy(1, 7);

        final JComboBox<CommPort> commPortsSpeedometer = new JComboBox<>();

        speedometerPanel = new SpeedometerPanel(group, commPortsSpeedometer);
        dialogBuilder.add(speedometerPanel).xy(3, 7);
        // dialogBuilder.nextLine(2);

        speedometerPanel.addPeer(commPortRadioPanel);
        speedometerPanel.addPeer(netBidibPanel);
        speedometerPanel.addPeer(serialOverTcpPortRadioPanel);

        commPortRadioPanel.addPeer(netBidibPanel);
        commPortRadioPanel.addPeer(speedometerPanel);
        commPortRadioPanel.addPeer(serialOverTcpPortRadioPanel);

        netBidibPanel.addPeer(commPortRadioPanel);
        netBidibPanel.addPeer(speedometerPanel);
        netBidibPanel.addPeer(serialOverTcpPortRadioPanel);

        serialOverTcpPortRadioPanel.addPeer(commPortRadioPanel);
        serialOverTcpPortRadioPanel.addPeer(speedometerPanel);
        serialOverTcpPortRadioPanel.addPeer(netBidibPanel);

        // set the comm ports after creation of the radio panel
        setCommPorts(commPortsCombo);
        setCommPortsSpeedometer(commPortsSpeedometer);
        setNetBidibServices(netBidibServiceInfoCombo);
        commPortRadioPanel.setEnabled(globalSettings.isSerialPortsEnabled());
        speedometerPanel.setEnabled(globalSettings.isSpeedometerEnabled());
        netBidibPanel.setEnabled(globalSettings.isNetBidibEnabled());
        serialOverTcpPortRadioPanel.setEnabled(globalSettings.isSerialOverTcpEnabled());

        commPortRadioPanel.setInitialValues(globalSettings);
        speedometerPanel.setInitialValues(globalSettings);
        netBidibPanel.setInitialValues(globalSettings);
        serialOverTcpPortRadioPanel.setInitialValues(globalSettings);

        selectedPortTypeModel = presentationModel.getBufferedModel(GlobalSettings.PROPERTY_SELECTED_PORTTYPE);

        // prepare the value models
        previousSelectedNetBidibHostModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_NETBIDIB_HOST);

        previousSelectedSpeedometerPortModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_SPEEDOMETER_PORT);

        previousSelectedSerialOverTcpHostModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_SERIAL_OVER_TCP_HOST);

        previousSelectedSerialSymLinkModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_SERIAL_SYMLINK);

        previousSelectedComPortModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_COM_PORT);

        previousSelectedSimulationFileNameModel =
            presentationModel.getBufferedModel(GlobalSettings.PROPERTY_PREVIOUS_SELECTED_SIMLATION_FILENAME);

        // select the radio button
        if (selectedPortTypeModel.getValue() != null) {
            PreferencesPortType selectedPortType = (PreferencesPortType) selectedPortTypeModel.getValue();

            switch (selectedPortType.getConnectionPortType()) {
                case SerialPort:
                case SerialSymLink:
                case SerialSimulation:
                    commPortRadioPanel.select(selectedPortType);
                    break;
                case NetBidibClient:
                    netBidibPanel.select(selectedPortType);
                    break;
                case SerialOverTcp:
                case SerialOverTcpSimulation:
                    serialOverTcpPortRadioPanel.select(selectedPortType);
                    break;
                default:
                    PreferencesPortType speedometerPortType =
                        PreferencesPortType
                            .getValue(ConnectionPortType.Speedometer.name() + ":"
                                + globalSettings.getPreviousSelectedSpeedometerPort());
                    speedometerPanel.select(speedometerPortType);
                    break;
            }
        }

        // add preferences model listener
        model.addPropertyChangeListener(event -> {
            switch (event.getPropertyName()) {
                case PreferencesModel.PROPERTY_DETECTED_NETBIDIB_SERVICES:
                    setNetBidibServices(netBidibServiceInfoCombo);
                    break;
                case PreferencesModel.PROPERTY_COMM_PORTS:
                    setCommPorts(commPortsCombo);
                    break;
                default:
                    break;
            }
        });

        contentPanel = dialogBuilder.build();
        contentPanel.setOpaque(false);

        return contentPanel;
    }

    private void fireSelectedPortTypeChanged(PreferencesPortType portType) {
        LOGGER.info("Selected port type has changed: {}", portType);

        if (selectedPortTypeModel != null) {
            selectedPortTypeModel.setValue(portType);
        }
    }

    private void setCommPorts(JComboBox<CommPort> commPortsCombo) {
        Set<CommPort> commPorts = new LinkedHashSet<>();
        CommPort activeCommPort = null;
        String activeCommPortName = null;

        // must get the correct connection from the globalSettings
        PreferencesPortType portType =
            ConnectionConfiguration
                .toPreferencesPortType(globalSettings.getConnectionConfigurations(),
                    ConnectionRegistry.CONNECTION_ID_MAIN);

        if (portType != null && ConnectionPortType.SerialPort.equals(portType.getConnectionPortType())) {
            activeCommPortName = extractPortName(portType.getConnectionName());
            if ("null".equals(activeCommPortName)) {
                activeCommPortName = null;
            }
            LOGGER.debug("The activeCommPortName: {}", activeCommPortName);
        }
        else if (StringUtils.isNotBlank(globalSettings.getPreviousSelectedComPort())) {
            activeCommPortName = extractPortName(globalSettings.getPreviousSelectedComPort());
            if ("null".equals(activeCommPortName)) {
                activeCommPortName = null;
            }
            LOGGER.debug("The activeCommPortName: {}", activeCommPortName);
        }

        for (CommPort commPort : model.getAvailableCommPorts()) {
            commPorts.add(commPort);
            if (SystemUtils.IS_OS_WINDOWS) {

                if (activeCommPortName != null && activeCommPortName.startsWith(commPort.getName())) {
                    activeCommPort = commPort;
                }
            }
            else {
                if (activeCommPortName != null && activeCommPortName.equals(commPort.getName())) {
                    activeCommPort = commPort;
                }
            }
        }

        LOGGER.debug("Add empty item if someone wants to release the port.");
        CommPort commPortNone = new CommPort(CommPort.NONE);
        commPorts.add(commPortNone);

        commPortsCombo.setModel(new DefaultComboBoxModel<CommPort>(commPorts.toArray(new CommPort[0])));

        if (activeCommPort != null) {
            LOGGER.debug("Select the active commport: {}", activeCommPort);
            commPortRadioPanel.setSelectedPort(activeCommPort);
        }
        else {
            // on linux and mac we must support symlinks ...
            if (StringUtils.isNotBlank(activeCommPortName) && !CommPort.NONE.equals(activeCommPortName)) {
                LOGGER.debug("Set the active symlink: {}", activeCommPortName);
                commPortRadioPanel.setSymlink(activeCommPortName);
            }
            else {
                LOGGER.debug("Select <none> item because no default active port is set.");
                commPortRadioPanel.setSelectedPort(commPortNone);
            }
        }
    }

    private void setCommPortsSpeedometer(JComboBox<CommPort> commPortsCombo) {
        Set<CommPort> commPorts = new LinkedHashSet<>();
        CommPort activeCommPort = null;
        String activeCommPortName = null;

        // must get the correct connection from the globalSettings
        PreferencesPortType portType =
            ConnectionConfiguration
                .toPreferencesPortType(globalSettings.getConnectionConfigurations(),
                    ConnectionRegistry.CONNECTION_ID_MAIN);

        if (portType != null && ConnectionPortType.Speedometer.equals(portType.getConnectionPortType())) {
            activeCommPortName = extractPortName(portType.getConnectionName());
            if ("null".equals(activeCommPortName)) {
                activeCommPortName = null;
            }
            LOGGER.debug("The activeCommPortName: {}", activeCommPortName);
        }
        else if (StringUtils.isNotBlank(globalSettings.getPreviousSelectedSpeedometerPort())) {
            activeCommPortName = extractPortName(globalSettings.getPreviousSelectedSpeedometerPort());
            if ("null".equals(activeCommPortName)) {
                activeCommPortName = null;
            }
            LOGGER.debug("The activeCommPortName: {}", activeCommPortName);
        }

        for (CommPort commPort : model.getAvailableCommPorts()) {
            commPorts.add(commPort);
            if (activeCommPortName != null && activeCommPortName.equals(commPort.getName())) {
                activeCommPort = commPort;
            }
        }

        LOGGER.debug("Add empty item if someone wants to release the port.");
        CommPort commPortNone = new CommPort(CommPort.NONE);
        commPorts.add(commPortNone);

        commPortsCombo.setModel(new DefaultComboBoxModel<CommPort>(commPorts.toArray(new CommPort[0])));

        if (activeCommPort != null) {
            LOGGER.debug("Select the active commport: {}", activeCommPort);
            speedometerPanel.setSelectedPort(activeCommPort);
        }
        else {
            LOGGER.debug("Select <none> item because no default active port is set.");
            speedometerPanel.setSelectedPort(commPortNone);
        }
    }

    private void setNetBidibServices(final JComboBox<NetBidibServiceInfo> netBidibServiceInfoCombo) {

        // must get the correct connection from the globalSettings
        PreferencesPortType portType =
                ConnectionConfiguration
                        .toPreferencesPortType(globalSettings.getConnectionConfigurations(),
                                ConnectionRegistry.CONNECTION_ID_MAIN);

        String activeNetBidibServiceName = null;
        if (portType != null && ConnectionPortType.NetBidibClient.equals(portType.getConnectionPortType())) {
            activeNetBidibServiceName = extractPortName(portType.getConnectionName());
            if ("null".equals(activeNetBidibServiceName)) {
                activeNetBidibServiceName = null;
            }
            LOGGER.debug("The activeNetBidibServiceName: {}", activeNetBidibServiceName);
        }
        else if (StringUtils.isNotBlank(globalSettings.getPreviousSelectedNetBidibHost())) {
            activeNetBidibServiceName = extractPortName(globalSettings.getPreviousSelectedNetBidibHost());
            if ("null".equals(activeNetBidibServiceName)) {
                activeNetBidibServiceName = null;
            }
            LOGGER.debug("The activeNetBidibServiceName: {}", activeNetBidibServiceName);
        }

        Set<NetBidibServiceInfo> netBidibServices = new LinkedHashSet<>();

        final Set<NetBidibServiceInfo> detectedNetBidibServices = model.getDetectedNetBidibServices();
        LOGGER.info("Detected netBidib services: {}", detectedNetBidibServices);

        NetBidibServiceInfo activeServiceInfo = null;

        for (NetBidibServiceInfo si : detectedNetBidibServices) {
            netBidibServices.add(si);

            if (activeNetBidibServiceName != null && activeNetBidibServiceName.equals(si.getQualifiedName())) {
                activeServiceInfo = si;
            }
        }
        final NetBidibServiceInfo serviceInfoNone = new NetBidibServiceInfo(NetBidibServiceInfo.NONE);
        netBidibServices.add(serviceInfoNone);

        netBidibServiceInfoCombo.setModel(new DefaultComboBoxModel<>(netBidibServices.toArray(new NetBidibServiceInfo[0])));

        if (activeServiceInfo != null) {
            LOGGER.debug("Select the active service info: {}", activeServiceInfo);
            netBidibPanel.setSelectedService(activeServiceInfo);
        }
        else {
            LOGGER.debug("Select <none> item because no default active port is set.");
            netBidibPanel.setSelectedService(serviceInfoNone);
        }
    }


    /**
     * Extract the portName to access the serial ports.
     * 
     * @param connectionUri
     *            the connection uri
     * @return the port name to access the serial port
     */
    private String extractPortName(String connectionUri) {

        // if (connectionUri != null && connectionUri.indexOf(" - ") > 0) {
        // connectionUri = connectionUri.substring(0, connectionUri.indexOf(" - "));
        // }
        return connectionUri;
    }

    private interface PortRadioPanel extends ActionListener {
        void selectDefault();
    }

    private class NetBidibPanel extends JPanel implements PortRadioPanel {
        private static final long serialVersionUID = 1L;

        public static final String DEFAULT_TCP_PORT = "62875";

        public static final String DEFAULT_TCP_HOST = "localhost";

        private JTextField textField;

        private JTextField portTextField;

        private JRadioButton textFieldButton;

        private JRadioButton serviceInfoButton;

        private JComboBox<NetBidibServiceInfo> netBidibServiceInfoCombo;

        private List<PortRadioPanel> peer = new ArrayList<>();

        public void addPeer(PortRadioPanel peer) {
            this.peer.add(peer);
        }

        public NetBidibPanel(ButtonGroup group, JComboBox<NetBidibServiceInfo> netBidibServiceInfoCombo) {
            super(new GridBagLayout());
            setOpaque(false);
            super.setEnabled(false);

            this.netBidibServiceInfoCombo = netBidibServiceInfoCombo;

            textFieldButton = new JRadioButton("");
            textFieldButton.setContentAreaFilled(false);

            serviceInfoButton = new JRadioButton("");
            serviceInfoButton.setContentAreaFilled(false);

            // Group the radio buttons.
            group.add(textFieldButton);
            group.add(serviceInfoButton);

            textField = new JTextField(20);
            final DocumentListener documentListener = new DocumentListener() {

                @Override
                public void removeUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void changedUpdate(DocumentEvent e) {
                    setText();
                }

                private void setText() {
                    if (textFieldButton.isSelected()) {
                        PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.NetBidibClient.name());

                        String host =
                            (StringUtils.isNotBlank(textField.getText()) ? textField.getText() : DEFAULT_TCP_HOST);
                        String port =
                            (StringUtils.isNotBlank(portTextField.getText()) ? portTextField.getText()
                                : DEFAULT_TCP_PORT);
                        String connectionName = String.format("%1$s:%2$s", host, port);
                        portType.setConnectionName(connectionName);
                        fireSelectedPortTypeChanged(portType);
                    }

                    if (previousSelectedNetBidibHostModel != null) {
                        previousSelectedNetBidibHostModel
                            .setValue(String.format("%1$s:%2$s", textField.getText(), portTextField.getText()));
                    }
                }
            };
            textField.getDocument().addDocumentListener(documentListener);

            portTextField = new JTextField(5);
            portTextField.setText(DEFAULT_TCP_PORT);
            portTextField.getDocument().addDocumentListener(documentListener);

            textFieldButton.addActionListener(NetBidibPanel.this);
            serviceInfoButton.addActionListener(NetBidibPanel.this);

            this.netBidibServiceInfoCombo.setRenderer(new DefaultListCellRenderer() {
                @Override
                public Component getListCellRendererComponent(JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                    Component renderer = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);

                    if (value instanceof NetBidibServiceInfo) {
                        NetBidibServiceInfo si = (NetBidibServiceInfo) value;
                        String text = NetBidibServiceInfo.format(si);
                        setText(text);
                    }

                    return renderer;
                }
            });

            // netBidib service info
            this.netBidibServiceInfoCombo.addActionListener( e -> {
                final NetBidibServiceInfo netBidibServiceInfo = (NetBidibServiceInfo) NetBidibPanel.this.netBidibServiceInfoCombo.getSelectedItem();
                LOGGER.debug("The selection of the netBidibServiceInfo has changed: {}", netBidibServiceInfo);

                if (netBidibServiceInfo != null) {


                    if (netBidibServiceInfo != null && !NetBidibServiceInfo.NONE.equals(netBidibServiceInfo.getQualifiedName())) {
                        LOGGER.info("The netBidib service was selected, auto select the netBidib service radio button.");
                        NetBidibPanel.this.serviceInfoButton.setSelected(true);
                    }

                    final PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.NetBidibClient.name());
                    String host = netBidibServiceInfo.getAddresses() != null && netBidibServiceInfo.getAddresses().length > 0 ? netBidibServiceInfo.getAddresses()[0].getHostAddress() : DEFAULT_TCP_HOST;
                    String port = String.valueOf(netBidibServiceInfo.getPort());

                    String connectionName = String.format("%1$s:%2$s", host, port);
                    portType.setConnectionName(connectionName);
                    fireSelectedPortTypeChanged(portType);
                }

                if (previousSelectedNetBidibHostModel != null) {
                    previousSelectedNetBidibHostModel
                            .setValue(String.format("%1$s:%2$s", textField.getText(), portTextField.getText()));
                }
            });

            GridBagConstraints gbc = new GridBagConstraints();
            gbc.insets = new Insets(0, 0, 5, 0);
            add(serviceInfoButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            gbc.gridwidth = 3;
            add(this.netBidibServiceInfoCombo, gbc);

            //gbc.gridwidth = 1;
            gbc = new GridBagConstraints();
            gbc.gridy = 1;
            gbc.insets = new Insets(0, 0, 5, 0);
            add(textFieldButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(textField, gbc);

            gbc.gridx = 2;
            gbc.weightx = 0.0;
            add(new JLabel(":"), gbc);

            gbc.gridx = 3;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(portTextField, gbc);
        }

        private void setSelectedService(final NetBidibServiceInfo serviceInfo) {
            this.netBidibServiceInfoCombo.setSelectedItem(serviceInfo);
            actionPerformed(null);
        }

        @Override
        public void actionPerformed(ActionEvent e) {

            textField.setEnabled(textFieldButton.isEnabled());
            portTextField.setEnabled(textFieldButton.isEnabled());

            if (e != null && (e.getSource() == textFieldButton || e.getSource() == serviceInfoButton)) {
                if (CollectionUtils.isNotEmpty(peer)) {
                    LOGGER.trace("Update peers from netBidibPort.");
                    for (PortRadioPanel p : peer) {
                        p.actionPerformed(e);
                    }
                }
            }

            if (e != null) {
                if (e.getSource() == textFieldButton) {
                    PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.NetBidibClient.name());
                    String connectionName = String.format("%1$s:%2$s", textField.getText(), portTextField.getText());
                    portType.setConnectionName(connectionName);
                    fireSelectedPortTypeChanged(portType);
                }
                else if (e.getSource() == serviceInfoButton) {
                    PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.NetBidibClient.name());
                    final NetBidibServiceInfo netBidibServiceInfo = (NetBidibServiceInfo) netBidibServiceInfoCombo.getSelectedItem();
                    String host = netBidibServiceInfo.getAddresses() != null && netBidibServiceInfo.getAddresses().length > 0 ? netBidibServiceInfo.getAddresses()[0].getHostAddress() : DEFAULT_TCP_HOST;
                    String port = String.valueOf(netBidibServiceInfo.getPort());

                    String connectionName = String.format("%1$s:%2$s", host, port);
                    portType.setConnectionName(connectionName);
                    fireSelectedPortTypeChanged(portType);
                }
            }
        }

        @Override
        public void setEnabled(boolean enabled) {
            boolean old = isEnabled();
            if (old != enabled) {
                super.setEnabled(enabled);

                serviceInfoButton.setEnabled(enabled);
                netBidibServiceInfoCombo.setEnabled(enabled);

                textFieldButton.setEnabled(enabled);
                portTextField.setEnabled(enabled);
                actionPerformed(null);

                if (!enabled) {
                    // check if we we have a selected option on this panel, if yes, remove it by selecting the default
                    // option of the peer
                    if (textFieldButton.isSelected() || serviceInfoButton.isSelected()) {
                        if (CollectionUtils.isNotEmpty(peer)) {
                            for (PortRadioPanel p : peer) {
                                p.selectDefault();
                            }
                        }
                    }
                }
            }
        }

        @Override
        public void selectDefault() {
        }

        private NetBidibServiceInfo getSelectedServiceInfo(String host, int port) {
            final DefaultComboBoxModel<NetBidibServiceInfo> comboModel = (DefaultComboBoxModel<NetBidibServiceInfo>) netBidibServiceInfoCombo.getModel();
            NetBidibServiceInfo selectedServiceInfo = null;
            for (int index = 0; index < comboModel.getSize(); index++) {
                NetBidibServiceInfo si = comboModel.getElementAt(index);
                if (si.getAddresses() != null && si.getAddresses().length > 0 && si.getAddresses()[0].getHostAddress().equals(host)) {

                    if (si.getPort() == port) {
                        // found the service info
                        selectedServiceInfo = si;
                        break;
                    }
                }
            }
            return selectedServiceInfo;
        }

        public void setInitialValues(GlobalSettingsInterface model) {

            String prevSelectedHost = model.getPreviousSelectedNetBidibHost();
            if (StringUtils.isNotBlank(prevSelectedHost)) {
                LOGGER.info("NetBidibPanel, current value of prevSelectedHost: {}", prevSelectedHost);
                try {
                    int index = prevSelectedHost.lastIndexOf(":");
                    String[] splited;
                    if (index > -1) {
                        splited = new String[2];
                        splited[0] = prevSelectedHost.substring(0, index);
                        splited[1] = prevSelectedHost.substring(index + 1);
                    }
                    else {
                        splited = new String[] { prevSelectedHost };
                    }

                    if (splited.length > 1) {
                        // check if we have this entry registered in the combobox already
                        NetBidibServiceInfo selectedServiceInfo =
                            getSelectedServiceInfo(splited[0], Integer.parseInt(splited[1]));

                        if (selectedServiceInfo != null) {
                            LOGGER.info("Found the initial selected service info: {}", selectedServiceInfo);
                            netBidibServiceInfoCombo.setSelectedItem(selectedServiceInfo);
                            serviceInfoButton.setSelected(true);
                        }
                    }

                    textField.setText(splited[0]);
                    if (splited.length > 1) {
                        portTextField.setText(splited[1]);
                    }
                }
                catch (Exception ex) {
                    LOGGER
                        .warn("Set the initial values in the NetBidibPanel failed, prevSelectedHost: {}",
                            prevSelectedHost, ex);
                }
            }
        }

        public void select(PreferencesPortType portType) {
            switch (portType.getConnectionPortType()) {
                case NetBidibClient:
                    textFieldButton.setSelected(false);
                    serviceInfoButton.setSelected(false);
                    String prevSelectedHost = portType.getConnectionName();
                    if (StringUtils.isNotBlank(prevSelectedHost)) {
                        try {
                            int index = prevSelectedHost.lastIndexOf(":");
                            String[] splited;
                            if (index > -1) {
                                splited = new String[2];
                                splited[0] = prevSelectedHost.substring(0, index);
                                splited[1] = prevSelectedHost.substring(index + 1);
                            }
                            else {
                                splited = new String[] { prevSelectedHost };
                            }

                            if (splited.length > 1) {
                                // check if we have this entry registered in the combobox already
                                NetBidibServiceInfo selectedServiceInfo =
                                    getSelectedServiceInfo(splited[0], Integer.parseInt(splited[1]));
                                if (selectedServiceInfo != null) {
                                    LOGGER.info("Found the initial selected service info: {}", selectedServiceInfo);
                                    netBidibServiceInfoCombo.setSelectedItem(selectedServiceInfo);
                                    serviceInfoButton.setSelected(true);
                                }
                                else {
                                    textFieldButton.setSelected(true);
                                }
                            }

                            textField.setText(splited[0]);
                            if (splited.length > 1) {
                                portTextField.setText(splited[1]);
                            }
                        }
                        catch (Exception ex) {
                            LOGGER
                                .warn("Set the selected value in the NetBidibPanel failed, prevSelectedHost: {}",
                                    prevSelectedHost, ex);
                        }
                    }
                    break;
                default:
                    break;
            }
        }
    }

    private class SpeedometerPanel extends JPanel implements PortRadioPanel {
        private static final long serialVersionUID = 1L;

        private JComboBox<CommPort> commPorts;

        private JRadioButton comboButton;

        private List<PortRadioPanel> peer = new ArrayList<>();

        public void addPeer(PortRadioPanel peer) {
            this.peer.add(peer);
        }

        public SpeedometerPanel(ButtonGroup group, JComboBox<CommPort> commPorts) {
            super(new GridBagLayout());
            setOpaque(false);
            super.setEnabled(false);

            this.commPorts = commPorts;

            comboButton = new JRadioButton("");
            comboButton.setContentAreaFilled(false);

            // Group the radio buttons.
            group.add(comboButton);

            this.commPorts.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    CommPort commPort = (CommPort) SpeedometerPanel.this.commPorts.getSelectedItem();
                    LOGGER.debug("The selection of the comm port has changed: {}", commPort);

                    if (commPort != null && !CommPort.NONE.equals(commPort.getName())) {
                        LOGGER.info("The COM port was selected, auto select the COM port radio button.");
                        comboButton.setSelected(true);
                    }

                    if (comboButton.isSelected()) {

                        PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.Speedometer.name());
                        portType.setConnectionName(commPort != null ? commPort.getName() : "");
                        LOGGER.info("Selectced port: {}", portType);
                        fireSelectedPortTypeChanged(portType);
                    }

                    if (previousSelectedSpeedometerPortModel != null) {
                        previousSelectedSpeedometerPortModel.setValue(commPort != null ? commPort.getName() : "");
                    }
                }
            });

            comboButton.addActionListener(SpeedometerPanel.this);

            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridy = 0;
            gbc.insets = new Insets(0, 0, 5, 0);
            add(comboButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(commPorts, gbc);
        }

        private void setSelectedPort(CommPort commPort) {
            commPorts.setSelectedItem(commPort);
            actionPerformed(null);
        }

        @Override
        public void actionPerformed(ActionEvent e) {

            commPorts.setEnabled(comboButton.isEnabled());

            if (e != null && e.getSource() == comboButton) {
                if (CollectionUtils.isNotEmpty(peer)) {
                    LOGGER.trace("Update peers from speedometerPort.");
                    for (PortRadioPanel p : peer) {
                        p.actionPerformed(e);
                    }
                }
            }

            if (e != null) {
                if (e.getSource() == comboButton) {
                    PreferencesPortType portType = PreferencesPortType.getValue(ConnectionPortType.Speedometer.name());
                    // get the selected port
                    portType
                        .setConnectionName(
                            (commPorts.getSelectedItem() != null ? commPorts.getSelectedItem().toString() : null));
                    fireSelectedPortTypeChanged(portType);
                }
            }
        }

        @Override
        public void setEnabled(boolean enabled) {
            boolean old = isEnabled();
            if (old != enabled) {
                super.setEnabled(enabled);
                comboButton.setEnabled(enabled);

                actionPerformed(null);

                if (!enabled) {
                    // check if we we have a selected option on this panel, if yes, remove it by selecting the default
                    // option of the peer
                    if (comboButton.isSelected()) {
                        if (CollectionUtils.isNotEmpty(peer)) {
                            for (PortRadioPanel p : peer) {
                                p.selectDefault();
                            }
                        }
                    }
                }
            }
        }

        @Override
        public void selectDefault() {
            PreferencesPortType portType = PreferencesPortType.getValue(ConnectionPortType.Speedometer.name());
            fireSelectedPortTypeChanged(portType);
        }

        public void setInitialValues(final GlobalSettingsInterface globalSettings) {
        }

        public void select(PreferencesPortType portType) {
            switch (portType.getConnectionPortType()) {
                case Speedometer:
                    comboButton.setSelected(true);
                    break;
                default:
                    break;
            }
        }
    }

    private class SerialOverTcpPortRadioPanel extends JPanel implements PortRadioPanel {
        private static final long serialVersionUID = 1L;

        public static final String DEFAULT_TCP_PORT = "62875";

        public static final String DEFAULT_TCP_HOST = "localhost";

        private JTextField textField;

        private JTextField portTextField;

        private JLabel labelSerialOverTcpMock;

        private JRadioButton textFieldButton;

        private JRadioButton serialOverTcpMockButton;

        private List<PortRadioPanel> peer = new ArrayList<>();

        public void addPeer(PortRadioPanel peer) {
            this.peer.add(peer);
        }

        public SerialOverTcpPortRadioPanel(ButtonGroup group) {
            super(new GridBagLayout());
            setOpaque(false);
            super.setEnabled(false);

            textFieldButton = new JRadioButton("");
            textFieldButton.setContentAreaFilled(false);
            serialOverTcpMockButton = new JRadioButton("");
            serialOverTcpMockButton.setContentAreaFilled(false);

            // Group the radio buttons.
            group.add(textFieldButton);
            group.add(serialOverTcpMockButton);

            textField = new JTextField(20);
            portTextField = new JTextField(5);
            portTextField.setText(DEFAULT_TCP_PORT);
            final DocumentListener documentListener = new DocumentListener() {

                @Override
                public void removeUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void changedUpdate(DocumentEvent e) {
                    setText();
                }

                private void setText() {
                    if (textFieldButton.isSelected()) {
                        PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.SerialOverTcp.name());

                        String host =
                            (StringUtils.isNotBlank(textField.getText()) ? textField.getText() : DEFAULT_TCP_HOST);
                        String port =
                            (StringUtils.isNotBlank(portTextField.getText()) ? portTextField.getText()
                                : DEFAULT_TCP_PORT);
                        String connectionName = String.format("%1$s:%2$s", host, port);
                        portType.setConnectionName(connectionName);
                        fireSelectedPortTypeChanged(portType);

                    }

                    if (previousSelectedSerialOverTcpHostModel != null) {
                        previousSelectedSerialOverTcpHostModel
                            .setValue(String.format("%1$s:%2$s", textField.getText(), portTextField.getText()));
                    }
                }
            };
            textField.getDocument().addDocumentListener(documentListener);
            portTextField.getDocument().addDocumentListener(documentListener);

            labelSerialOverTcpMock = new JLabel(Resources.getString(MiscSettingsPanel.class, "serialOverTcpMock"));
            labelSerialOverTcpMock.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    serialOverTcpMockButton.setSelected(true);

                    // set selected does not fire the action event and therefore we must fire the change
                    PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialOverTcpSimulation.name());
                    fireSelectedPortTypeChanged(portType);
                }
            });

            textFieldButton.addActionListener(SerialOverTcpPortRadioPanel.this);
            serialOverTcpMockButton.addActionListener(SerialOverTcpPortRadioPanel.this);

            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridy = 0;
            gbc.insets = new Insets(0, 0, 5, 0);
            add(textFieldButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(textField, gbc);

            gbc.gridx = 2;
            gbc.weightx = 0.0;
            add(new JLabel(":"), gbc);

            gbc.gridx = 3;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(portTextField, gbc);

            gbc = new GridBagConstraints();
            gbc.gridy = 1;
            add(serialOverTcpMockButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.gridwidth = 2;
            gbc.weightx = 1.0;
            add(labelSerialOverTcpMock, gbc);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            labelSerialOverTcpMock.setEnabled(serialOverTcpMockButton.isEnabled());
            textField.setEnabled(textFieldButton.isEnabled());
            portTextField.setEnabled(textFieldButton.isEnabled());

            if (e != null && (e.getSource() == textFieldButton || e.getSource() == serialOverTcpMockButton)) {
                if (peer != null) {
                    if (CollectionUtils.isNotEmpty(peer)) {
                        LOGGER.trace("Update peers from plainTcpPort.");
                        for (PortRadioPanel p : peer) {
                            p.actionPerformed(e);
                        }
                    }
                }
            }

            if (e != null) {
                if (e.getSource() == serialOverTcpMockButton) {
                    PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialOverTcpSimulation.name());
                    fireSelectedPortTypeChanged(portType);
                }
                else if (e.getSource() == textFieldButton) {
                    PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialOverTcp.name());

                    String connectionName = String.format("%1$s:%2$s", textField.getText(), portTextField.getText());
                    portType.setConnectionName(connectionName);
                    fireSelectedPortTypeChanged(portType);
                }
            }
        }

        @Override
        public void setEnabled(boolean enabled) {
            boolean old = isEnabled();
            if (old != enabled) {
                super.setEnabled(enabled);

                textFieldButton.setEnabled(enabled);
                serialOverTcpMockButton.setEnabled(enabled);
                actionPerformed(null);

                if (!enabled) {
                    // check if we we have a selected option on this panel, if yes, remove it by selecting the default
                    // option of the peer
                    if (serialOverTcpMockButton.isSelected() || textFieldButton.isSelected()) {
                        if (CollectionUtils.isNotEmpty(peer)) {
                            for (PortRadioPanel p : peer) {
                                p.selectDefault();
                            }
                        }
                    }
                }
            }
        }

        @Override
        public void selectDefault() {
            serialOverTcpMockButton.setSelected(true);
            PreferencesPortType portType =
                PreferencesPortType.getValue(ConnectionPortType.SerialOverTcpSimulation.name());
            fireSelectedPortTypeChanged(portType);
        }

        public void setInitialValues(GlobalSettingsInterface model) {

            String prevSelectedHost = model.getPreviousSelectedSerialOverTcpHost();
            if (StringUtils.isNotBlank(prevSelectedHost)) {
                LOGGER.info("SerialOverTcpPortRadioPanel, current value of prevSelectedHost: {}", prevSelectedHost);
                try {
                    String[] splited = prevSelectedHost.split(":");

                    textField.setText(splited[0]);
                    if (splited.length > 1) {
                        portTextField.setText(splited[1]);
                    }
                }
                catch (Exception ex) {
                    LOGGER
                        .warn("Set the initial values in the SerialOverTcpPortRadioPanel failed, prevSelectedHost: {}",
                            prevSelectedHost, ex);
                }
            }
        }

        public void select(PreferencesPortType portType) {
            switch (portType.getConnectionPortType()) {
                case SerialOverTcp:
                    textFieldButton.setSelected(false);

                    String prevSelectedHost = portType.getConnectionName();
                    if (StringUtils.isNotBlank(prevSelectedHost)) {
                        try {
                            String[] splited = prevSelectedHost.split(":");

                            textField.setText(splited[0]);
                            if (splited.length > 1) {
                                portTextField.setText(splited[1]);
                            }
                        }
                        catch (Exception ex) {
                            LOGGER
                                .warn(
                                    "Set the selected values in the SerialOverTcpPortRadioPanel failed, prevSelectedHost: {}",
                                    prevSelectedHost, ex);
                        }
                    }
                    textFieldButton.setSelected(true);
                    break;
                default:
                    serialOverTcpMockButton.setSelected(true);
                    break;
            }
        }
    }

    private class CommPortRadioPanel extends JPanel implements PortRadioPanel {
        private static final long serialVersionUID = 1L;

        private JComboBox<CommPort> commPortsCombo;

        private JTextField textFieldSerialSymLink;

        private JLabel labelSerialMock;

        private JRadioButton comboButton;

        private JRadioButton textFieldButton;

        private JRadioButton serialMockButton;

        private final JTextField simulationFileName;

        private List<PortRadioPanel> peer = new ArrayList<>();

        public void addPeer(PortRadioPanel peer) {
            this.peer.add(peer);
        }

        public CommPortRadioPanel(ButtonGroup group, JComboBox<CommPort> commPortsCombo) {
            super(new GridBagLayout());
            setOpaque(false);
            super.setEnabled(false);

            this.commPortsCombo = commPortsCombo;

            comboButton = new JRadioButton("");
            comboButton.setContentAreaFilled(false);
            textFieldButton = new JRadioButton("");
            textFieldButton.setContentAreaFilled(false);
            serialMockButton = new JRadioButton("");
            serialMockButton.setContentAreaFilled(false);

            // Group the radio buttons.
            group.add(comboButton);
            group.add(textFieldButton);
            group.add(serialMockButton);

            textFieldSerialSymLink = new JTextField(20);
            textFieldSerialSymLink.getDocument().addDocumentListener(new DocumentListener() {

                @Override
                public void removeUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void insertUpdate(DocumentEvent e) {
                    setText();
                }

                @Override
                public void changedUpdate(DocumentEvent e) {
                    setText();
                }

                private void setText() {
                    if (textFieldButton.isSelected()) {
                        PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.SerialSymLink.name());
                        portType.setConnectionName(textFieldSerialSymLink.getText());
                        fireSelectedPortTypeChanged(portType);
                    }

                    if (previousSelectedSerialSymLinkModel != null) {
                        previousSelectedSerialSymLinkModel.setValue(textFieldSerialSymLink.getText());
                    }
                }
            });

            this.commPortsCombo.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    CommPort commPort = (CommPort) CommPortRadioPanel.this.commPortsCombo.getSelectedItem();
                    LOGGER.debug("The selection of the comm port has changed: {}", commPort);

                    if (commPort != null && !CommPort.NONE.equals(commPort.getName())) {
                        LOGGER.info("The COM port was selected, auto select the COM port radio button.");
                        comboButton.setSelected(true);
                    }

                    if (comboButton.isSelected()) {

                        PreferencesPortType portType =
                            PreferencesPortType.getValue(ConnectionPortType.SerialPort.name());
                        portType.setConnectionName(commPort != null ? commPort.getName() : "");
                        portType.setSerialNumber(commPort != null ? commPort.getSerialNumber() : null);
                        LOGGER.info("Selectced port: {}", portType);

                        fireSelectedPortTypeChanged(portType);
                    }

                    if (previousSelectedComPortModel != null) {
                        previousSelectedComPortModel.setValue(commPort != null ? commPort.getName() : "");
                    }
                }
            });

            simulationFileName = new JTextField();
            simulationFileName.getDocument().addDocumentListener(new DefaultDocumentListener() {
                @Override
                protected void valueChanged() {
                    LOGGER.info("The simulation filename has changed: {}", simulationFileName.getText());

                    if (previousSelectedSimulationFileNameModel != null) {
                        previousSelectedSimulationFileNameModel.setValue(simulationFileName.getText());
                    }

                    PreferencesPortType portType = prepareSimulationPortType(simulationFileName);
                    fireSelectedPortTypeChanged(portType);
                }
            });

            labelSerialMock = new JLabel(Resources.getString(MiscSettingsPanel.class, "serialMock"));
            labelSerialMock.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    serialMockButton.setSelected(true);

                    // set selected does not fire the action event and therefore we must fire the change
                    PreferencesPortType portType = prepareSimulationPortType(simulationFileName);

                    fireSelectedPortTypeChanged(portType);
                }
            });

            comboButton.addActionListener(CommPortRadioPanel.this);
            textFieldButton.addActionListener(CommPortRadioPanel.this);
            serialMockButton.addActionListener(CommPortRadioPanel.this);

            GridBagConstraints gbc = new GridBagConstraints();
            gbc.insets = new Insets(0, 0, 5, 0);
            add(comboButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            gbc.gridwidth = 3;
            add(this.commPortsCombo, gbc);

            gbc.gridwidth = 1;
            gbc = new GridBagConstraints();
            gbc.gridy = 1;
            gbc.insets = new Insets(0, 0, 5, 0);
            add(textFieldButton, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 0.0;
            gbc.insets = new Insets(0, 0, 5, 5);
            add(new JLabel("Symlink"), gbc);

            gbc.insets = new Insets(0, 0, 5, 0);
            gbc.gridx = 2;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            gbc.gridwidth = 2;
            add(textFieldSerialSymLink, gbc);

            // serial simulation
            gbc = new GridBagConstraints();
            gbc.gridy = 2;
            add(serialMockButton, gbc);

            gbc = new GridBagConstraints();
            gbc.gridy = 2;
            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 0.0;
            gbc.insets = new Insets(0, 0, 0, 5);
            add(labelSerialMock, gbc);

            gbc.gridx = 2;
            gbc.weightx = 1.0;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.insets = new Insets(0, 0, 0, 5);
            add(simulationFileName, gbc);

            final JButton selectSimulationButton =
                new JButton(Resources.getString(ConnectionPanel.class, "select-simulation"));
            selectSimulationButton
                .setToolTipText(Resources.getString(ConnectionPanel.class, "select-simulation.tooltip"));
            selectSimulationButton.addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    final FileFilter ff = new XmlFileFilter();

                    String storedWorkingDirectory = wizardSettings.getWorkingDirectory(WORKING_DIR_SIMULATION_KEY);

                    final FileDialog dialog =
                        new FileDialog(contentPanel, FileDialog.OPEN, storedWorkingDirectory, null, ff) {
                            @Override
                            public void approve(final String fileName) {
                                File file = new File(fileName);

                                String selectedFile = file.getPath();
                                LOGGER.info("The selected simulation file: {}", selectedFile);
                                simulationFileName.setText(selectedFile);

                                final String workingDir = Paths.get(fileName).getParent().toString();
                                LOGGER.info("Save current workingDir: {}", workingDir);

                                wizardSettings.setWorkingDirectory(WORKING_DIR_SIMULATION_KEY, workingDir);
                            }
                        };
                    dialog.showDialog();
                }
            });

            gbc.gridx = 3;
            gbc.weightx = 0.0;
            gbc.insets = new Insets(0, 0, 0, 0);
            add(selectSimulationButton, gbc);
        }

        private static final String WORKING_DIR_SIMULATION_KEY = "simulation";

        private void setSelectedPort(CommPort commPort) {
            commPortsCombo.setSelectedItem(commPort);
            actionPerformed(null);
        }

        private void setSymlink(String symlink) {
            textFieldSerialSymLink.setText(symlink);
            actionPerformed(null);
        }

        private PreferencesPortType prepareSimulationPortType(final JTextField simulationFileName) {
            PreferencesPortType portType = PreferencesPortType.getValue(ConnectionPortType.SerialSimulation.name());
            String simulationFilename = simulationFileName.getText();
            portType.setConnectionName(simulationFilename);
            LOGGER.info("Use the simulationFilename: {}", simulationFilename);
            return portType;
        }

        @Override
        public void actionPerformed(ActionEvent e) {

            commPortsCombo.setEnabled(comboButton.isEnabled());
            textFieldSerialSymLink.setEnabled(textFieldButton.isEnabled());
            labelSerialMock.setEnabled(serialMockButton.isEnabled());

            if (e != null && (e.getSource() == comboButton || e.getSource() == textFieldButton
                || e.getSource() == serialMockButton)) {
                if (peer != null) {
                    if (CollectionUtils.isNotEmpty(peer)) {
                        LOGGER.trace("Update peers from commPort.");
                        for (PortRadioPanel p : peer) {
                            p.actionPerformed(e);
                        }
                    }
                }
            }

            if (e != null) {
                if (e.getSource() == serialMockButton) {
                    PreferencesPortType portType = prepareSimulationPortType(simulationFileName);

                    fireSelectedPortTypeChanged(portType);
                }
                else if (e.getSource() == comboButton) {
                    PreferencesPortType portType = PreferencesPortType.getValue(ConnectionPortType.SerialPort.name());
                    // get the selected port
                    portType
                        .setConnectionName(
                            (commPortsCombo.getSelectedItem() != null ? commPortsCombo.getSelectedItem().toString() : null));
                    fireSelectedPortTypeChanged(portType);
                }
                else if (e.getSource() == textFieldButton) {
                    PreferencesPortType portType =
                        PreferencesPortType.getValue(ConnectionPortType.SerialSymLink.name());
                    portType.setConnectionName(textFieldSerialSymLink.getText());
                    fireSelectedPortTypeChanged(portType);
                }
            }
        }

        @Override
        public void setEnabled(boolean enabled) {
            boolean old = isEnabled();
            if (old != enabled) {
                super.setEnabled(enabled);

                comboButton.setEnabled(enabled);
                textFieldButton.setEnabled(enabled);
                serialMockButton.setEnabled(enabled);
                actionPerformed(null);

                if (!enabled) {
                    // check if we we have a selected option on this panel, if yes, remove it by selecting the default
                    // option of the peer
                    if (serialMockButton.isSelected() || textFieldButton.isSelected() || comboButton.isSelected()) {
                        if (CollectionUtils.isNotEmpty(peer)) {
                            for (PortRadioPanel p : peer) {
                                p.selectDefault();
                            }
                        }
                    }
                }
            }
        }

        @Override
        public void selectDefault() {
            serialMockButton.setSelected(true);
            PreferencesPortType portType = PreferencesPortType.getValue(ConnectionPortType.SerialSimulation.name());
            fireSelectedPortTypeChanged(portType);
        }

        public void setInitialValues(GlobalSettingsInterface model) {
            textFieldSerialSymLink.setText(model.getPreviousSelectedSerialSymLink());

            simulationFileName.setText(model.getPreviousSelectedSimulationFileName());
        }

        public void select(PreferencesPortType portType) {
            LOGGER.info("Selected the portType: {}", portType);

            switch (portType.getConnectionPortType()) {
                case SerialPort:
                    comboButton.setSelected(true);
                    break;
                case SerialSymLink:
                    textFieldButton.setSelected(true);
                    textFieldSerialSymLink.setText(portType.getConnectionName());
                    break;
                default:
                    serialMockButton.setSelected(true);
                    simulationFileName.setText(portType.getConnectionName());
                    break;
            }
        }
    }

    private static class DefaultDocumentListener implements DocumentListener {

        @Override
        public void changedUpdate(DocumentEvent e) {
            valueChanged();
        }

        @Override
        public void removeUpdate(DocumentEvent e) {
            valueChanged();
        }

        @Override
        public void insertUpdate(DocumentEvent e) {
            valueChanged();
        }

        protected void valueChanged() {

        }
    }

    private final class XmlFileFilter extends FileFilter {

        public static final String SUFFIX_XML = "xml";

        private static final String PATTERN_XML_FILENAME = ".+\\.xml$";

        @Override
        public boolean accept(File file) {
            boolean result = false;

            if (file != null) {
                if (file.isDirectory()) {
                    result = true;
                }
                else if (file.toString() != null) {
                    String extension = FilenameUtils.getExtension(file.toString());
                    if (SUFFIX_XML.equalsIgnoreCase(extension)) {

                        // check if the pattern matches
                        Matcher m = Pattern.compile(PATTERN_XML_FILENAME).matcher(file.getName());
                        if (m.matches()) {
                            LOGGER.trace("The pattern matches!");

                            result = true;
                        }
                    }
                }
            }
            return result;
        }

        @Override
        public String getDescription() {
            return Resources.getString(ConnectionPanel.class, "filter") + " (*." + SUFFIX_XML + ")";
        }
    }

}
