package org.bidib.wizard.mvc.stepcontrol.controller;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections4.CollectionUtils;
import org.bidib.jbidibc.messages.AccessoryState;
import org.bidib.jbidibc.messages.AccessoryStateOptions;
import org.bidib.jbidibc.messages.AccessoryStateOptions.Options;
import org.bidib.jbidibc.messages.BidibLibrary;
import org.bidib.jbidibc.messages.accessory.ByteOptionsValue;
import org.bidib.jbidibc.messages.accessory.OptionsValue;
import org.bidib.jbidibc.messages.enums.AccessoryExecutionState;
import org.bidib.jbidibc.messages.enums.AccessoryStateOptionsKeys;
import org.bidib.jbidibc.messages.utils.ProductUtils;
import org.bidib.jbidibc.messages.utils.ThreadFactoryBuilder;
import org.bidib.wizard.api.model.Accessory;
import org.bidib.wizard.api.model.NodeInterface;
import org.bidib.wizard.api.model.listener.AccessoryListListener;
import org.bidib.wizard.api.model.listener.AccessoryListener;
import org.bidib.wizard.api.model.listener.CvDefinitionListener;
import org.bidib.wizard.api.model.listener.DefaultNodeListListener;
import org.bidib.wizard.api.model.listener.PortListListener;
import org.bidib.wizard.api.service.node.SwitchingNodeService;
import org.bidib.wizard.client.common.controller.FeedbackPortStatusChangeProvider;
import org.bidib.wizard.client.common.view.statusbar.StatusBar;
import org.bidib.wizard.core.model.connection.ConnectionRegistry;
import org.bidib.wizard.core.service.SettingsService;
import org.bidib.wizard.model.ports.MotorPort;
import org.bidib.wizard.model.ports.SoundPort;
import org.bidib.wizard.model.status.SoundPortStatus;
import org.bidib.wizard.mvc.main.model.MainModel;
import org.bidib.wizard.mvc.main.view.component.DefaultBusyFrame;
import org.bidib.wizard.mvc.main.view.panel.listener.TabStatusListener;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlAspect;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlModel;
import org.bidib.wizard.mvc.stepcontrol.model.StepControlModel.OperationModeEnum;
import org.bidib.wizard.mvc.stepcontrol.model.TurnTableType;
import org.bidib.wizard.mvc.stepcontrol.view.InvalidAspectException;
import org.bidib.wizard.mvc.stepcontrol.view.StepControlPanel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.jgoodies.common.collect.ArrayListModel;

public class StepControlController implements CvDefinitionListener, StepControlControllerInterface, PortListListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(StepControlController.class);

    private StepControlPanel stepControlPanel;

    private final StepControlModel stepControlModel;

    private final MainModel mainModel;

    private final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider;

    private final AccessoryListener accessoryListener;

    private final AccessoryListListener accessoryListListener;

    private NodeInterface controlledNode;

    private Accessory controlledAccessory;

    private Accessory operationalAccessory;

    private Accessory soundAccessory;

    private final ScheduledExecutorService accessoryStateWorkers =
        Executors
            .newScheduledThreadPool(1,
                new ThreadFactoryBuilder().setNameFormat("accessoryStateWorkers-thread-%d").build());

    @Autowired
    private SwitchingNodeService switchingService;

    @Autowired
    private SettingsService settingsService;

    private final StatusBar statusBar;

    public StepControlController(final MainModel mainModel,
        final FeedbackPortStatusChangeProvider feedbackPortStatusChangeProvider, final StatusBar statusBar) {
        this.mainModel = mainModel;
        this.feedbackPortStatusChangeProvider = feedbackPortStatusChangeProvider;
        this.statusBar = statusBar;

        stepControlModel = new StepControlModel();
        stepControlModel.setMaxConfiguredAspects(MAX_CONFIGURED_ASPECTS);

        accessoryListener = new AccessoryListener() {

            @Override
            public void macrosChanged() {

            }

            @Override
            public void labelChanged(String label) {

            }

            @Override
            public void accessoryStateChanged(Integer accessoryId, Integer aspect) {

                LOGGER.info("The accessory state has changed, accessory: {}, aspect: {}", accessoryId, aspect);

                if (accessoryId == null) {
                    LOGGER.info("No accessoryId delivered, skip processing.");
                    return;
                }

                // emergency stop must be signaled specially
                if (accessoryId == ACCESSORY_ID_CONTROLLING) {
                    boolean emergencyStop = false;
                    if (aspect != null) {
                        if (aspect.intValue() == BidibLibrary.BIDIB_ACCESSORY_ASPECT_ESTOP) {
                            LOGGER.warn("Emergency stop was signalled!");

                            emergencyStop = true;
                            if (stepControlModel != null) {
                                stepControlModel.setOperationalMode(OperationModeEnum.emergencyStop);
                            }

                            // shutdown all tasks
                            cancelScheduleAccessoryStateCheck();
                        }
                        else if (aspect.intValue() == BidibLibrary.BIDIB_ACCESSORY_ASPECT_UNKNOWN) {
                            LOGGER.warn("Illegal aspect was signalled!");
                        }
                    }

                    // other aspects must trigger the animation

                    if (controlledAccessory != null) {
                        AccessoryState state = controlledAccessory.getAccessoryState();
                        AccessoryStateOptions options = controlledAccessory.getAccessoryStateOptions();
                        AccessoryExecutionState executionState = controlledAccessory.getAccessoryExecutionState();

                        LOGGER
                            .info("Current accessory state: {},  options: {}, executionState: {}", state, options,
                                executionState);

                        // notify the execution state to the panel
                        stepControlPanel.executionStateChanged(executionState, accessoryId, aspect, state);

                        if (options != null) {
                            Options stateOptions = options.getOptions();

                            if (stateOptions != null) {
                                try {

                                    Integer currentAngle = null;

                                    OptionsValue<?> optionsValue =
                                        stateOptions
                                            .getOptionsValue(AccessoryStateOptionsKeys.STATE_OPTION_CURRENT_ANGLE);
                                    if (optionsValue != null) {
                                        currentAngle = ByteOptionsValue.getIntValue(optionsValue);

                                        double currentDegrees = 1.5d * currentAngle;
                                        LOGGER.info("Current angle: {}, degrees: {}", currentAngle, currentDegrees);

                                        if (stepControlModel != null) {
                                            stepControlModel.setTurntableCurrentDegrees(currentDegrees);
                                        }

                                        stepControlPanel.setTurntableDegrees(currentDegrees);
                                    }

                                    optionsValue =
                                        stateOptions
                                            .getOptionsValue(AccessoryStateOptionsKeys.STATE_OPTION_TARGET_ANGLE);
                                    if (optionsValue != null) {
                                        int targetAngle = ByteOptionsValue.getIntValue(optionsValue);

                                        double targetDegrees = 1.5d * targetAngle;
                                        LOGGER
                                            .info("Target angle: {}, degrees: {}, currentAngle: {}, emergencyStop: {}",
                                                targetAngle, targetDegrees, currentAngle, emergencyStop);

                                        if (stepControlModel != null) {
                                            stepControlModel.setTurntableTargetDegrees(targetDegrees);
                                        }

                                        if (currentAngle != null && !emergencyStop
                                            && executionState == AccessoryExecutionState.RUNNING) {
                                            // check if the current angle and the target angle are different
                                            if (targetAngle != currentAngle) {
                                                LOGGER.info("targetAngle != currentAngle, start the scheduler.");
                                                scheduleAccessoryStateCheck();
                                            }
                                            else if (executionState == AccessoryExecutionState.RUNNING) {
                                                LOGGER.info("executionState is running, start the scheduler.");
                                                scheduleAccessoryStateCheck();

                                            }
                                        }
                                        else {
                                            LOGGER
                                                .info(
                                                    "The execution state is no longer running! Current executionState: {}",
                                                    executionState);

                                            cancelScheduleAccessoryStateCheck();
                                        }
                                    }
                                }
                                catch (Exception ex) {
                                    LOGGER.warn("Get the current turntable angle failed.", ex);
                                }
                            }
                        }
                    }

                }
                else if (accessoryId == ACCESSORY_ID_OPERATING) {
                    // operational accessory
                    LOGGER.info("Received accessory id 1: operational accessory, aspect: {}", aspect);
                    if (aspect != null) {
                        switch (aspect) {
                            case ASPECT_ID_OPERATING:
                                if (stepControlModel != null) {

                                    OperationModeEnum oldOpMode = stepControlModel.getOperationalMode();
                                    LOGGER.info("Set the step control operational mode, oldOpMode: {}", oldOpMode);
                                    stepControlModel.setOperationalMode(OperationModeEnum.operational);

                                    if (controlledNode != null && operationalAccessory != null
                                        && controlledAccessory != null
                                        && !(oldOpMode == OperationModeEnum.operational)) {
                                        LOGGER.info("Query the controlled accessory state.");
                                        // read the current state of the controlled accessory
                                        switchingService
                                            .queryAccessoryState(ConnectionRegistry.CONNECTION_ID_MAIN,
                                                controlledNode.getSwitchingNode(), Arrays.asList(controlledAccessory));
                                    }
                                    else {
                                        LOGGER.info("Do not query the controlled accessory state.");
                                    }
                                }
                                break;
                            case ASPECT_ID_HOMING:
                                if (stepControlModel != null) {
                                    LOGGER.info("Set the step control to homing mode.");
                                    stepControlModel.setOperationalMode(OperationModeEnum.homingInProgress);
                                }
                                break;
                            default:
                                if (stepControlModel != null) {
                                    LOGGER.info("Set the step control to non-operational mode.");
                                    stepControlModel.setOperationalMode(OperationModeEnum.emergencyStop);
                                }
                                break;
                        }
                    }
                    else {
                        if (stepControlModel != null) {
                            LOGGER.info("Set the step control to non-operational mode.");
                            stepControlModel.setOperationalMode(OperationModeEnum.emergencyStop);
                        }
                    }
                }
                else if (accessoryId == ACCESSORY_ID_SOUND) {
                    // sound accessory
                    LOGGER.info("Received sound accessory change, current aspect: {}", aspect);
                    if (aspect != null) {
                        switch (aspect.intValue()) {
                            case 0:
                                stepControlModel.setSoundActive(false);
                                break;
                            default:
                                stepControlModel.setSoundActive(true);
                                break;
                        }
                    }
                    else {
                        stepControlModel.setSoundActive(true);
                    }
                }
                else {
                    LOGGER.info("Received accessory id > 0");
                }
            }
        };

        accessoryListListener = new AccessoryListListener() {

            @Override
            public void pendingChangesChanged() {

            }

            @Override
            public void listChanged() {
                LOGGER.info("The accessory list has changed.");

                if (controlledAccessory != null) {
                    // free the listener
                    controlledAccessory.removeAccessoryListener(accessoryListener);
                    controlledAccessory = null;
                }

                if (operationalAccessory != null) {
                    // free the listener
                    operationalAccessory.removeAccessoryListener(accessoryListener);
                    operationalAccessory = null;
                }

                if (soundAccessory != null) {
                    // free the listener
                    soundAccessory.removeAccessoryListener(accessoryListener);
                    soundAccessory = null;
                }

                // reset the operational mode
                stepControlModel.setOperationalMode(OperationModeEnum.unknown);
                stepControlModel.setSelectedAspect(null);
                stepControlModel.setSoundActive(true);

                // check if the node is a StepControl and register listener for accessory 0
                if (controlledNode != null) {
                    LOGGER.info("The currently controlled node is a StepControl.");

                    // ensure console is visible
                    CvConsoleController.ensureConsoleVisible();

                    if (CollectionUtils.isNotEmpty(controlledNode.getAccessories())) {
                        // the first accessory is the worker accessory
                        Accessory accessory = controlledNode.getAccessories().get(ACCESSORY_ID_CONTROLLING);

                        LOGGER.info("Add listener to controlled accessory: {}", accessory);
                        final List<Accessory> accessoryToQueryState = new LinkedList<>();

                        controlledAccessory = accessory;
                        controlledAccessory.addAccessoryListener(accessoryListener);

                        operationalAccessory = mainModel.getAccessories().get(ACCESSORY_ID_OPERATING);
                        operationalAccessory.addAccessoryListener(accessoryListener);
                        accessoryToQueryState.add(operationalAccessory);

                        soundAccessory = mainModel.getAccessories().get(ACCESSORY_ID_SOUND);
                        if (soundAccessory != null) {
                            soundAccessory.addAccessoryListener(accessoryListener);
                            accessoryToQueryState.add(soundAccessory);
                        }

                        final Integer operationModeAccessoryNumber = accessory.getOperationModeAccessoryNumber();
                        if (operationModeAccessoryNumber != null) {
                            LOGGER
                                .info(
                                    "The accessory has an operationModeAccessoryNumber: {}. Read only the operational accessory.",
                                    operationModeAccessoryNumber);
                            // read the current state of accessory ACCESSORY_ID_OPERATING
                            switchingService
                                .queryAccessoryState(ConnectionRegistry.CONNECTION_ID_MAIN,
                                    controlledNode.getSwitchingNode(), accessoryToQueryState);
                        }
                        else {
                            LOGGER.info("Query the accessory state of the operational and execution accessory.");
                            accessoryToQueryState.add(controlledAccessory);

                            // read the current state of accessory 0 and 1
                            switchingService
                                .queryAccessoryState(ConnectionRegistry.CONNECTION_ID_MAIN,
                                    controlledNode.getSwitchingNode(), accessoryToQueryState);
                        }
                    }
                }
            }

            @Override
            public void accessoryChanged(Integer accessoryId) {

            }
        };
    }

    public void start(final TabStatusListener tabStatusListener) {

        LOGGER.info("Use changeProvider: {}", feedbackPortStatusChangeProvider);

        // TODO add listener for accessories

        stepControlPanel =
            new StepControlPanel(mainModel, stepControlModel, settingsService, tabStatusListener,
                feedbackPortStatusChangeProvider, this, this.statusBar);
        stepControlPanel.createComponent();

        // add the node list listener

        final PortListListener motorPortListListener = new PortListListener() {

            @Override
            public void listChanged() {

                if (controlledNode != null) {
                    List<MotorPort> motorPorts = new LinkedList<>();
                    motorPorts.addAll(controlledNode.getMotorPorts());

                    if (CollectionUtils.isNotEmpty(motorPorts)) {
                        // get the first motor port
                        MotorPort motorPort = motorPorts.get(0);
                        LOGGER.info("Fetched first motor port: {}", motorPort);
                        stepControlModel.setMotorPort(motorPort);
                    }
                    else {
                        stepControlModel.setMotorPort(null);
                    }
                }
                else {
                    stepControlModel.setMotorPort(null);
                }
            }

            @Override
            public Class<?> getPortClass() {
                return MotorPort.class;
            }
        };

        final PortListListener soundPortListListener = new PortListListener() {

            @Override
            public void listChanged() {
                LOGGER.info("The list of sound port has changed.");
                List<SoundPort> soundPorts = new LinkedList<>();
                if (controlledNode != null) {
                    // soundPorts.addAll(controlledNode.getSoundPorts());
                    soundPorts.addAll(controlledNode.getSoundPorts());
                    LOGGER.info("Fetched sound ports: {}", soundPorts);
                }
                stepControlModel.setSoundPorts(soundPorts);
            }

            @Override
            public Class<?> getPortClass() {
                return SoundPort.class;
            }
        };

        // TODO add the input port listener
        // final PortListener<InputPortStatus> inputPortListListener = new PortListener<InputPortStatus>() {
        //
        // @Override
        // public void labelChanged(Port<InputPortStatus> port, String label) {
        // }
        //
        // @Override
        // public void statusChanged(final NodeInterface node, Port<InputPortStatus> port) {
        // LOGGER.info("The port status has changed, node: {}, port: {}", node, port);
        // if (port.getId() == 2) {
        // // this is the homing input
        // }
        // }
        //
        // @Override
        // public void configChanged(Port<InputPortStatus> port) {
        // }
        //
        // @Override
        // public Class<?> getPortClass() {
        // return InputPort.class;
        // }
        // };

        mainModel.addNodeListListener(new DefaultNodeListListener() {

            @Override
            public void nodeChanged(final NodeInterface node) {

                final NodeInterface selectedNode = mainModel.getSelectedNode();

                // check if the node has really changed
                if (selectedNode != null && selectedNode.equals(controlledNode)) {
                    LOGGER.warn("The selected node is the controlled node. Skip update of node data.");
                    return;
                }

                LOGGER.info("The selected node has changed, current controlledAccessory: {}", controlledAccessory);

                if (controlledAccessory != null) {
                    // free the listener
                    controlledAccessory.removeAccessoryListener(accessoryListener);

                    controlledAccessory = null;
                }

                if (controlledNode != null) {
                    controlledNode.removeAccessoryListListener(accessoryListListener);
                    controlledNode.removeCvDefinitionListener(StepControlController.this);

                    LOGGER.info("Remove the StepControlController as portListListener for MotorPorts.");
                    controlledNode.removePortListListener(MotorPort.class, motorPortListListener);
                    stepControlModel.setMotorPort(null);

                    // LOGGER.info("Remove the StepControlController as portListListener for InputPorts.");
                    // controlledNode.removePortListListener(InputPort.class, StepControlController.this);

                    LOGGER.info("Remove the StepControlController as portListListener for SoundPorts.");
                    controlledNode.removePortListListener(SoundPort.class, soundPortListListener);

                    stepControlModel.setSoundPorts(null);

                    // release the controlled node
                    controlledNode = null;
                }

                StepControlController.this.mainModel.removeCvDefinitionListener(StepControlController.this);

                // check if the node is a StepControl and register listener for accessory 0
                if (selectedNode != null && ProductUtils.isStepControl(selectedNode.getUniqueId())) {
                    LOGGER.info("The currently selected node is a StepControl.");
                    controlledNode = selectedNode;

                    controlledNode.addCvDefinitionListener(StepControlController.this);

                    StepControlController.this.mainModel.addCvDefinitionListener(StepControlController.this);

                    // trigger load of CV values
                    cvDefinitionChanged();

                    controlledNode.addAccessoryListListener(accessoryListListener);
                    // trigger the listener
                    accessoryListListener.listChanged();

                    if (CollectionUtils.isNotEmpty(controlledNode.getAccessories())) {
                        // the first accessory is the worker accessory
                        Accessory accessory = controlledNode.getAccessories().get(0);

                        controlledAccessory = accessory;
                        accessory.addAccessoryListener(accessoryListener);
                    }
                    else {
                        LOGGER.info("No accessories on controlled node found.");
                    }

                    LOGGER.info("Add the StepControlController as portListListener for MotorPorts.");
                    controlledNode.addPortListListener(MotorPort.class, motorPortListListener);

                    if (CollectionUtils.isNotEmpty(controlledNode.getMotorPorts())) {
                        stepControlModel.setMotorPort(controlledNode.getMotorPorts().get(0));
                    }

                    // LOGGER.info("Add the StepControlController as portListListener for InputPorts.");
                    // controlledNode.addPortListListener(InputPort.class, StepControlController.this);

                    LOGGER.info("Add the StepControlController as portListListener for SoundPorts.");
                    controlledNode.addPortListListener(SoundPort.class, soundPortListListener);

                    stepControlModel.setSoundPorts(controlledNode.getSoundPorts());
                }

            }
        });

    }

    public StepControlPanel getComponent() {
        return stepControlPanel;
    }

    @Override
    public void triggerLoadCvValues() {
        LOGGER.info("Load the CV for the StepControl.");

        if (stepControlPanel != null) {
            LOGGER.info("Let the stepControlPanel initiate loading the CV values: {}", stepControlPanel);

            stepControlPanel.triggerLoadCvValues();
        }
    }

    @Override
    public void cvDefinitionChanged() {
        LOGGER.info("The CV definition has changed.");

        if (stepControlPanel != null) {
            LOGGER.info("Notify the stepControl panel: {}", stepControlPanel);

            stepControlPanel.cvDefinitionChanged();
        }
    }

    @Override
    public void cvDefinitionValuesChanged(final boolean read, final List<String> changedNames) {
        LOGGER.info("The CV definition values have changed, read: {}", read);

        if (stepControlPanel != null) {
            LOGGER.info("Notify the stepControl panel of the changed CV defintion values: {}", stepControlPanel);

            stepControlPanel.cvDefinitionValuesChanged(read, changedNames);
        }
    }

    @Override
    public ArrayListModel<StepControlAspect> getConfigureAspectsListModel() {

        // return the prepared aspects from the stepControl model
        final List<StepControlAspect> aspects = new LinkedList<>();

        // get the aspects from the model
        aspects.addAll(stepControlModel.getStepControlAspects());

        if (TurnTableType.round == stepControlModel.getTurnTableType()) {
            for (StepControlAspect currentAspect : stepControlModel.getStepControlAspects()) {
                LOGGER.info("Prepare aspect opposite to save from aspect: {}", currentAspect);

                if (currentAspect.getOppositeAspect() != null) {
                    StepControlAspect aspectOpposite = (StepControlAspect) currentAspect.getOppositeAspect();
                    aspectOpposite.setPosition(currentAspect.getOppositePosition());
                    aspectOpposite.setPolarity(currentAspect.getOppositePolarity());

                    if (aspectOpposite.isValid()) {

                        LOGGER.info("Prepared the aspect opposite to save: {}", aspectOpposite);
                        aspects.add(aspectOpposite);
                    }
                    else {
                        LOGGER.warn("The opposite aspect is not valid: {}", aspectOpposite);

                        throw new InvalidAspectException();
                    }
                }
                else {
                    LOGGER.warn("The opposite aspect is not available for aspect: {}", currentAspect);

                    StepControlAspect aspectOpposite =
                        new StepControlAspect(null, currentAspect.getOppositePosition(),
                            currentAspect.getOppositePolarity());

                    if (aspectOpposite.isValid()) {
                        // keep the opposite aspect
                        currentAspect.setOppositeAspect(aspectOpposite);

                        LOGGER.info("Prepared the aspect opposite to save: {}", aspectOpposite);
                        aspects.add(aspectOpposite);
                    }
                    else {
                        LOGGER.warn("The opposite aspect is not valid: {}", aspectOpposite);

                        throw new InvalidAspectException();
                    }

                }
            }
        }

        Collections.sort(aspects, new Comparator<StepControlAspect>() {

            @Override
            public int compare(StepControlAspect o1, StepControlAspect o2) {
                return Long.compare(o1.getPosition(), o2.getPosition());
            }
        });

        ArrayListModel<StepControlAspect> arrayListModel = new ArrayListModel<>();
        arrayListModel.addAll(aspects);
        return arrayListModel;
        // return stepControlModel.getStepControlAspectsListModel();
    }

    public NodeInterface getSelectedNode() {
        return mainModel.getSelectedNode();
    }

    private ScheduledFuture<?> taskFuture;

    public void scheduleAccessoryStateCheck() {

        // check if the push interval value is active
        Integer pushInterval = stepControlModel.getPushInterval();
        if (pushInterval != null && pushInterval > 0) {

            LOGGER.info("The updates of the accessory state are delivered via push notification.");
            return;
        }

        // add a task to the worker to check the accessory state
        taskFuture = accessoryStateWorkers.schedule(() -> {
            try {
                LOGGER.info("Check the state of the controlledAccessory: {}", controlledAccessory);

                // release the task future
                taskFuture = null;

                if (controlledNode != null) {

                    // read the current state of accessory 0
                    switchingService
                        .queryAccessoryState(ConnectionRegistry.CONNECTION_ID_MAIN, controlledNode.getSwitchingNode(),
                            Arrays.asList(controlledAccessory));
                }
            }
            catch (Exception ex) {
                LOGGER.warn("Check the state of the controlledAccessory failed for bidibNode: {}", controlledNode, ex);
            }
        }, 250, TimeUnit.MILLISECONDS);
    }

    private void cancelScheduleAccessoryStateCheck() {
        LOGGER.info("Cancel the accessory state check.");
        // shutdown all tasks
        try {
            if (taskFuture != null) {
                taskFuture.cancel(true);

                taskFuture = null;
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Stop scheduled task failed.", ex);
        }

    }

    @Override
    public void setMotorPortValue(final MotorPort motorPort) {
        LOGGER.info("Set the motor port value: {}", motorPort);

        switchingService
            .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, controlledNode.getSwitchingNode(), motorPort);
    }

    @Override
    public Class<?> getPortClass() {
        return MotorPort.class;
    }

    @Override
    public void triggerSoundPort(int portId, final SoundPortStatus soundPortStatus) {
        LOGGER.info("Activate the sound port value: {}, soundPortStatus: {}", portId, soundPortStatus);
        SoundPort soundPort = new SoundPort();
        soundPort.setId(portId);
        soundPort.setStatus(soundPortStatus);
        switchingService
            .setPortStatus(ConnectionRegistry.CONNECTION_ID_MAIN, controlledNode.getSwitchingNode(), soundPort);
    }

    @Override
    public void listChanged() {
        // TODO Auto-generated method stub

    }

    @Override
    public void setSoundActive(boolean soundActive) {
        LOGGER.info("Activate the sound accessory, soundActive: {}", soundActive);

        if (controlledNode != null) {
            try {
                DefaultBusyFrame.setWaitCursor(stepControlPanel.getComponent());
                Accessory accessory = new Accessory();
                accessory.setId(ACCESSORY_ID_SOUND);

                switchingService
                    .setAccessoryAspect(ConnectionRegistry.CONNECTION_ID_MAIN, controlledNode.getSwitchingNode(),
                        accessory, soundActive ? 2 : 0);
            }
            finally {
                DefaultBusyFrame.setDefaultCursor(stepControlPanel.getComponent());
            }
        }
        else {
            LOGGER.warn("Activate the sound accessory is skipped. The controlled node is not avaialable.");
        }
    }

    @Override
    public void activateAspect(final Accessory accessory, int aspectNumber) {
        LOGGER.info("Start accessory with aspect: {}, accessory: {}", aspectNumber, accessory);

        try {
            DefaultBusyFrame.setWaitCursor(stepControlPanel.getComponent());

            switchingService
                .setAccessoryAspect(ConnectionRegistry.CONNECTION_ID_MAIN, getSelectedNode().getSwitchingNode(),
                    accessory, aspectNumber);
        }
        finally {
            DefaultBusyFrame.setDefaultCursor(stepControlPanel.getComponent());
        }
    }

    @Override
    public void storeAccessory(Accessory accessory) {
        LOGGER.info("Store the accessory: {}", accessory);

        try {
            DefaultBusyFrame.setWaitCursor(stepControlPanel.getComponent());

            switchingService
                .saveAccessory(ConnectionRegistry.CONNECTION_ID_MAIN, getSelectedNode().getSwitchingNode(), accessory);
        }
        finally {
            DefaultBusyFrame.setDefaultCursor(stepControlPanel.getComponent());
        }
    }
}
