package org.bidib.wizard.mvc.worklist.view;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListModel;

import org.bidib.wizard.api.event.WorkListItemEvent;
import org.bidib.wizard.api.locale.Resources;
import org.bidib.wizard.client.common.table.AbstractEmptyTable;
import org.bidib.wizard.client.common.view.DockKeys;
import org.bidib.wizard.mvc.worklist.controller.listener.WorkListControllerListener;
import org.bidib.wizard.mvc.worklist.model.ActionTableCellRenderer;
import org.bidib.wizard.mvc.worklist.model.WorkItemListModel;
import org.bidib.wizard.mvc.worklist.model.WorkItemModel;
import org.bidib.wizard.mvc.worklist.model.WorkListItemTableModel;
import org.bushe.swing.event.annotation.AnnotationProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jgoodies.binding.list.SelectionInList;
import com.jgoodies.forms.builder.FormBuilder;
import com.jgoodies.forms.debug.FormDebugPanel;
import com.jidesoft.grid.RolloverTableUtils;
import com.jidesoft.grid.SortableTableModel;
import com.vlsolutions.swing.docking.DockKey;
import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState.Location;
import com.vlsolutions.swing.docking.DockingDesktop;

public class WorkListView implements Dockable {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkListView.class);

    private static final String ENCODED_DIALOG_COLUMN_SPECS = "pref, 3dlu, fill:50dlu:grow";

    private static final String ENCODED_DIALOG_ROW_SPECS = "fill:50dlu:grow";

    private JPanel contentPanel;

    private final DockingDesktop desktop;

    private final WorkItemListModel workItemListModel;

    private final WorkListControllerListener controller;

    private SelectionInList<WorkItemModel> workItemSelection;

    private ButtonsCellEditorRenderer buttonEditor;

    public WorkListView(final DockingDesktop desktop, final WorkListControllerListener controller,
        final WorkItemListModel workItemListModel) {
        this.desktop = desktop;
        this.controller = controller;
        this.workItemListModel = workItemListModel;

        DockKeys.DOCKKEY_WORK_LIST_VIEW.setName(Resources.getString(getClass(), "title"));
        // turn off autohide and close features
        // DockKeys.DOCKKEY_WORK_LIST_VIEW.setCloseEnabled(false);
        // DockKeys.DOCKKEY_WORK_LIST_VIEW.setAutoHideEnabled(false);
        DockKeys.DOCKKEY_WORK_LIST_VIEW.setLocation(Location.HIDDEN);

    }

    private JPanel createPanel() {
        LOGGER.info("Create the content panel.");

        FormBuilder dialogBuilder = null;
        boolean debugDialog = false;
        if (debugDialog) {
            JPanel panel = new FormDebugPanel();
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }
        else {
            JPanel panel = new JPanel(new BorderLayout());
            dialogBuilder =
                FormBuilder.create().columns(ENCODED_DIALOG_COLUMN_SPECS).rows(ENCODED_DIALOG_ROW_SPECS).panel(panel);
        }

        workItemSelection =
            new SelectionInList<WorkItemModel>((ListModel<WorkItemModel>) workItemListModel.getWorkItemListModel());

        final SortableTableModel tableModel = new SortableTableModel(new WorkListItemTableModel(workItemSelection));

        final AbstractEmptyTable workItemsTable =
            new AbstractEmptyTable(tableModel, Resources.getString(getClass(), "empty_table")) {
                private static final long serialVersionUID = 1L;

            };

        workItemsTable.adjustRowHeight();

        this.buttonEditor = new ButtonsCellEditorRenderer(d -> {
            LOGGER.info("Pressed button on row: {}, action: {}", d.getRow(), d.getAction());

            final int selectedRow = d.getRow();

            int sortedRow = tableModel.getSortedRowAt(selectedRow);
            WorkListItemTableModel workListItemTableModel = (WorkListItemTableModel) tableModel.getActualModel();
            final WorkItemModel selected = workListItemTableModel.getRow(sortedRow);

            switch (d.getAction()) {
                case DELETE:
                    LOGGER.info("Remove selected: {}", selected);

                    // ask user before remove the item
                    int result =
                        JOptionPane
                            .showConfirmDialog(JOptionPane.getFrameForComponent(desktop.getRootPane()),
                                Resources.getString(WorkListView.class, "delete-item.text"),
                                Resources.getString(WorkListView.class, "delete-item.title"),
                                JOptionPane.OK_CANCEL_OPTION);
                    if (result == JOptionPane.OK_OPTION) {
                        try {
                            workItemListModel.removeWorkListItem(selected);

                            if (workItemListModel.getWorkItems().isEmpty()) {
                                // if the list is empty we must clear the rollover editor
                                RolloverTableUtils.cancelRollover(workItemsTable, null);
                                workItemsTable.updateUI();
                            }
                        }
                        catch (Exception ex) {
                            LOGGER.warn("Remove item or cancel rollover failed.", ex);
                        }
                    }
                    break;
                case EDIT:
                    LOGGER.info("Perform action on row: {}", d.getRow());
                    try {
                        controller.applyAction(selected);

                        // we must clear the rollover editor
                        RolloverTableUtils.cancelRollover(workItemsTable, null);
                        workItemsTable.updateUI();

                    }
                    catch (Exception ex) {
                        LOGGER.warn("Perform action of item failed.", ex);
                    }
                default:
                    break;
            }
        });

        workItemsTable.getColumnModel().getColumn(0).setPreferredWidth(60);

        workItemsTable.getColumnModel().getColumn(1).setCellRenderer(new ActionTableCellRenderer());

        workItemsTable.getColumnModel().getColumn(2).setCellEditor(buttonEditor);
        workItemsTable.getColumnModel().getColumn(2).setCellRenderer(buttonEditor);
        int columnWidth = 35;
        workItemsTable.getColumnModel().getColumn(2).setPreferredWidth(columnWidth);
        workItemsTable.getColumnModel().getColumn(2).setMaxWidth(100);
        workItemsTable.getColumnModel().getColumn(2).setMinWidth(columnWidth);

        RolloverTableUtils.install(workItemsTable);

        final JScrollPane positionsTableScrollPane = new JScrollPane(workItemsTable);

        dialogBuilder.add(positionsTableScrollPane).xyw(1, 1, 3);

        contentPanel = dialogBuilder.build();

        // add the eventbus processing
        AnnotationProcessor.process(this);

        return contentPanel;
    }

    @Override
    public Component getComponent() {
        if (contentPanel == null) {
            contentPanel = createPanel();
        }
        return contentPanel;
    }

    @Override
    public DockKey getDockKey() {
        return DockKeys.DOCKKEY_WORK_LIST_VIEW;
    }

    public void addWorkListItem(final WorkListItemEvent item) {
        LOGGER.info("Received worklist item event: {}", item);

        this.workItemListModel.addWorkListItem(item);
    }
}
